/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 500 $
 * $Date: 2013-08-16 19:04:13 +0800 (Fri, 16 Aug 2013) $
 */
#ifndef INDII_IMAGE_IMAGEMANIPULATION_HPP
#define INDII_IMAGE_IMAGEMANIPULATION_HPP

#include "wx/image.h"

#include "boost/numeric/ublas/matrix.hpp"
#include "boost/numeric/ublas/matrix_sparse.hpp"

namespace indii {
/**
 * Image channel element type.
 */
typedef unsigned char channel_element;

/**
 * Dense image channel. Values on \f$[0,1]\f$.
 */
typedef boost::numeric::ublas::matrix<channel_element> channel;
 
/**
 * Sparse image channel. Values on \f$[0,1]\f$.
 */
typedef boost::numeric::ublas::mapped_matrix<channel_element> sparse_channel;

/**
 * Image mask element type.
 */
typedef bool mask_element;

/**
 * Dense image mask.
 */
typedef boost::numeric::ublas::matrix<mask_element> mask;
 
/**
 * Sparse image channel.
 */
typedef boost::numeric::ublas::mapped_matrix<mask_element> sparse_mask;

/**
 * Threshold value.
 *
 * @param x Value to threshold.
 * @param a Threshold point.
 * @param edge Edge softness.
 *
 * @return Thresholded value.
 */
float threshold(const float x, const float a, const float edge);

/**
 * Calculate saturation.
 *
 * @param r Red value.
 * @param g Green value.
 * @param b Blue value.
 *
 * @return Saturation.
 */
unsigned char saturation(const unsigned char r, const unsigned char g,
    const unsigned char b);

/**
 * Scale channel or mask.
 *
 * @tparam T Channel or mask type.
 *
 * @param src Source channel or mask.
 * @param dst Destination channel or mask.
 *
 * @p src is sampled to build @p dst.
 */
template <class T>
void scale(const T& src, T& dst);

/**
 * Hide image.
 *
 * @param image Image.
 *
 * The alpha channel across the entire image is set to transparent.
 */
void hide(wxImage& image);

/**
 * Hide image, with mask.
 *
 * @tparam M Mask type.
 *
 * @param image Image.
 * @param m Mask.
 *
 * The alpha channel of the area indicated by @p m is made transparent.
 */
template <class M>
void hide(wxImage& image, const M& m);

/**
 * Show image.
 *
 * @param image Image.
 *
 * The alpha channel across the entire image is set to opaque.
 */
void show(wxImage& image);

/**
 * Show image, with mask.
 *
 * @tparam M Mask type.
 *
 * @param image Image.
 * @param m Mask.
 *
 * The alpha channel of the area indicated by @p m is made opaque.
 */
template <class M>
void show(wxImage& image, const M& m);

/**
 * Show image, with mask.
 *
 * @tparam M Mask type.
 *
 * @param image Image.
 * @param rect Rectangle to show.
 *
 * The alpha channel of the area indicated by @p rect is made opaque.
 */
void show(wxImage& image, const wxRect& rect);

/**
 * Mask channel.
 *
 * @tparam C1 channel type.
 * @tparam C2 channel type.
 * @tparam M mask type.
 *
 * @param c The channel.
 * @param m The mask.
 * @param result The result.
 */
template <class C1, class C2, class M>
void maskChannel(const C1& c, const M& m, C2& result);
 
/**
 * Update alpha of image, with channel.
 *
 * @tparam C Channel type.
 *
 * @param image Image to update.
 * @param c Alpha channel.
 *
 * The alpha channel is updated according to the alpha values provided by
 * @p c.
 */
template <class C>
void applyAlpha(wxImage& image, const C& c);

/**
 * Update alpha channel of image, with channel and mask.
 *
 * @tparam M Mask type.
 * @tparam C Channel type.
 *
 * @param image Image to update.
 * @param m Mask defining update region.
 * @param c Alpha channel.
 *
 * The alpha channel of the area indicated by @p m is updated according to
 * the alpha values provided by @p c.
 */
template <class M, class C>
void applyAlpha(wxImage& image, const M& m, const C& c);

/**
 * Return a list of file types and extensions for use with wxFileDialog.
 *
 * @param save Is this for a save dialog?
 */
wxString wildcards(const bool save = false);

}

#ifdef __WXMSW__
/* defined as macros in windows.h, which clashes with std::min and std::max,
 * so undef */
#undef min
#undef max
#endif

inline unsigned char indii::saturation(const unsigned char r,
    const unsigned char g, const unsigned char b) {
  unsigned int mn, mx;
  unsigned char s;

  mn = static_cast<unsigned int>(std::min(std::min(r,g),b));
  mx = static_cast<unsigned int>(std::max(std::max(r,g),b));
  if (mx > 0) {
    s = static_cast<unsigned char>(255 - mn*255/mx);
  } else {
    s = 0;
  }

  return s;
}   

inline float indii::threshold(const float x, const float a,
    const float edge) {
  /* pre-condition */
  assert (x >= 0.0f && x <= 1.0f);
  assert (a >= 0.0f && a <= 1.0f);
  assert (edge >= 0.0 && edge <= 1.0);

  if (edge == 0.0f) {
    /* limit case */
    return (x >= a) ? 1.0f : 0.0f;
  } else {
    //return 1.0f / (1.0f + expf(-logf(255.0f)*(x - a)));
    return 1.0f / (1.0f + std::pow(255.0f, (a - x) / edge));
  }
}

template <class T>
void indii::scale(const T& src, T& dst) {
  int srcX, srcY, dstX, dstY;
  int srcWidth, srcHeight, dstWidth, dstHeight;
  double widthFactor, heightFactor;
  
  srcWidth = src.size2();
  srcHeight = src.size1();
  
  dstWidth = dst.size2();
  dstHeight = dst.size1();

  widthFactor = (double)srcWidth / dstWidth;
  heightFactor = (double)srcHeight / dstHeight;
  
  dst.clear();
  
  for (dstY = 0; dstY < dstHeight; dstY++) {
    for (dstX = 0; dstX < dstWidth; dstX++) {
      srcX = static_cast<int>(widthFactor*dstX);
      srcY = static_cast<int>(heightFactor*dstY);
      if (src(srcY,srcX) > 0.0) {
        dst(dstY,dstX) = src(srcY,srcX);
      }
    }
  }
}

template <class M>
void indii::hide(wxImage& image, const M& m) {
  /* pre-conditions */
  assert ((unsigned int)image.GetWidth() == m.size2());
  assert ((unsigned int)image.GetHeight() == m.size1());

  int x, y;
  typename M::const_iterator1 iter1;
  typename M::const_iterator2 iter2;

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }

  for (iter1 = m.begin1(); iter1 != m.end1(); iter1++) {
    for (iter2 = iter1.begin(); iter2 != iter1.end(); iter2++) {
      x = iter2.index2();
      y = iter2.index1();
      if (m(y,x)) {
        image.SetAlpha(x, y, (unsigned char)0);
      }
    }
  }
}

template <class M>
void indii::show(wxImage& image, const M& m) {
  /* pre-conditions */
  assert ((unsigned int)image.GetWidth() == m.size2());
  assert ((unsigned int)image.GetHeight() == m.size1());

  int x, y;
  typename M::const_iterator1 iter1;
  typename M::const_iterator2 iter2;

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }

  for (iter1 = m.begin1(); iter1 != m.end1(); iter1++) {
    for (iter2 = iter1.begin(); iter2 != iter1.end(); iter2++) {
      x = static_cast<int>(iter2.index2());
      y = static_cast<int>(iter2.index1());
      if (m(y,x)) {
        image.SetAlpha(x, y, (unsigned char)255);
      }
    }
  }
}

template <class C1, class C2, class M>
void indii::maskChannel(const C1& c, const M& m, C2& result) {
  /* pre-conditions */
  assert (result.size1() == c.size1());
  assert (result.size2() == c.size2());
  assert (result.size1() == m.size1());
  assert (result.size2() == m.size2());

  unsigned int x, y;
  typename M::const_iterator1 iter1;
  typename M::const_iterator2 iter2;

  for (iter1 = m.begin1(); iter1 != m.end1(); iter1++) {
    for (iter2 = iter1.begin(); iter2 != iter1.end(); iter2++) {
      x = iter2.index2();
      y = iter2.index1();
      if (m(y,x)) {
        result(y,x) = c(y,x);
      }
    }
  }
}

template <class C>
void indii::applyAlpha(wxImage& image, const C& c) {
  /* pre-conditions */
  assert ((unsigned int)image.GetWidth() == c.size2());
  assert ((unsigned int)image.GetHeight() == c.size1());

  int x, y;
  const int width = image.GetWidth();
  const int height = image.GetHeight();

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }
  
  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {
      image.SetAlpha(x, y, c(y,x));
    }
  }
}

template <class M, class C>
void indii::applyAlpha(wxImage& image, const M& m, const C& c) {
  /* pre-conditions */
  assert ((unsigned int)image.GetWidth() == m.size2());
  assert ((unsigned int)image.GetHeight() == m.size1());
  assert ((unsigned int)image.GetWidth() == c.size2());
  assert ((unsigned int)image.GetHeight() == c.size1());

  int x, y;
  typename M::const_iterator1 iter1;
  typename M::const_iterator2 iter2;

  if (!image.HasAlpha()) {
    image.InitAlpha();
  }

  for (iter1 = m.begin1(); iter1 != m.end1(); iter1++) {
    for (iter2 = iter1.begin(); iter2 != iter1.end(); iter2++) {
      x = iter2.index2();
      y = iter2.index1();
      if (m(y,x)) {
        image.SetAlpha(x, y, c(y,x));
      }
    }
  }
}

#endif
