/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 500 $
 * $Date: 2013-08-16 19:04:13 +0800 (Fri, 16 Aug 2013) $
 */
#ifndef INDII_IMAGE_FILEIMAGERESOURCE_HPP
#define INDII_IMAGE_FILEIMAGERESOURCE_HPP

#include "ImageResource.hpp"
#include "ImageCache.hpp"

namespace indii {
/**
 * File based image resource manager.
 */
class FileImageResource : public ImageResource {
public:
  /**
   * Constructor.
   *
   * @param path Path name of the image.
   */
  FileImageResource(const wxString& path);

  /**
   * Destructor.
   */
  virtual ~FileImageResource();

  virtual unsigned int getWidth();
  
  virtual unsigned int getHeight();
  
  virtual wxImage* get(const int width = 0, const int height = 0,
      const bool high = false);

  virtual wxImage calc(const wxRect& rect, const unsigned int width = 0,
      const unsigned int height = 0, const bool high = false);

private:
  /**
   * Original image.
   */
  wxImage* original;
  
  /**
   * Resized image cache.
   */
  ImageCache* lowCache;
  
  /**
   * Resized image cache, high quality.
   */
  ImageCache* highCache;
};
}

inline unsigned int indii::FileImageResource::getWidth() {
  return original->GetWidth();
}

inline unsigned int indii::FileImageResource::getHeight() {
  return original->GetHeight();
}

inline wxImage* indii::FileImageResource::get(const int width,
    const int height, const bool high) {
  if (high) {
    return highCache->get(width, height);
  } else {
    return lowCache->get(width, height);
  }
}

inline wxImage indii::FileImageResource::calc(const wxRect& rect,
    const unsigned int width, const unsigned int height, const bool high) {
  return get(width, height, high)->GetSubImage(rect);
}

#endif
