/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 500 $
 * $Date: 2013-08-16 19:04:13 +0800 (Fri, 16 Aug 2013) $
 */
#ifndef INDII_IMAGE_COLOURSPACE_HPP
#define INDII_IMAGE_COLOURSPACE_HPP

#include "boost/numeric/ublas/vector.hpp"

namespace indii {
/**
 * Colour space.
 */
class ColourSpace {
public:
  /*
   * Types of colour tuples.
   */
  typedef boost::numeric::ublas::vector<unsigned char,
      boost::numeric::ublas::bounded_array<unsigned char,3> > rgb_t;
  typedef boost::numeric::ublas::vector<float,
      boost::numeric::ublas::bounded_array<float,3> > hsl_t;

  /**
   * Constructor.
   */
  ColourSpace();

  /**
   * Get red channel lightness proportion.
   *
   * @return Red channel proportion.
   */
  float getRedMix();

  /**
   * Get green channel lightness proportion.
   *
   * @return Green channel proportion.
   */
  float getGreenMix();

  /**
   * Get blue channel lightness proportion.
   *
   * @return Blue channel proportion.
   */
  float getBlueMix();

  /**
   * Set channel proportions for lightness.
   * 
   * @param r Red channel proportion.
   * @param g Green channel proportion.
   * @param b Blue channel proportion.
   */
  void setLightness(const float r, const float g, const float b);

  /**
   * Convert RGB to HSL coordinates using greyscale parameters.
   *
   * @param in Input RGB values.
   * @param[out] out Output HSL values.
   */
  void rgb2hsl(const rgb_t& rgb, hsl_t& hsl);

  /**
   * Convert RGB to L using greyscale parameters.
   *
   * @param r Red channel value.
   * @param g Green channel value.
   * @param b Blue channel value.
   *
   * @return l Lightness.
   */
  float rgb2l(const unsigned char r, const unsigned char g,
      const unsigned char b);
  
  /**
   * Convert RGB to HSL coordinates using greyscale parameters.
   *
   * @param in Input HSL values.
   * @param[out] out Output RGB values.
   */
  void hsl2rgb(const hsl_t& hsl, rgb_t& rgb);

  /**
   * Round and cast float to unsigned char.
   */
  static unsigned char uround(const float val);

  /**
   * Bound value above and below.
   *
   * @param lower Lower bound.
   * @param value Value.
   * @param upper Upper bound.
   */
  static float bound(const float lower, const float value, const float upper);

  /**
   * Bound value above .
   *
   * @param value Value.
   * @param upper Upper bound.
   */
  static float upper_bound(const float value, const float upper);

  /**
   * Bound value below.
   *
   * @param lower Lower bound.
   * @param value Value.
   */
  static float lower_bound(const float lower, const float value);
  
private:
  /**
   * Lightness proportions.
   */
  float greyR, greyG, greyB;

};
}

#define BASE_R 6.0f
#define BASE_G 2.0f
#define BASE_B 4.0f

inline float indii::ColourSpace::getRedMix() {
  return greyR;
}

inline float indii::ColourSpace::getGreenMix() {
  return greyG;
}

inline float indii::ColourSpace::getBlueMix() {
  return greyB;
}

inline float indii::ColourSpace::rgb2l(const unsigned char r, const unsigned char g,
    const unsigned char b) {
  return greyR*r + greyG*g + greyB*b;
}


inline void indii::ColourSpace::rgb2hsl(const rgb_t& rgb, hsl_t& hsl) {
  float& h = hsl(0);
  float& s = hsl(1);
  float& l = hsl(2);

  const unsigned char& r = rgb(0);
  const unsigned char& g = rgb(1);
  const unsigned char& b = rgb(2);
  float k1, k2, k3;

  /* lightness */
  l = greyR*r + greyG*g + greyB*b;

  /* hue & saturation */
  if (r == g && g == b) {
    // special case for greys
    h = 0.0f;
    s = 0.0f;
  } else if (r >= g) {
    if (g >= b) {
      // r > g > b
      k1 = static_cast<float>(g - b);
      k2 = static_cast<float>(r - b);
      k3 = static_cast<float>(r + b);
      h = k1/k2;
      s = k2/k3;
    } else if (r >= b) {
      // r > b > g
      k1 = static_cast<float>(b - g);
      k2 = static_cast<float>(r - g);
      k3 = static_cast<float>(r + g);
      h = BASE_R - k1/k2;
      s = k2/k3;
    } else {
      // b > r > g
      k1 = static_cast<float>(r - g);
      k2 = static_cast<float>(b - g);
      k3 = static_cast<float>(b + g);
      h = BASE_B + k1/k2;
      s = k2/k3;
    }
  } else {
    if (r >= b) {
      // g > r > b
      k1 = static_cast<float>(r - b);
      k2 = static_cast<float>(g - b);
      k3 = static_cast<float>(g + b);
      h = BASE_G - k1/k2;
      s = k2/k3;
    } else if (g >= b) {
      // g > b > r
      k1 = static_cast<float>(b - r);
      k2 = static_cast<float>(g - r);
      k3 = static_cast<float>(g + r);
      h = BASE_G + k1/k2;
      s = k2/k3;
    } else {
      // b > g > r
      k1 = static_cast<float>(g - r);
      k2 = static_cast<float>(b - r);
      k3 = static_cast<float>(b + r);
      h = BASE_B - k1/k2;
      s = k2/k3;
    }
  }

  /* relaxed IEEE compliance, so make sure result bounded as expected */
  //h = std::max(0.0f, std::min(h, BASE_R));
  //s = std::max(0.0f, std::min(s, 1.0f));
  l = /*std::max(0.0f, */std::min(l, 255.0f)/*)*/;
  
  /* post-condition */
  assert (h >= 0.0f && h < BASE_R);
  assert (s >= 0.0f && s <= 1.0f);
  assert (l >= 0.0f && l <= 255.0f);
}

inline void indii::ColourSpace::hsl2rgb(const hsl_t& hsl, rgb_t& rgb) {
  float r, g, b;
  float k1, k2, k3;

  const float& h = hsl(0);
  const float& s = hsl(1);
  const float& l = hsl(2);

  assert (h >= 0.0f && h < BASE_R);
  assert (s >= 0.0f && s <= 1.0f);
  assert (l >= 0.0f && l <= 255.0f);

  if (s == 0.0f) {
    // special case for greys
    r = l;
    g = l;
    b = l;
  } else {
    k2 = (s - 1.0f)/(s + 1.0f);
    switch ((unsigned char)h) {
      case 0:
        // r > g > b
        k1 = h;
        k3 = k1 + k2*(k1 - 1.0f);
        r = l/(greyR + greyG*k3 - greyB*k2);
        g = k3*r;
        b = -k2*r;
        break;
      case 1:
        // g > r > b
        k1 = h - BASE_G;
        k3 = -k1 - k2*(k1 + 1.0f);
        g = l/(greyG + greyR*k3 - greyB*k2);
        r = k3*g;
        b = -k2*g;
        break;
      case 2:
        // g > b > r
        k1 = h - BASE_G;
        k3 = k1 + k2*(k1 - 1.0f);  
        g = l/(greyG + greyB*k3 - greyR*k2);
        b = k3*g;
        r = -k2*g;
        break;
      case 3:
        // b > g > r
        k1 = h - BASE_B;
        k3 = -k1 - k2*(k1 + 1.0f);  
        b = l/(greyB + greyG*k3 - greyR*k2);
        g = k3*b;
        r = -k2*b;
        break;
      case 4:
        // b > r > g
        k1 = h - BASE_B;
        k3 = k1 + k2*(k1 - 1.0f);  
        b = l/(greyB + greyR*k3 - greyG*k2);
        r = k3*b;
        g = -k2*b;
        break;
      case 5:
        // r > b > g
        k1 = h - BASE_R;
        k3 = -k1 - k2*(k1 + 1.0f);  
        r = l/(greyR + greyB*k3 - greyG*k2);
        b = k3*r;
        g = -k2*r;
        break;
      default:
        assert(false);
    }
  }

  rgb(0) = uround(r);
  rgb(1) = uround(g);
  rgb(2) = uround(b);
}

inline unsigned char indii::ColourSpace::uround(const float val) {
  if (val >= 255.0f) {
    return 255;
  } else if (val <= 0.0) {
    return 0;
  } else {
    return static_cast<unsigned char>(val+0.5);
  }
}

inline float indii::ColourSpace::bound(const float lower,
    const float value, const float upper) {
  if (!(value >= lower)) { // handles nan
    return lower;
  } else if (value > upper) {
    return upper;
  } else {
 	 return value;
  }
}

inline float indii::ColourSpace::upper_bound(const float value,
    const float upper) {
  return (value > upper) ? upper : value;
}

inline float indii::ColourSpace::lower_bound(const float lower,
    const float value) {
  return (value < lower) ? lower : value;
}

#endif

