/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 500 $
 * $Date: 2013-08-16 19:04:13 +0800 (Fri, 16 Aug 2013) $
 */
#include "PreviewImage.hpp"

#include "Help.hpp"
#include "../images/indii.hpp"

#include "wx/dcbuffer.h"
#include "wx/settings.h"
#include "wx/dcgraph.h"

using namespace indii;

#define MAX_ZOOM_IN 16
#define MAX_ZOOM_OUT 24

PreviewImage::PreviewImage(wxWindow* parent, ImageResource* res, Model* model,
    Controller* control) :
  wxScrolledWindow(parent, wxID_ANY, wxDefaultPosition, wxDefaultSize,
      wxNO_FULL_REPAINT_ON_RESIZE | wxCLIP_CHILDREN | wxHSCROLL | wxVSCROLL
          | wxBORDER_NONE), res(res), model(model), control(control),
      zoomNumerator(1), zoomDenominator(1), overlay(true), doClear(true) {
  watch(model);

  SetCursor(wxCursor(wxCURSOR_HAND));
  SetBackgroundStyle(wxBG_STYLE_CUSTOM);
}

void PreviewImage::zoomIn() {
  if (zoomDenominator == 1) {
    if (zoomNumerator < MAX_ZOOM_IN) {
      ++zoomNumerator;
    }
  } else {
    --zoomDenominator;
  }
  clear(false);

  int width = res->getWidth() * zoomNumerator / zoomDenominator;
  int height = res->getHeight() * zoomNumerator / zoomDenominator;
  SetVirtualSize(width, height);
  SetScrollRate(1, 1);
  Refresh();

  /* post-condition */
  assert(zoomNumerator >= 1 && zoomDenominator >= 1);
}

void PreviewImage::zoomOut() {
  if (zoomNumerator == 1) {
    if (zoomDenominator < MAX_ZOOM_OUT) {
      ++zoomDenominator;
    }
  } else {
    --zoomNumerator;
  }
  clear();

  int width = res->getWidth() * zoomNumerator / zoomDenominator;
  int height = res->getHeight() * zoomNumerator / zoomDenominator;
  SetVirtualSize(width, height);
  SetScrollRate(1, 1);
  Refresh();

  /* post-condition */
  assert(zoomNumerator >= 1 && zoomDenominator >= 1);
}

void PreviewImage::zoomNormal() {
  zoomNumerator = 1;
  zoomDenominator = 1;
  clear();

  int width = res->getWidth();
  int height = res->getHeight();
  SetVirtualSize(width, height);
  SetScrollRate(1, 1);
  Refresh();

  /* post-condition */
  assert(zoomNumerator == 1 && zoomDenominator == 1);
}

void PreviewImage::zoomFit() {
  const int clientWidth = GetClientSize().GetWidth();
  const int clientHeight = GetClientSize().GetHeight();
  int width, height;

  if (clientWidth > 0 && clientHeight > 0) {
    res->fitInside(clientWidth, clientHeight, &width, &height);
    zoomNumerator = std::min(width/res->getWidth(), height/res->getHeight());
    zoomDenominator = std::max(res->getWidth()/width, res->getHeight()/height);
    if (zoomNumerator == 0) {
      zoomNumerator = 1;
    } else if (zoomNumerator > MAX_ZOOM_IN) {
      zoomNumerator = MAX_ZOOM_IN;
    }
    if (zoomDenominator == 0) {
      zoomDenominator = 1;
    } else if (zoomDenominator > MAX_ZOOM_OUT) {
      zoomDenominator = MAX_ZOOM_OUT;
    }
    clear();

    width = res->getWidth()*zoomNumerator/zoomDenominator;
    height = res->getHeight()*zoomNumerator/zoomDenominator;
    SetVirtualSize(width, height);
    SetScrollRate(1, 1);
    Refresh();
  }

  /* post-condition */
  assert (zoomNumerator == 1 || zoomDenominator == 1);
}

void PreviewImage::refresh() {
  Refresh();
  Update();
}

void PreviewImage::clear(const bool on) {
  doClear = on;
}

void PreviewImage::notify() {
  clear(false);
  refresh();
}

void PreviewImage::OnPaint(wxPaintEvent& evt) {
  wxPaintDC base(this);

  #if wxUSE_GRAPHICS_CONTEXT
  wxGCDC dc(base);
  #else
  wxPaintDC& dc = base;
  #endif
  DoPrepareDC(dc);

  /* background */
  #ifdef __WXMSW__
  if (doClear) {
    dc.SetBackground(wxBrush(*_img_indii_stipple));
    dc.Clear();
  }
  doClear = true;
  #else
  base.SetBackground(wxBrush(*_img_indii_stipple));
  base.Clear();
  #endif

  /* scale */
  const int width = res->getWidth()/zoomDenominator;
  const int height = res->getHeight()/zoomDenominator;
  dc.SetUserScale(zoomNumerator, zoomNumerator);

  /* centre */
  const int clientWidth = GetClientSize().GetWidth();
  const int clientHeight = GetClientSize().GetHeight();
  int centreX = 0;
  int centreY = 0;
  wxPoint origin = dc.GetDeviceOrigin();
  if (width * zoomNumerator < clientWidth) {
    centreX = (clientWidth - width * zoomNumerator) / 2;
    origin.x = centreX;
  }
  if (height * zoomNumerator < clientHeight) {
    centreY = (clientHeight - height * zoomNumerator) / 2;
    origin.y = centreY;
  }
  dc.SetDeviceOrigin(origin.x, origin.y);

  /* shadow */
  //wxBitmap bmp1(width + 8, height + 8, 32);
  //wxMemoryDC dc1(bmp1);
  //dc1.SetBackground(wxBrush(*wxBLACK));
  //dc1.Clear();
  //wxImage img1(bmp1.ConvertToImage());
  //hide(img1);
  //show(img1, wxRect(1, 1, width, height));
  //dc.DrawBitmap(img1.Blur(1), 0, 0, true);

  /* scrolling */
  wxRect rect;
  int xUnits, yUnits, xScale, yScale, xPixels, yPixels;

  GetViewStart(&xUnits, &yUnits);
  GetScrollPixelsPerUnit(&xScale, &yScale);
  xPixels = xUnits * xScale;
  yPixels = yUnits * yScale;

  /* update */
  wxRegionIterator upd(GetUpdateRegion());
  while (upd) {
    /* expand rect by one pixel all sides, helps when scrolling */
    rect.x = std::max(std::min(((int) upd.GetX() + xPixels) / zoomNumerator
        - centreX - 1, width - 1), 0);
    rect.y = std::max(std::min(((int) upd.GetY() + yPixels) / zoomNumerator
        - centreY - 1, height - 1), 0);
    rect.width = std::max(std::min(((int) upd.GetW() + zoomNumerator - 1)
        / zoomNumerator + 2, width - rect.x), 0);
    rect.height = std::max(std::min(((int) upd.GetH() + zoomNumerator - 1)
        / zoomNumerator + 2, height - rect.y), 0);

    if (rect.width > 0 && rect.height > 0) {
      wxImage fg(rect.width, rect.height);
      model->calcFg(rect, width, height, fg);
      #ifdef __WXMSW__
      if (overlay && zoomNumerator == 1) {
        /* double buffer */
        wxBitmap fg2(fg);
        wxMemoryDC dc2(fg2);
        #if wxUSE_GRAPHICS_CONTEXT
        wxGCDC dc3(dc2);
        #else
        wxPaintDC& dc3 = dc2;
        #endif
        dc3.SetDeviceOrigin(-rect.x, -rect.y);
        //dc3.SetUserScale(zoomNumerator, zoomNumerator);

        model->calcAn(rect, width, height, dc3);
        dc.DrawBitmap(fg2, rect.x, rect.y);
      } else {
        /* when zoomed in, double buffering means res of control points is
         * low, prefer the slight flicker in this case */
        dc.DrawBitmap(fg, rect.x, rect.y);
        if (overlay) {
          model->calcAn(rect, width, height, dc);
        }
      }
      #else
      dc.DrawBitmap(fg, rect.x, rect.y);
      if (overlay) {
        model->calcAn(rect, width, height, dc);
      }
      #endif
    }

    upd++;
  }
}

void PreviewImage::OnSize(wxSizeEvent& evt) {
  Refresh();
}

void PreviewImage::OnMouseWheel(wxMouseEvent& evt) {
  /* pan view */
  int x, y, xUnit, yUnit;

  GetViewStart(&x, &y);
  GetScrollPixelsPerUnit(&xUnit, &yUnit);

  #ifndef WX_28
  if (evt.GetWheelAxis() == 0) {
  #endif
    y -= evt.GetWheelRotation();
  #ifndef WX_28
  } else {
    x -= evt.GetWheelRotation();
  }
  #endif
  Scroll(x, y);
  clear(false);
}

void PreviewImage::OnMouseDown(wxMouseEvent& evt) {
  mouseX = evt.GetX();
  mouseY = evt.GetY();

  int x0, y0, x, y;
  getLocation(evt, &x0, &y0, &x, &y);

  if (control != NULL) {
    control->down(x0, y0, x, y, evt);
  }
  #ifdef __WXMSW__
  CaptureMouse();
  #endif
  //clear(false);
}

void PreviewImage::OnMouseUp(wxMouseEvent& evt) {
  int x0, y0, x, y;
  getLocation(evt, &x0, &y0, &x, &y);

  if (control != NULL) {
    control->up(x0, y0, x, y, evt);
  }
  #ifdef __WXMSW__
  ReleaseMouse();
  #endif
  //clear(false);
}

void PreviewImage::OnMouseDoubleClick(wxMouseEvent& evt) {
  mouseX = evt.GetX();
  mouseY = evt.GetY();

  int x0, y0, x, y;
  getLocation(evt, &x0, &y0, &x, &y);

  if (control != NULL) {
    control->doubleDown(x0, y0, x, y, evt);
  }
  #ifdef __WXMSW__
  CaptureMouse();
  #endif
  //clear(false);
}

void PreviewImage::OnMouseMove(wxMouseEvent& evt) {
  if (evt.Dragging()) {
    int x0, y0, x, y, dx0, dy0, dx, dy;
    getLocation(evt, &x0, &y0, &x, &y);

    dx = evt.GetX() - mouseX;
    dy = evt.GetY() - mouseY;
    dx0 = dx * zoomDenominator / zoomNumerator;
    dy0 = dy * zoomDenominator / zoomNumerator;

    if (control != NULL) {
      control->drag(x0, y0, x, y, dx, dy, dx0, dy0, evt);
    }
    if (control == NULL || evt.GetSkipped()) {
      /* default scroll action */
      GetViewStart(&x, &y);

      Scroll(x - dx, y - dy);
      evt.Skip(false);
    }

    mouseX = evt.GetX();
    mouseY = evt.GetY();

    clear(false);
    Update();
  }
}

void PreviewImage::OnEraseBackground(wxEraseEvent& evt) {
  //
}

void PreviewImage::OnHelp(wxHelpEvent& evt) {
  Help::show(510);
}

void PreviewImage::getLocation(wxMouseEvent& evt, int* x0, int* y0, int* x,
    int* y) {
  /* pre-condition */
  assert (x0 != NULL && y0 != NULL && x != NULL && y != NULL);

  const int width = res->getWidth() * zoomNumerator / zoomDenominator;
  const int height = res->getHeight() * zoomNumerator / zoomDenominator;
  const int clientWidth = GetClientSize().GetWidth();
  const int clientHeight = GetClientSize().GetHeight();

  int centreX = 0, centreY = 0, xUnit, yUnit;

  GetViewStart(x, y);
  GetScrollPixelsPerUnit(&xUnit, &yUnit);
  if (width < clientWidth) {
    centreX = (clientWidth - width) / 2;
  }
  if (height < clientHeight) {
    centreY = (clientHeight - height) / 2;
  }

  *x *= xUnit;
  *y *= yUnit;
  *x += evt.GetX() - centreX;
  *y += evt.GetY() - centreY;
  *x0 = *x * zoomDenominator / zoomNumerator;
  *y0 = *y * zoomDenominator / zoomNumerator;
}

BEGIN_EVENT_TABLE(PreviewImage, wxWindow)
EVT_PAINT(PreviewImage::OnPaint) EVT_LEFT_DOWN(PreviewImage::OnMouseDown)
EVT_LEFT_UP(PreviewImage::OnMouseUp)
EVT_LEFT_DCLICK(PreviewImage::OnMouseDoubleClick)
EVT_MOUSEWHEEL(PreviewImage::OnMouseWheel)
EVT_MOTION(PreviewImage::OnMouseMove)
EVT_ERASE_BACKGROUND(PreviewImage::OnEraseBackground)
EVT_SIZE(PreviewImage::OnSize)
EVT_HELP(wxID_ANY, PreviewImage::OnHelp)
END_EVENT_TABLE()
