/*
 * Copyright (C) 2004 Jimmy Do <crispyleaves@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "timer-applet.h"
#include "start-timer-dialog.h"
#include "manage-presets-window.h"
#include "add-profile-dialog.h"
#include "prefs-dialog.h"
#include "about-dialog.h"
/* tree view column constants */
#include "profile-manager.h"
#include <gconf/gconf-client.h>
#include <panel-applet-gconf.h>
#include <libgnome/libgnome.h>
/* for GNOME_STOCK_TIMER */
#include <libgnomeui/libgnomeui.h>
#include <libgnome/gnome-help.h>
#include <libxml/parser.h>
#include <time.h>

#define DEFAULT_NOTIFICATION_SOUND_PATH		SOUNDS_DIR "/clicked.wav"

#define TIMER_APPLET_DATA_PATH			"/.gnome2/timer-applet"
#define TIMER_APPLET_DATA_FILE_VERSION		"1.0"
#define TIMER_APPLET_MENU_FILE_NAME		"GNOME_TimerApplet.xml"
#define PRESETS_SUBMENU_PATH			"/popups/popup/Presets"
#define TA_MENU_COMMAND_PAUSE_TIMER		"/commands/PauseTimer"
#define TA_MENU_COMMAND_RESUME_TIMER		"/commands/ResumeTimer"
#define TA_MENU_COMMAND_RESTART_TIMER		"/commands/RestartTimer"
#define TA_MENU_COMMAND_STOP_TIMER		"/commands/StopTimer"
#define TA_DONT_SHOW_ZEROS			TRUE

#define TIMER_TIMEOUT_INTERVAL			500


/**
 * Callback prototypes
 */
static void
on_pause_timer_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname);
static void
on_resume_timer_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname);
static void
on_restart_timer_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname);
static void
on_stop_timer_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname);
static void
on_presets_submenu_item_activate (BonoboUIComponent *uic, TimerApplet *timer_applet, const gchar *verb);
static void
on_manage_presets_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname);
static void
on_preferences_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname);
static void
on_help_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname);
static void
on_about_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname);
static void
on_main_button_clicked (GtkWidget *button, TimerApplet *timer_applet);
static gboolean
on_button_press_hack (GtkWidget *widget, GdkEventButton *event, TimerApplet *timer_applet);
static void
on_preferences_change (GConfClient *client, guint cnxn_id, GConfEntry *entry, TimerApplet *timer_applet);
static gboolean
on_timer_timeout (TimerApplet *timer_applet);
static void
on_presets_changed (GObject *obj, TimerApplet *timer_applet);
static void
on_applet_change_background (PanelApplet *applet, PanelAppletBackgroundType type,
				GdkColor *color, GdkPixmap *pixmap, gpointer user_data);
static void
on_applet_change_orient (PanelApplet *applet, PanelAppletOrient orient, TimerApplet *timer_applet);
static void
on_applet_change_size (PanelApplet *applet, gint size, TimerApplet *timer_applet);
static void
on_applet_instance_destroy (GtkWidget *applet, TimerApplet *timer_applet);


static const BonoboUIVerb timer_applet_verbs [] = {
	BONOBO_UI_UNSAFE_VERB ("PauseTimer", on_pause_timer_activate),
	BONOBO_UI_UNSAFE_VERB ("ResumeTimer", on_resume_timer_activate),
	BONOBO_UI_UNSAFE_VERB ("RestartTimer", on_restart_timer_activate),
	BONOBO_UI_UNSAFE_VERB ("StopTimer", on_stop_timer_activate),
	BONOBO_UI_UNSAFE_VERB ("ManagePresets", on_manage_presets_activate),
	BONOBO_UI_UNSAFE_VERB ("Preferences", on_preferences_activate),
	BONOBO_UI_UNSAFE_VERB ("Help", on_help_activate),
	BONOBO_UI_UNSAFE_VERB ("About", on_about_activate),
	BONOBO_UI_VERB_END
};

static GtkListStore *timer_presets_list = NULL;
static guint presets_changed_sig_id = 0;
static GtkWidget *manage_presets_window = NULL;
static GtkWidget *about_dialog;

/**
 * Returns the remaining time as a gdouble. Returns a negative number
 * if the time has been exceeded.
 */
static gdouble
calc_remaining_time (TimerApplet *timer_applet)
{
	g_assert (timer_applet);
	
	if (timer_applet->timer) {
		/* NOTE: timer may be either running or paused */
		return timer_applet->cur_profile_duration - g_timer_elapsed (timer_applet->timer, NULL);
	}
	else {
		/* Means that the timer either has not been started or has
		 * completely finished a countdown. Either way, we return
		 * zero because the timer hasn't been set or the timer has finished.
		 */
		g_assert (!timer_applet->timer_running);
		return 0;
	}
}

/**
 * Set the timer button's tooltip to the given string
 */
static void
set_tooltip (TimerApplet *timer_applet, gchar *tooltip_str)
{
	gtk_tooltips_set_tip (GTK_TOOLTIPS (timer_applet->main_tooltips), timer_applet->main_button, tooltip_str, tooltip_str);
}

/**
 * Set the timer button's tooltip to indicate that the timer is not running
 */
static void
set_tooltip_to_stop_str (TimerApplet *timer_applet)
{
	set_tooltip (timer_applet, _("Click to start timer"));
}

/**
 * Given a number of seconds, this function returns a string representing
 * the remaining amount of time. The caller should free the returned string
 * when it's finished being used.
 * show_all indicates whether this function should return hours:minutes:seconds.
 * shorten_zero indicates whether a "shortened" string (currenty, an empty string)
 * should be returned when remaining_seconds is equal to zero.
 * If show_all == false, it returns either hours:minutes or minutes:seconds,
 * depending on how much time is remaining, so that the user isn't given
 * information he doesn't need.
 */
static gchar *
construct_time_str (gdouble remaining_seconds, gboolean show_all, gboolean shorten_zero)
{
	guint hours, minutes, seconds;
	gchar *time_str;

	g_assert (remaining_seconds >= 0);
	
	if (!shorten_zero || remaining_seconds > 0)
	{
		hours = (guint)(remaining_seconds / 3600);
		minutes = (guint)((remaining_seconds - (hours * 3600)) / 60);
		seconds = (guint)(remaining_seconds - (hours *3600) - (minutes * 60));

		g_assert (hours >= 0);
		g_assert (minutes >= 0);
		g_assert (seconds >= 0);
		g_assert (hours <= MAX_NUM_HOURS);
		g_assert (minutes <= 59);
		g_assert (seconds <= 59);
	
		/* strip hour indication field if we don't time hours and skip seconds
		 * indicator if we are.
		 */

		if (show_all) {
			/* hours:minutes:seconds */
			time_str = g_strdup_printf (_("%.2d:%.2d:%.2d"), hours, minutes, seconds);
		}
		else {
			if ((hours > 0) || (minutes > 14)) {
				/* hours:minutes */
				time_str = g_strdup_printf (_("%.2d:%.2d"), hours, minutes);
			}
			else {
				/* minutes:seconds */
				time_str = g_strdup_printf (_("%.2d:%.2d"), minutes, seconds);
			}
		}
	}
	else
	{
		time_str = g_strdup ("");
	}

	return time_str;
}

/**
 * Given a profile name and its associated duration of seconds,
 * this function returns a string that can be used to display the profile.
 * Caller is responsible for freeing the string when done.
 */
static gchar *
construct_display_name (const gchar *name, gdouble duration)
{
	gchar *display_name;
	gchar *time_str;
	
	time_str = construct_time_str (duration, TRUE /* always show hours:minutes:seconds */, FALSE /* don't shorten zero */);
	/* <name of preset> (<number of hours, minutes, and seconds>) */
	display_name = g_strdup_printf (_("%s (%s)"), name, time_str);
	g_free (time_str);
	time_str = NULL;
	
	return display_name;
}



/**
 * A hack used in construct_applet_content to remove the gap
 * between the applet and the right-click context menu
 */
static void
force_no_focus_padding (GtkWidget *widget)
{
	gboolean first_time = TRUE;;
	
	if (first_time) {
		gtk_rc_parse_string ("\n"
				     "   style \"timer-applet-button-style\"\n"
				     "   {\n"
				     "      GtkWidget::focus-line-width=0\n"
				     "      GtkWidget::focus-padding=0\n"
				     "   }\n"
				     "\n"
				     "   widget \"*.timer-applet-button\" style \"timer-applet-button-style\"\n"
				     "\n");
		first_time = FALSE;
	}
	
	gtk_widget_set_name (widget, "timer-applet-button");
}


/**
 * Fills in the applet widget with a button that contains an icon and optional label.
 */
static void
construct_applet_content (TimerApplet *timer_applet)
{
	GtkWidget *button_layout_box;
	GtkTooltips *tooltips;
	gint panel_size;
	PanelAppletOrient panel_orient;
	
	g_assert (timer_applet->main_button == NULL);
	g_assert (timer_applet->main_image == NULL);
	g_assert (timer_applet->main_remaining_time_label == NULL);
	
	panel_size = panel_applet_get_size (timer_applet->applet);
	panel_orient = panel_applet_get_orient (timer_applet->applet);
	if (panel_orient == PANEL_APPLET_ORIENT_LEFT ||
			panel_orient == PANEL_APPLET_ORIENT_RIGHT ||
			panel_size >= GNOME_Vertigo_PANEL_MEDIUM) {
		/* Layout the timer image and, optionally,
		 * remaining time vertically
		 */
		button_layout_box = gtk_vbox_new (FALSE, 0);
	}
	else {
		/* Layout the timer image and, optionally,
		 * remaining time horizontally
		 */
		button_layout_box = gtk_hbox_new (FALSE, 2);
	}
	
	timer_applet->main_button_layout_box = button_layout_box;
	
	timer_applet->main_image = gtk_image_new_from_stock (GNOME_STOCK_TIMER, GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_box_pack_start (GTK_BOX (button_layout_box), timer_applet->main_image, TRUE, TRUE, 0);
	gtk_widget_show (timer_applet->main_image);
	
	if (panel_applet_gconf_get_bool (timer_applet->applet, SHOW_REMAINING_TIME_KEY, NULL)) {
		gchar *remaining_time_str;
	
		/* calc_remaining_time may return a negative number if the user decides to show
		 * the remaining time just as the timer finishes its countdown
		 */
		remaining_time_str = construct_time_str (MAX (0, calc_remaining_time (timer_applet)), FALSE, TA_DONT_SHOW_ZEROS);
		timer_applet->main_remaining_time_label = gtk_label_new (remaining_time_str);
		g_free (remaining_time_str);
		remaining_time_str = NULL;
		
		gtk_box_pack_start (GTK_BOX (button_layout_box), timer_applet->main_remaining_time_label, TRUE, TRUE, 0);
		gtk_widget_show (timer_applet->main_remaining_time_label);
	}
	
	g_object_set (G_OBJECT (button_layout_box), "sensitive", timer_applet->timer_running, NULL);
	
	timer_applet->main_button = gtk_button_new ();
	g_object_set (G_OBJECT (timer_applet->main_button), "relief", GTK_RELIEF_NONE, NULL);
	gtk_container_add (GTK_CONTAINER (timer_applet->main_button), button_layout_box);
	gtk_widget_show (button_layout_box);
	
	tooltips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS (tooltips), timer_applet->main_button, "unknown", "unknown remaining time");
	timer_applet->main_tooltips = tooltips;
	
	set_tooltip_to_stop_str (timer_applet);
	
	gtk_container_add (GTK_CONTAINER (timer_applet->applet), timer_applet->main_button);
	gtk_widget_show (timer_applet->main_button);
	/* FIXME: hack from charpicker applet - fixes gap when opening right-click menu */
	force_no_focus_padding (timer_applet->main_button);
	
	g_signal_connect (G_OBJECT (timer_applet->main_button), "clicked", G_CALLBACK (on_main_button_clicked), timer_applet);
	/* FIXME: hack from charpicker applet - fixes to allow middle- and right-clicks on applet */
	g_signal_connect (G_OBJECT (timer_applet->main_button), "button_press_event", G_CALLBACK (on_button_press_hack), timer_applet);
}


/**
 * Destroys the current applet contents and reconstructs the applet.
 * This function is used when the user switches between showing or not showing
 * the remaining time label.
 */
static void
update_main_button (TimerApplet *timer_applet)
{
	g_assert (timer_applet);
	g_assert (timer_applet->main_button);
	
	gtk_widget_destroy (timer_applet->main_button);
	timer_applet->main_button = NULL;
	timer_applet->main_image = NULL;
	timer_applet->main_remaining_time_label = NULL;
	construct_applet_content (timer_applet);
}

/**
 * Saves the list of timer profiles into a file
 */
static void
save_timer_presets (void)
{
	xmlDocPtr doc;
	xmlNodePtr root;
	
	doc = xmlNewDoc ((xmlChar *) "1.0");
	root = xmlNewDocNode (doc, NULL, (xmlChar *) "timerapplet", NULL);
	xmlDocSetRootElement (doc, root);
	xmlNewProp (root, (xmlChar *) "version", (xmlChar *) TIMER_APPLET_DATA_FILE_VERSION);
	
	{
		GtkTreeIter iter;
		gboolean valid;
		
		valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (timer_presets_list), &iter);
		
		while (valid) {
			gchar *profile_name;
			gdouble profile_duration;
			
			gtk_tree_model_get (GTK_TREE_MODEL (timer_presets_list), &iter,
						PROFILE_NAME_COL, &profile_name,
						PROFILE_DURATION_COL, &profile_duration,
						-1);
			
			{
				xmlNodePtr node;
				gchar *profile_duration_str;
				
				node = xmlNewChild (root, NULL, (xmlChar *) "profile", NULL);
				profile_duration_str = g_strdup_printf ("%f", profile_duration);
				
				g_assert (profile_duration_str);
				
				xmlNewProp (node, (xmlChar *) "name", (xmlChar *) profile_name);
				xmlNewProp (node, (xmlChar *) "duration", (xmlChar *) profile_duration_str);
				
				g_free (profile_duration_str);
				profile_duration_str = NULL;
			}
			
			g_free (profile_name);
			profile_name = NULL;
			
			valid = gtk_tree_model_iter_next (GTK_TREE_MODEL (timer_presets_list), &iter);
		}
	}
	
	{
		gchar *file_path;
		
		file_path = g_strdup_printf ("%s%s", g_get_home_dir (), TIMER_APPLET_DATA_PATH);
		xmlSaveFormatFile (file_path, doc, 1);
		g_free (file_path);
		file_path = NULL;
	}
	
	g_assert (doc);
	xmlFreeDoc (doc);
	
	g_print ("Saved presets.\n");
	g_signal_emit (timer_presets_list, presets_changed_sig_id, 0, NULL);
}

/**
 * Load the list of timer profiles from the data file
 */
static void
load_timer_presets (void)
{
	xmlDocPtr doc;
	xmlNodePtr root;
	xmlNodePtr node;
	
	/* Load from file */
	{
		gchar *file_path;
		
		file_path = g_strdup_printf ("%s%s", g_get_home_dir (), TIMER_APPLET_DATA_PATH);
		doc = xmlParseFile (file_path);
		g_free (file_path);
		file_path = NULL;
	}
	
	if (!doc) {
		/* If the file doesn't exist, we create an empty one */
		save_timer_presets ();
		return;
	}
	
	root = xmlDocGetRootElement (doc);
	if (!root || xmlStrcmp (root->name, (xmlChar *) "timerapplet")) {
		/* the file was corrupted */
		xmlFreeDoc (doc);
		save_timer_presets ();
		return;
	}
	
	node = root->xmlChildrenNode;
	
	while (node) {
		if (xmlStrcmp (node->name, (xmlChar *) "profile") == 0) {
			xmlChar *profile_name;
			gdouble profile_duration;
			
			profile_name = xmlGetProp (node, (xmlChar *) "name");
			
			/* Get the duration and convert it to a gdouble */
			{
				xmlChar *profile_duration_str;
				profile_duration_str = xmlGetProp (node, (xmlChar *) "duration");
				g_assert (profile_duration_str);
				profile_duration = atof ((const char *)profile_duration_str);
				xmlFree (profile_duration_str);
				profile_duration_str = NULL;
			}
			
			g_assert (profile_name);
			g_assert (profile_duration >= 0);
			timer_applet_add_preset ((const gchar *)profile_name, profile_duration);
			
			xmlFree (profile_name);
			profile_name = NULL;
		}
		node = node->next;
	}
	
	xmlFreeDoc (doc);
	doc = NULL;
}

/**
 * Updates the optional remaining time label and the tooltip to
 * reflect the give amount of time.
 */
static void
set_ui_time (TimerApplet *timer_applet, gdouble seconds_remaining)
{
	g_assert (timer_applet);
	g_assert (seconds_remaining >= 0);

	if (timer_applet->main_remaining_time_label) {
		gchar *applet_time_str;
		applet_time_str = construct_time_str (seconds_remaining, FALSE, TA_DONT_SHOW_ZEROS);
		g_assert (GTK_IS_LABEL (timer_applet->main_remaining_time_label));
		gtk_label_set_text (GTK_LABEL (timer_applet->main_remaining_time_label), applet_time_str);
		g_free (applet_time_str);
		applet_time_str = NULL;
	}
	
	{
		gchar *tooltip_str;
		gchar *tooltip_time_str;
		
		/* Always show hours:minutes:seconds in tooltip */
		tooltip_time_str = construct_time_str (seconds_remaining, TRUE, FALSE /* don't shorten zero */);
		
		g_assert (timer_applet->cur_profile_name);
		if (strlen (timer_applet->cur_profile_name) > 0) {
			/* <remaining time in hours:minutes:seconds> (<name of preset>) */
			tooltip_str = g_strdup_printf (_("%s (%s)"), tooltip_time_str, timer_applet->cur_profile_name);
		}
		else {
			tooltip_str = g_strdup_printf ("%s", tooltip_time_str);
		}
		g_free (tooltip_time_str);
		tooltip_time_str = NULL;
		
		set_tooltip (timer_applet, tooltip_str);
		g_free (tooltip_str);
		tooltip_str = NULL;
	}
}

/**
 * Returns true if the timer is currently running,
 * and false if the timer is paused or stopped.
 */
static gboolean
timer_is_running (TimerApplet *timer_applet)
{
	if (timer_applet->timer_running) {
		g_assert (timer_applet->timer);
	}
	return timer_applet->timer_running;
}

/**
 * Returns true if the timer is currently paused,
 * and false if the timer is running or stopped.
 */
static gboolean
timer_is_paused (TimerApplet *timer_applet)
{
	return (timer_applet->timer != NULL && !timer_applet->timer_running);
}

/**
 * Activate the timer by properly sensitizing the timer icon
 * and label. Requires that timer_applet->timer is non-NULL
 * and running. timer_applet->timer_running should be TRUE.
 */
static void
activate_timer (TimerApplet *timer_applet)
{
	BonoboUIComponent *popup_component;

	g_assert (timer_applet);
	g_assert (timer_applet->timer);
	g_assert (timer_applet->timer_running);
	
	popup_component = panel_applet_get_popup_component (PANEL_APPLET (timer_applet->applet));

	/* Show the PauseTimer menu item. */
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_PAUSE_TIMER,
					"hidden", "0",
					NULL);

	/* Hide the ResumeTimer menu item. */
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_RESUME_TIMER,
					"hidden", "1",
					NULL);

	/* Show the RestartTimer menu item. */
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_RESTART_TIMER,
					"hidden", "0",
					NULL);

	/* Show the StopTimer menu item. */
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_STOP_TIMER,
					"hidden", "0",
					NULL);

	g_assert (timer_applet->main_button_layout_box);
	{
		gboolean is_sensitive;
		g_object_get (G_OBJECT (timer_applet->main_button_layout_box), "sensitive", &is_sensitive, NULL);
		g_assert (!is_sensitive);
	}

	g_object_set (G_OBJECT (timer_applet->main_button_layout_box), "sensitive", TRUE, NULL);
}

/**
 * Resume the timer. Should only be called when the
 * timer is paused, not while it's stopped or running.
 */
static void
resume_timer (TimerApplet *timer_applet)
{
	g_assert (timer_is_paused (timer_applet));
	g_assert (timer_applet->timer);
	g_assert (timer_applet->timer_running == FALSE);

	g_timer_continue (timer_applet->timer);
	timer_applet->timer_running = TRUE;
		
	activate_timer (timer_applet);
}

/**
 * Pause the timer. Note: the user may pause the timer just as
 * the timer is about to finish, meaning that calc_remaining_time
 * may return a negative value.
 */
static void
pause_timer (TimerApplet *timer_applet)
{
	BonoboUIComponent *popup_component;

	g_assert (timer_applet != NULL);
	g_assert (timer_applet->timer);
	
	g_timer_stop (timer_applet->timer);
	timer_applet->timer_running = FALSE;

	set_ui_time (timer_applet, MAX (0, calc_remaining_time (timer_applet)));
	set_tooltip (timer_applet, _("Paused. Click to resume timer."));
	g_assert (timer_applet->main_button_layout_box);
	g_object_set (G_OBJECT (timer_applet->main_button_layout_box), "sensitive", FALSE, NULL);

	popup_component = panel_applet_get_popup_component (PANEL_APPLET (timer_applet->applet));

	/* Hide the PauseTimer menu item. */
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_PAUSE_TIMER,
					"hidden", "1",
					NULL);

	/* Show the ResumeTimer menu item. */
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_RESUME_TIMER,
					"hidden", "0",
					NULL);
}

/**
 * Stop the timer. Can be called when the timer has finished a countdown,
 * or when the user explicitly resets the timer to zero.
 */
static void
stop_timer (TimerApplet *timer_applet)
{
	BonoboUIComponent *popup_component;

	g_assert (timer_applet);
	g_assert (timer_applet->timer);
	g_assert (timer_is_running (timer_applet) || timer_is_paused (timer_applet));

	/* calc_remaining_time (timer_applet) can be a negative number
	 * if called from on_timer_timeout when the timer finishes,
	 * or it can be >= 0 if the user explicitly resets the timer to zero
	 * while the timer is running or paused.
	 */

	g_timer_stop (timer_applet->timer);
	g_timer_reset (timer_applet->timer);
	g_timer_destroy (timer_applet->timer);
	timer_applet->timer = NULL;
	timer_applet->timer_running = FALSE;
	g_assert (calc_remaining_time (timer_applet) == 0);

	set_ui_time (timer_applet, 0);
	set_tooltip_to_stop_str (timer_applet);

	g_assert (timer_applet->main_button_layout_box);
	g_object_set (G_OBJECT (timer_applet->main_button_layout_box), "sensitive", FALSE, NULL);

	popup_component = panel_applet_get_popup_component (PANEL_APPLET (timer_applet->applet));

	/* Hide the PauseTimer menu item. */
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_PAUSE_TIMER,
					"hidden", "1",
					NULL);

	/* Hide the RestartTimer menu item. */
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_RESTART_TIMER,
					"hidden", "1",
					NULL);

	/* Hide the StopTimer menu item. */
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_STOP_TIMER,
					"hidden", "1",
					NULL);
}

/**
 * Construct a string representing the current time.
 * User should free the string when done using it.
 */
static gchar *
construct_current_time_str (void)
{
	time_t current_time;
	struct tm *time_info;
	char time_str[256];
	gchar *utf8_str;

	time (&current_time);
	time_info = localtime (&current_time);

	g_assert (time_info);
	
	{
		size_t num_chars;
		/* "hh:mm AM/PM" format string for strftime */
		num_chars = strftime (time_str, sizeof (time_str), _("%I:%M %p"), time_info);
		g_assert (num_chars > 0);
	}

	utf8_str = g_locale_to_utf8 (time_str, -1, NULL, NULL, NULL);

	return utf8_str;
}

/**
 * Notify the user that the timer is done by displaying a dialog box
 * and optionally playing a sound
 */
static void
notify_timer_done (TimerApplet *timer_applet)
{
	GtkWidget *dialog;
	gchar *cur_time_str;
	gchar *timer_done_str;

	g_assert (timer_applet);

	cur_time_str = construct_current_time_str ();

	g_assert (timer_applet->cur_profile_name);
	if (strlen (timer_applet->cur_profile_name) > 0) {
		/* "<timer name>" finished at <finish time> */
		timer_done_str = g_strdup_printf (_("\"%s\" finished at %s"),
							timer_applet->cur_profile_name,
							cur_time_str);
	}
	else {
		/* Timer finished at <finish time> */
		timer_done_str = g_strdup_printf (_("Timer finished at %s"), cur_time_str);
	}
	g_free (cur_time_str);
	cur_time_str = NULL;
		
	dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_INFO, GTK_BUTTONS_OK, timer_done_str);
	g_free (timer_done_str);
	timer_done_str = NULL;
	g_object_set (G_OBJECT (dialog), "title", _("Timer Finished"), NULL);
	
	if (panel_applet_gconf_get_bool (timer_applet->applet, PLAY_NOTIFICATION_SOUND_KEY, NULL)) {
		gboolean use_custom_sound;
		gchar *notification_sound_path;
		use_custom_sound = panel_applet_gconf_get_bool (timer_applet->applet, USE_CUST_NOTIFICATION_SOUND_KEY, NULL);

		if (use_custom_sound) {
			g_print ("Using custom notification sound.\n");
			notification_sound_path = panel_applet_gconf_get_string (timer_applet->applet,
										CUST_NOTIFICATION_SOUND_PATH_KEY,
										NULL);
			g_assert (notification_sound_path);
			if (strlen (notification_sound_path) == 0) {
				g_print ("No custom notification sound specified. Using default sound.\n");
				g_free (notification_sound_path);
				notification_sound_path = g_strdup (DEFAULT_NOTIFICATION_SOUND_PATH);
			}
		}
		else {
			g_print ("Using default notification sound.\n");
			notification_sound_path = g_strdup (DEFAULT_NOTIFICATION_SOUND_PATH);
		}
		
		g_assert (notification_sound_path);
		g_print ("Playing notification sound: %s\n", notification_sound_path);
		gnome_sound_play (notification_sound_path);

		g_free (notification_sound_path);
		notification_sound_path = NULL;
	}

	/* Make the applet button un-clickable until the user dismisses the dialog box */

	g_assert (timer_applet->main_button);
	g_object_set (G_OBJECT (timer_applet->main_button), "sensitive", FALSE, NULL);

	if (panel_applet_gconf_get_bool (timer_applet->applet, NOTIFICATION_DIALOG_ALWAYS_ON_TOP_KEY, NULL)) {
		gtk_window_set_keep_above (GTK_WINDOW (dialog), TRUE);
	}
	
	if (panel_applet_gconf_get_bool (timer_applet->applet, NOTIFICATION_DIALOG_STICKY_KEY, NULL)) {
		gtk_window_stick (GTK_WINDOW (dialog));
	}

	if (timer_applet->main_remaining_time_label) {
		gtk_label_set_text (GTK_LABEL (timer_applet->main_remaining_time_label), _("Finished"));
	}
	
	gtk_dialog_run (GTK_DIALOG (dialog));

	if (timer_applet->main_remaining_time_label) {
		gtk_label_set_text (GTK_LABEL (timer_applet->main_remaining_time_label), "");
	}

	g_object_set (G_OBJECT (timer_applet->main_button), "sensitive", TRUE, NULL);
	gtk_widget_destroy (dialog);
}

static void
row_ref_to_iter (GtkTreeModel *tree_model, GtkTreeRowReference *row_ref, GtkTreeIter *tree_iter)
{
	GtkTreePath *path;
	gboolean iter_set;
	
	g_assert (tree_model);
	g_assert (row_ref);
	g_assert (tree_iter);
	
	path = gtk_tree_row_reference_get_path (row_ref);
	iter_set = gtk_tree_model_get_iter (GTK_TREE_MODEL (tree_model), tree_iter, path);
	g_assert (iter_set);
	g_free (path);
	path = NULL;
}

static GtkTreeRowReference *
row_ref_from_iter (GtkTreeModel *tree_model, GtkTreeIter *tree_iter)
{
	GtkTreeRowReference *new_row_ref;
	GtkTreePath *tree_path;
	
	g_assert (tree_model);
	g_assert (tree_iter);
	
	tree_path = gtk_tree_model_get_path (GTK_TREE_MODEL (tree_model), tree_iter);
	g_assert (tree_path);
	new_row_ref = gtk_tree_row_reference_new (GTK_TREE_MODEL (tree_model), tree_path);
	g_assert (new_row_ref);
	g_free (tree_path);
	tree_path = NULL;
	
	return new_row_ref;
}

static void 
update_presets_submenu (TimerApplet *timer_applet)
{
	BonoboUIComponent *popup_component;

	popup_component = panel_applet_get_popup_component (PANEL_APPLET (timer_applet->applet));
	
	/* Remove existing menu items. */
	int i;
	for (i = 0;; i++) {
		gchar *item_path = g_strdup_printf (PRESETS_SUBMENU_PATH "/Preset_%d", i);
		if (bonobo_ui_component_path_exists (popup_component, item_path, NULL)) {
			bonobo_ui_component_rm (popup_component, item_path, NULL);
			g_print ("Removed presets submenu item: %s\n", item_path);
			g_free (item_path);
			item_path = NULL;
		}
		else {
			g_free (item_path);
			item_path = NULL;
			break;
		}
	}

	{
		GtkTreeIter iter;
		gboolean valid;
		
		valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (timer_presets_list), &iter);
		while (valid) {
			gchar *preset_name;
			gdouble preset_duration;
			gchar *preset_display_name;
			gint preset_index;
			
			gtk_tree_model_get (GTK_TREE_MODEL (timer_presets_list), &iter,
								PROFILE_NAME_COL, &preset_name,
								PROFILE_DURATION_COL, &preset_duration,
								-1);
			g_assert (preset_name);
			preset_display_name = construct_display_name (preset_name, preset_duration);
			g_free (preset_name);
			preset_name = NULL;
			
			/* Determine this preset's index. */
			{
				GtkTreeRowReference *row_ref = row_ref_from_iter (GTK_TREE_MODEL (timer_presets_list), &iter);
				GtkTreePath *path = gtk_tree_row_reference_get_path (row_ref);
				gint *path_indices = gtk_tree_path_get_indices (path);
				g_assert (gtk_tree_path_get_depth (path) == 1);
				preset_index = path_indices[0];
				gtk_tree_path_free (path);
				path = NULL;
			}
			
			gchar *verb = g_strdup_printf ("Preset_%d", preset_index);
			BonoboUINode *node = bonobo_ui_node_new ("menuitem");	
			bonobo_ui_node_set_attr (node, "name", verb);
			bonobo_ui_node_set_attr (node, "verb", verb);
			bonobo_ui_node_set_attr (node, "label", preset_display_name);
			g_print ("Added presets submenu item: %s\n", verb);
			bonobo_ui_component_set_tree (popup_component, PRESETS_SUBMENU_PATH, node, NULL);
			bonobo_ui_component_add_verb (popup_component, verb, (BonoboUIVerbFn)on_presets_submenu_item_activate, timer_applet);
			g_free (verb);
			verb = NULL;
			
			g_free (preset_display_name);
			preset_display_name = NULL;

			valid = gtk_tree_model_iter_next (GTK_TREE_MODEL (timer_presets_list), &iter);
		}
	}
}

/**
 * Shows a dialog box to ask whether the user would like to
 * resume a paused timer. This function resumes the timer or
 * shows the "Start Timer" dialog, depending on what button
 * the user clicks on.
 * Requires that the timer is paused.
 */
static void
ask_user_resume_timer (TimerApplet *timer_applet)
{
	GtkWidget *ask_resume_dialog;
	
	g_assert (timer_is_paused (timer_applet));

	ask_resume_dialog = gtk_dialog_new_with_buttons (_("Resume Timer"),
							NULL,
							GTK_DIALOG_DESTROY_WITH_PARENT,
							GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
							_("_Start Over"), GTK_RESPONSE_NO,
							_("_Resume Timer"), GTK_RESPONSE_YES,
							NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (ask_resume_dialog), GTK_RESPONSE_YES);

	{
		GtkWidget *main_vbox;
		GtkWidget *hbox;
		GtkWidget *image;
		GtkWidget *label;
		gchar *remaining_time_str;
		gchar *dialog_header_text;
		gchar *dialog_body_text;
		gchar *dialog_text;

		main_vbox = GTK_DIALOG (ask_resume_dialog)->vbox;
		hbox = gtk_hbox_new (FALSE, 0);
		image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION, GTK_ICON_SIZE_DIALOG);
		remaining_time_str = construct_time_str (timer_applet_get_remaining_time (timer_applet), TRUE, FALSE /* don't shorten zero */);

		g_assert (timer_applet->cur_profile_name);
		if (strlen (timer_applet->cur_profile_name) > 0) {
			/* Resume the "<timer name>" timer? */
			dialog_header_text = g_strdup_printf(_("Resume the \"%s\" timer?"), timer_applet->cur_profile_name);
		}
		else {
			dialog_header_text = g_strdup_printf(_("Resume the timer?"));
		}
		/* The timer is currently paused. Would you like to resume countdown from <remaining time>? */
		dialog_body_text = g_strdup_printf(_("The timer is currently paused. Would you like to resume countdown from %s?"), remaining_time_str);
		dialog_text = g_strdup_printf("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
						dialog_header_text, dialog_body_text);
		g_free (dialog_header_text);
		dialog_header_text = NULL;
		g_free (dialog_body_text);
		dialog_body_text = NULL;
		g_free (remaining_time_str);
		remaining_time_str = NULL;
		label = gtk_label_new (dialog_text);
		g_free (dialog_text);
		dialog_text = NULL;
		gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, FALSE, 0);
				
		g_object_set (G_OBJECT (ask_resume_dialog),
					"border-width", 6,
					"has-separator", FALSE,
					"resizable", FALSE,
					NULL);
		g_object_set (G_OBJECT (main_vbox), "spacing", 12, NULL);
		g_object_set (G_OBJECT (hbox),
				"spacing", 12,
				"border-width", 6,
				NULL);
		g_object_set (G_OBJECT (image),
				"yalign", 0.0,
				NULL);
		g_object_set (G_OBJECT (label),
				"use-markup", TRUE,
				"wrap", TRUE,
				"yalign", 0.0,
				NULL);

		gtk_widget_show (image);
		gtk_widget_show (label);
		gtk_widget_show (hbox);
	}

	{
		gint dialog_result;
		dialog_result = gtk_dialog_run (GTK_DIALOG (ask_resume_dialog));
		gtk_widget_destroy (ask_resume_dialog);
		ask_resume_dialog = NULL;

		switch (dialog_result) {
			case GTK_RESPONSE_YES:
				resume_timer (timer_applet);
				break;

			case GTK_RESPONSE_NO:
				start_timer_dialog_open_with_name_and_duration (timer_applet,
										timer_applet->cur_profile_name,
										timer_applet->cur_profile_duration);
				break;


			case GTK_RESPONSE_CANCEL:
				/* do nothing */
				break;

			case GTK_RESPONSE_DELETE_EVENT:
				/* do nothing */
				break;

			default:
				g_assert_not_reached ();
		}
	}
}

/**
 * Returns a pointer to the list of timer presets that's
 * shared among all instances of this applet
 */
GtkListStore *
timer_applet_get_presets_list (void)
{
	return timer_presets_list;
}

/**
 * Indicate which timer preset should run. If preset_name is NULL,
 * then the timer will be a "one-time" instant timer.
 *
 * NOTE: preset_name must not point to the same string as
 * timer_applet->cur_profile_name.
 */
static void
set_current_preset (TimerApplet *timer_applet, const gchar *preset_name, gdouble timer_duration)
{
	g_assert (preset_name != NULL);
	g_assert (preset_name != timer_applet->cur_profile_name);

	g_assert (timer_applet->cur_profile_name);
	g_free (timer_applet->cur_profile_name);
	timer_applet->cur_profile_name = g_strdup (preset_name);
	timer_applet->cur_profile_duration = timer_duration;
	set_ui_time (timer_applet, timer_duration);
}

/**
 * Start the timer with the given preset name and duration.
 * preset_name can be an empty string to indicate a one-time timer.
 * preset_name cannot be NULL.
 */
void
timer_applet_start_timer (TimerApplet *timer_applet, const gchar *preset_name, gdouble timer_duration)
{
	g_assert (timer_applet);
	g_assert (preset_name);
	g_assert (timer_applet->timer_running == FALSE);
	g_assert (timer_duration >= 0);

	set_current_preset (timer_applet, preset_name, timer_duration);

	/* If the timer was paused, we kept timer_applet->timer
	 * around just to keep track of how much time was left.
	 * Now that we're starting another timer, we can just get rid
	 * of the current timer_applet->timer and create a new one.
	 */
	if (timer_applet->timer) {
		g_timer_reset (timer_applet->timer);
		g_timer_destroy (timer_applet->timer);
		timer_applet->timer = NULL;
	}

	g_assert (timer_applet->timer == NULL);
	timer_applet->timer = g_timer_new ();
	timer_applet->timer_running = TRUE;

	activate_timer (timer_applet);
}

/**
 * Used to add a new timer profile with the given name and duration
 */
GtkTreeRowReference *
timer_applet_add_preset (const gchar *name, gdouble duration)
{
	GtkTreeIter iter;
	gchar *display_name;
	
	g_assert (timer_presets_list);
	
	display_name = construct_display_name (name, duration);
	
	gtk_list_store_append (GTK_LIST_STORE (timer_presets_list), &iter);
	gtk_list_store_set (GTK_LIST_STORE (timer_presets_list), &iter,
				PROFILE_DISPLAY_NAME_COL, display_name,
				PROFILE_NAME_COL, name,
				PROFILE_DURATION_COL, duration,
				-1);
				
	g_free (display_name);
	display_name = NULL;
	
	save_timer_presets ();
	
	return row_ref_from_iter (GTK_TREE_MODEL (timer_presets_list), &iter);
}

/**
 * Remove the given timer preset referenced by row_ref
 */
void
timer_applet_remove_preset (GtkTreeRowReference *row_ref)
{
	GtkTreeIter iter;
	gboolean iter_reset;
	
	g_assert (timer_presets_list);
	
	row_ref_to_iter (GTK_TREE_MODEL (timer_presets_list), row_ref, &iter);
	iter_reset = gtk_list_store_remove (GTK_LIST_STORE (timer_presets_list), &iter);
	
	save_timer_presets ();
}

/**
 * Update the timer preset at row_ref to the given name and duration
 */
void
timer_applet_update_preset (GtkTreeRowReference *row_ref, gchar *name, gdouble duration)
{
	GtkTreeIter iter;
	gchar *display_name;
	
	g_assert (timer_presets_list);
	
	display_name = construct_display_name (name, duration);
	
	row_ref_to_iter (GTK_TREE_MODEL (timer_presets_list), row_ref, &iter);
	
	gtk_list_store_set (GTK_LIST_STORE (timer_presets_list), &iter,
				PROFILE_DISPLAY_NAME_COL, display_name,
				PROFILE_NAME_COL, name,
				PROFILE_DURATION_COL, duration,
				-1);
				
	g_free (display_name);
	display_name = NULL;
	
	save_timer_presets ();
}

/**
 * Returns the name and duration for the profile at the given row.
 * User should free 'name' when done.
 */
void
timer_applet_get_profile_name_and_duration (TimerApplet *timer_applet, GtkTreeRowReference *row_ref, gchar **name, gdouble *duration)
{
	GtkTreeIter iter;
	
	g_assert (timer_applet);
	g_assert (row_ref);
	g_assert (name);
	g_assert (duration);
	
	g_assert (timer_presets_list);
	
	row_ref_to_iter (GTK_TREE_MODEL (timer_presets_list), row_ref, &iter);
	
	gtk_tree_model_get (GTK_TREE_MODEL (timer_presets_list), &iter,
				PROFILE_NAME_COL, name,
				PROFILE_DURATION_COL, duration,
				-1);
}

/**
 * Returns the number of seconds left in a paused timer that
 * has not finished. Should only be called when the timer isn't running.
 */
gdouble
timer_applet_get_remaining_time (TimerApplet *timer_applet)
{
	g_assert (!timer_applet->timer_running);
	/* calc_remaining_time may return a negative number while
	 * the timer is paused because the user could have paused
	 * the timer just as it was about to finish
	 */
	return MAX (0, calc_remaining_time (timer_applet));
}

/**
 * Initialize the given timer_applet
 */
void
timer_applet_init (TimerApplet *timer_applet, PanelApplet *panel_applet)
{
	BonoboUIComponent        *popup_component;
	g_assert (timer_applet);
	g_assert (panel_applet);
	
	timer_applet->applet = panel_applet;
	panel_applet_set_flags (timer_applet->applet, PANEL_APPLET_EXPAND_MINOR);
	
	timer_applet->timer_running = FALSE;
	timer_applet->timer = NULL;
	if (timer_presets_list == NULL) {
		/* Perform one-time initialization */

		timer_presets_list = gtk_list_store_new (3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_DOUBLE);
		
		presets_changed_sig_id = g_signal_newv ("presets-changed",
						G_TYPE_FROM_CLASS (GTK_LIST_STORE_GET_CLASS (timer_presets_list)),
						G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
						NULL /* class closure */,
						NULL /* accumulator */,
						NULL /* accu_data */,
						g_cclosure_marshal_VOID__VOID,
						G_TYPE_NONE /* return type */,
						0 /* n_params */,
						NULL /* param_types */);
						
		load_timer_presets ();

		gnome_window_icon_set_default_from_file (PIXMAPS_DIR "/timer-applet-icon.png");

		g_assert (manage_presets_window == NULL);
		manage_presets_window = manage_presets_window_new ();
		g_assert (manage_presets_window);
		
		about_dialog = NULL;
	}
	timer_applet->cur_profile_name = g_strdup ("");
	timer_applet->cur_profile_duration = 0;
	
	timer_applet->main_button = NULL;
	timer_applet->main_button_layout_box = NULL;
	timer_applet->main_image = NULL;
	timer_applet->main_remaining_time_label = NULL;
	timer_applet->main_tooltips = NULL;
	
	timer_applet->start_timer_dialog = NULL;
	timer_applet->start_timer_profile_manager_widget = NULL;
	
	{
		GConfClient *gconf_client;
		gchar *prefs_key;
		
		gconf_client = gconf_client_get_default ();
		
		/* Apply the GConf schema to this instance's GConf preferences */
		panel_applet_add_preferences (timer_applet->applet, "/schemas/apps/timer-applet/prefs", NULL);
		prefs_key = panel_applet_get_preferences_key (timer_applet->applet);
		g_print ("Adding notification for preferences key: %s\n", prefs_key);
		timer_applet->gconf_notification_id = gconf_client_notify_add (gconf_client,
									prefs_key,
									(GConfClientNotifyFunc) on_preferences_change,
									timer_applet, NULL, NULL);
		g_free (prefs_key);
		prefs_key = NULL;
		
		g_object_unref (G_OBJECT (gconf_client));
	}
	
	timer_applet->prefs_dialog = prefs_dialog_new (timer_applet->applet);
	g_assert (timer_applet->prefs_dialog);
	
	construct_applet_content (timer_applet);
	
	/* Setup right-click menu */
	panel_applet_setup_menu_from_file (timer_applet->applet,
					   NULL,
					   TIMER_APPLET_MENU_FILE_NAME,
					   NULL,
					   timer_applet_verbs,
					   timer_applet);
	popup_component = panel_applet_get_popup_component (PANEL_APPLET (timer_applet->applet));
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_PAUSE_TIMER,
					"hidden", "1",
					NULL);
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_RESUME_TIMER,
					"hidden", "1",
					NULL);
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_RESTART_TIMER,
					"hidden", "1",
					NULL);
	bonobo_ui_component_set_prop (popup_component,
					TA_MENU_COMMAND_STOP_TIMER,
					"hidden", "1",
					NULL);

	start_timer_dialog_create (timer_applet);
	update_presets_submenu (timer_applet);
	
	timer_applet->presets_changed_handler_id = g_signal_connect (G_OBJECT (timer_presets_list), "presets-changed", (GCallback)on_presets_changed, timer_applet);
	
	g_timeout_add (TIMER_TIMEOUT_INTERVAL, (GSourceFunc) on_timer_timeout, timer_applet);

	g_assert (timer_applet->applet);

	g_signal_connect (G_OBJECT (timer_applet->applet), "change-background",
				G_CALLBACK (on_applet_change_background), NULL);
	g_signal_connect (G_OBJECT (timer_applet->applet), "change-orient",
				G_CALLBACK (on_applet_change_orient), timer_applet);
	g_signal_connect (G_OBJECT (timer_applet->applet), "change-size",
				G_CALLBACK (on_applet_change_size), timer_applet);
	g_signal_connect (G_OBJECT (timer_applet->applet), "destroy",
				G_CALLBACK (on_applet_instance_destroy), timer_applet);
	
	gtk_widget_show (GTK_WIDGET (timer_applet->applet));	
}






/**
 * Callbacks for context menu
 */

static void
on_pause_timer_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname)
{
	g_assert (timer_applet);

	pause_timer (timer_applet);
}

static void
on_resume_timer_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname)
{
	g_assert (timer_applet);

	resume_timer (timer_applet);
}

static void
on_restart_timer_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname)
{
	gchar *preset_name;
	g_assert (timer_applet);

	stop_timer (timer_applet);

	g_assert (timer_applet->cur_profile_duration >= 0);

	/* We need to create a copy of the string first because cur_profile_name will be
	 * freed before being set to the new preset name (which in this case is the same name).
	 */
	preset_name = g_strdup (timer_applet->cur_profile_name);
	timer_applet_start_timer (timer_applet, preset_name, timer_applet->cur_profile_duration);
	g_free (preset_name);
	preset_name = NULL;
}

static void
on_stop_timer_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname)
{
	g_assert (timer_applet);

	stop_timer (timer_applet);
}
	
static void
on_presets_submenu_item_activate (BonoboUIComponent *uic, TimerApplet *timer_applet, const gchar *verb)
{
	int preset_index;
	
	g_assert (timer_applet);
	
	if (sscanf (verb, "Preset_%d", &preset_index) != 1) {
		g_print ("ERROR: Invalid preset submenu verb: %s\n", verb);
		return;
	}

	GtkTreeIter iter;
	{
		GtkTreePath *path = gtk_tree_path_new_from_indices (preset_index, -1);
		gboolean iter_set = gtk_tree_model_get_iter (GTK_TREE_MODEL (timer_presets_list), &iter, path);
		g_assert (iter_set);
		gtk_tree_path_free (path);
	}

	{
		gchar *preset_name;
		gdouble preset_duration;
				
		gtk_tree_model_get (GTK_TREE_MODEL (timer_presets_list), &iter,
							PROFILE_NAME_COL, &preset_name,
							PROFILE_DURATION_COL, &preset_duration,
							-1);
		if (timer_is_running (timer_applet) || timer_is_paused (timer_applet)) {
			stop_timer (timer_applet);
		}
		timer_applet_start_timer (timer_applet, preset_name, preset_duration);
		g_print ("Activated this preset: %s (index = %d)\n", preset_name, preset_index);
		g_free (preset_name);
		preset_name = NULL;
	}
}
 
static void
on_manage_presets_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname)
{
	g_assert (timer_applet);
	
	manage_presets_window_open (GTK_DIALOG (manage_presets_window));
}

static void
on_preferences_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname)
{
	g_assert (timer_applet->prefs_dialog);

	prefs_dialog_open (GTK_DIALOG (timer_applet->prefs_dialog));
}

static void
on_help_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname)
{
	GError *err = NULL;
	
	g_assert (timer_applet);

	if (!gnome_help_display ("timer-applet.xml", NULL, &err)) {
		g_assert (err);
		g_print ("Could not display help for Timer Applet: %s\n", err->message);
		g_error_free (err);
		err = NULL;
	}
}

static void
on_about_activate (BonoboUIComponent *component, TimerApplet *timer_applet, const gchar *cname)
{
	g_assert (timer_applet);
	
	/* NOTE: about_dialog is automatically set to NULL
	 * through a callback when the dialog is closed.
	 */
	about_dialog_show (&about_dialog);
}

/**
 * Other callbacks
 */

static void
on_main_button_clicked (GtkWidget *button, TimerApplet *timer_applet)
{
	g_assert (timer_applet);
	
	if (timer_is_running (timer_applet)) {
		g_assert (timer_applet->timer);
		pause_timer (timer_applet);
	}
	else if (timer_is_paused (timer_applet)) {
		ask_user_resume_timer (timer_applet);
	}
	else {
		start_timer_dialog_open (timer_applet);
	}
}

/**
 * A hack used in construct_applet_content to allow middle-clicks and right-clicks in the applet
 */
static gboolean
on_button_press_hack (GtkWidget *widget, GdkEventButton *event, TimerApplet *timer_applet)
{
	if (event->button == 3 || event->button == 2) {
		gtk_propagate_event (GTK_WIDGET (timer_applet->applet), (GdkEvent *)event);
		
		return TRUE;
	}
	
	return FALSE;
}

/**
 * Called when GConf notices a change caused by someone editing a preference in the
 * GConf configuration editor or through another Timer Applet's preferences dialog box.
 * This is called for each instance of Timer Applet.
 */
static void
on_preferences_change (GConfClient *client, guint cnxn_id, GConfEntry *entry, TimerApplet *timer_applet)
{
	if (timer_applet->applet) {
		prefs_dialog_update (GTK_DIALOG (timer_applet->prefs_dialog));

		/* If this handler is called while the applet is being removed from the panel,
		 * prefs_dialog_update() can cause the applet's "destroy" handler to be run
		 * before the code below gets to execute.
		 * Specifically, the "destroy" handler is run when prefs_dialog_update() changes
		 * the "active" property of one of its checkboxes from TRUE to FALSE or FALSE to
		 * TRUE. The "destroy" handler, however, does NOT run if a checkbox's value is
		 * unchanged, such as setting it to TRUE when it's already TRUE or setting it to
		 * FALSE when it's already FALSE. 
		 *
		 * Therefore, we check again to see if timer_applet->applet is NULL, which indicates
		 * that the applet's "destroy" handler has already been run.
		 */
		if (timer_applet->applet) {
			update_main_button (timer_applet);
		}
	}
}

/**
 * Timeout handler that gets called periodically to update the timer
 */
static gboolean
on_timer_timeout (TimerApplet *timer_applet)
{
	g_assert (timer_applet);

	/* Check if the applet is still in the panel */
	if (timer_applet->applet) {	
		if (timer_applet->timer_running) {
			gdouble remaining_time;
			
			g_assert (timer_applet->timer);
	
			remaining_time = calc_remaining_time (timer_applet);
			
			if (remaining_time >= 0) {
				set_ui_time (timer_applet, remaining_time);
			}
			else {
				stop_timer (timer_applet);
				notify_timer_done (timer_applet);
			}
		}
	
		return TRUE; /* don't remove the timeout handler */
	}
	else {
		return FALSE; /* remove the timeout handler */
	}
}

static void
on_presets_changed (GObject *obj, TimerApplet *timer_applet)
{
	g_assert (timer_applet);
	
	g_print ("Presets changed.\n");
	update_presets_submenu (timer_applet);
}

static void
on_applet_change_background (PanelApplet *applet, PanelAppletBackgroundType type,
				GdkColor *color, GdkPixmap *pixmap, gpointer user_data)
{
	GtkRcStyle *rc_style;
	GtkStyle *style;

	gtk_widget_set_style (GTK_WIDGET (applet), NULL);
	rc_style = gtk_rc_style_new ();
	gtk_widget_modify_style (GTK_WIDGET (applet), rc_style);
	gtk_rc_style_unref (rc_style);

	switch (type) {
		case PANEL_NO_BACKGROUND:

			break;

		case PANEL_COLOR_BACKGROUND:
			gtk_widget_modify_bg (GTK_WIDGET (applet),
						GTK_STATE_NORMAL, color);
			break;
		
		case PANEL_PIXMAP_BACKGROUND:
			style = gtk_style_copy (GTK_WIDGET (applet)->style);
			if (style->bg_pixmap[GTK_STATE_NORMAL]) {
				g_object_unref (style->bg_pixmap[GTK_STATE_NORMAL]);
			}
			style->bg_pixmap[GTK_STATE_NORMAL] = g_object_ref (pixmap);
			gtk_widget_set_style (GTK_WIDGET (applet), style);
			g_object_unref (style);
			break;
	}
}

static void
on_applet_change_orient (PanelApplet *applet, PanelAppletOrient orient, TimerApplet *timer_applet)
{
	update_main_button (timer_applet);
}

static void
on_applet_change_size (PanelApplet *applet, gint size, TimerApplet *timer_applet)
{
	update_main_button (timer_applet);
}

static void
on_applet_instance_destroy (GtkWidget *applet, TimerApplet *timer_applet)
{
	/* When the applet gets destroyed, we set timer_applet->applet to NULL to indicate
	 * that we want on_timer_timeout to remove itself by returning FALSE
	 */
	timer_applet->applet = NULL;
	
	/* Remove GConf notification */
	{
		GConfClient *gconf_client;
		
		gconf_client = gconf_client_get_default ();
		
		gconf_client_notify_remove (gconf_client, timer_applet->gconf_notification_id);
		timer_applet->gconf_notification_id = 0;
		
		g_object_unref (G_OBJECT (gconf_client));
	}

	g_signal_handler_disconnect (G_OBJECT (timer_presets_list), timer_applet->presets_changed_handler_id);
	timer_applet->presets_changed_handler_id = 0;
}
