/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2012
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"
#include "tickr_html_entities.h"

/*
 * Build selection f_list from full feed f_list.
 *
 * Always use get/set_feed_list() and get/set_feed_selection() to access
 * FList *feed_list and FList *feed_selection defined in tickr_main.c.
 */
int build_feed_selection_from_feed_list()
{
	FList	*new_selection = NULL, *node = get_feed_list();

	if (IS_FLIST(node)) {
		for (node = f_list_first(node); IS_FLIST(node); node = node->next)
			if (node->selected)
				new_selection = f_list_add_at_end(new_selection,\
					node->url, node->title, TRUE, node->rank);
		if (IS_FLIST(new_selection)) {
			new_selection = f_list_first(new_selection);
			set_feed_selection(new_selection);
			return OK;
		} else {
			set_feed_selection(NULL);
			return SELECTION_EMPTY;
		}
	} else {
		set_feed_selection(NULL);
		return SELECTION_ERROR;
	}
}

/*
 * feed_index_in_selection starts at 0
 * (these funcs do nothing in single selection mode)
 */
/* Always reload -> OK like that ? */
void current_feed()
{
	FList	*selection = get_feed_selection();
	int	f_index;

	if (get_ticker_env()->selection_mode == MULTIPLE) {
		if ((f_index = f_list_index(selection)) == 0)
			set_feed_selection(f_list_last(selection));
		else if (f_index > 0)
			set_feed_selection(f_list_nth(selection, f_index /* (rank = f_index + 1) - 1 */));
		get_ticker_env()->reload_rq = TRUE;
	}
}

/* Always reload -> OK like that ? */
void first_feed()
{
	if (get_ticker_env()->selection_mode == MULTIPLE) {
		set_feed_selection(f_list_first(get_feed_selection()));
		get_ticker_env()->reload_rq = TRUE;
	}
}

/* Always reload -> OK like that ? */
void last_feed()
{
	if (get_ticker_env()->selection_mode == MULTIPLE) {
		set_feed_selection(f_list_last(get_feed_selection()));
		get_ticker_env()->reload_rq = TRUE;
	}
}

gboolean previous_feed()
{
	FList	*selection = get_feed_selection();
	int	f_index;

	if (get_ticker_env()->selection_mode == MULTIPLE) {
		if ((f_index = f_list_index(selection)) > 1 || f_index == 0) {
			if (f_index > 1)
				set_feed_selection(f_list_nth(selection, f_index - 1));
			else if (f_index == 0)
				set_feed_selection(f_list_nth(selection, f_list_count(selection) - 1));
			get_ticker_env()->reload_rq = TRUE;
			return TRUE;
		} else if (f_index == 1)
			info_win_wait("This is already the first feed", INFO_WIN_WAIT_TIMEOUT);
	}
	return FALSE;
}

gboolean next_feed()
{
	FList	*selection = get_feed_selection();
	int	f_index;

	if (get_ticker_env()->selection_mode == MULTIPLE) {
		if ((f_index = f_list_index(selection)) > 0) {
			get_ticker_env()->reload_rq = TRUE;
			return TRUE;
		} else
			info_win_wait("This is already the last feed", INFO_WIN_WAIT_TIMEOUT);
	}
	return FALSE;
}

/*
 * - Single mode: (re)load resrc->id = text file if it exists or rss feed if valid.
 * - Multiple mode: load sequentially all selected feeds in feed list.
 *
 * In case of url: load_resource() -> get_feed() -> parse_rss20/rss10/atom_xml_file().
 */
int load_resource(Resource *resrc, FList *feed_selection)
{
	int	rss_status, exit_status;

	if (resrc->fp != NULL) {
		fclose(resrc->fp);
		resrc->fp = NULL;
	}
	if (resrc->fp_extra != NULL) {
		fclose(resrc->fp_extra);
		resrc->fp_extra = NULL;
	}
	resrc->xml_dump[0] = '\0';
	resrc->type = RESRC_TYPE_UNDETERMINED;
	resrc->format = RSS_FORMAT_UNDETERMINED;

	if (get_ticker_env()->selection_mode == MULTIPLE) {
		if (IS_FLIST(feed_selection)) {
			str_n_cpy(resrc->id, feed_selection->url, FILE_NAME_MAXLEN);
			if (IS_FLIST(feed_selection->next))
				feed_selection = feed_selection->next;
			else if (IS_FLIST(f_list_first(feed_selection)))
				feed_selection = f_list_first(feed_selection);
			set_feed_selection(feed_selection);
		} else  {
			warning(TRUE, 1, "No feed selected - ",
				"Switching to single selection mode");
			get_ticker_env()->selection_mode = SINGLE;
			resrc->id[0] = '\0';
			/*warning(TRUE, 1, "No feed selection available - "
				"Switching to single selection mode");
			get_ticker_env()->selection_mode = SINGLE;
			resrc->id[0] = '\0';*/
		}
	}

	if (resrc->id[0] == '\0') {
		resrc->type = RESRC_UNSPECIFIED;
		fprintf(STD_ERR, "No resource specified\n");
		exit_status = RESOURCE_UNSPECIFIED;
	} else {
		/*
		 * Is there an existing file ?
		 */
		if ((resrc->fp = g_fopen(resrc->id, "rb")) == NULL) {
			/*
			 * If not, have we a valid rss url instead ?
			 */
			if ((rss_status = get_feed(resrc, get_params())) != OK) {
				if (rss_status != FEED_UNPARSABLE && rss_status != FEED_EMPTY &&\
						rss_status != RSS_NO_ITEM_ELEMENT &&\
						rss_status != ATOM_NO_ENTRY_ELEMENT &&\
						rss_status != FEED_DOWNLOAD_ERROR &&\
						rss_status != FEED_FORMAT_ERROR &&\
						rss_status != CONNECT_TOO_MANY_ERRORS) {
					warning(get_ticker_env()->selection_mode == MULTIPLE,
						2, "get_feed() error ", itoa2(rss_status));
				}
				resrc->id[0] ='\0';
				if (rss_status == CONNECT_TOO_MANY_ERRORS)
					exit_status = CONNECT_TOO_MANY_ERRORS;
				else
					exit_status = RESOURCE_INVALID;
			} else {
				resrc->type = RESRC_URL;
				exit_status = OK;
			}
		} else {
			resrc->type = RESRC_FILE;
			exit_status = OK;
		}
	}

	if (exit_status == OK) {
		if (format_resource(resrc, XML_DUMP) == RESOURCE_FORMAT_ERROR) {
			big_error(RESOURCE_FORMAT_ERROR, 1, "format_resource() error");
			exit_status = RESOURCE_FORMAT_ERROR;
		}
		if (resrc->type == RESRC_URL)
			if (format_resource(resrc, XML_DUMP_EXTRA) == RESOURCE_FORMAT_ERROR) {
				big_error(RESOURCE_FORMAT_ERROR, 1, "format_resource() error");
				exit_status = RESOURCE_FORMAT_ERROR;
			}
	}
	return exit_status;
}

/*
 * Strip html tags,
 * then 'translate' html entities,
 * then check for lines longer than LINE_MAXLEN and insert newline's if any,
 * then resrc->fp refers to a "formatted" file.
 */
int format_resource(Resource *resrc, const char *file_name)
{
	FILE		*fp, *tmp_fp;
	char		*str1, *str2, entity[16];
	size_t		str1_size = 8 * 1024;
	int		is_inside_tag = FALSE, c, i;
	gchar		*p;

	if (strcmp(file_name, XML_DUMP) == 0)
		fp = resrc->fp;
	else if (strcmp(file_name, XML_DUMP_EXTRA) == 0)
		fp = resrc->fp_extra;
	else
		return RESOURCE_FORMAT_ERROR;

	/* Strip html tags.*/
	tmp_fp = open_new_datafile_with_name(TMP1, "wb+");
	i = 0;
	while ((c = fgetc(fp)) != (int)EOF) {
		if (get_params()->strip_html_tags == 'y') {
			if (c == '<') {
				is_inside_tag = TRUE;
				i = 0;
			} else if (c == '>')
				is_inside_tag = FALSE;
		} else
			is_inside_tag = FALSE;
		/* A little bit confusing here... */
		if (!is_inside_tag && !(get_params()->strip_html_tags == 'y' && c == '>'))
			fputc((int)c, tmp_fp);
		else if (i++ < 1)
				fputc((int)' ', tmp_fp);
	}
	fclose(fp);
	fp = tmp_fp;
	fseek(fp, 0, SEEK_SET);

	/* 'Translate' html entities */
	tmp_fp = open_new_datafile_with_name(TMP2, "wb+");
	while ((c = fgetc(fp)) != (int)EOF) {
		if (c == '&') {
			i = 0;
			do {
				entity[i++] = (char)c;
				if (c == ';')
					break;
			} while ((c = fgetc(fp)) != (int)EOF && i < 16);
			entity[i] = '\0';
			i = 0;
			while (html_entity[i] != NULL) {
				if (strcmp(entity, html_entity[i]) == 0 ||\
						strcmp(entity, html_entity[i + 1]) == 0) {
					fputs(html_entity[i + 2], tmp_fp);
					break;
				} else
					i += 3;
			}
		} else
			fputc((int)c, tmp_fp);
	}
	fclose(fp);
	fp = tmp_fp;
	fseek(fp, 0, SEEK_SET);

	/* Newline stuff */
	tmp_fp = open_new_datafile_with_name(file_name, "wb+");
	str1 = malloc2(str1_size * sizeof(char));
	str2 = l_str_new(NULL);
#ifndef G_OS_WIN32
	while (getline(&str1, &str1_size, fp) != -1)
#else
	while (fgets(str1, str1_size, fp) != NULL)
#endif
		str2 = l_str_cat(str2, str1);
	free2(str1);
	/* XML_DMP: Check for lines longer than LINE_MAXLEN and insert newline's if any */
	if (strcmp(file_name, XML_DUMP) == 0) {
		i = 0;
		for (p = str2; *p != '\0'; p = g_utf8_find_next_char(p, NULL)) {
			if (*p == '\n')
				i = 0;
			/*
			 * If the word we are parsing isn't longer than WORD_MAXLEN
			 * chars, we replace the 1st ' ' found after it with '\n',
			 * otherwise the word is cut off.
			 */
			if (i >= LINE_MAXLEN - WORD_MAXLEN)
				if (i >= LINE_MAXLEN || g_unichar_isspace(*p)) {
					*p = '\n';
					i = 0;
					continue;
				}
			i++;
		}
	} else {
		/* WHAT TODO FOR: XML_DUMP_EXTRA ?
		 * Remove '\n' ? Not sure so doing nothing */
		/*for (p = str2; *p != '\0'; p = g_utf8_find_next_char(p, NULL))
			if (*p == '\n')*/
	}
	fputs(str2, tmp_fp);
	l_str_free(str2);
	fclose(fp);
	fp = tmp_fp;
	fseek(fp, 0, SEEK_SET);

	if (strcmp(file_name, XML_DUMP) == 0)
		resrc->fp = fp;
	else if (strcmp(file_name, XML_DUMP_EXTRA) == 0)
		resrc->fp_extra = fp;

	/* Remove tmp files */
	g_remove(get_datafile_full_name_from_name(TMP1));
	g_remove(get_datafile_full_name_from_name(TMP2));
	return OK;
}

/*
 * *** File names, paths and dirs stuff ***
 */
void set_tickr_icon_to_dialog(GtkWindow *dialog)
{
	gtk_window_set_icon_from_file(dialog, get_imagefile_full_name_from_name(TICKR_ICON), NULL);
}

/* Open/create file in data dir from name.
 * Data dir = config files, not images files.
 */
FILE *open_new_datafile_with_name(const char *name, const char *mode_str)
{
	char	file_name[FILE_NAME_MAXLEN + 1];
	FILE	*fp;

	str_n_cpy(file_name, get_datafile_full_name_from_name(name), FILE_NAME_MAXLEN);
	if ((fp = g_fopen(file_name, mode_str)) == NULL) {
		if (mode_str[0] == 'w')
			big_error(CREATE_FILE_ERROR, 4, "Error creating file ", file_name, ": ", strerror(errno));
		else if (mode_str[0] == 'r')
			big_error(OPEN_FILE_ERROR, 4, "Error opening file ", file_name, ": ", strerror(errno));
	}
	return fp;
}

/* Get full path and name for file in data dir from name.
 * Data dir = config files, not images files.
 */
char *get_datafile_full_name_from_name(const char *name)
{
	static char	file_name[FILE_NAME_MAXLEN + 1];

	snprintf(file_name, FILE_NAME_MAXLEN + 1 - 2, "%s%c%s", get_datadir_full_path(), SEPARATOR_CHAR, name);
	if (get_instance_id() != 0)
		str_n_cat(file_name, itoa2(get_instance_id()), 2);
	return file_name;
}

/* Now, image files */
char *get_imagefile_full_name_from_name(const char *name)
{
	static char	file_name[FILE_NAME_MAXLEN + 1];

#ifndef G_OS_WIN32
	snprintf(file_name, FILE_NAME_MAXLEN + 1, "%s%c%s", IMAGES_PATH, SEPARATOR_CHAR, name);
#else
	snprintf(file_name, FILE_NAME_MAXLEN + 1, "%s%c%s%c%s",
			get_progfiles_dir(), SEPARATOR_CHAR, IMAGES_PATH, SEPARATOR_CHAR, name);
#endif
	return file_name;
}

/* Data dir = config files, not images files */
char *get_datadir_full_path()
{
	static char	full_path[TMPSTR_SIZE + 1];

#ifndef G_OS_WIN32
	snprintf(full_path, TMPSTR_SIZE + 1, "%s%c%s", usr_home_dir(), SEPARATOR_CHAR, TICKR_DIR_NAME);
#else
	snprintf(full_path, TMPSTR_SIZE + 1, "%s%c%s", get_appdata_dir_utf8(), SEPARATOR_CHAR, TICKR_DIR_NAME);
#endif
	return full_path;
}

char *usr_home_dir()
{
	static char	str[TMPSTR_SIZE+ 1] = "";

#ifndef G_OS_WIN32
	return str_n_cpy(str, getpwuid(getuid())->pw_dir, TMPSTR_SIZE);
#endif
	/* For win32 version, no home dir - See win32 specific funcs in libetm. */
	return str;
}

/* Fix non-ascii (for instance cyrillic) user name in app data dir issue on win32 .*/
#ifdef G_OS_WIN32
const char *get_appdata_dir_utf8()
{
	static char	file_name[FILE_NAME_MAXLEN + 1];
	gchar		*filename_utf8;

	if ((filename_utf8 = g_utf16_to_utf8((const gunichar2 *)get_appdata_dir_w(), FILE_NAME_MAXLEN, NULL, NULL, NULL)) != NULL) {
		str_n_cpy(file_name, filename_utf8, FILE_NAME_MAXLEN);
		g_free(filename_utf8);
		return (const char *)file_name;
	} else
		return NULL;
}
#endif
