/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2011
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

#if USE_GUI
static gint esc_key_pressed(GtkWidget *dialog2, GdkEventKey *event, gpointer unused)
{
	unused = unused;
	if (event->keyval == GDK_Escape) {
		gtk_dialog_response(GTK_DIALOG(dialog2), GTK_RESPONSE_CLOSE);
		return TRUE;
	} else
		return FALSE;
}

static void force_quit_dialog(GtkWidget *dialog2)
{
	gtk_dialog_response(GTK_DIALOG(dialog2), GTK_RESPONSE_CLOSE);
}

/*
 * pick new file dialog win
 */
void open_new_txt_file(Resource *resrc)
{
	TickerEnv	*env = get_ticker_env();
	GtkWidget	*dialog;
	char		*file_name2, tmp[TMPSTR_SIZE + 1];
	int		resource_flag;
	Resource	resrc_bak;
	int		selection_mode_bak = get_ticker_env()->selection_mode;

	gtk_window_set_keep_above(GTK_WINDOW(env->win), FALSE);

	dialog = gtk_file_chooser_dialog_new(
			"Text File Picker", GTK_WINDOW(env->win),
			GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CLOSE,
			GTK_STOCK_OK, GTK_RESPONSE_OK,
			NULL);

	set_news_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	gtk_widget_show_all(dialog);
	/*
	 * backup last valid opened resource (if any)
	 */
	str_n_cpy(resrc_bak.id, resrc->id, FILE_NAME_MAXLEN);

	while (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
		file_name2 = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
		str_n_cpy(resrc->id, file_name2, FILE_NAME_MAXLEN);
		g_free(file_name2);

		env->suspend_rq = TRUE;
		get_ticker_env()->selection_mode = SINGLE;
		if ((resource_flag = load_resource(resrc, NULL)) == OK) {
			env->reload_rq = TRUE;
			env->suspend_rq = FALSE;
			break;
		} else {
			snprintf(tmp, TMPSTR_SIZE + 1, "load_resource() error %d", resource_flag);
			info_win(APP_NAME" - Error", tmp, INFO_ERROR, FALSE);
			str_n_cpy(resrc->id, resrc_bak.id, FILE_NAME_MAXLEN);
			get_ticker_env()->selection_mode = selection_mode_bak;
			load_resource(resrc, NULL);
			env->suspend_rq = FALSE;
		}
	}
	gtk_widget_destroy(dialog);
	check_main_win_always_on_top();
}

static void get_layout_dims(GtkWidget *view, int *layout_width, int *layout_height, char *str)
{
	PangoLayout	*p_layout;

	p_layout = pango_layout_new(gtk_widget_get_pango_context(view));
	pango_layout_set_text(p_layout, str, -1);
	pango_layout_get_pixel_size2(p_layout, layout_width, layout_height);
	if (p_layout != NULL)
		g_object_unref(p_layout);
}

void help_win()
{
#ifndef G_OS_WIN32
# define FONT			"DejaVu Sans Mono 8" /* 8 9 10 ok */
#else
# define FONT			"Courier New 8"
#endif
#define HLPWIN_FG_COLOR		"#000000ff"	/* black */
#define HLPWIN_BG_COLOR		"#ffffffff"	/* white */

	GtkWidget		*dialog, *sc_win, *view;
	GtkTextBuffer		*buf;
	PangoFontDescription	*font_des;
	GdkColor		fg_color, bg_color;
	guint16			fgc_alpha, bgc_alpha;
	char			*txt;
	int			width, height;
	int			i;

	gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

	dialog = gtk_dialog_new_with_buttons(
			APP_NAME" Quick Help", GTK_WINDOW(get_ticker_env()->win),
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
			NULL);

	set_news_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	sc_win = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sc_win), GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), sc_win);
	gtk_container_set_border_width(GTK_CONTAINER(sc_win), 5);

	txt = l_str_new(NULL);
	buf = gtk_text_buffer_new(NULL);

	view = gtk_text_view_new_with_buffer(buf);
	gtk_text_view_set_editable(GTK_TEXT_VIEW(view), FALSE);
	gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(view), FALSE);
	font_des = pango_font_description_from_string(FONT);
	gtk_widget_modify_font(view, font_des);

	get_gdk_color_from_hexstr(HLPWIN_FG_COLOR, &fg_color, &fgc_alpha);
	get_gdk_color_from_hexstr(HLPWIN_BG_COLOR, &bg_color, &bgc_alpha);
	/*
	 * not using gtk_widget_modify_fg/bg() but gtk_widget_modify_text/base()
	 * because it's specifically for editable text
	 */
	if (FALSE) {	/* one possible extra setting = 'same colors as ticker's' = 'yes' */
		gtk_widget_modify_text(view, GTK_STATE_NORMAL, &(get_params()->fg_color));
		gtk_widget_modify_base(view, GTK_STATE_NORMAL, &(get_params()->bg_color));
	} else if (TRUE) { /* widget-level-defined */
		gtk_widget_modify_text(view, GTK_STATE_NORMAL, &fg_color);
		gtk_widget_modify_base(view, GTK_STATE_NORMAL, &bg_color);
	} else if (FALSE) { /* same as label inside window */
		gtk_widget_modify_text(view, GTK_STATE_NORMAL,
			(const GdkColor *)&(gtk_widget_get_style(get_ticker_env()->win)->fg[GTK_STATE_NORMAL]));
		gtk_widget_modify_base(view, GTK_STATE_NORMAL,
			(const GdkColor *)&(gtk_widget_get_style(get_ticker_env()->win)->bg[GTK_STATE_NORMAL]));
	}

	gtk_text_view_set_left_margin(GTK_TEXT_VIEW(view), 15);
	gtk_text_view_set_right_margin(GTK_TEXT_VIEW(view), 15);
	gtk_container_add(GTK_CONTAINER(sc_win), view);

	txt = l_str_cat(txt, "\n");
	for (i = 0; get_help_str0()[i] != NULL; i++)
		txt = l_str_cat(txt, get_help_str0()[i]);
	for (i = 0; get_help_str1()[i] != NULL; i++)
		txt = l_str_cat(txt, get_help_str1()[i]);

	gtk_text_buffer_set_text(buf, txt, -1);
	get_layout_dims(view, &width, &height, txt);
	l_str_free(txt);
	if (width < 400)
		width = 400;
	else if (width > 800)
		width = 800;
	if (height > 450)
		height = 450;
	gtk_widget_set_size_request(sc_win, width + 30 + 15 + 15, height);

	gtk_widget_show_all(dialog);
	gtk_dialog_run(GTK_DIALOG(dialog));

	gtk_widget_destroy(dialog);
	check_main_win_always_on_top();
}

static gint license_win()
{
	GtkWidget	*dialog, *label;
	char		*txt;
	int		i;

	gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

	dialog = gtk_dialog_new_with_buttons(
			APP_NAME" License", GTK_WINDOW(get_ticker_env()->win),
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
			NULL);

	set_news_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_container_set_border_width(GTK_CONTAINER(dialog), 15);
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_NONE);

	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	txt = l_str_new(NULL);
#ifndef G_OS_WIN32
	txt = l_str_cat(txt, "<small>");
#endif
	for (i = 0; get_license_str1()[i] != NULL; i++)
		txt = l_str_cat(txt, get_license_str1()[i]);
	txt = l_str_cat(txt, "<a href=\"http://www.gnu.org/licenses/\">");
	txt = l_str_cat(txt, get_license_str2());
	txt = l_str_cat(txt, "</a>");
#ifndef G_OS_WIN32
	txt = l_str_cat(txt, "</small>");
#endif
	txt = l_str_cat(txt, "\n");

	label = gtk_label_new(txt);
	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), label);
	l_str_free(txt);

	gtk_widget_show_all(dialog);
	gtk_dialog_run(GTK_DIALOG(dialog));

	gtk_widget_destroy(dialog);
	check_main_win_always_on_top();
	return TRUE;
}

void about_win()
{
	GtkWidget	*dialog, *table, *image = NULL, *label[4];
	GtkWidget	*license_but, *close_but;
	gint		response;

	gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

	dialog = gtk_dialog_new_with_buttons(
			"About "APP_NAME, GTK_WINDOW(get_ticker_env()->win),
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			NULL);

	license_but = gtk_dialog_add_button(GTK_DIALOG(dialog), "License", GTK_RESPONSE_NONE);
	close_but = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);

	set_news_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	g_signal_connect(G_OBJECT(license_but), "clicked", G_CALLBACK(license_win), NULL);
	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	table = gtk_table_new(7, 1, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 5);
	gtk_container_set_border_width(GTK_CONTAINER(table), 15);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);

	image = gtk_image_new_from_file(get_imagefile_full_name_from_name(NEWS_LOGO));
	gtk_table_attach_defaults(GTK_TABLE(table), image, 0, 1, 0, 1);

	label[0] = gtk_label_new("\n<b>"APP_NAME" version "APP_VERSION_NUMBER" - RSS Feed Reader</b>");
	gtk_label_set_use_markup(GTK_LABEL(label[0]), TRUE);
	gtk_table_attach_defaults(GTK_TABLE(table), label[0], 0, 1, 1, 2);
	label[1] = gtk_label_new("<small>Copyright (C) Emmanuel Thomas-Maurin 2009-2011</small>");
	gtk_label_set_use_markup(GTK_LABEL(label[1]), TRUE);
	gtk_table_attach_defaults(GTK_TABLE(table), label[1], 0, 1, 2, 3);
	label[2] = gtk_label_new("<a href=\"mailto:"APP_SUPPORT_EMAIL"\">"APP_SUPPORT_EMAIL"</a>");
	gtk_label_set_use_markup(GTK_LABEL(label[2]), TRUE);
	gtk_table_attach_defaults(GTK_TABLE(table), label[2], 0, 1, 3, 4);
	label[3] = gtk_label_new("<a href=\""APP_WEBSITE"\">"APP_WEBSITE"</a>");
	gtk_label_set_use_markup(GTK_LABEL(label[3]), TRUE);
	gtk_table_attach_defaults(GTK_TABLE(table), label[3], 0, 1, 4, 5);

	gtk_widget_show_all(dialog);
	while ((response = gtk_dialog_run(GTK_DIALOG(dialog))) != GTK_RESPONSE_CLOSE && \
		response != GTK_RESPONSE_CLOSE);

	gtk_widget_destroy(dialog);
	check_main_win_always_on_top();
}

/*
 * info, warning or error
 * === should format txt in case it's a long string ===
 */
void info_win(const char *title, const char *txt, info_type info, gboolean selectable)
{
	GtkWidget	*dialog, *table, *image = NULL, *label[2];
	gint		layout_width, layout_height;

	if (get_params()->disable_popups == 'y')
		return;
	gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

	dialog = gtk_dialog_new_with_buttons(
			title, GTK_WINDOW(get_ticker_env()->win),
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_OK, GTK_RESPONSE_OK,
			NULL);

	set_news_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	table = gtk_table_new(1, 3, FALSE);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);
	gtk_container_set_border_width(GTK_CONTAINER(table), 0);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);

	if (info == INFO) {
		/* better without any image if info only so do nothing here */
		/*image = gtk_image_new_from_stock(GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_DIALOG);*/
	} else if (info == INFO_WARNING) {
		image = gtk_image_new_from_stock(GTK_STOCK_DIALOG_WARNING, GTK_ICON_SIZE_DIALOG);
		gtk_table_attach_defaults(GTK_TABLE(table), image, 0, 1, 0, 1);
	} else if (info == INFO_ERROR) {
		image = gtk_image_new_from_stock(GTK_STOCK_DIALOG_ERROR, GTK_ICON_SIZE_DIALOG);
		gtk_table_attach_defaults(GTK_TABLE(table), image, 0, 1, 0, 1);
	}

	label[0] = gtk_label_new(txt);
	/* better if disabled
	gtk_label_set_use_markup(GTK_LABEL(label[0]), TRUE);*/
	pango_layout_get_pixel_size(gtk_label_get_layout(GTK_LABEL(label[0])),
		&layout_width, &layout_height);
	if (info == INFO) {
		if (layout_width > get_ticker_env()->screen_w - 60)
			gtk_widget_set_size_request(label[0], get_ticker_env()->screen_w - 60, -1);
	} else {
		if (layout_width > get_ticker_env()->screen_w - 120)	/* extra 60 for image width */
			gtk_widget_set_size_request(label[0], get_ticker_env()->screen_w - 120, -1);
	}
	if (selectable)
		gtk_label_set_selectable(GTK_LABEL(label[0]), TRUE);
	gtk_table_attach_defaults(GTK_TABLE(table), label[0], 1, 2, 0, 1);
	label[1] = gtk_label_new("");
	gtk_table_attach_defaults(GTK_TABLE(table), label[1], 2, 3, 0, 1);

	gtk_widget_show_all(dialog);
	gtk_dialog_run(GTK_DIALOG(dialog));

	gtk_widget_destroy(dialog);
	check_main_win_always_on_top();
}

/* delay in ms */
void info_win_wait(const char *txt, int delay)
{
	GtkWidget	*win;
	GtkWidget	*table, *image, *label[2];
	gint		layout_width, layout_height;

	if (get_params()->disable_popups == 'y')
		return;
	gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

	win = gtk_window_new(GTK_WINDOW_POPUP);

	set_news_icon_to_dialog(GTK_WINDOW(win));
	gtk_window_set_position(GTK_WINDOW(win), INFO_WIN_WAIT_POS);

	table = gtk_table_new(1, 3, FALSE);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);
	gtk_container_set_border_width(GTK_CONTAINER(table), 0);
	gtk_container_add(GTK_CONTAINER(GTK_WINDOW(win)), table);

	image = gtk_image_new_from_stock(GTK_STOCK_DIALOG_WARNING, GTK_ICON_SIZE_DIALOG);
	gtk_table_attach_defaults(GTK_TABLE(table), image, 0, 1, 0, 1);

	label[0] = gtk_label_new(txt);
	/* better if disabled
	gtk_label_set_use_markup(GTK_LABEL(label[0]), TRUE);*/
	pango_layout_get_pixel_size(gtk_label_get_layout(GTK_LABEL(label[0])),
		&layout_width, &layout_height);
	if (layout_width > get_ticker_env()->screen_w - 120)
		gtk_widget_set_size_request(label[0], get_ticker_env()->screen_w - 120, -1);
	gtk_table_attach_defaults(GTK_TABLE(table), label[0], 1, 2, 0, 1);
	label[1] = gtk_label_new("");
	gtk_table_attach_defaults(GTK_TABLE(table), label[1], 2, 3, 0, 1);

	gtk_widget_show_all(win);
	/* so that win will popup immediately */
	while (gtk_events_pending())
		gtk_main_iteration();

#ifndef G_OS_WIN32
	sleep(delay / 1000);
#else
	Sleep(delay);
#endif
	gtk_widget_destroy(win);
	check_main_win_always_on_top();
}

int question_win(const char *txt)
{
	GtkWidget	*dialog, *table, *image, *label[2];
	gint		response;

	gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

	dialog = gtk_dialog_new_with_buttons(
			APP_NAME, GTK_WINDOW(get_ticker_env()->win),
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_NO, GTK_RESPONSE_NO,
			GTK_STOCK_YES, GTK_RESPONSE_YES,
			NULL);

	set_news_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
	/* why always no?
	gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_NO);*/

	table = gtk_table_new(1, 3, FALSE);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);
	gtk_container_set_border_width(GTK_CONTAINER(table), 0);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);

	image = gtk_image_new_from_stock(GTK_STOCK_DIALOG_QUESTION,
							GTK_ICON_SIZE_DIALOG);
	gtk_table_attach_defaults(GTK_TABLE(table), image, 0, 1, 0, 1);
	label[0] = gtk_label_new(txt);
	gtk_table_attach_defaults(GTK_TABLE(table), label[0], 1, 2, 0, 1);
	label[1] = gtk_label_new("");
	gtk_table_attach_defaults(GTK_TABLE(table), label[1], 2, 3, 0, 1);

	gtk_widget_show_all(dialog);
	response = gtk_dialog_run(GTK_DIALOG(dialog));

	gtk_widget_destroy(dialog);
	check_main_win_always_on_top();
	return ((response == GTK_RESPONSE_YES) ? YES : NO);
}

void win_with_spinner(spinner_win_mode mode, const char *txt)
{
	static GtkWidget	*win;
	GtkWidget		*hbox, *label;
#ifndef G_OS_WIN32
#ifndef WITHOUT_GTK_SPINNER
	static GtkWidget	*spinner;
	GtkWidget		*space_label;
#endif
#endif
	if (mode == WIN_WITH_SPINNER_OPEN) {
		gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

		win = gtk_window_new(GTK_WINDOW_POPUP);
		set_news_icon_to_dialog(GTK_WINDOW(win));
		gtk_window_set_position(GTK_WINDOW(win), GTK_WIN_POS_CENTER);
		gtk_window_set_keep_above(GTK_WINDOW(win), TRUE);

		hbox = gtk_hbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(GTK_WINDOW(win)), hbox);
#ifndef G_OS_WIN32
#ifndef WITHOUT_GTK_SPINNER
		spinner = gtk_spinner_new();
		space_label = gtk_label_new("    ");
#endif
#endif
		label = gtk_label_new(txt);
#ifndef G_OS_WIN32
#ifndef WITHOUT_GTK_SPINNER
		gtk_container_add(GTK_CONTAINER(GTK_BOX(hbox)), spinner);
		gtk_container_add(GTK_CONTAINER(GTK_BOX(hbox)), space_label);
#endif
#endif
		gtk_container_add(GTK_CONTAINER(GTK_BOX(hbox)), label);
		gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(hbox)), 15);

		gtk_widget_show_all(win);
		/* so that win will popup immediately */
		while (gtk_events_pending())
			gtk_main_iteration();
#ifndef G_OS_WIN32
#ifndef WITHOUT_GTK_SPINNER
		gtk_spinner_start(GTK_SPINNER(spinner));
#endif
#endif
	} else if (mode == WIN_WITH_SPINNER_CLOSE) {
#ifndef G_OS_WIN32
#ifndef WITHOUT_GTK_SPINNER
		gtk_spinner_stop(GTK_SPINNER(spinner));
#endif
#endif
		gtk_widget_destroy(win);
		check_main_win_always_on_top();
	} else
		warning("win_with_spinner():", "Invalid mode", "", "", FALSE);
}
#endif
