/*
 *	NEWS - GTK-based RSS Ticker - Copyright (C) Emmanuel Thomas-Maurin 2009-2011
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "news.h"
#include "news_html_entities.h"

#ifdef G_OS_WIN32
extern FILE	*stdout_fp, *stderr_fp;
#endif
static int	counter = 0;

/* these funcs do nothing in single selection mode -> is that ok? */
void current_feed()
{
	if (get_ticker_env()->selection_mode == MULTIPLE) {
		if (counter > 0 )
			counter--;
		get_ticker_env()->reload_rq = TRUE;
	}
}

void first_feed()
{
	if (get_ticker_env()->selection_mode == MULTIPLE) {
		counter = 0;
		get_ticker_env()->reload_rq = TRUE;
	}
}

void last_feed()
{
	if (get_ticker_env()->selection_mode == MULTIPLE) {
		while (get_selected_url_list()[counter] != NULL)
			counter++;
		counter--;
		get_ticker_env()->reload_rq = TRUE;
	}
}

gboolean previous_feed()
{
	if (get_ticker_env()->selection_mode == MULTIPLE) {
		if (counter >= 2 ) {
			counter -= 2;
			get_ticker_env()->reload_rq = TRUE;
			return TRUE;
		} else {
			info_win_wait("This is already the first Feed", INFO_WIN_WAIT_TIMEOUT);
			return FALSE;
		}
	} else
		return FALSE;
}

gboolean next_feed()
{
	if (get_ticker_env()->selection_mode == MULTIPLE) {
		if (get_selected_url_list()[counter] != NULL) {
			get_ticker_env()->reload_rq = TRUE;
			return TRUE;
		} else {
			info_win_wait("This is already the last Feed", INFO_WIN_WAIT_TIMEOUT);
			return FALSE;
		}
	} else
		return FALSE;
}

/*
 * - single mode: (re)load text file if it exists or rss feed if valid
 * - multiple mode: (re)load sequentially all rss feeds in selected_url[]
 *
 * in case of url: load_resource() -> get_feed() -> parse_rss20/rss10/atom_xml_file()
 *
 * (we should check/convert file content to utf8 asap and before parsing
 * xml file)
 */
int load_resource(Resource *resrc, char **selected_urls)
{
#ifdef G_OS_WIN32
	gchar		*win32_filename;
#endif
	int		rss_status, exit_status;
	char		tmp[256];

	if (resrc->fp != NULL) {
		fclose(resrc->fp);
		resrc->fp = NULL;
	}
	resrc->xml_dump[0] = '\0';
	resrc->type = RESRC_TYPE_UNDETERMINED;
	resrc->format = RSS_FORMAT_UNDETERMINED;

	if (get_ticker_env()->selection_mode == MULTIPLE) {
		if (selected_urls[counter] == NULL) {
			counter = 0;
			/* we must re-init selected_urls before loading */
			selected_urls = init_selected_url_list();
			if (load_selected_url_list((char **)selected_urls) != OK) {
				warning("No feed selection available", "-",
					"Switching to single selection mode", "", TRUE);
				get_ticker_env()->selection_mode = SINGLE;
			}
		}
		if (selected_urls[counter] != NULL)
			str_n_cpy(resrc->id, selected_urls[counter], FILE_NAME_MAXLEN);
		else
			resrc->id[0] = '\0';
		counter++;
	}

	if (resrc->id[0] == '\0') {
		resrc->type = RESRC_UNSPECIFIED;
		fprintf(STD_ERR, "No resource specified\n");
		exit_status = RESOURCE_UNSPECIFIED;

	} else {
		/*
		 * is there an existing file?
		 */
#ifndef G_OS_WIN32
		if ((resrc->fp = g_fopen(resrc->id, "rb")) == NULL) {
#else
		win32_filename = g_win32_locale_filename_from_utf8(resrc->id);
		if ((resrc->fp = g_fopen(win32_filename, "rb")) == NULL) {
			g_free(win32_filename);
#endif
			/*
			 * if not, have we a valid rss url instead?
			 */
			if ((rss_status = get_feed(resrc, get_params())) != OK) {
				if (rss_status != RSS_UNPARSABLE && rss_status != RSS_EMPTY &&\
						rss_status != RSS_NO_ITEM_ELEMENT &&\
						rss_status != ATOM_NO_ENTRY_ELEMENT &&\
						rss_status != RSS_DOWNLOAD_ERROR &&\
						rss_status != RSS_FORMAT_ERROR &&\
						rss_status != CONNECT_TOO_MANY_ERRORS) {
					snprintf(tmp, 256, "get_feed() error %d", rss_status);
					warning(tmp, "", "", "",
						get_ticker_env()->selection_mode == MULTIPLE);
				}
				resrc->id[0] ='\0';
				if (rss_status == CONNECT_TOO_MANY_ERRORS)
					exit_status = CONNECT_TOO_MANY_ERRORS;
				else
					exit_status = RESOURCE_INVALID;
			} else {
				resrc->type = RESRC_URL;
				exit_status = OK;
			}
		} else {
			resrc->type = RESRC_FILE;
			exit_status = OK;
		}
	}

	if (exit_status == OK) {
		if (format_resource(resrc) == RESOURCE_FORMAT_ERROR) {
			big_error(RESOURCE_FORMAT_ERROR,
				"format_resource() error", "", "", "");
			exit_status = RESOURCE_FORMAT_ERROR;
		}
	}
	return exit_status;
}

/*
 * strip html tags
 * then check for lines longer than LINE_MAXLEN and insert newline's if any
 * then resrc->fp refers to a "formatted" file
 */
int format_resource(Resource *resrc)
{
	FILE		*tmp_fp;
	char		*str1, *str2, entity[16];
	size_t		str1_size = 8 * 1024;
	int		is_inside_tag = FALSE, c, i;
	gchar		*p;

	/* strip html tags */
	tmp_fp = open_new_datafile_with_name(TMP1, "wb+");
	i = 0;
	while ((c = fgetc(resrc->fp)) != (int)EOF) {
		if (get_params()->strip_html_tags == 'y') {
			if (c == '<') {
				is_inside_tag = TRUE;
				i = 0;
			} else if (c == '>')
				is_inside_tag = FALSE;
		} else
			is_inside_tag = FALSE;
		/* a little bit confusing here... */
		if (!is_inside_tag && !(get_params()->strip_html_tags == 'y' && c == '>'))
			fputc((int)c, tmp_fp);
		else if (i++ < 1)
				fputc((int)' ', tmp_fp);
	}
	fclose(resrc->fp);
	resrc->fp = tmp_fp;
	fseek(resrc->fp, 0, SEEK_SET);

	/* 'translate' html entities */
	tmp_fp = open_new_datafile_with_name(TMP2, "wb+");
	while ((c = fgetc(resrc->fp)) != (int)EOF) {
		if (c == '&') {
			i = 0;
			do {
				entity[i++] = (char)c;
				if (c == ';')
					break;
			} while ((c = fgetc(resrc->fp)) != (int)EOF && i < 16);
			entity[i] = '\0';
			i = 0;
			while (html_entity[i] != NULL) {
				if (strcmp(entity, html_entity[i]) == 0 ||\
						strcmp(entity, html_entity[i + 1]) == 0) {
					fputs(html_entity[i + 2], tmp_fp);
					break;
				} else
					i += 3;
			}
		} else
			fputc((int)c, tmp_fp);
	}
	fclose(resrc->fp);
	resrc->fp = tmp_fp;
	fseek(resrc->fp, 0, SEEK_SET);

	/* check for lines longer than LINE_MAXLEN and insert newline's if any */
	tmp_fp = open_new_datafile_with_name(XMLDUMP, "wb+");
	str1 = malloc2(str1_size * sizeof(char));
	str2 = l_str_new(NULL);
#ifndef G_OS_WIN32
	while (getline(&str1, &str1_size, resrc->fp) != -1)
#else
	while (fgets(str1, str1_size, resrc->fp) != NULL)
#endif
		str2 = l_str_cat(str2, str1);
	free2(str1);
	i = 0;
	for (p = str2; *p != '\0'; p = g_utf8_find_next_char(p, NULL)) {
		if (*p == '\n')
			i = 0;
		/*
		 * if the word we are parsing isn't longer than WORD_MAXLEN
		 * chars, we replace the 1st ' ' found after it with a '\n'
		 * otherwise the word is cut off
		 */
		if (i >= LINE_MAXLEN - WORD_MAXLEN)
			if (i >= LINE_MAXLEN || g_unichar_isspace(*p)) {
				*p = '\n';
				i = 0;
				continue;
			}
		i++;
	}
	fputs(str2, tmp_fp);
	l_str_free(str2);
	fclose(resrc->fp);
	resrc->fp = tmp_fp;
	fseek(resrc->fp, 0, SEEK_SET);
	return OK;
}

/*
 * *** file names, paths and dirs stuff ***
 */
void set_news_icon_to_dialog(GtkWindow *dialog)
{
	gtk_window_set_icon_from_file(dialog, get_imagefile_full_name_from_name(NEWS_ICON), NULL);
}

/* open/create file in data dir from name
 * data dir = config files, not images files
 */
FILE *open_new_datafile_with_name(const char *name, const char *mode_str)
{
	char	file_name[FILE_NAME_MAXLEN + 1];
#ifdef G_OS_WIN32
	gchar	*win32_filename;
#endif
	FILE	*fp;

	str_n_cpy(file_name, get_datafile_full_name_from_name(name), FILE_NAME_MAXLEN);
#ifdef G_OS_WIN32
	if ((win32_filename = g_win32_locale_filename_from_utf8(file_name)) != NULL) {
		str_n_cpy(file_name, win32_filename, FILE_NAME_MAXLEN);
		g_free(win32_filename);
	}
#endif
	if ((fp = g_fopen(file_name, mode_str)) == NULL) {
		if (mode_str[0] == 'w')
			big_error(CREATE_FILE_ERROR, "Error creating file:", file_name, " - ", strerror(errno));
		else if (mode_str[0] == 'r')
			big_error(OPEN_FILE_ERROR, "Error opening file:", file_name, " - ", strerror(errno));
	}
	return fp;
}

/* get full path and name for file in data dir from name
 * data dir = config files, not images files
 */
const char *get_datafile_full_name_from_name(const char *name)
{
	static char	file_name[FILE_NAME_MAXLEN + 1];
#ifdef G_OS_WIN32
	gchar		*win32_filename;
#endif

	snprintf(file_name, FILE_NAME_MAXLEN + 1 - 2, "%s%c%s", get_datadir_full_path(), SEPARATOR_CHAR, name);
	if (get_instance_id() != 0)
		str_n_cat(file_name, itoa2(get_instance_id()), 2);
#ifdef G_OS_WIN32
	if ((win32_filename = g_win32_locale_filename_from_utf8(file_name)) != NULL) {
		str_n_cpy(file_name, win32_filename, FILE_NAME_MAXLEN);
		g_free(win32_filename);
	}
#endif
	return (const char *)file_name;
}

/* now, image files */
const char *get_imagefile_full_name_from_name(const char *name)
{
	static char	file_name[FILE_NAME_MAXLEN + 1];
#ifdef G_OS_WIN32
	gchar		*win32_filename;
#endif

#ifndef G_OS_WIN32
	snprintf(file_name, FILE_NAME_MAXLEN + 1, "%s%c%s", IMAGES_PATH, SEPARATOR_CHAR, name);
#else
	snprintf(file_name, FILE_NAME_MAXLEN + 1, "%s%c%s%c%s",
			get_progfiles_dir(), SEPARATOR_CHAR, IMAGES_PATH, SEPARATOR_CHAR, name);
	if ((win32_filename = g_win32_locale_filename_from_utf8(file_name)) != NULL) {
		str_n_cpy(file_name, win32_filename, FILE_NAME_MAXLEN);
		g_free(win32_filename);
	}
#endif
	return (const char *)file_name;
}

/* data dir = config files, not images files */
const char *get_datadir_full_path()
{
	static char	full_path[TMPSTR_SIZE + 1];

#ifndef G_OS_WIN32
	snprintf(full_path, TMPSTR_SIZE + 1, "%s%c%s", usr_home_dir(), SEPARATOR_CHAR, NEWS_DIR_NAME);
#else
	snprintf(full_path, TMPSTR_SIZE + 1, "%s%c%s", get_appdata_dir(), SEPARATOR_CHAR, NEWS_DIR_NAME);
#endif
	return (const char *)full_path;
}

const char *usr_home_dir()
{
	static char	str[TMPSTR_SIZE+ 1] = "";

#ifndef G_OS_WIN32
	return (const char *)str_n_cpy(str, getpwuid(getuid())->pw_dir, TMPSTR_SIZE);
#endif
	/* for win32 version, no home dir - see win32 specific funcs in libetm */
	return (const char *)str;
}
