/*
 *	NEWS - GTK-based RSS Ticker - Copyright (C) Emmanuel Thomas-Maurin 2009-2011
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "news.h"

#if USE_GUI

static char	url_array[NURLMAX][FILE_NAME_MAXLEN + 1];
static char	*url[NURLMAX + 1];
static char	selected_url_array[NURLMAX][FILE_NAME_MAXLEN + 1];
static char	*selected_url[NURLMAX + 1];

/* (re)assign all pointers to all array elements as pointer list is NULL
 * terminated - don't modify array content */
char **init_url_list()
{
	int	i = 0;

	for (i = 0; i < NURLMAX; i++)
		url[i] = url_array[i];
	url[NURLMAX] = NULL;
	return (char **)url;
}

/* (re)assign all pointers to all array elements as pointer list is NULL
 * terminated - don't modify array content */
char **init_selected_url_list()
{
	int	i = 0;

	for (i = 0; i < NURLMAX; i++)
		selected_url[i] = selected_url_array[i];
	selected_url[NURLMAX] = NULL;
	return (char **)selected_url;
}

char **get_url_list()
{
	return (char **)url;
}

char **get_selected_url_list()
{
	return (char **)selected_url;
}

/*
 * fill selected_url_array[NURLMAX][FILE_NAME_MAXLEN + 1] with lines from
 * url list file

 * entry format in url list file:
 *	["*" (selected) or "-" (unselected) + url [+ ">" + title] + '\n']
 *
 * entry max length = FILE_NAME_MAXLEN
 * see also: (UN)SELECTED_URL_CHAR in news.h
 *
 * we pass a reference to (char **)selected_p_url[NURLMAX + 1] which has been
 * set as p_selected_url[i] = selected_url_array[i] and p_selected_url[NURLMAX] = NULL
 */
int load_selected_url_list(char **p_selected_url)
{
	char	tmp[FILE_NAME_MAXLEN + 1];
	int	i, j;

	if (load_url_list(p_selected_url) == OK) {
		for (i = 0; i < NURLMAX && p_selected_url[i] != NULL; i++) {
			if (p_selected_url[i][0] == SELECTED_URL_CHAR) {
				str_n_cpy(tmp, str_without_title(p_selected_url[i]), FILE_NAME_MAXLEN);
				for (j = 0; j < FILE_NAME_MAXLEN; j++) {
					if (tmp[j] == '\n') {
						tmp[0] = '\0';
						break;
					}
				}
				str_n_cpy(p_selected_url[i], tmp + 1, FILE_NAME_MAXLEN);
			} else
				p_selected_url[i][0] = '\0';
		}
		p_selected_url[i] = NULL;
		sort_url_list(p_selected_url, 0);	/* tmp + 1 -> we compare after 1 char (* or -) */
		if (p_selected_url[0] != NULL)
			return OK;
		else
			return SELECTION_EMPTY;
	} else {
		p_selected_url[0] = NULL;
		return SELECTION_ERROR;
	}
}

/*
 * fill url_array[NURLMAX][FILE_NAME_MAXLEN + 1] with lines from url list file
 *
 * entry format in url list file:
 *	["*" (selected) or "-" (unselected) + url [+ ">" + title] + '\n']
 *
 * entry max length = FILE_NAME_MAXLEN
 * see also: (UN)SELECTED_URL_CHAR in news.h
 *
 * we pass a reference to (char **)p_url[NURLMAX + 1] which has been set as
 * p_url[i] = url_array[i] and p_url[NURLMAX] = NULL
 */
int load_url_list(char **p_url)
{
	char	*listfname, listfname1[TMPSTR_SIZE + 1], listfname2[TMPSTR_SIZE + 1];
	FILE	*fp;
	char	*tmp1, tmp2[TMPSTR_SIZE + 1];
	size_t	tmp1_size = FILE_NAME_MAXLEN + 1;
	int	i, j;

	str_n_cpy(listfname1, get_datafile_full_name_from_name(URLLIST), TMPSTR_SIZE);
#ifndef G_OS_WIN32
	snprintf(listfname2, TMPSTR_SIZE + 1, "%s%c%s", INSTALL_PATH, SEPARATOR_CHAR, URLLIST);
#else
	snprintf(listfname2, TMPSTR_SIZE + 1, "%s%c%s%c%s",
		get_progfiles_dir(), SEPARATOR_CHAR, NEWS_DIR_NAME, SEPARATOR_CHAR, URLLIST);
#endif
	listfname = listfname1;

	if ((fp = g_fopen(listfname, "rb")) == NULL) {
		if (question_win("No URL list has been saved yet. Use sample one ?") == YES)
			listfname = listfname2;
		else
			return LOAD_URL_LIST_ERROR;
	}
	if ((fp = g_fopen(listfname, "rb")) != NULL) {
		tmp1 = malloc2(tmp1_size * sizeof(char));
		i = 0;
#ifndef G_OS_WIN32
		while (getline(&tmp1, &tmp1_size, fp) != EOF && i < NURLMAX) {
#else
		while (fgets(tmp1, tmp1_size, fp) != NULL && i < NURLMAX) {
#endif
			for (j = 0; j < FILE_NAME_MAXLEN; j++)
				if (tmp1[j] == '\n') {
					tmp1[j] = '\0';
					break;
				}
			if (tmp1[0] != '\0')
				str_n_cpy(p_url[i++], tmp1, FILE_NAME_MAXLEN);
		}
		p_url[i] = NULL;
		if (i >= NURLMAX) {
			snprintf(tmp2, TMPSTR_SIZE + 1, "Max number of URL's (= %d) is reached.\n"
				"(You may set NURLMAX to a higher value in news.h and recompile.)",
				NURLMAX);
				warning(tmp2, "", "", "", FALSE);
		}
		free2(tmp1);
		fclose(fp);
		return OK;
	} else {
		warning("Can't load URL list:", listfname, "-", strerror(errno),
			FALSE);
		return LOAD_URL_LIST_ERROR;
	}
}

/*
 * sort list then remove empty and duplicated entries
 * we compare after <shift> chars
 */
void sort_url_list(char **p_url, int shift)
{
	char	*p_url2[NURLMAX + 1], *tmp, empty_str[] = "";
	int	list_len, min, i, j;

	/*
	 * copy p_url[] to p_url2[]
	 */
	for (i = 0; i < NURLMAX && p_url[i] != NULL; i++)
		p_url2[i] = p_url[i];
	list_len = i;
	/*
	 * sort array (selection sort)
	 */
	for (i = 0; i < list_len; i++) {
		min = i;
		for (j = i  + 1; j < list_len; j++) {
			/* + shift because we compare after shift chars */
			if (strcmp(p_url2[min] + shift, p_url2[j] + shift) > 0)
				min = j;
		}
		tmp = p_url2[i];
		p_url2[i] = p_url2[min];
		p_url2[min] = tmp;
	}
	/*
	 * remove duplicated and empty entries
	 * as we can have one entry with title (url + title) and the same without title
	 * (url only), we use str_without_title() to compare strings because we want to
	 * keep entries with title and remove similar entries without title
	 */
	for (i = 0; i < list_len; i++)
		for (j = i  + 1; j < list_len; j++) {
			/* + shift because we compare after shift chars */
			if (strcmp(str_without_title(p_url2[i] + shift), str_without_title(p_url2[j] + shift)) == 0) {
				if (!str_has_title(p_url2[i]))
					p_url2[i] = empty_str;
				else
					p_url2[j] = empty_str;
			}
		}

	for (i = 0, j = 0; i < list_len; i++)
		if (p_url2[i] != empty_str)
			p_url[j++] = p_url2[i];
	p_url[j] = NULL;
}

int save_url_list(char **p_url)
{
	char	*listfname, tmp[FILE_NAME_MAXLEN + 1];
	FILE	*fp;
	int	i, j;

	listfname = (char *)get_datafile_full_name_from_name(URLLIST);
	if ((fp = g_fopen(listfname, "wb")) != NULL) {
		for (i = 0, j = 0; p_url[i] != NULL && j < NURLMAX; i++) {
			str_n_cpy(tmp, p_url[i], FILE_NAME_MAXLEN);
			if (tmp[0] != '\0') {
				fprintf(fp, "%s\n", tmp);
				j++;
			}
		}
		fclose(fp);
		return OK;
	} else {
		warning("Can't save URL list:", listfname, "-", strerror(errno),
			FALSE);
		return SAVE_URL_LIST_ERROR;
	}
}

/*
 * actually set entry as empty instead of removing it
 * -> need to sort list afterwards
 */
void remove_url_from_list(char **p_url, int rank)
{
	if (rank >= 0 && rank < NURLMAX)
		p_url[rank][0] = '\0';
}

gboolean str_has_title(char *str)
{
	int	i;

	for (i = 0; i < FILE_NAME_MAXLEN && str[i] != '\0'; i++)
		if (str[i]  == TITLE_TAG_CHAR)
			return TRUE;
	return FALSE;
}

char *str_without_title(char *str)
{
	static char	str2[16][FILE_NAME_MAXLEN + 1];
	static int	count = -1;
	int		i;

	count++;
	count &= 15;

	str_n_cpy(str2[count], str, FILE_NAME_MAXLEN);
	for (i = 0; i < FILE_NAME_MAXLEN && str2[count][i] != '\0'; i++)
		if (str2[count][i]  == TITLE_TAG_CHAR) {
			str2[count][i] = '\0';
			break;
		}
	return str2[count];
}
#endif
