/* -*- mode: c; c-file-style: "gnu" -*-
 * rfc2817-client.c -- RFC2817 test client
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file rfc2817-client.c
 * This is a very simplistic RFC2817 test client. All it does is to
 * connect to a server and issue a request with an Upgrade: TLS/1.0
 * header. There is no failiure handling - if the remote end does not
 * support this functionality, this program will fail miserably.
 *
 * On the other hand, if it does, this little client should just work.
 */

#include "system.h"
#include "compat/compat.h"

#if defined (HAVE_ARGP_H) && defined(HAVE_ARGP_PARSE)
#include <argp.h>
#endif
#include <sys/types.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <gnutls/gnutls.h>

#define MAX_BUF 16384 /**< Maximum buffer size. */

char *rfc2817_port = "80"; /**< Port to connect to. */
char *rfc2817_server = "127.0.0.1"; /**< Server to connect to. */
char *rfc2817_path = "/"; /**< Path to request. */

/** @internal Bug report address.
 * @note Used by the argp suite.
 */
const char *argp_program_bug_address = "<algernon@bonehunter.rulez.org>";
/** @internal Program version.
 * @note Used by the argp suite.
 */
const char *argp_program_version = "rfc2817-client (Thy/" VERSION ")";

/** @internal Argp options.
 */
static struct argp_option rfc2817_options[] = {
  {"host", 'h', "HOST", 0, "Remote host to connect to", 1},
  {"port", 'p', "PORT", 0, "Remote port to connect to", 1},
  {0, 0, 0, 0, NULL, 0}
};
static error_t _rfc2817_parse_opt (int key, char *arg,
				   struct argp_state *state);
/** @internal Argp parser definition.
 */
static struct argp argp =
  {rfc2817_options, _rfc2817_parse_opt, "[PATH]",
   "rfc2817-client -- Simple RFC2817 client\v"
   "The optional PATH argument will be the requested path.",
   NULL, NULL, NULL};

/** @internal Parse one option.
 * See the argp docs for details.
 */
static error_t
_rfc2817_parse_opt (int key, char *arg, struct argp_state *state)
{
  switch (key)
    {
    case 'h':
      rfc2817_server = bhc_strdup (arg);
      break;
    case 'p':
      rfc2817_port = bhc_strdup (arg);
      break;
    case ARGP_KEY_ARG:
      rfc2817_path = bhc_strdup (arg);
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

/** The main code of the client.
 */
int
main (int argc, char **argv)
{
  int err, ret;
  int sd;
  static struct sockaddr_in sa;
  char buffer[MAX_BUF + 1];
  char *sb;
  gnutls_session session;
  gnutls_certificate_credentials xcred;
  const int comp_prio[] = { GNUTLS_COMP_ZLIB, GNUTLS_COMP_LZO,
			    GNUTLS_COMP_NULL, 0 };

  /* Parse options */
  argp_parse (&argp, argc, argv, 0, NULL, NULL);

  /* Connect to the remote host */
  sd = socket (AF_INET, SOCK_STREAM, 0);
  sa.sin_family = AF_INET;
  sa.sin_port = htons (atoi (rfc2817_port));
  inet_pton (AF_INET, rfc2817_server, &sa.sin_addr);

  err = connect (sd, (struct sockaddr *)&sa, sizeof (sa));
  if (err < 0)
    {
      fprintf (stderr, "Connect error\n");
      exit (1);
    }

  /* Issue the request with an Upgrade: header */
  asprintf (&sb, "GET %s HTTP/1.1\r\n"
	    "Host: %s\r\n"
	    "User-Agent: %s\r\n"
	    "Upgrade: TLS/1.0\r\n"
	    "Connection: upgrade\r\n\r\n", rfc2817_path, rfc2817_server,
	    argp_program_version);
  fputs (sb, stdout);
  write (sd, sb, strlen (sb));
  free (sb);

  /* Wait the reply. Only a HTTP 101 header comes in, which we assume
     to be smaller than MAX_BUF (16k), so only one read is done
     here. */
  ret = read (sd, buffer, MAX_BUF);
  if (ret >= 0)
    buffer[ret] = '\0';
  fputs (buffer, stdout);

  /* Initialise GnuTLS stuff */
  gnutls_global_init ();
  gnutls_certificate_allocate_credentials (&xcred);
  gnutls_init (&session, GNUTLS_CLIENT);
  gnutls_set_default_priority (session);
  gnutls_compression_set_priority (session, comp_prio);

  gnutls_credentials_set (session, GNUTLS_CRD_CERTIFICATE, xcred);

  gnutls_transport_set_ptr (session, (gnutls_transport_ptr)((long)sd));

  gnutls_server_name_set (session, GNUTLS_NAME_DNS, rfc2817_server,
			  strlen (rfc2817_server));

  /* Handshake */
  ret = gnutls_handshake (session);
  if (ret < 0)
    {
      fprintf (stderr, "*** Handshake failed\n");
      gnutls_perror (ret);
    }
  else
    {
      ret = 1;

      /* Send the closing request. */
      asprintf (&sb, "OPTIONS %s HTTP/1.1\r\n"
		"Host: %s\r\n"
		"User-Agent: %s\r\n"
		"Connection: close\r\n\r\n", rfc2817_path, rfc2817_server,
		argp_program_version);
      fputs (sb, stdout);
      gnutls_record_send (session, sb, strlen (sb));
      free (sb);
    }

  /* And read everything until the connection gets closed. We can do
     that, since we explictly disabled keep-alive with Connection:
     close. */
  while (ret > 0)
    {
      ret = gnutls_record_recv (session, buffer, MAX_BUF);
      if (ret < 0)
	fprintf (stderr, "*** Error: %s\n", gnutls_strerror (ret));
      else if (ret > 0)
	{
	  buffer[ret] = '\0';
	  fputs (buffer, stdout);
	}
    };

  if (ret >= 0)
    {
      fputs ("\n", stdout);
      gnutls_bye (session, GNUTLS_SHUT_RDWR);
    }

  shutdown (sd, SHUT_RDWR);
  close (sd);

  gnutls_deinit (session);
  gnutls_certificate_free_credentials (xcred);

  gnutls_global_deinit ();

  return 0;
}

/* arch-tag: a5a844ba-540f-4ca9-b382-12f92e5c952e */
