/* -*- mode: c; c-file-style: "gnu" -*-
 * src/authorisers/htaccess.c -- .htaccess Authoriser for Thy.
 * Copyright (C) 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy-Auth.
 *
 * Thy-Auth is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Thy-Auth is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file htaccess.c
 * This file implements Apache-style .htaccess authorisation support
 * for Thy-Auth. See the apache docs for details.
 */

#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "compat/compat.h"
#include "methods/passwd.h"
#include "misc.h"

/** @defgroup ta_htaccess thy-auth-htaccess
 * @{
 */

/** Structure to hold info retrieved from a .htaccess file.
 * This structure is used for tracking .htaccess infos.
 */
struct
{
  char *name; /**< Value of AuthName. */
  char *userfile; /**< Value of AuthUserFile. */
  char *groupfile; /**< Value of AuthGroupFile. */
  char *ruser; /**< Value of "Require user". */
  char *rgroup; /**< Value of "Require group". */
  int auth; /**< Zero if no authentication is needed, 1 otherwise. */
  int valid_user; /**< Non-zero if "Require valid-user" was seen, zero
		     otherwise. */
} _thy_htaccess;

/** @internal Reset #_thy_htaccess.
 * Frees all elements, and resets #_thy_htaccess.
 */
static void
_thy_auth_htaccess_free (void)
{
  free (_thy_htaccess.name);
  free (_thy_htaccess.userfile);
  free (_thy_htaccess.groupfile);
  free (_thy_htaccess.ruser);
  free (_thy_htaccess.rgroup);

  memset (&_thy_htaccess, 0, sizeof (_thy_htaccess));

  _thy_htaccess.auth = -1;
  _thy_htaccess.valid_user = -1;
}

/** @internal Read a .htaccess file.
 * Reads a given .htaccess file, and puts all the relevant info from
 * it into #_thy_htaccess.
 *
 * @param fn is the filename to parse.
 *
 * @returns Zero on success, -1 otherwise.
 *
 * @note Always call _thy_auth_htaccess_free() before calling this!
 */
static int
_thy_auth_htaccess_read (const char *fn)
{
  FILE *f;
  char *line = NULL;
  int len;
  size_t n = 0;

  _thy_auth_htaccess_free ();

  f = fopen (fn, "r");
  if (f == NULL)
    return -1;

  while ((len = getline (&line, &n, f)) != -1)
    {
      int l = 0;

      /* Remove the trailing \r or \n. */
      while (l < len)
	{
	  if (line[l] == '\r' || line[l] == '\n')
	    line[l] = '\0';
	  l++;
	}

      if (!strncasecmp (line, "AuthType ", 9))
	{
	  if (!strncasecmp (&line[9], "Basic", 5))
	    _thy_htaccess.auth = 1;
	  else if (!strncasecmp (&line[9], "None", 4))
	    _thy_htaccess.auth = 0;
	  continue;
	}
      if (!strncasecmp (line, "AuthUserFile ", 13))
	{
	  _thy_htaccess.userfile = bhc_strdup (&line[13]);
	  continue;
	}
      if (!strncasecmp (line, "AuthGroupFile ", 14))
	{
	  _thy_htaccess.groupfile = bhc_strdup (&line[14]);
	  continue;
	}
      if (!strncasecmp (line, "Require ", 8))
	{
	  if (!strncasecmp (&line[8], "valid-user", 10))
	    _thy_htaccess.valid_user = 1;
	  else if (!strncasecmp (&line[8], "group ", 6))
	    _thy_htaccess.rgroup = bhc_strdup (&line[8+6]);
	  else if (!strncasecmp (&line[8], "user ", 5))
	    _thy_htaccess.ruser = bhc_strdup (&line[8+5]);
	  continue;
	}
      if (!strncasecmp (line, "AuthName \"", 10))
	{
	  _thy_htaccess.name = bhc_strdup (&line[10]);
	  l = strlen (_thy_htaccess.name);
	  _thy_htaccess.name[l-1] = '\0';
	}
    }

  fclose (f);
  return 0;
}

void
_thy_auth_do_one_auth (int argc, char **argv)
{
  char *user, *pw, *realm, *htaccess;
  int i = 0;

  if (argc < 5)
    {
      thy_auth_respond ("0\n", 2);
      return;
    }

  /* Get stuff */
  user = argv[0];
  pw = argv[1];
  realm = argv[2];
  htaccess = argv[4];

  if (_thy_auth_htaccess_read (htaccess) != 0)
    {
      thy_auth_respond ("0\n", 2);
      return;
    }

  if (_thy_htaccess.auth != 1 || _thy_htaccess.name == NULL)
    {
      thy_auth_respond ("0\n", 2);
      return;
    }

  if (_thy_htaccess.ruser)
    {
      if (strcmp (_thy_htaccess.ruser, user))
	{
	  thy_auth_respond ("0\n", 2);
	  return;
	}
    }
  if (_thy_htaccess.rgroup && _thy_htaccess.groupfile)
    {
      char **members =
	thy_auth_passwd_group_get (_thy_htaccess.groupfile,
				   _thy_htaccess.rgroup);
      size_t found = 0;

      if (!members)
	{
	  thy_auth_respond ("0\n", 2);
	  return;
	}

      i = 0;
      while (members[i])
	{
	  if (!strcmp (members[i], user))
	    found = 1;
	  free (members[i]);
	  i++;
	}
      free (members);

      if (!found)
	{
	  thy_auth_respond ("0\n", 2);
	  return;
	}
    }

  /* Authenticate */
  i = thy_auth_passwd_auth (_thy_htaccess.userfile, user, pw);

  /* Response */
  thy_auth_respond ((i == 0) ? "1\n" : "0\n", 2);
}

void
_thy_auth_do_one_check (int argc, char **argv)
{
  char *tmp;
  int len;

  if (argc < 1)
    {
      thy_auth_respond ("0\n", 2);
      return;
    }

  if (_thy_auth_htaccess_read (argv[0]) != 0)
    {
      thy_auth_respond ("0\n", 2);
      return;
    }

  if (_thy_htaccess.auth != 1 || _thy_htaccess.name == NULL)
    {
      thy_auth_respond ("0\n", 2);
      return;
    }

  thy_auth_respond ("1\n", 2);

  /* Write out the length & the string itself. */
  len = asprintf (&tmp, "%05u\n%s\n", strlen (_thy_htaccess.name),
		  _thy_htaccess.name);
  thy_auth_respond (tmp, len);

}

/** thy-auth-htaccess main core.
 */
int
main (int argc, char **argv)
{
  signal (SIGPIPE, SIG_IGN);

  _thy_auth_htaccess_free ();

  for (;;)
    thy_auth_do_one ();

  return 0;
}

/** @} */

/* arch-tag: af35774d-4b1f-4922-b1d2-18e059c11ef7 */
