/***************************************************************************
                           th-fake-dialog.c
                           ----------------
    begin                : Wed Feb 16 2005
    copyright            : (C) 2005 by Tim-Philipp Müller
    email                : tim centricular net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This class tries to emulate a GtkDialog's handling of buttons and     *
 *    responses, but is not a top-level window like GtkDialog is. Most     *
 *    code is taken straight from gtkdialog.c                              *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "th-fake-dialog.h"
#include "th-marshal.h"

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <string.h>

/* signals */

enum 
{
	RESPONSE,
	LAST_SIGNAL
};

struct _ThFakeDialogPrivate
{
	gint      default_response_id;
	gboolean  default_response_id_set;
};

/* using helper struct so we can represent response id 0 */
typedef struct _ResponseData ResponseData;
struct _ResponseData
{
	gint response_id;
};

static void             fake_dialog_class_init    (ThFakeDialogClass *klass);

static void             fake_dialog_instance_init (ThFakeDialog *fd);

static void             fake_dialog_finalize      (GObject *object);

static void             fake_dialog_set_default_response_real (ThFakeDialog *dialog, 
                                                               gint response_id);

static ResponseData    *fake_dialog_get_response_data (GtkWidget *widget,
                                                       gboolean   create);

/* variables */

static GObjectClass  *parent_class; /* NULL */

static guint          fake_dialog_signals[LAST_SIGNAL];

/***************************************************************************
 *
 *   fake_dialog_update_spacings
 *
 ***************************************************************************/

static void
fake_dialog_update_spacings (ThFakeDialog *fdialog)
{
	GtkWidget *widget;
	gint       content_area_border;
	gint       button_spacing;
	gint       action_area_border;
  
	widget = GTK_WIDGET (gtk_dialog_new ());

	gtk_widget_style_get (widget,
	                      "content_area_border", &content_area_border,
	                      "button_spacing", &button_spacing,
	                      "action_area_border", &action_area_border,
	                      NULL);

	gtk_container_set_border_width (GTK_CONTAINER (fdialog->vbox),
	                                content_area_border);

	gtk_box_set_spacing (GTK_BOX (fdialog->action_area),
	                     button_spacing);

	gtk_container_set_border_width (GTK_CONTAINER (fdialog->action_area),
	                                action_area_border);

	gtk_widget_destroy (widget);
}

/***************************************************************************
 *
 *   fake_dialog_style_set
 *
 ***************************************************************************/

static void
fake_dialog_style_set (GtkWidget *fdialog, GtkStyle *prev_style)
{
	fake_dialog_update_spacings (TH_FAKE_DIALOG (fdialog));
}


/***************************************************************************
 *
 *   fake_dialog_class_init
 *
 ***************************************************************************/

static void
fake_dialog_class_init (ThFakeDialogClass *klass)
{
	GtkWidgetClass *widget_class;
	GObjectClass   *object_class; 

	object_class = G_OBJECT_CLASS (klass);
	widget_class = GTK_WIDGET_CLASS (klass);
	
	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = fake_dialog_finalize;
	widget_class->style_set = fake_dialog_style_set;

	fake_dialog_signals[RESPONSE] =
		g_signal_new ("response",
		              G_OBJECT_CLASS_TYPE (klass),
		              G_SIGNAL_RUN_LAST,
		              G_STRUCT_OFFSET (ThFakeDialogClass, response),
		              NULL, NULL,
		              th_marshal_VOID__INT,
		              G_TYPE_NONE, 1,
		              G_TYPE_INT);
}

/***************************************************************************
 *
 *   fake_dialog_parent_set
 *
 *   big hack to make the default response stuff work even 
 *     if fake dialog doesn't call fake_dialog_run()
 *
 ***************************************************************************/

static void
fake_dialog_parent_set (ThFakeDialog *fdialog, GtkObject *old_parent, GtkWidget *widget)
{
	GtkWidget *parent;

	parent = gtk_widget_get_parent (GTK_WIDGET (fdialog));

	if (parent == NULL || !fdialog->priv->default_response_id_set)
		return;

	while (gtk_widget_get_parent (parent))
		parent = gtk_widget_get_parent (parent);

	if (GTK_IS_WINDOW (parent))
	{
		fake_dialog_set_default_response_real (fdialog, fdialog->priv->default_response_id);
	}
	else
	{
		g_warning ("ThFakeDialog's default response id stuff needs to be\n"
		           "\timplemented properly. In the mean time, please call\n"
		           "th_fake_dialog_set_default_response() after the fake\n"
		           "dialog has been embedded into a GtkWindow.\n");
	}
}

/***************************************************************************
 *
 *   fake_dialog_instance_init
 *
 ***************************************************************************/

static void
fake_dialog_instance_init (ThFakeDialog *fdialog)
{
	fdialog->priv = g_new0 (ThFakeDialogPrivate, 1);

	fdialog->priv->default_response_id = 0;
	fdialog->priv->default_response_id_set = FALSE;

	fdialog->action_area = gtk_hbutton_box_new ();

	gtk_button_box_set_layout (GTK_BUTTON_BOX (fdialog->action_area),
	                           GTK_BUTTONBOX_END);  

	gtk_box_pack_end (GTK_BOX (fdialog), fdialog->action_area, FALSE, TRUE, 0);

	fdialog->separator = gtk_hseparator_new ();
	gtk_box_pack_end (GTK_BOX (fdialog), fdialog->separator, FALSE, TRUE, 0);

	gtk_widget_show (fdialog->separator);
	gtk_widget_show (fdialog->action_area);

	fdialog->vbox = GTK_WIDGET (fdialog); /* for compatibility */

	g_signal_connect (fdialog, "parent-set", 
	                  G_CALLBACK (fake_dialog_parent_set),
	                  fdialog);
}

/***************************************************************************
 *
 *   fake_dialog_finalize
 *
 ***************************************************************************/

static void
fake_dialog_finalize (GObject *object)
{
	ThFakeDialog *fake_dialog;

	fake_dialog = (ThFakeDialog*) object;

	memset (fake_dialog->priv, 0xab, sizeof (ThFakeDialogPrivate));
	g_free (fake_dialog->priv);
	fake_dialog->priv = NULL;

	/* chain up */
	parent_class->finalize (object);
}


/***************************************************************************
 *
 *   th_fake_dialog_get_type
 *
 ***************************************************************************/

GType
th_fake_dialog_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThFakeDialogClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) fake_dialog_class_init,
			NULL, NULL,
			sizeof (ThFakeDialog),
			0,
			(GInstanceInitFunc) fake_dialog_instance_init
		};

		type = g_type_register_static (GTK_TYPE_VBOX, "ThFakeDialog", &info, G_TYPE_FLAG_ABSTRACT);
	}

	return type;
}

/***************************************************************************
 *
 *   th_fake_dialog_response
 *
 ***************************************************************************/

void
th_fake_dialog_response (ThFakeDialog *fdialog, gint response_id)
{
  g_return_if_fail (TH_IS_FAKE_DIALOG (fdialog));

  g_signal_emit (fdialog,
		 fake_dialog_signals[RESPONSE],
		 0,
		 response_id);
}

/***************************************************************************
 *
 *   fake_dialog_get_response_data
 *
 ***************************************************************************/

static ResponseData *
fake_dialog_get_response_data (GtkWidget *widget, gboolean create)
{
	ResponseData *rdata;

	rdata = g_object_get_data (G_OBJECT (widget), "th-fake-dialog-response-data");

	if (rdata == NULL && create)
	{
		rdata = g_new0 (ResponseData, 1);
      
		g_object_set_data_full (G_OBJECT (widget),
		                        "th-fake-dialog-response-data",
		                        rdata,
		                        g_free);
	}

	return rdata;
}

/***************************************************************************
 *
 *   th_fake_dialog_set_default_response
 *
 *   Sets the last widget in the dialog's action area with the given 
 *    response_id as the default widget for the dialog. Pressing "Enter" 
 *    normally activates the default widget.
 *
 ***************************************************************************/

void
th_fake_dialog_set_default_response (ThFakeDialog *dialog, gint response_id)
{
	g_return_if_fail (TH_IS_FAKE_DIALOG (dialog));

	dialog->priv->default_response_id = response_id;
	dialog->priv->default_response_id_set = TRUE;
}

/***************************************************************************
 *
 *   fake_dialog_set_default_response_real
 *
 *   We can't do this right away in th_fake_dialog_set_default_response(),
 *    because we need to be in a GtkWindow first for this to work, which
 *    we are not when we are created (not so much a problem for the real
 *    GtkDialog, because it is a GtkWindow itself).
 *
 ***************************************************************************/

static void
fake_dialog_set_default_response_real (ThFakeDialog *dialog, gint response_id)
{
	GList *children;
	GList *tmp_list;

	g_return_if_fail (TH_IS_FAKE_DIALOG (dialog));

	children = gtk_container_get_children (GTK_CONTAINER (dialog->action_area));

	tmp_list = children;
	while (tmp_list != NULL)
	{
		GtkWidget *widget = tmp_list->data;
		ResponseData *rd = fake_dialog_get_response_data (widget, FALSE);

		if (rd && rd->response_id == response_id)
		{
			GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_DEFAULT);
			gtk_widget_grab_default (widget);
			gtk_widget_grab_focus (widget);
		}
	    
		tmp_list = g_list_next (tmp_list);
	}

	g_list_free (children);
}

/***************************************************************************
 *
 *   fake_dialog_action_widget_activated
 *
 ***************************************************************************/

static void
fake_dialog_action_widget_activated (GtkWidget *widget, ThFakeDialog *fdialog)
{
  ResponseData *rdata;
  gint          response_id;
  
  g_return_if_fail (TH_IS_FAKE_DIALOG (fdialog));

  response_id = GTK_RESPONSE_NONE;
  
  rdata = fake_dialog_get_response_data (widget, TRUE);

  g_assert (rdata != NULL);
  
  response_id = rdata->response_id;

  th_fake_dialog_response (fdialog, response_id);
}

/***************************************************************************
 *
 *   th_fake_dialog_add_action_widget
 *
 ***************************************************************************/

void
th_fake_dialog_add_action_widget (ThFakeDialog *fdialog,
                                  GtkWidget    *child,
                                  gint          response_id)
{
	ResponseData *rdata;
	gint          signal_id = 0;
  
	g_return_if_fail (TH_IS_FAKE_DIALOG (fdialog));
	g_return_if_fail (GTK_IS_WIDGET (child));

	rdata = fake_dialog_get_response_data (child, TRUE);

	rdata->response_id = response_id;

	if (GTK_IS_BUTTON (child))
		signal_id = g_signal_lookup ("clicked", GTK_TYPE_BUTTON);
	else
		signal_id = GTK_WIDGET_GET_CLASS (child)->activate_signal != 0;

	if (signal_id)
	{
		GClosure *closure;

		closure = g_cclosure_new_object (G_CALLBACK (fake_dialog_action_widget_activated),
		                                 G_OBJECT (fdialog));

		g_signal_connect_closure_by_id (child,
		                                signal_id,
		                                0,
		                                closure,
		                                FALSE);
	}
	else g_warning ("Only 'activatable' widgets can be packed into the action area of a ThFakeDialog");

	gtk_box_pack_end (GTK_BOX (fdialog->action_area),
	                  child, FALSE, TRUE, 0);
  
	if (response_id == GTK_RESPONSE_HELP)
		gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (fdialog->action_area), child, TRUE);
}

/***************************************************************************
 *
 *   th_fake_dialog_add_button
 *
 ***************************************************************************/

GtkWidget*
th_fake_dialog_add_button (ThFakeDialog  *fdialog,
                           const gchar   *button_text,
                           gint           response_id)
{
	GtkWidget *button;
  
	g_return_val_if_fail (TH_IS_FAKE_DIALOG (fdialog), NULL);
	g_return_val_if_fail (button_text != NULL, NULL);

	button = gtk_button_new_from_stock (button_text);

	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  
	gtk_widget_show (button);
  
	th_fake_dialog_add_action_widget (fdialog,
	                                  button,
                                    response_id);

	return button;
}

typedef struct
{
  GtkDialog *dialog;
  gint response_id;
  gboolean destroyed;
  GMainLoop *loop;
} RunInfo;

static void
shutdown_loop (RunInfo *ri)
{
	if (g_main_loop_is_running (ri->loop))
		g_main_loop_quit (ri->loop);
}

static void
run_unmap_handler (ThFakeDialog *dialog, gpointer data)
{
	RunInfo *ri = data;

	shutdown_loop (ri);
}

static void
run_response_handler (ThFakeDialog *fdialog,
                      gint          response_id,
                      gpointer      data)
{
	RunInfo *ri;

	ri = data;

	ri->response_id = response_id;

	shutdown_loop (ri);
}

static void
run_destroy_handler (ThFakeDialog *fdialog, gpointer data)
{
	RunInfo *ri = data;
    
	shutdown_loop (ri);

	ri->destroyed = TRUE;
}

/***************************************************************************
 *
 *   th_fake_dialog_run
 *
 ***************************************************************************/

gint
th_fake_dialog_run (ThFakeDialog *fdialog)
{
	RunInfo ri = { NULL, GTK_RESPONSE_NONE, FALSE, NULL };
	gulong  response_id;
	gulong  unmap_id;
	gulong  destroy_id;
  
	g_return_val_if_fail (TH_IS_FAKE_DIALOG (fdialog), -1);

	if (fdialog->priv->default_response_id_set)
	{
		fake_dialog_set_default_response_real (fdialog, fdialog->priv->default_response_id);
	}

	g_object_ref (fdialog);

	response_id = g_signal_connect (fdialog,
	                  "response",
	                  G_CALLBACK (run_response_handler),
	                  &ri);
  
	unmap_id = g_signal_connect (fdialog,
	               "unmap",
	               G_CALLBACK (run_unmap_handler),
	               &ri);
  
	destroy_id = g_signal_connect (fdialog,
	                "destroy",
	                G_CALLBACK (run_destroy_handler),
	                &ri);

	ri.loop = g_main_loop_new (NULL, FALSE);
	ri.destroyed = FALSE;

	g_main_loop_run (ri.loop);

	g_main_loop_unref (ri.loop);

	ri.loop = NULL;

	if (ri.destroyed == FALSE)
	{
		g_signal_handler_disconnect (fdialog, response_id);
		g_signal_handler_disconnect (fdialog, unmap_id);
		g_signal_handler_disconnect (fdialog, destroy_id);
	}
	else
	{
		ri.response_id = TH_RESPONSE_DESTROYED;
	}

	g_object_unref (fdialog);

	return ri.response_id;
}

