/***************************************************************************
                            th-disc-drive-pool.c
                            --------------------
    begin                : Mon May 17 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "th-device-pool.h"
#include "th-disc-drive-pool.h"
#include "th-marshal.h"
#include "th-utils.h"

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <glib/gi18n.h>

/* number of seconds after which we try to find DVD drives
 *  using other mechanisms than HAL, if we haven't found
 *  any drives until then that is. */
#define TH_DISC_DRIVE_NON_HAL_DRIVE_DETECT_DELAY 3

enum
{
 DRIVE_ADDED,
 DRIVE_REMOVED,
 DISC_CHANGED,
 NO_DRIVE_FOUND,
 NUM_SIGNALS
};

struct _ThDiscDrivePoolPrivate 
{
	LibHalContext *hal_ctx;
	GData         *drives;        /* maps drive_udi => ThDiscDrive, and disc_udi => ThDiscDrive */
	gulong         find_timeout;
};

static void           disc_drive_pool_class_init              (ThDiscDrivePoolClass *klass);

static void           disc_drive_pool_instance_init           (ThDiscDrivePool *cp);

static void           disc_drive_pool_finalize                (GObject *object);

static void           disc_drive_pool_add_drive               (ThDiscDrivePool *ddpool, 
                                                               const gchar     *udi);

static void           disc_drive_pool_remove_drive            (ThDiscDrivePool *ddpool, 
                                                               const gchar     *udi);

static void           disc_drive_pool_add_disc                (ThDiscDrivePool *ddpool, 
                                                               const gchar     *disc_udi);

static void           disc_drive_pool_remove_disc             (ThDiscDrivePool *ddpool, 
                                                               const gchar     *disc_udi);

static void           disc_drive_pool_add_disc_to_drive       (ThDiscDrivePool *ddpool,
                                                               ThDiscDrive     *drive,
                                                               const gchar     *disc_udi);

static void           disc_drive_pool_remove_disc_from_drive  (ThDiscDrivePool *ddpool, 
                                                               ThDiscDrive     *drive,
                                                               const gchar     *disc_udi);

static gboolean       disc_drive_pool_have_udi                (ThDiscDrivePool *ddpool, 
                                                               const gchar     *udi);

static ThDiscDrive   *disc_drive_pool_get_drive_from_udi      (ThDiscDrivePool *ddpool, 
                                                               const gchar     *udi);

static ThDiscDrive   *disc_drive_pool_get_drive_from_disc_udi (ThDiscDrivePool *ddpool, 
                                                               const gchar     *disc_udi);

static gboolean       disc_drive_pool_no_drives_cb (ThDiscDrivePool *ddpool);


/* ThDeviceTracker interface */

static void           disc_pool_set_context      (ThDeviceTracker *tracker, 
                                                  LibHalContext   *ctx);

static void           disc_pool_device_added     (ThDeviceTracker *tracker,
                                                  const char      *udi);

static void           disc_pool_device_removed   (ThDeviceTracker *tracker,
                                                  const char      *udi);

static void           disc_pool_device_new_cap   (ThDeviceTracker *tracker,
                                                  const char      *udi, 
                                                  const char      *cap);

static void           disc_pool_device_lost_cap  (ThDeviceTracker *tracker,
                                                  const char      *udi, 
                                                  const char      *cap);

static void           disc_pool_device_prop_mod  (ThDeviceTracker *tracker,
                                                  const char      *udi, 
                                                  const char      *key, 
                                                  gboolean         is_removed, 
                                                  gboolean         is_added);

#ifndef USE_NEW_DBUS_HAL_API

static void           disc_pool_device_condition (ThDeviceTracker *tracker,
                                                  const char      *udi, 
                                                  const char      *cond_name, 
                                                  DBusMessage     *msg);

#else

static void           disc_pool_device_condition (ThDeviceTracker *tracker,
                                                  const char      *udi, 
                                                  const char      *cond_name, 
                                                  const gchar     *cond_detail);
#endif

/* variables */

static guint          ddp_signals[NUM_SIGNALS];  /* all 0 */

static GObjectClass  *ddp_parent_class;          /* NULL */


/***************************************************************************
 *
 *   disc_drive_pool_class_init
 *
 ***************************************************************************/

static void
disc_drive_pool_class_init (ThDiscDrivePoolClass *klass)
{
	GObjectClass  *object_class; 

	object_class = G_OBJECT_CLASS(klass);
	
	ddp_parent_class = g_type_class_peek_parent(klass);

	object_class->finalize = disc_drive_pool_finalize;

	ddp_signals[DRIVE_ADDED] =
	     g_signal_new("drive-added",
	                  G_TYPE_FROM_CLASS (object_class),
	                  G_SIGNAL_RUN_LAST,
	                  G_STRUCT_OFFSET (ThDiscDrivePoolClass, drive_added),
	                  NULL, NULL,
	                  th_marshal_VOID__OBJECT,
	                  G_TYPE_NONE, 1, TH_TYPE_DISC_DRIVE);

	ddp_signals[DRIVE_REMOVED] =
	     g_signal_new("drive-removed",
	                  G_TYPE_FROM_CLASS (object_class),
	                  G_SIGNAL_RUN_FIRST,
	                  G_STRUCT_OFFSET (ThDiscDrivePoolClass, drive_removed),
	                  NULL, NULL,
	                  th_marshal_VOID__OBJECT,
	                  G_TYPE_NONE, 1, TH_TYPE_DISC_DRIVE);

	ddp_signals[DISC_CHANGED] =
	     g_signal_new("disc-changed",
	                  G_TYPE_FROM_CLASS (object_class),
	                  G_SIGNAL_RUN_FIRST,
	                  G_STRUCT_OFFSET (ThDiscDrivePoolClass, disc_changed),
	                  NULL, NULL,
	                  th_marshal_VOID__OBJECT,
	                  G_TYPE_NONE, 1, TH_TYPE_DISC_DRIVE);

	ddp_signals[NO_DRIVE_FOUND] =
	     g_signal_new("no-drive-found",
	                  G_TYPE_FROM_CLASS (object_class),
	                  G_SIGNAL_RUN_FIRST,
	                  G_STRUCT_OFFSET (ThDiscDrivePoolClass, no_drive_found),
	                  NULL, NULL,
	                  th_marshal_VOID__VOID,
	                  G_TYPE_NONE, 0);
}

/***************************************************************************
 *
 *   disc_drive_pool_instance_init
 *
 ***************************************************************************/

static void
disc_drive_pool_instance_init (ThDiscDrivePool *ddpool)
{
	guint delay = TH_DISC_DRIVE_NON_HAL_DRIVE_DETECT_DELAY * 1000;

	ddpool->priv = g_new0 (ThDiscDrivePoolPrivate, 1);

	g_datalist_init (&ddpool->priv->drives);

	if (th_utils_running_in_valgrind ())
	{
		gchar *t;

		delay = delay * 10;
		
		t = g_strdup_printf ("Thoggen has detected that it is running inside valgrind.\n"
		                     "Increasing delay for non-HAL DVD drive detection\n"
		                     "to %u seconds.\n", delay / 1000);
		
		th_utils_vg_style_print (t);
		
		g_free (t);
	}

	ddpool->priv->find_timeout = g_timeout_add (delay,
	                                            (GSourceFunc) disc_drive_pool_no_drives_cb,
	                                            ddpool);
}

/***************************************************************************
 *
 *   disc_drive_pool_finalize
 *
 ***************************************************************************/

static void
disc_drive_pool_finalize (GObject *object)
{
	ThDiscDrivePool *ddpool;
	
	ddpool = (ThDiscDrivePool*) object;

	g_print("ThDiscDrivePool: finalize\n");
	
	g_datalist_clear (&ddpool->priv->drives);
	ddpool->priv->drives = NULL;

	if (ddpool->priv->find_timeout > 0)
		g_source_remove (ddpool->priv->find_timeout);

	memset (ddpool->priv, 0xff, sizeof (ThDiscDrivePoolPrivate));
	g_free (ddpool->priv);

	/* chain up */
	ddp_parent_class->finalize (object);
}

/***************************************************************************
 *
 *   disc_pool_device_tracker_init
 *
 ***************************************************************************/

static void
disc_pool_device_tracker_init (ThDeviceTrackerIface *iface)
{
	iface->set_context    = disc_pool_set_context;
	iface->device_added   = disc_pool_device_added;
	iface->device_removed = disc_pool_device_removed;
	iface->new_cap        = disc_pool_device_new_cap;
	iface->lost_cap       = disc_pool_device_lost_cap;
	iface->prop_mod       = disc_pool_device_prop_mod;
	iface->dev_condition  = disc_pool_device_condition;
}

/***************************************************************************
 *
 *   th_disc_drive_pool_get_type
 *
 ***************************************************************************/

GType
th_disc_drive_pool_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThDiscDrivePoolClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) disc_drive_pool_class_init,
			NULL, NULL,
			sizeof (ThDiscDrivePool),
			0,
			(GInstanceInitFunc) disc_drive_pool_instance_init
		};

		static const GInterfaceInfo device_tracker_info =
		{
			(GInterfaceInitFunc) disc_pool_device_tracker_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (G_TYPE_OBJECT, "ThDiscDrivePool", &info, 0);
	
		/* register ThDeviceTracker interface */
		g_type_add_interface_static (type, TH_TYPE_DEVICE_TRACKER, &device_tracker_info);
	}

	return type;
}



/***************************************************************************
 *
 *   th_disc_drive_pool_new
 *
 ***************************************************************************/

ThDiscDrivePool *
th_disc_drive_pool_new (void)
{
	return (ThDiscDrivePool *) g_object_new (TH_TYPE_DISC_DRIVE_POOL, NULL);
}

/***************************************************************************
 *
 *   disc_drive_pool_have_udi
 *
 *   Returns TRUE if this is either a drive udi or a disc udi
 *
 ***************************************************************************/

static gboolean
disc_drive_pool_have_udi (ThDiscDrivePool *ddpool, const gchar *udi)
{
	g_return_val_if_fail (TH_IS_DISC_DRIVE_POOL (ddpool), FALSE);
	g_return_val_if_fail (udi != NULL, FALSE);
	
	return (g_datalist_get_data (&ddpool->priv->drives, udi) != NULL);
}

/***************************************************************************
 *
 *   disc_drive_pool_get_drive_from_udi
 *
 *   Returns the disc drive for a given disc drive udi if we know it
 *
 ***************************************************************************/

static ThDiscDrive *
disc_drive_pool_get_drive_from_udi (ThDiscDrivePool *ddpool, const gchar *udi)
{
	ThDiscDrive *drive;
	gchar       *drive_udi;

	g_return_val_if_fail (TH_IS_DISC_DRIVE_POOL (ddpool), NULL);
	g_return_val_if_fail (udi != NULL, NULL);

	drive = (ThDiscDrive*) g_datalist_get_data (&ddpool->priv->drives, udi);
	
	if (drive == NULL)
		return NULL;

	g_object_get (drive, "udi", &drive_udi, NULL);
	
	/* In this case the udi is the udi of a
	 *  disc that is in one of our drives. */
	if (!drive_udi || !g_str_equal (udi, drive_udi))
	{
		g_free (drive_udi);
		return NULL;
	}

	g_free (drive_udi);

	return TH_DISC_DRIVE (drive);
}

/***************************************************************************
 *
 *   disc_drive_pool_get_drive_from_disc_udi
 *
 *   Returns the disc drive for a given disc udi
 *
 ***************************************************************************/

static ThDiscDrive *
disc_drive_pool_get_drive_from_disc_udi (ThDiscDrivePool *ddpool, const gchar *disc_udi)
{
	ThDiscDrive *drive;
	gchar       *drive_udi;
	
	g_return_val_if_fail (TH_IS_DISC_DRIVE_POOL (ddpool), NULL);
	g_return_val_if_fail (disc_udi != NULL, NULL);

	drive = (ThDiscDrive*) g_datalist_get_data (&ddpool->priv->drives, disc_udi);
	
	if (drive == NULL)
		return NULL;

	g_object_get (drive, "udi", &drive_udi, NULL);
	
	g_return_val_if_fail (drive_udi != NULL, NULL);

	/* In this case the udi is the udi of the drive itself 
	 *  and not of the disc that is in one of our drives. */
	if (g_str_equal (disc_udi, drive_udi))
	{
		g_free (drive_udi);
		return NULL;
	}
	
	g_free (drive_udi);
	
	return drive;
}


/***************************************************************************
 *
 *   disc_drive_pool_add_disc_to_drive
 *
 ***************************************************************************/

static void
disc_drive_pool_add_disc_to_drive (ThDiscDrivePool *ddpool, 
                                   ThDiscDrive     *drive,
                                   const gchar     *disc_udi)
{
	gchar *label = NULL;

	if (disc_drive_pool_get_drive_from_disc_udi (ddpool, disc_udi))
		return;
	
	g_datalist_set_data (&ddpool->priv->drives, disc_udi, drive);
	
	if (hal_device_property_exists (ddpool->priv->hal_ctx, disc_udi, "volume.label"))
		label = hal_device_get_property_string (ddpool->priv->hal_ctx, disc_udi, "volume.label");
	
	if (label == NULL)
		label = strdup ("(no title)");

	g_object_set (drive, "disc-title", label, NULL);

	g_signal_emit (ddpool, ddp_signals[DISC_CHANGED], 0, drive);
	
	hal_free_string (label);
}

/***************************************************************************
 *
 *   disc_drive_pool_add_disc
 *
 ***************************************************************************/

static void
disc_drive_pool_add_disc (ThDiscDrivePool *ddpool, 
                          const gchar     *disc_udi)
{
	ThDiscDrive *drive;
	gchar       *drive_udi;

	if (!hal_device_property_exists (ddpool->priv->hal_ctx, disc_udi, "info.parent"))
		return;

	drive_udi = hal_device_get_property_string (ddpool->priv->hal_ctx, disc_udi, "info.parent");
	
	if (drive_udi == NULL)
		return;

	drive = disc_drive_pool_get_drive_from_udi (ddpool, drive_udi);
	
	if (drive)
	{
		disc_drive_pool_add_disc_to_drive (ddpool, drive, disc_udi);
	}
	else
	{
		g_warning ("DVD disc inserted into a drive we don't know about yet?!\n"
		           "disc_udi  = %s\ndrive_udi = %s", disc_udi, drive_udi);
	}
	
	hal_free_string (drive_udi);
}

/***************************************************************************
 *
 *   disc_drive_pool_remove_disc_from_drive
 *
 ***************************************************************************/

static void
disc_drive_pool_remove_disc_from_drive (ThDiscDrivePool *ddpool, 
                                        ThDiscDrive     *drive,
                                        const gchar     *disc_udi)
{
	g_datalist_remove_data (&ddpool->priv->drives, disc_udi);
	
	g_object_set (drive, "disc-title", NULL, NULL);
	
	g_signal_emit (ddpool, ddp_signals[DISC_CHANGED], 0, drive);
}

/***************************************************************************
 *
 *   disc_drive_pool_remove_disc
 *
 *   wasteful, but analog to _add_disc() and _add_disc_to_drive()
 *
 ***************************************************************************/

static void
disc_drive_pool_remove_disc (ThDiscDrivePool *ddpool, 
                             const gchar     *disc_udi)
{
	ThDiscDrive *drive;
	
	drive = disc_drive_pool_get_drive_from_disc_udi (ddpool, disc_udi);
	
	if (drive == NULL)
		return;

	disc_drive_pool_remove_disc_from_drive (ddpool, drive, disc_udi);
}

/***************************************************************************
 *
 *   disc_drive_pool_get_drive_info
 *
 ***************************************************************************/

static gboolean
disc_drive_pool_get_drive_info (ThDiscDrivePool *disc_pool,
                                const gchar     *udi,
                                gchar          **p_device,
                                gchar          **p_vendor,
                                gchar          **p_product,
                                gboolean        *p_req_polling)
{
	LibHalContext *ctx;
	gchar         *device;
	gchar         *vendor = NULL;
	gchar         *product = NULL;

	ctx = disc_pool->priv->hal_ctx;

	device = hal_device_get_property_string (ctx, udi, "block.device");
	g_return_val_if_fail (device != NULL, FALSE);

	/* do polling ourselves if HAL doesn't do it for us */
	*p_req_polling = FALSE;
	if (hal_device_property_exists (ctx, udi, "storage.media_check_enabled")
	 && !hal_device_get_property_bool (ctx, udi, "storage.media_check_enabled"))
		*p_req_polling = TRUE;
	
	if (hal_device_property_exists (ctx, udi, "info.vendor"))
		vendor = hal_device_get_property_string (ctx, udi, "info.vendor");
	if (vendor == NULL && hal_device_property_exists (ctx, udi, "block.vendor"))
		vendor = hal_device_get_property_string (ctx, udi, "block.vendor");

	if (hal_device_property_exists (ctx, udi, "info.product"))
		product = hal_device_get_property_string (ctx, udi, "info.product");
	if (product == NULL && hal_device_property_exists (ctx, udi, "block.product"))
		product = hal_device_get_property_string (ctx, udi, "block.product");
	
	if (vendor  == NULL && product == NULL)
	{
		vendor = strdup ("Generic");
		product = strdup ("DVD Drive");
	}
	else if (vendor == NULL)
	{
		vendor = strdup ("DVD Drive");
	}
	else if (product == NULL)
	{
		product = strdup ("DVD Drive");
	}

	*p_device  = g_strdup (device);
	*p_vendor  = g_strdup (vendor);
	*p_product = g_strdup (product);
	
	hal_free_string (device);
	hal_free_string (vendor);
	hal_free_string (product);
	
	return TRUE;
}

/***************************************************************************
 *
 *   disc_drive_pool_add_drive
 *
 ***************************************************************************/

static void
disc_drive_pool_add_drive (ThDiscDrivePool *disc_pool, const gchar *udi)
{
	ThDiscDrive *drive;
	gboolean     req_polling;
	gchar       *device, *vendor, *product;
	
	g_return_if_fail (TH_IS_DISC_DRIVE_POOL (disc_pool));
	
	if (disc_drive_pool_get_drive_from_udi (disc_pool, udi))
		return;

	if (!disc_drive_pool_get_drive_info (disc_pool, udi, &device, &vendor, &product, &req_polling))
		g_return_if_reached ();

	drive = th_disc_drive_new (udi, device, vendor, product);

	if (th_disc_drive_pool_add_drive (disc_pool, drive)) {	
	  if (req_polling) {
	    g_object_set (drive, 
	        "poll-media-required", TRUE, 
	        "poll-media", TRUE, 
	        NULL);
	  }
	}

	if (disc_pool->priv->find_timeout > 0)
	{
		g_source_remove (disc_pool->priv->find_timeout);
		disc_pool->priv->find_timeout = 0;
	}
	g_free (device);
	g_free (vendor);
	g_free (product);

	g_object_unref (drive); /* disc_pool->priv->drives kept a ref */
}


/***************************************************************************
 *
 *   disc_drive_pool_remove_drive
 *
 *   Does nothing if we don't have this device as disc drive
 *
 ***************************************************************************/

static void           
disc_drive_pool_remove_drive (ThDiscDrivePool *ddpool, const gchar *udi)
{
	ThDiscDrive *drive;
	
	g_return_if_fail (TH_IS_DISC_DRIVE_POOL (ddpool));

	drive = disc_drive_pool_get_drive_from_udi (ddpool, udi);
	
	if (drive == NULL)
		return;

	g_signal_emit (ddpool, ddp_signals[DRIVE_REMOVED], 0, drive);
	
	g_datalist_remove_data (&ddpool->priv->drives, udi);
	
	g_object_unref (drive);
}

/***************************************************************************
 *
 *   disc_pool_device_is_dvd_disc
 *
 ***************************************************************************/

static gboolean
disc_pool_device_is_dvd_disc (ThDiscDrivePool *disc_pool, const gchar *udi)
{
	LibHalContext *ctx; 
	gboolean       is_dvd_rom, is_dvd_foo, is_unknown;
	char          *disc_type;
	
	ctx = disc_pool->priv->hal_ctx;

	if (!hal_device_property_exists (ctx, udi, "info.capabilities")
	 || !hal_device_query_capability (ctx, udi, "volume")
	 || !hal_device_property_exists (ctx, udi, "volume.is_disc")
	 || !hal_device_property_exists (ctx, udi, "volume.disc.type"))
		return FALSE;
	 
	disc_type = hal_device_get_property_string (ctx, udi, "volume.disc.type");
	if (disc_type == NULL)
		return FALSE;
	
	is_dvd_rom = g_str_equal (disc_type, "dvd_rom");
	is_dvd_foo = g_str_has_prefix (disc_type, "dvd_");
	is_unknown = g_str_equal (disc_type, "unknown");
	
	hal_free_string (disc_type);
	
	/* There seem to be occasions where HAL (wrongly) says a 
	 *  dvd_rom medium is blank and contains no data. Hence, we
	 *  will just assume that there is a DVD in the drive if the
	 *  disc type is 'dvd_rom', and only check whether the disc
	 *  has any data according to HAL if it is some other type
	 *  (which would be rewritable disc and the like) */
	
	if (is_dvd_rom)
		return TRUE;

	if (is_dvd_foo
	 && hal_device_property_exists (ctx, udi, "volume.disc.has_data")
	 && hal_device_get_property_bool (ctx, udi, "volume.disc.has_data"))
		return TRUE;
	
	/* If the disc type is unknown, assume it's a DVD if it is larger
	 *  than 1GB (false positives are better than unrecognised DVDs) */
	if (is_unknown
	 && hal_device_property_exists (ctx, udi, "volume.size")
	 && hal_device_property_exists (ctx, udi, "volume.fstype"))
	{
		gboolean might_be_a_dvd = FALSE;
		guint64  size;
		char    *fstype;

		size = hal_device_get_property_uint64 (ctx, udi, "volume.size");
		fstype = hal_device_get_property_string (ctx, udi, "volume.fstype");

		if (g_ascii_strcasecmp (fstype, "udf") == 0  &&  size > 1*1024*1024*1024)
			might_be_a_dvd = TRUE;

		hal_free_string (fstype);

		return might_be_a_dvd;
	}

	return FALSE;
}

/***************************************************************************
 *
 *   disc_pool_device_is_dvd_drive
 *
 ***************************************************************************/

static gboolean
disc_pool_device_is_dvd_drive (ThDiscDrivePool *disc_pool, const gchar *udi)
{
	LibHalContext *ctx = disc_pool->priv->hal_ctx;

	if (!hal_device_property_exists (ctx, udi, "storage.cdrom.dvd")
	 || !hal_device_get_property_bool (ctx, udi, "storage.cdrom.dvd"))
		return FALSE;

	return TRUE;
}

/***************************************************************************
 *
 *   disc_pool_set_context
 *
 ***************************************************************************/

static void
disc_pool_set_context (ThDeviceTracker *tracker, LibHalContext *ctx)
{
	/* Note: when the HAL daemon/dbus system message 
	 *  bus is restarted, the context might change */
	TH_DISC_DRIVE_POOL(tracker)->priv->hal_ctx = ctx;
}

/***************************************************************************
 *
 *   disc_pool_device_added
 *
 *   We are watching for two things here: DVD drives, and DVD discs
 *
 ***************************************************************************/

static void
disc_pool_device_added (ThDeviceTracker *tracker, const char *udi)
{
	ThDiscDrivePool *disc_pool;
	
	disc_pool = TH_DISC_DRIVE_POOL (tracker);

	if (disc_pool_device_is_dvd_drive (disc_pool, udi))
	{
		disc_drive_pool_add_drive (disc_pool, udi);
	}
	else if (disc_pool_device_is_dvd_disc (disc_pool, udi))
	{
		disc_drive_pool_add_disc (disc_pool, udi);
	}
}

/***************************************************************************
 *
 *   disc_pool_device_removed
 *
 ***************************************************************************/

static void
disc_pool_device_removed (ThDeviceTracker *tracker, const char *udi)
{
	ThDiscDrivePool *disc_pool;
	
	disc_pool = TH_DISC_DRIVE_POOL (tracker);
	
	if (disc_drive_pool_get_drive_from_udi (disc_pool, udi))
	{
		disc_drive_pool_remove_drive (disc_pool, udi);
	}
	else if (disc_drive_pool_get_drive_from_disc_udi (disc_pool, udi))
	{
		disc_drive_pool_remove_disc (disc_pool, udi);
	}
}


/***************************************************************************
 *
 *   disc_pool_device_new_cap
 *
 ***************************************************************************/

static void
disc_pool_device_new_cap (ThDeviceTracker *tracker, const char *udi, const char *cap)
{
	if (g_str_equal (cap, "storage.cdrom"))
	{
		/* everything else will be checked there ... */
		disc_pool_device_prop_mod (tracker, udi, "storage.cdrom.dvd", FALSE, FALSE);
	}
}

/***************************************************************************
 *
 *   disc_pool_device_lost_cap
 *
 ***************************************************************************/

static void
disc_pool_device_lost_cap (ThDeviceTracker *tracker, const gchar *udi, const gchar *cap)
{
	if (disc_drive_pool_have_udi (TH_DISC_DRIVE_POOL (tracker), udi))
	{
		g_print ("%s [%s] [%s]\n", __FUNCTION__, udi, cap);
	}
}

/***************************************************************************
 *
 *   disc_pool_device_prop_mod
 *
 ***************************************************************************/

static void
disc_pool_device_prop_mod (ThDeviceTracker *tracker, 
                           const gchar     *udi, 
                           const gchar     *key, 
                           gboolean         is_removed, 
                           gboolean         is_added)
{
	ThDiscDrivePool *disc_pool = NULL;
	LibHalContext   *ctx;

	/* g_print ("%s [%s] [%s]\n", __FUNCTION__, udi, key); */
	
	if (!key || !g_str_equal (key, "storage.cdrom.dvd"))
		return;

	disc_pool = TH_DISC_DRIVE_POOL (tracker);
	ctx = disc_pool->priv->hal_ctx;

	/* could be because drive is removed, or driver is unloaded etc. */
	if (is_removed 
	 || !hal_device_property_exists (ctx, udi, "storage.cdrom.dvd")
	 || !hal_device_get_property_bool (ctx, udi, "storage.cdrom.dvd"))
	{
		disc_drive_pool_remove_drive (disc_pool, udi);
		return;
	}
	
	if (is_added || hal_device_get_property_bool (ctx, udi, "storage.cdrom.dvd"))
	{
		if (!disc_drive_pool_get_drive_from_udi (disc_pool, udi))
			disc_drive_pool_add_drive (disc_pool, udi);
	}
}

/***************************************************************************
 *
 *   disc_pool_device_condition
 *
 ***************************************************************************/

static void
disc_pool_device_condition (ThDeviceTracker *tracker, 
                            const gchar     *udi, 
                            const gchar     *cond_name, 
#ifndef USE_NEW_DBUS_HAL_API
                            DBusMessage     *msg)
#else
                            const gchar     *cond_detail)
#endif
{
	ThDiscDrivePool *disc_pool;

	disc_pool = TH_DISC_DRIVE_POOL (tracker);

	if (disc_drive_pool_have_udi (disc_pool, udi))
	{
		g_print ("%s [%s] [%s]\n", __FUNCTION__, udi, cond_name);
	}
}

/***************************************************************************
 *
 *   disc_drive_pool_find_drives_via_proc_sys_dev_cdrom
 *
 ***************************************************************************/

static gboolean
disc_drive_pool_find_drives_via_proc_sys_dev_cdrom (ThDiscDrivePool *ddpool)
{
	gboolean   ret = FALSE;
	gsize      clen;
	gchar     *content, **lines, **l, *devs, *dvd;

	if (!g_file_get_contents ("/proc/sys/dev/cdrom/info", &content, &clen, NULL))
		return FALSE;

	lines = g_strsplit (content, "\n", -1);

	devs = dvd = NULL;
	for (l = lines;  l && *l;  ++l)
	{
		if (g_str_has_prefix (*l, "drive name:"))
			devs = *l + strlen ("drive name:");
		else if (g_str_has_prefix (*l, "Can read DVD:"))
			dvd = *l + strlen ("can read DVD:");
	}

	if (devs && dvd)
	{
		while (*devs == '\t') ++devs;
		while (*dvd == '\t') ++dvd;
		do
		{
			gchar *dev, *end;

			if ((end = strchr (devs, ' ')))
				dev = g_strndup (devs, end - devs);
			else if ((end = strchr (devs, '\t')))
				dev = g_strndup (devs, end - devs);
			else
				dev = g_strdup (devs);

			if (!dev || *dev == 0x00)
			{
				g_free (dev);
				break;
			}

			if (atoi (dvd) == 1)
			{
				ThDiscDrive *drive;
				gchar       *device;

				device = g_strdup_printf ("/dev/%s", dev);	
				if (!g_datalist_get_data (&ddpool->priv->drives, device))
				{
					g_message ("Found DVD drive '%s'\n", device);

					drive = th_disc_drive_new (device, device, _("DVD Drive"), " ");

					g_datalist_set_data_full (&ddpool->priv->drives, device, drive, g_object_unref);
	
					g_signal_emit (ddpool, ddp_signals[DRIVE_ADDED], 0, drive);

					g_object_set (drive, 
					              "poll-media-required", TRUE, 
					              "poll-media", TRUE, 
					              NULL);

					ret = TRUE;
				}
				g_free (device);
			}

			devs = end;
			while (devs && *devs == ' ')
				++devs;

			g_free (dev);
		}
		while (devs && *devs);
	}

	g_strfreev (lines);
	g_free (content);

	return ret;
}

/***************************************************************************
 *
 *   disc_drive_pool_find_drives_via_env_variable
 *
 ***************************************************************************/

static gboolean
disc_drive_pool_find_drives_via_env_variable (ThDiscDrivePool *ddpool)
{
	ThDiscDrive *drive;
	const gchar *dev;
	GError      *err = NULL;
	
	dev = g_getenv ("THOGGEN_DVD_DRIVE_DEVICE");
	if (dev == NULL || !g_path_is_absolute (dev))
		return FALSE;

	if (!th_utils_device_might_be_dvd_drive (dev, &err))
	{
		g_warning ("Failed to check dvd device '%s' (set via THOGGEN_DVD_DRIVE_DEVICE\n"
		           "\tenvironment variable): %s\n", dev, err->message);
		g_error_free (err);
		return FALSE;
	}
	
	if (g_datalist_get_data (&ddpool->priv->drives, dev))
		g_return_val_if_reached (TRUE); /* shouldn't happen */

	drive = th_disc_drive_new (dev, dev, _("DVD Drive"), " ");

	g_datalist_set_data_full (&ddpool->priv->drives, dev, drive, g_object_unref);
	
	g_signal_emit (ddpool, ddp_signals[DRIVE_ADDED], 0, drive);

	g_object_set (drive, 
	              "poll-media-required", TRUE, 
	              "poll-media", TRUE, 
	              NULL);

	return TRUE;
}

/***************************************************************************
 *
 *   disc_drive_pool_no_drives_cb
 *
 ***************************************************************************/

static gboolean
disc_drive_pool_no_drives_cb (ThDiscDrivePool *ddpool)
{
	ddpool->priv->find_timeout = 0;

	g_print ("Trying to find drives using backup method ...\n");
	
	if (!disc_drive_pool_find_drives_via_env_variable (ddpool)
	 && !disc_drive_pool_find_drives_via_proc_sys_dev_cdrom (ddpool))
	{
		g_signal_emit (ddpool, ddp_signals[NO_DRIVE_FOUND], 0);
	}

	return FALSE; /* only run once */
}

/***************************************************************************
 *
 *   th_disc_drive_pool_get_drive_from_udi
 *
 ***************************************************************************/

ThDiscDrive *
th_disc_drive_pool_get_drive_from_udi (ThDiscDrivePool * pool, const gchar * udi)
{
  g_return_val_if_fail (TH_IS_DISC_DRIVE_POOL (pool), NULL);
  g_return_val_if_fail (udi != NULL, NULL);

  return disc_drive_pool_get_drive_from_udi (pool, udi);
}

/***************************************************************************
 *
 *   th_disc_drive_pool_add_drive
 *
 ***************************************************************************/

gboolean
th_disc_drive_pool_add_drive (ThDiscDrivePool * pool, ThDiscDrive * drive)
{
  gchar *drive_udi = NULL;

  g_return_val_if_fail (TH_IS_DISC_DRIVE_POOL (pool), FALSE);
  g_return_val_if_fail (TH_IS_DISC_DRIVE (drive), FALSE);

  g_object_get (drive, "udi", &drive_udi, NULL);
  g_return_val_if_fail (drive_udi != NULL, FALSE);

  if (disc_drive_pool_get_drive_from_udi (pool, drive_udi))
    return FALSE;

  g_datalist_set_data_full (&pool->priv->drives, drive_udi,
      g_object_ref (drive), g_object_unref);
	
  g_signal_emit (pool, ddp_signals[DRIVE_ADDED], 0, drive);

  return TRUE;
}

