/*
    Theseus - maximum likelihood superpositioning of macromolecular structures

    Copyright (C) 2004-2010 Douglas L. Theobald

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the:

    Free Software Foundation, Inc.,
    59 Temple Place, Suite 330,
    Boston, MA  02111-1307  USA

    -/_|:|_|_\-
*/

#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <float.h>
#include <ctype.h>
#include "CovMat.h"
#include "DLTutils.h"
#include "Error.h"
#include "Embed.h"
#include "DLTmath.h"
#include "pdbIO.h"
#include "Threads.h"
#include "pdbUtils.h"


void
CoordsCopyXYZ(Coords *coords1, const Coords *coords2)
{
    memcpy(coords1->x, coords2->x, coords2->vlen * sizeof(double));
    memcpy(coords1->y, coords2->y, coords2->vlen * sizeof(double));
    memcpy(coords1->z, coords2->z, coords2->vlen * sizeof(double));
}


void
CoordsCopy(Coords *coords1, const Coords *coords2)
{
    strncpy(coords1->filename, coords2->filename, FILENAME_MAX-1);
    memcpy(coords1->resSeq, coords2->resSeq, coords2->vlen * sizeof(int));
    memcpy(coords1->x, coords2->x, coords2->vlen * sizeof(double));
    memcpy(coords1->y, coords2->y, coords2->vlen * sizeof(double));
    memcpy(coords1->z, coords2->z, coords2->vlen * sizeof(double));
    memcpy(coords1->o, coords2->o, coords2->vlen * sizeof(double));
}


/* copy coords, omitting the given atom */
void
CoordsDelete(CoordsArray *cdsA, const int omit)
{
    int             i, j;

    if(omit == cdsA->vlen - 1)
    {
        cdsA->vlen--;
        cdsA->avecoords->vlen--;

        for (i = 0; i < cdsA->cnum; ++i)
            cdsA->coords[i]->vlen--;
    }
    else
    {
        cdsA->vlen--;
        cdsA->avecoords->vlen--;

        for (i = 0; i < cdsA->cnum; ++i)
            cdsA->coords[i]->vlen--;

        memmove(&cdsA->avecoords->resSeq[omit], &cdsA->avecoords->resSeq[omit+1],
                (cdsA->vlen - omit) * sizeof(int));
        memmove(&cdsA->avecoords->x[omit], &cdsA->avecoords->x[omit+1],
                (cdsA->vlen - omit) * sizeof(double));
        memmove(&cdsA->avecoords->y[omit], &cdsA->avecoords->y[omit+1],
                (cdsA->vlen - omit) * sizeof(double));
        memmove(&cdsA->avecoords->z[omit], &cdsA->avecoords->z[omit+1],
                (cdsA->vlen - omit) * sizeof(double));

        for (j = 0; j < cdsA->cnum; ++j)
        {
            memmove(&cdsA->coords[j]->resSeq[omit], &cdsA->coords[j]->resSeq[omit+1],
                    (cdsA->vlen - omit) * sizeof(int));
            memmove(&cdsA->coords[j]->x[omit], &cdsA->coords[j]->x[omit+1],
                    (cdsA->vlen - omit) * sizeof(double));
            memmove(&cdsA->coords[j]->y[omit], &cdsA->coords[j]->y[omit+1],
                    (cdsA->vlen - omit) * sizeof(double));
            memmove(&cdsA->coords[j]->z[omit], &cdsA->coords[j]->z[omit+1],
                    (cdsA->vlen - omit) * sizeof(double));
        }
    }
}


void
CoordsArrayCopy(CoordsArray *cdsA1, const CoordsArray *cdsA2)
{
    int             i;

    cdsA1->vlen = cdsA2->vlen;
    cdsA1->cnum = cdsA2->cnum;
    cdsA1->pdbA = cdsA2->pdbA;

    for (i = 0; i < cdsA2->cnum; ++i)
        CoordsCopyAll(cdsA1->coords[i], cdsA2->coords[i]);

    CoordsCopyAll(cdsA1->avecoords, cdsA2->avecoords);

    AlgorithmCopy(cdsA1->algo, cdsA2->algo);
    StatisticsCopy(cdsA1->stats, cdsA2->stats);

    memcpy(cdsA1->var, cdsA2->var, cdsA2->vlen * sizeof(double));
    memcpy(cdsA1->w, cdsA2->w, cdsA2->vlen * sizeof(double));
}


void
AlgorithmCopy(Algorithm *algo1, const Algorithm *algo2)
{
    memcpy(algo1, algo2, sizeof(Algorithm));
    algo1->argv = NULL; /* DLT debug -- these should be copied */
    algo1->infiles = NULL; /* DLT debug -- these should be copied */
    if (algo2->selection != NULL)
    {
        algo1->selection = (char *) malloc((strlen(algo2->selection) + 1) * sizeof(char));
        strcpy(algo1->selection, algo2->selection);
    }
    if (algo2->atomslxn != NULL)
    {
        algo1->atomslxn = (char *) malloc((strlen(algo2->atomslxn) + 1) * sizeof(char));
        strcpy(algo1->atomslxn, algo2->atomslxn);
    }
}


void
StatisticsCopy(Statistics *stats1, const Statistics *stats2)
{
    memcpy(stats1, stats2, sizeof(Statistics));
}


void
CoordsCopyAll(Coords *coords1, const Coords *coords2)
{
    int             i;

    coords1->vlen = coords2->vlen;
    coords1->aalen = coords2->aalen;
    coords1->model  = coords2->model;

    strncpy(coords1->filename, coords2->filename, FILENAME_MAX - 1);

    memcpy(coords1->resName_space, coords2->resName_space, coords2->vlen * 4 * sizeof(char));
    memcpy(coords1->chainID, coords2->chainID, coords2->vlen * sizeof(char));
    memcpy(coords1->resSeq, coords2->resSeq, coords2->vlen * sizeof(int));
    memcpy(coords1->x, coords2->x, coords2->vlen * sizeof(double));
    memcpy(coords1->y, coords2->y, coords2->vlen * sizeof(double));
    memcpy(coords1->z, coords2->z, coords2->vlen * sizeof(double));
    memcpy(coords1->o, coords2->o, coords2->vlen * sizeof(double));
    memcpy(coords1->b, coords2->b, coords2->vlen * sizeof(double));
    memcpy(coords1->residual_x, coords2->residual_x, coords2->vlen * sizeof(double));
    memcpy(coords1->residual_y, coords2->residual_y, coords2->vlen * sizeof(double));
    memcpy(coords1->residual_z, coords2->residual_z, coords2->vlen * sizeof(double));

    MatCpySym(coords1->matrix, (const double **) coords2->matrix, 3);

    memcpy(coords1->center, coords2->center, 3 * sizeof(double));
    memcpy(coords1->last_center, coords2->last_center, 3 * sizeof(double));
    memcpy(coords1->translation, coords2->translation, 3 * sizeof(double));

    coords1->RMSD_from_mean  = coords2->RMSD_from_mean;
    coords1->wRMSD_from_mean = coords2->wRMSD_from_mean;

    for (i = 0; i < 2; ++i)
        coords1->evals[i]  = coords2->evals[i];
}


void
PDBCoordsCopyAll(PDBCoords *coords1, const PDBCoords *coords2)
{
    coords1->vlen = coords2->vlen;
    coords1->model  = coords2->model;

    strncpy(coords1->filename, coords2->filename, FILENAME_MAX-1);

    memcpy(coords1->record_space, coords2->record_space, coords2->vlen * 8 * sizeof(char));
    memcpy(coords1->name_space, coords2->name_space, coords2->vlen * 4 * sizeof(char));
    memcpy(coords1->resName_space, coords2->resName_space, coords2->vlen * 4 * sizeof(char));
    memcpy(coords1->segID_space, coords2->segID_space, coords2->vlen * 8 * sizeof(char));
    memcpy(coords1->element_space, coords2->element_space, coords2->vlen * 4 * sizeof(char));
    memcpy(coords1->charge_space, coords2->charge_space, coords2->vlen * 4 * sizeof(char));

    memcpy(coords1->serial, coords2->serial, coords2->vlen * sizeof(int));
    memcpy(coords1->Hnum, coords2->Hnum, coords2->vlen * sizeof(char));
    memcpy(coords1->altLoc, coords2->altLoc, coords2->vlen * sizeof(char));
    memcpy(coords1->xchainID, coords2->xchainID, coords2->vlen * sizeof(char));
    memcpy(coords1->chainID, coords2->chainID, coords2->vlen * sizeof(char));
    memcpy(coords1->resSeq, coords2->resSeq, coords2->vlen * sizeof(int));
    memcpy(coords1->iCode, coords2->iCode, coords2->vlen * sizeof(char));
    memcpy(coords1->x, coords2->x, coords2->vlen * sizeof(double));
    memcpy(coords1->y, coords2->y, coords2->vlen * sizeof(double));
    memcpy(coords1->z, coords2->z, coords2->vlen * sizeof(double));
    memcpy(coords1->occupancy, coords2->occupancy, coords2->vlen * sizeof(double));
    memcpy(coords1->tempFactor, coords2->tempFactor, coords2->vlen * sizeof(double));

    MatCpySym(coords1->matrix, (const double **) coords2->matrix, 3);
    memcpy(coords1->translation, coords2->translation, 3 * sizeof(double));
}


void
CoordsAdd(Coords *coords1, const Coords *coords2)
{
    int             i;

    coords1->vlen = coords2->vlen;

    for (i = 0; i < coords2->vlen; ++i)
    {
        coords1->x[i] += coords2->x[i];
        coords1->y[i] += coords2->y[i];
        coords1->z[i] += coords2->z[i];
    }

    coords1->RMSD_from_mean  += coords2->RMSD_from_mean;
    coords1->wRMSD_from_mean += coords2->wRMSD_from_mean;
}


void
RotMatAddIp(double **mat1, const double **mat2)
{
    int             i, j;

    for (i = 0; i < 3; ++i)
        for (j = 0; j < 3; ++j)
            mat1[i][j] += mat2[i][j];
}


void
NormJKCoords(CoordsArray *cdsA)
{
    int             i, j, k;
    double          jackknife = (double) cdsA->algo->jackknife;

    for (k = 0; k < cdsA->cnum; ++k)
        for (i = 0; i < 3; ++i)
            for (j = 0; j < 3; ++j)
                cdsA->coords[k]->matrix[i][j] =
                    cdsA->coords[k]->jackmat[i][j] / jackknife;

    for (k = 0; k < cdsA->cnum; ++k)
        ClosestRotMatIp(cdsA->coords[k]->matrix);

    for (k = 0; k < cdsA->cnum; ++k)
        for (i = 0; i < 3; ++i)
            cdsA->coords[k]->translation[i] = 
                cdsA->coords[k]->jktranslation[i] / jackknife;

    cdsA->jkcoords->RMSD_from_mean  /= jackknife;
    cdsA->jkcoords->wRMSD_from_mean /= jackknife;
}


void
FillSegIDWithResSeq(PDBCoords *coords_to, const Coords *coords_from)
{
    int             i, rn;
    char            resnum[5] ;

    for (i = 0; i < coords_to->vlen; ++i)
    {
         rn = coords_from->resSeq[i];

         if (rn != 0)
         {
             sprintf(resnum, "%04d", rn);
             strncpy(coords_to->segID[i], resnum, 4);
         }
    }
}


/* copies the info for a Coords struct to a PDBCoords struct,
   for the most part */
void
CopyCoords2PDB(PDBCoords *pdbcoords, const Coords *coords)
{
    int             i;

    pdbcoords->vlen = coords->vlen;
    pdbcoords->model = coords->model;
    strncpy(pdbcoords->filename, coords->filename, FILENAME_MAX-1);

    MatCpySym(pdbcoords->matrix, (const double **) coords->matrix, 3);

    for (i = 0; i < 3; ++i)
        pdbcoords->translation[i]  = coords->translation[i];

    for (i = 0; i < coords->vlen; ++i)
    {
        strcpy(pdbcoords->record[i], "ATOM");
        pdbcoords->serial[i]     = i+1;
        pdbcoords->Hnum[i]       = ' ';

        if (strncmp(coords->resName[i], "ADE", 3) == 0 ||
            strncmp(coords->resName[i], "CYT", 3) == 0 ||
            strncmp(coords->resName[i], "GUA", 3) == 0 ||
            strncmp(coords->resName[i], "THY", 3) == 0 ||
            strncmp(coords->resName[i], "URA", 3) == 0 ||
            strncmp(coords->resName[i], "  A", 3) == 0 ||
            strncmp(coords->resName[i], "  C", 3) == 0 ||
            strncmp(coords->resName[i], "  G", 3) == 0 ||
            strncmp(coords->resName[i], "  T", 3) == 0 ||
            strncmp(coords->resName[i], " DA", 3) == 0 || /* remediated PDB residue names */
            strncmp(coords->resName[i], " DC", 3) == 0 || /* remediated PDB residue names */
            strncmp(coords->resName[i], " DG", 3) == 0 || /* remediated PDB residue names */
            strncmp(coords->resName[i], " DT", 3) == 0 || /* remediated PDB residue names */
            strncmp(coords->resName[i], " DI", 3) == 0 || /* remediated PDB residue names */
            strncmp(coords->resName[i], " DU", 3) == 0 || /* remediated PDB residue names */
            strncmp(coords->resName[i], "  U", 3) == 0)
            strcpy(pdbcoords->name[i], "P  ");
        else
            strcpy(pdbcoords->name[i], "CA ");

        pdbcoords->altLoc[i]     = ' ';
        strncpy(pdbcoords->resName[i], coords->resName[i], 3);
        pdbcoords->xchainID[i]   = ' ';
        pdbcoords->chainID[i]    = coords->chainID[i];
        pdbcoords->resSeq[i]     = coords->resSeq[i];
        pdbcoords->iCode[i]      = ' ';
        pdbcoords->x[i]          = coords->x[i];
        pdbcoords->y[i]          = coords->y[i];
        pdbcoords->z[i]          = coords->z[i];
        pdbcoords->occupancy[i]  = coords->o[i];
        pdbcoords->tempFactor[i] = coords->b[i];
        strncpy(pdbcoords->segID[i], "    ", 4);
        strncpy(pdbcoords->element[i], "  ", 2);
        strncpy(pdbcoords->charge[i], "  ", 2);
    }
}


int
NMRCheckPDBCoordsArray(PDBCoordsArray *pdbA)
{
    int             i;
    int             vlen = pdbA->coords[0]->vlen;

    for(i = 1; i < pdbA->cnum; ++i)
    {
        if (pdbA->coords[i]->vlen != vlen)
        {
            fprintf(stderr,
                    "\n  WARNING20: PDB coordinates %d and %d are of unequal length [%d vs %d]. \n\n",
                    0, i, vlen, pdbA->coords[i]->vlen);
            //PrintTheseusTag();
            //exit(EXIT_FAILURE);
        }
    }

    return(vlen);
}


int
NMRCheckCoordsArray(CoordsArray *pdbA)
{
    int             i;
    int             vlen = pdbA->coords[0]->vlen;

    for(i = 1; i < pdbA->cnum; ++i)
    {
        if (pdbA->coords[i]->vlen != vlen)
        {
            fprintf(stderr,
                    "\n  WARNING20: PDB coordinates %d and %d are of unequal length [%d vs %d]. \n\n",
                    0, i, vlen, pdbA->coords[i]->vlen);
            //PrintTheseusTag();
            //exit(EXIT_FAILURE);
        }
    }

    return(vlen);
}


/* matrix and translation need to be in PDBCoords structure */
void
TransformPDBCoordsIp(PDBCoords *pdbcoords)
{
    int             i;
    double          xt, yt, zt;
    double         *x = pdbcoords->x, *y = pdbcoords->y, *z = pdbcoords->z;
    const double    transx = pdbcoords->translation[0];
    const double    transy = pdbcoords->translation[1];
    const double    transz = pdbcoords->translation[2];
    const double  **rmat = (const double **) pdbcoords->matrix;
    const double    rmat00 = rmat[0][0], rmat01 = rmat[0][1], rmat02 = rmat[0][2],
                    rmat10 = rmat[1][0], rmat11 = rmat[1][1], rmat12 = rmat[1][2],
                    rmat20 = rmat[2][0], rmat21 = rmat[2][1], rmat22 = rmat[2][2];

    for (i = 0; i < pdbcoords->vlen; ++i)
    {
/*         MatPrint(pdbcoords->matrix, 3); */
/*             fprintf(stderr, */
/*                     " translation =    %8.3f %8.3f %8.3f \n", */
/*                     pdbcoords->translation[0], */
/*                     pdbcoords->translation[1], */
/*                     pdbcoords->translation[2]); */
/*             fprintf(stderr, */
/*                     " before:         x = %8.3f  y = %8.3f  z = %8.3f \n", */
/*                     pdbcoords->x[i], pdbcoords->y[i], pdbcoords->z[i]); */

        xt = x[i] - transx;
        yt = y[i] - transy;
        zt = z[i] - transz;

        x[i] = (xt * rmat00) + (yt * rmat10) + (zt * rmat20);
        y[i] = (xt * rmat01) + (yt * rmat11) + (zt * rmat21);
        z[i] = (xt * rmat02) + (yt * rmat12) + (zt * rmat22);

/*             fprintf(stderr, */
/*                     " after rotation: x = %8.3f  y = %8.3f  z = %8.3f \n", */
/*                     pdbcoords->x[i], pdbcoords->y[i], pdbcoords->z[i]); */
/*                     exit(0); */
    }
}


void
RotateCoordsOp(const Coords *coords1, const double **rmat, Coords *coords2)
{
    int             i;
    double          xt, yt, zt;
    double         *x1 = coords1->x, *y1 = coords1->y, *z1 = coords1->z,
                   *x2 = coords2->x, *y2 = coords2->y, *z2 = coords2->z;
    const double    rmat00 = rmat[0][0], rmat01 = rmat[0][1], rmat02 = rmat[0][2],
                    rmat10 = rmat[1][0], rmat11 = rmat[1][1], rmat12 = rmat[1][2],
                    rmat20 = rmat[2][0], rmat21 = rmat[2][1], rmat22 = rmat[2][2];

    for (i = 0; i < coords1->vlen; ++i)
    {
        xt = x1[i];
        yt = y1[i];
        zt = z1[i];

        x2[i] = xt * rmat00 + yt * rmat10 + zt * rmat20;
        y2[i] = xt * rmat01 + yt * rmat11 + zt * rmat21;
        z2[i] = xt * rmat02 + yt * rmat12 + zt * rmat22;
    }
}


void
RotateCoordsIp(Coords *coords, const double **rmat)
{
    int             i;
    double          xt, yt, zt;
    double         *x = coords->x, *y = coords->y, *z = coords->z;
    const double    rmat00 = rmat[0][0], rmat01 = rmat[0][1], rmat02 = rmat[0][2],
                    rmat10 = rmat[1][0], rmat11 = rmat[1][1], rmat12 = rmat[1][2],
                    rmat20 = rmat[2][0], rmat21 = rmat[2][1], rmat22 = rmat[2][2];

    for (i = 0; i < coords->vlen; ++i)
    {
        xt = x[i];
        yt = y[i];
        zt = z[i];

        x[i] = (xt * rmat00) + (yt * rmat10) + (zt * rmat20);
        y[i] = (xt * rmat01) + (yt * rmat11) + (zt * rmat21);
        z[i] = (xt * rmat02) + (yt * rmat12) + (zt * rmat22);
    }
}


void
RotateCoordsArrayIp(CoordsArray *cdsA, const double **rmat)
{
    int             i;

    for (i = 0; i < cdsA->cnum; ++i)
         RotateCoordsIp(cdsA->coords[i], rmat);

    RotateCoordsIp(cdsA->avecoords, rmat);
}


void
TransformCoordsIp(Coords *coords)
{
    const double  **rmat = (const double **) coords->matrix;
    const double    transx = coords->center[0];
    const double    transy = coords->center[1];
    const double    transz = coords->center[2];
    int             i;
    double          xt, yt, zt;
    double         *x = coords->x, *y = coords->y, *z = coords->z;
    const double    rmat00 = rmat[0][0], rmat01 = rmat[0][1], rmat02 = rmat[0][2],
                    rmat10 = rmat[1][0], rmat11 = rmat[1][1], rmat12 = rmat[1][2],
                    rmat20 = rmat[2][0], rmat21 = rmat[2][1], rmat22 = rmat[2][2];

    for (i = 0; i < coords->vlen; ++i)
    {
        xt = x[i] + transx;
        yt = y[i] + transy;
        zt = z[i] + transz;

        x[i] = (xt * rmat00) + (yt * rmat10) + (zt * rmat20);
        y[i] = (xt * rmat01) + (yt * rmat11) + (zt * rmat21);
        z[i] = (xt * rmat02) + (yt * rmat12) + (zt * rmat22);
    }
}


void
ScaleCoords(Coords *coords, const double scale)
{
    int             i;
    double         *x = coords->x, *y = coords->y, *z = coords->z;

    for (i = 0; i < coords->vlen; ++i)
    {
        x[i] *= scale;
        y[i] *= scale;
        z[i] *= scale;
    }
}


double
NormalizeWeightsOcc(double *w, double *o, int vlen)
{
    int             i;
    double          weightsum;
    double          normalize;

    weightsum = 0.0;
    for (i = 0; i < vlen; ++i)
        weightsum += o[i] * w[i];

    normalize = vlen / weightsum;

    for (i = 0; i < vlen; ++i)
        w[i] *= normalize;

    return(normalize);
}


double
NormalizeWeights(double *w, int vlen)
{
    int             i;
    double          weightsum;
    double          normalize;

/* normalize by trace of weight matrix */
    weightsum = 0.0;
    for (i = 0; i < vlen; ++i)
        weightsum += w[i];

    /* printf("\nweightsum: %f ", weightsum); */

    normalize = vlen / weightsum;
/* printf("normalize: %f", normalize); */
    for (i = 0; i < vlen; ++i)
        w[i] *= normalize;

    return(normalize);

/* normalize by determinant of weight matrix */
/*     weightsum = 1.0; */
/*     for (i = 0; i < coords->vlen; ++i) */
/*         weightsum *= w[i]; */
/*  */
/*     normalize = pow(weightsum, -1.0 / coords->vlen); */
/*  */
/*     for (i = 0; i < coords->vlen; ++i) */
/*         w[i] *= normalize; */

/* normalize by trace of covariance matrix */
/*     weightsum = 0.0; */
/*     for (i = 0; i < coords->vlen; ++i) */
/*         weightsum += vars[i]; */
/*  */
/*     normalize = coords->vlen / weightsum; */
/*  */
/*     for (i = 0; i < coords->vlen; ++i) */
/*         w[i] *= normalize; */

/* normalize by determinant of covariance matrix */
/*     weightsum = 1.0; */
/*     for (i = 0; i < coords->vlen; ++i) */
/*         weightsum *= vars[i]; */
/*  */
/*     normalize = pow(weightsum, 1.0 / coords->vlen); */
/*  */
/*     for (i = 0; i < coords->vlen; ++i) */
/*         w[i] *= normalize; */
}


void
CenMass(Coords *coords)
{
    int             i;
    const int       vlen = coords->vlen;
    double          xsum, ysum, zsum;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;

    xsum = ysum = zsum = 0.0;
    for (i = 0; i < vlen; ++i)
    {
        xsum += x[i];
        ysum += y[i];
        zsum += z[i];
    }

    coords->center[0] = xsum / vlen;
    coords->center[1] = ysum / vlen;
    coords->center[2] = zsum / vlen;
}


void
CenMassWtOp(Coords *coords, const CoordsArray *weights)
{
    int             i;
    double          tempx, tempy, tempz;
    const double   *wts = (const double *) weights->w;
    double          wti, wtsum;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;

    tempx = tempy = tempz = wtsum = 0.0;
    for (i = 0; i < coords->vlen; ++i)
    {
        wti = wts[i];
        wtsum += wti;
        tempx += (wti * x[i]);
        tempy += (wti * y[i]);
        tempz += (wti * z[i]);
    }

    coords->center[0] = tempx / wtsum;
    coords->center[1] = tempy / wtsum;
    coords->center[2] = tempz / wtsum;

/*     printf("\n********** %f %f %f", coords->center[0], coords->center[1], coords->center[2]); */
/*     fflush(NULL); */
}


void
CenMassWtIp(Coords *coords, const double *wts)
{
    int             i;
    double          tempx, tempy, tempz;
    double          wti, wtsum;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;

    tempx = tempy = tempz = wtsum = 0.0;
    for (i = 0; i < coords->vlen; ++i)
    {
        wti = wts[i];
//        printf("wt[%3d] = %8.3f\n", i, wti);

        wtsum += wti;
        tempx += (wti * x[i]);
        tempy += (wti * y[i]);
        tempz += (wti * z[i]);
    }

    coords->center[0] = tempx / wtsum;
    coords->center[1] = tempy / wtsum;
    coords->center[2] = tempz / wtsum;

/*     printf("wtsum = %8.3f\n", wtsum); */
/*     printf("wtocc = %8.3f\n", wtocc); */
/*     fflush(NULL); */

/*      printf("\nDT: % 8.3f % 8.3f % 8.3f\n",  */
/*            coords->center[0], coords->center[1], coords->center[2]);  */
/*      fflush(NULL);  */
}


void
CenMassCovOp(Coords *coords, const CoordsArray *weights)
{
    double          tempx, tempy, tempz;
    int             i, j;
    double         *covx = coords->covx,
                   *covy = coords->covy,
                   *covz = coords->covz;
    double          wtsum;

/* #include "internmat.h" */
/* #include "LedoitWolf.h" */
/* for (i = 0; i < coords->vlen; ++i) */
/*     for (j = 0; j < coords->vlen; ++j) */
/*         weights->CovMat[i][j] = internmat[i][j]; */
/*  */
/* CalcLedoitCovMatPar((CoordsArray *) weights); */
/* CovInvWeightLAPACK((CoordsArray *) weights); */
/* NormalizeCovMat(weights->WtMat, coords->vlen); */

    wtsum = 0.0;
    for (i = 0; i < coords->vlen; ++i)
        for (j = 0; j < coords->vlen; ++j)
            wtsum += weights->WtMat[i][j];

    tempx = tempy = tempz = 0.0;
    for (i = 0; i < coords->vlen; ++i)
    {
        tempx += covx[i];
        tempy += covy[i];
        tempz += covz[i];
    }

    coords->center[0] = tempx / wtsum;
    coords->center[1] = tempy / wtsum;
    coords->center[2] = tempz / wtsum;

/*     printf("\n% f % f % f", */
/*            coords->center[0], coords->center[1], coords->center[2]); */
/*     fflush(NULL); */
}


void
CenMassCov(Coords *coords, const double **wtmat)
{
    double          tempx, tempy, tempz;
    int             i, j;
    double         *covx = coords->covx,
                   *covy = coords->covy,
                   *covz = coords->covz;
    double          wtsum;

    CalcCovCoords(coords, wtmat);

    wtsum = 0.0;
    for (i = 0; i < coords->vlen; ++i)
        for (j = 0; j < coords->vlen; ++j)
            wtsum += wtmat[i][j];

    tempx = tempy = tempz = 0.0;
    for (i = 0; i < coords->vlen; ++i)
    {
        tempx += covx[i];
        tempy += covy[i];
        tempz += covz[i];
    }

    coords->center[0] = tempx / wtsum;
    coords->center[1] = tempy / wtsum;
    coords->center[2] = tempz / wtsum;
}


/*void*/
/*CenMassCovOcc(Coords *coords, const double **wtmat)*/
/*{*/
/*    double          tempx, tempy, tempz;*/
/*    int             i, j;*/
/*    const double   *occ = (const double *) coords->o;*/
/*    double         *covx = coords->covx,*/
/*                   *covy = coords->covy,*/
/*                   *covz = coords->covz;*/
/*    double          wtsum;*/
/**/
/*    wtsum = 0.0;*/
/*    for (i = 0; i < coords->vlen; ++i)*/
/*        for (j = 0; j < coords->vlen; ++j)*/
/*            wtsum += occ[i] * occ[j] * wtmat[i][j];*/
/**/
/*    tempx = tempy = tempz = 0.0;*/
/*    for (i = 0; i < coords->vlen; ++i)*/
/*    {*/
/*        tempx += occ[i] * covx[i];*/
/*        tempy += occ[i] * covy[i];*/
/*        tempz += occ[i] * covz[i];*/
/*    }*/
/**/
/*    coords->center[0] = tempx / wtsum;*/
/*    coords->center[1] = tempy / wtsum;*/
/*    coords->center[2] = tempz / wtsum;*/
/*}*/
/**/
/**/
/*void*/
/*CenMassWtOpOcc(Coords *coords, const CoordsArray *weights)*/
/*{*/
/*    int             i;*/
/*    double          tempx, tempy, tempz;*/
/*    const double   *wts = (const double *) weights->w;*/
/*    const double   *occ = (const double *) coords->o;*/
/*    double          wti, wtsum;*/
/*    const double   *x = (const double *) coords->x,*/
/*                   *y = (const double *) coords->y,*/
/*                   *z = (const double *) coords->z;*/
/**/
/*    tempx = tempy = tempz = wtsum = 0.0;*/
/*    for (i = 0; i < coords->vlen; ++i)*/
/*    {*/
/*        wti = wts[i] * occ[i];*/
/*        wtsum += wti;*/
/*        tempx += (wti * x[i]);*/
/*        tempy += (wti * y[i]);*/
/*        tempz += (wti * z[i]);*/
/*    }*/
/**/
/*    coords->center[0] = tempx / wtsum;*/
/*    coords->center[1] = tempy / wtsum;*/
/*    coords->center[2] = tempz / wtsum;*/
/*}*/
/**/
/**/
/*void*/
/*CenMassOcc(Coords *coords)*/
/*{*/
/*    int             i;*/
/*    double          tempx, tempy, tempz, occi, occsum;*/
/*    const double   *occ = (const double *) coords->o;*/
/*    const double   *x = (const double *) coords->x,*/
/*                   *y = (const double *) coords->y,*/
/*                   *z = (const double *) coords->z;*/
/**/
/*    tempx = tempy = tempz = occsum = 0.0;*/
/*    for (i = 0; i < coords->vlen; ++i)*/
/*    {*/
/*        occi = occ[i];*/
/*        occsum += occi;*/
/*        tempx += (occi * x[i]);*/
/*        tempy += (occi * y[i]);*/
/*        tempz += (occi * z[i]);*/
/*    }*/
/**/
/*    coords->center[0] = tempx / occsum;*/
/*    coords->center[1] = tempy / occsum;*/
/*    coords->center[2] = tempz / occsum;*/
/*}*/


void
CenMassOccVec(Coords *coords, double *cenmass)
{
    int             i;
    double          tempx, tempy, tempz, occi, occsum;
    const double   *occ = (const double *) coords->o;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;

    tempx = tempy = tempz = occsum = 0.0;
    for (i = 0; i < coords->vlen; ++i)
    {
        occi = occ[i];
        occsum += occi;
        tempx += (occi * x[i]);
        tempy += (occi * y[i]);
        tempz += (occi * z[i]);
    }

    cenmass[0] = tempx / occsum;
    cenmass[1] = tempy / occsum;
    cenmass[2] = tempz / occsum;
}


void
CenMassWtIpOcc(Coords *coords, const double *wts)
{
    int             i;
    double          tempx, tempy, tempz;
    const double   *occ = (const double *) coords->o;
    double          wti, wtsum;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;

    tempx = tempy = tempz = wtsum = 0.0;
    for (i = 0; i < coords->vlen; ++i)
    {
        wti = wts[i] * occ[i];
        wtsum += wti;
        tempx += (wti * x[i]);
        tempy += (wti * y[i]);
        tempz += (wti * z[i]);
    }

    coords->center[0] = tempx / wtsum;
    coords->center[1] = tempy / wtsum;
    coords->center[2] = tempz / wtsum;
//    printf("wtsum = %8.3f\n", wtsum);
//    fflush();

/*     printf("\n% 8.3f % 8.3f % 8.3f",  */
/*            coords->center[0], coords->center[1], coords->center[2]);  */
/*     fflush(NULL); */ 
}




/*                 coordsi->x[j] = avex[j]*rmat00 + avey[j]*rmat01 + avez[j]*rmat02; */
/*                 coordsi->y[j] = avex[j]*rmat10 + avey[j]*rmat11 + avez[j]*rmat12; */
/*                 coordsi->z[j] = avex[j]*rmat20 + avey[j]*rmat21 + avez[j]*rmat22; */

void
CenMassWtIpEM(Coords *coords, const Coords *avecds, const double *wts)
{
    int             i;
    double          tempx, tempy, tempz;
    const double   *occ = (const double *) coords->o;
    double          wti, wtsum;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;
    double          rmat00, rmat01, rmat02,
                    rmat10, rmat11, rmat12,
                    rmat20, rmat21, rmat22;
    double        **rmat = coords->matrix;
    double         *avex = avecds->x,
                   *avey = avecds->y,
                   *avez = avecds->z;

    rmat00 = rmat[0][0], rmat01 = rmat[0][1], rmat02 = rmat[0][2],
    rmat10 = rmat[1][0], rmat11 = rmat[1][1], rmat12 = rmat[1][2],
    rmat20 = rmat[2][0], rmat21 = rmat[2][1], rmat22 = rmat[2][2];

    tempx = tempy = tempz = wtsum = 0.0;
    for (i = 0; i < coords->vlen; ++i)
    {
        wti = wts[i];
//        printf("wt[%3d] = %8.3f\n", i, wti);
        
        if (occ[i] == 1.0)
        {
            //wtsum += 1.0;
            wtsum += wti;
            tempx += (wti * x[i]);
            tempy += (wti * y[i]);
            tempz += (wti * z[i]);
        }
        else if (occ[i] == 0.0)
        {
            tempx += (wti * (avex[i]*rmat00 + avey[i]*rmat01 + avez[i]*rmat02));
            tempy += (wti * (avex[i]*rmat10 + avey[i]*rmat11 + avez[i]*rmat12));
            tempz += (wti * (avex[i]*rmat20 + avey[i]*rmat21 + avez[i]*rmat22));

/*             tempx += (wti * (avex[i]*rmat00 + avey[i]*rmat10 + avez[i]*rmat20)); */
/*             tempy += (wti * (avex[i]*rmat01 + avey[i]*rmat11 + avez[i]*rmat21)); */
/*             tempz += (wti * (avex[i]*rmat02 + avey[i]*rmat12 + avez[i]*rmat22)); */
        }
    }

    coords->center[0] = tempx / wtsum;
    coords->center[1] = tempy / wtsum;
    coords->center[2] = tempz / wtsum;

/*     printf("wtsum = %8.3f\n", wtsum); */
/*     fflush(NULL); */

/*     printf("\nEM: % 8.3f % 8.3f % 8.3f",  */
/*            coords->center[0], coords->center[1], coords->center[2]);  */
/*     fflush(NULL);  */
}


/*void*/
/*CenMassCovOpOcc(Coords *coords, const CoordsArray *weights)*/
/*{*/
/*    double          tempx, tempy, tempz;*/
/*    int             i, j;*/
/*    const double   *occ = (const double *) coords->o;*/
/*    double         *covx = coords->covx,*/
/*                   *covy = coords->covy,*/
/*                   *covz = coords->covz;*/
/*    double          wtsum;*/
/**/
/*    wtsum = 0.0;*/
/*    for (i = 0; i < coords->vlen; ++i)*/
/*        for (j = 0; j < coords->vlen; ++j)*/
/*            wtsum += occ[i] * occ[j] * weights->WtMat[i][j];*/
/**/
/*    tempx = tempy = tempz = 0.0;*/
/*    for (i = 0; i < coords->vlen; ++i)*/
/*    {*/
/*        tempx += occ[i] * covx[i];*/
/*        tempy += occ[i] * covy[i];*/
/*        tempz += occ[i] * covz[i];*/
/*    }*/
/**/
/*    coords->center[0] = tempx / wtsum;*/
/*    coords->center[1] = tempy / wtsum;*/
/*    coords->center[2] = tempz / wtsum;*/
/*}*/


void
ApplyCenter(Coords *coords, const double cenx, const double ceny, const double cenz)
{
    int             i;
    double         *x = coords->x, *y = coords->y, *z = coords->z;

    for (i = 0; i < coords->vlen; ++i)
    {
        x[i] -= cenx;
        y[i] -= ceny;
        z[i] -= cenz;
    }
}


void
ApplyCenterIp(Coords *coords)
{
    int             i;
    double         *x = coords->x, *y = coords->y, *z = coords->z;
    const double    cenx = coords->center[0],
                    ceny = coords->center[1],
                    cenz = coords->center[2];

    for (i = 0; i < coords->vlen; ++i)
    {
        x[i] -= cenx;
        y[i] -= ceny;
        z[i] -= cenz;
    }
}


void
ApplyNegCenterIp(Coords *coords)
{
    int             i;
    double         *x = coords->x, *y = coords->y, *z = coords->z;
    const double    cenx = coords->center[0],
                    ceny = coords->center[1],
                    cenz = coords->center[2];

    for (i = 0; i < coords->vlen; ++i)
    {
        x[i] += cenx;
        y[i] += ceny;
        z[i] += cenz;
    }
}


/* apply the center from 'coords_base' to 'coords' */
void
ApplyCenterOp(Coords *coords1, const Coords *coords2)
{
    int             i;
    double         *x1 = coords1->x, *y1 = coords1->y, *z1 = coords1->z;
    const double   *x2 = coords2->x, *y2 = coords2->y, *z2 = coords2->z;
    const double    cenx = coords2->center[0],
                    ceny = coords2->center[1],
                    cenz = coords2->center[2];

    for (i = 0; i < coords1->vlen; ++i)
    {
        x1[i] = x2[i] - cenx;
        y1[i] = y2[i] - ceny;
        z1[i] = z2[i] - cenz;
    }
}


void
TransCoordsIp(Coords *coords, const double *trans)
{
    int             i;
    double         *x = coords->x, *y = coords->y, *z = coords->z;
    const double    transx = trans[0],
                    transy = trans[1],
                    transz = trans[2];

    for (i = 0; i < coords->vlen; ++i)
    {
        x[i] += transx;
        y[i] += transy;
        z[i] += transz;
    }
}


void
NegTransCoordsIp(Coords *coords, const double *trans)
{
    int             i;
    double         *x = coords->x, *y = coords->y, *z = coords->z;
    const double    transx = trans[0],
                    transy = trans[1],
                    transz = trans[2];

    for (i = 0; i < coords->vlen; ++i)
    {
        x[i] -= transx;
        y[i] -= transy;
        z[i] -= transz;
    }
}


/* Copy average coords to atoms with 0 occupancy, part of the EM algorithm's
   E-step for estimating missing data */
void
EM_MissingCoords(CoordsArray *cdsA)
{
    int             i, j;
    double         *avex = cdsA->avecoords->x, *avey = cdsA->avecoords->y,
                   *avez = cdsA->avecoords->z;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsi;

    for (i = 0; i < cdsA->cnum; ++i)
    {
        coordsi = (Coords *) coords[i];

        for (j = 0; j < cdsA->vlen; ++j)
        {
            if (coordsi->o[j] == 0.0)
            {
                coordsi->x[j] = avex[j];
                coordsi->y[j] = avey[j];
                coordsi->z[j] = avez[j];
            }
        }
    }
}


void
EM_MissingCoordsOp(CoordsArray *baseA, CoordsArray *scratchA)
{
    int             i, j;
    double        **rmat = NULL;
    double         *avex = scratchA->avecoords->x,
                   *avey = scratchA->avecoords->y,
                   *avez = scratchA->avecoords->z;
    const Coords  **coords = (const Coords **) baseA->coords;
    Coords         *coordsi;

    double          rmat00, rmat01, rmat02,
                    rmat10, rmat11, rmat12,
                    rmat20, rmat21, rmat22;

    for (i = 0; i < baseA->cnum; ++i)
    {
        coordsi = (Coords *) coords[i];

        rmat = scratchA->coords[i]->matrix;

        rmat00 = rmat[0][0], rmat01 = rmat[0][1], rmat02 = rmat[0][2],
        rmat10 = rmat[1][0], rmat11 = rmat[1][1], rmat12 = rmat[1][2],
        rmat20 = rmat[2][0], rmat21 = rmat[2][1], rmat22 = rmat[2][2];

        for (j = 0; j < baseA->vlen; ++j)
        {
            if (coordsi->o[j] == 0.0)
            {
                coordsi->x[j] = avex[j]*rmat00 + avey[j]*rmat01 + avez[j]*rmat02;
                coordsi->y[j] = avex[j]*rmat10 + avey[j]*rmat11 + avez[j]*rmat12;
                coordsi->z[j] = avex[j]*rmat20 + avey[j]*rmat21 + avez[j]*rmat22;

                coordsi->x[j] -= scratchA->coords[i]->translation[0];
                coordsi->y[j] -= scratchA->coords[i]->translation[1];
                coordsi->z[j] -= scratchA->coords[i]->translation[2];
            }
        }
    }
}


void
AveCoordsNovec(CoordsArray *cdsA)
{
    int             i, j;
    double          xtmp, ytmp, ztmp, /* otmp,  */btmp;
    double         *avex = cdsA->avecoords->x,
                   *avey = cdsA->avecoords->y,
                   *avez = cdsA->avecoords->z,
                   *aveo = cdsA->avecoords->o,
                   *aveb = cdsA->avecoords->b;
    const int       cnum = cdsA->cnum;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsj;
    double          invcnum = 1.0 / (double) cnum;

    for (i = 0; i < cdsA->vlen; ++i)
    {
        xtmp = ytmp = ztmp = /* otmp = */ btmp = 0.0;
        for (j = 0; j < cnum; ++j)
        {
            coordsj = (Coords *) coords[j];
            xtmp += coordsj->x[i];
            ytmp += coordsj->y[i];
            ztmp += coordsj->z[i];
            /* otmp += coordsj->o[i]; */
            btmp += coordsj->b[i];
        }

        avex[i] = xtmp * invcnum;
        avey[i] = ytmp * invcnum;
        avez[i] = ztmp * invcnum;
        aveo[i] = 1.0;
        aveb[i] = btmp * invcnum;
    }
}


/* Calculate the ML estimate of a hierarchical mean, where the atoms
   are normally distributed with hyper-mean zero */
/* See also below, which is probably more valid (only the weighted mean has zero centroid) */
/* 2009-06-11 */
double
HierAveCoords2(CoordsArray *cdsA)
{
    int             i, j;
    double         *avex = cdsA->avecoords->x,
                   *avey = cdsA->avecoords->y,
                   *avez = cdsA->avecoords->z,
                   *aveo = cdsA->avecoords->o,
                   *aveb = cdsA->avecoords->b;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsj;
    double          invcnum, psi;


    psi = 0.0;
    for (i = 0; i < vlen; ++i)
        psi += (avex[i]*avex[i] + avey[i]*avey[i] + avez[i]*avez[i]);
    psi /= (3.0 * vlen);

    memset(avex, 0, vlen * sizeof(double));
    memset(avey, 0, vlen * sizeof(double));
    memset(avez, 0, vlen * sizeof(double));
    memset(aveb, 0, vlen * sizeof(double));

    for (j = 0; j < cnum; ++j)
    {
        coordsj = (Coords *) coords[j];

		for (i = 0; i < vlen; ++i)
		{
			avex[i] += coordsj->x[i];
			avey[i] += coordsj->y[i];
			avez[i] += coordsj->z[i];
			aveb[i] += coordsj->b[i];
		}
    }

    for (i = 0; i < vlen; ++i)
    {
        invcnum = 1.0 / ((double) cnum + cdsA->var[i] / psi);
        //printf("\ninvcnum = %e  %e", invcnum, 1.0/cnum);
    
		avex[i] *= invcnum;
		avey[i] *= invcnum;
		avez[i] *= invcnum;
		aveo[i] = 1.0;
		aveb[i] *= invcnum;
    }
    
    return(psi);
}


/* Calculate the ML estimate of a hierarchical mean, where the variance-weighted atoms 
   are normally distributed with hyper-mean zero */
/* 2009-06-11 */
double
HierAveCoords(CoordsArray *cdsA)
{
    int             i, j;
    double         *avex = cdsA->avecoords->x,
                   *avey = cdsA->avecoords->y,
                   *avez = cdsA->avecoords->z,
                   *aveo = cdsA->avecoords->o,
                   *aveb = cdsA->avecoords->b;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsj;
    double          invcnum, psi, norm;


    psi = norm = 0.0;
    for (i = 0; i < vlen; ++i)
    {
        norm += 1.0 / cdsA->var[i];
        psi += (avex[i]*avex[i]/cdsA->var[i] + 
                avey[i]*avey[i]/cdsA->var[i] + 
                avez[i]*avez[i]/cdsA->var[i]);
    }
    psi /= (3.0 * norm);

    memset(avex, 0, vlen * sizeof(double));
    memset(avey, 0, vlen * sizeof(double));
    memset(avez, 0, vlen * sizeof(double));
    memset(aveb, 0, vlen * sizeof(double));

    for (j = 0; j < cnum; ++j)
    {
        coordsj = (Coords *) coords[j];

		for (i = 0; i < vlen; ++i)
		{
			avex[i] += coordsj->x[i];
			avey[i] += coordsj->y[i];
			avez[i] += coordsj->z[i];
			aveb[i] += coordsj->b[i];
		}
    }

    for (i = 0; i < vlen; ++i)
    {
        invcnum = 1.0 / ((double) cnum + 1.0 / psi);
        //printf("\ninvcnum = %e  %e", invcnum, 1.0/cnum);
    
		avex[i] *= invcnum;
		avey[i] *= invcnum;
		avez[i] *= invcnum;
		aveo[i] = 1.0;
		aveb[i] *= invcnum;
    }
    
    return(psi);
}


void
AveCoords(CoordsArray *cdsA)
{
    int             i, j;
    double         *avex = cdsA->avecoords->x,
                   *avey = cdsA->avecoords->y,
                   *avez = cdsA->avecoords->z,
                   *aveo = cdsA->avecoords->o,
                   *aveb = cdsA->avecoords->b;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsj;
    double          invcnum = 1.0 / (double) cnum;

    memset(avex, 0, vlen * sizeof(double));
    memset(avey, 0, vlen * sizeof(double));
    memset(avez, 0, vlen * sizeof(double));
    memset(aveb, 0, vlen * sizeof(double));

    for (j = 0; j < cnum; ++j)
    {
        coordsj = (Coords *) coords[j];

		for (i = 0; i < vlen; ++i)
		{
			avex[i] += coordsj->x[i];
			avey[i] += coordsj->y[i];
			avez[i] += coordsj->z[i];
			aveb[i] += coordsj->b[i];
		}
    }

    for (i = 0; i < vlen; ++i)
    {
		avex[i] *= invcnum;
		avey[i] *= invcnum;
		avez[i] *= invcnum;
		aveo[i] = 1.0;
		aveb[i] *= invcnum;
    }
}

void
AveCoordsTB(CoordsArray *cdsA, int omit)
{
    int             i, j;
    double         *avex = cdsA->avecoords->x,
                   *avey = cdsA->avecoords->y,
                   *avez = cdsA->avecoords->z,
                   *aveo = cdsA->avecoords->o,
                   *aveb = cdsA->avecoords->b;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsj;
    double          invcnum = 1.0 / (double) (cnum-1);

    memset(avex, 0, vlen * sizeof(double));
    memset(avey, 0, vlen * sizeof(double));
    memset(avez, 0, vlen * sizeof(double));
    memset(aveb, 0, vlen * sizeof(double));

    for (j = 0; j < cnum; ++j)
    {
    
        if (j == omit)
            continue;

        coordsj = (Coords *) coords[j];

		for (i = 0; i < vlen; ++i)
		{
			avex[i] += coordsj->x[i];
			avey[i] += coordsj->y[i];
			avez[i] += coordsj->z[i];
			aveb[i] += coordsj->b[i];
		}
    }

    for (i = 0; i < vlen; ++i)
    {
		avex[i] *= invcnum;
		avey[i] *= invcnum;
		avez[i] *= invcnum;
		aveo[i] = 1.0;
		aveb[i] *= invcnum;
    }
}



static void
*AveCds_pth(void *avedata_ptr)
{
    AveData        *avedata = (AveData *) avedata_ptr;
    int             i, j;
    double          xtmp, ytmp, ztmp, /* otmp,  */btmp;
    double         *avex = avedata->cdsA->avecoords->x,
                   *avey = avedata->cdsA->avecoords->y,
                   *avez = avedata->cdsA->avecoords->z,
                   *aveo = avedata->cdsA->avecoords->o,
                   *aveb = avedata->cdsA->avecoords->b;
    const int       cnum = avedata->cnum;
    double          invcnum = 1.0 / cnum;
    const Coords  **coords = (const Coords **) avedata->cdsA->coords;
    Coords         *coordsj;

    for (i = avedata->start; i < avedata->end; ++i)
    {
        xtmp = ytmp = ztmp = /* otmp = */ btmp = 0.0;
        for (j = 0; j < cnum; ++j)
        {
            coordsj = (Coords *) coords[j];
            xtmp += coordsj->x[i];
            ytmp += coordsj->y[i];
            ztmp += coordsj->z[i];
            /* otmp += coordsj->o[i]; */
            btmp += coordsj->b[i];
        }

        avex[i] = xtmp * invcnum;
        avey[i] = ytmp * invcnum;
        avez[i] = ztmp * invcnum;
        aveo[i] = 1.0;
        aveb[i] = btmp * invcnum;
    }

    pthread_exit((void *) 0);
}


void
AveCoords_pth(CoordsArray *cdsA, AveData **avedata, pthread_t *callThd,
              pthread_attr_t *attr, const int thrdnum)
{
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    int             i, rc = 0, incr;

    incr = vlen / thrdnum;

	for (i = 0; i < thrdnum - 1; ++i)
	{
        avedata[i]->cdsA = cdsA;
        avedata[i]->start = i * incr;
        avedata[i]->end = i*incr + incr;
        avedata[i]->vlen = vlen;
        avedata[i]->cnum = cnum;

        rc = pthread_create(&callThd[i], attr, AveCds_pth, (void *) avedata[i]);

        if (rc)
        {
            printf("ERROR811: return code from pthread_create() %d is %d\n", i, rc);
            exit(EXIT_FAILURE);
        }
	}

	avedata[thrdnum - 1]->cdsA = cdsA;
	avedata[thrdnum - 1]->start = (thrdnum - 1) * incr;
	avedata[thrdnum - 1]->end = vlen;
	avedata[thrdnum - 1]->vlen = vlen;
	avedata[thrdnum - 1]->cnum = cnum;

	rc = pthread_create(&callThd[thrdnum - 1], attr, AveCds_pth, (void *) avedata[thrdnum - 1]);

	if (rc)
	{
		printf("ERROR811: return code from pthread_create() %d is %d\n", i, rc);
		exit(EXIT_FAILURE);
	}

    for (i = 0; i < thrdnum; ++i)
    {
        rc = pthread_join(callThd[i], (void **) NULL);

        if (rc)
        {
            printf("ERROR812: return code from pthread_join() %d is %d\n", i, rc);
            exit(EXIT_FAILURE);
        }
    }

    return;
}


/* void */
/* AveCoordsOcc(CoordsArray *cdsA) */
/* { */
/*     int             i, j; */
/*     double          xtmp, ytmp, ztmp, btmp, occ; */
/*     double         *avex = cdsA->avecoords->x, */
/*                    *avey = cdsA->avecoords->y, */
/*                    *avez = cdsA->avecoords->z, */
/*                    *aveo = cdsA->avecoords->o, */
/*                    *aveb = cdsA->avecoords->b; */
/*     const Coords  **coords = (const Coords **) cdsA->coords; */
/*     Coords         *coordsj; */
/*     const double    cnum = cdsA->cnum; */
/*     double          invdf, occsum; */
/*  */
/*     for (i = 0; i < cdsA->vlen; ++i) */
/*     { */
/*         xtmp = ytmp = ztmp = btmp = 0.0; */
/*         occsum = 0.0; */
/*         for (j = 0; j < cnum; ++j) */
/*         { */
/*             coordsj = (Coords *) coords[j]; */
/*             occ = coordsj->o[i]; */
/*             occsum += occ; */
/*             xtmp += occ * coordsj->x[i]; */
/*             ytmp += occ * coordsj->y[i]; */
/*             ztmp += occ * coordsj->z[i]; */
/*             btmp += occ * coordsj->b[i]; */
/*         } */
/*  */
/*         //invdf = 1.0 / cdsA->df[i]; */
/*         invdf = 1.0 / occsum; */
/*  */
/*         avex[i] = xtmp * invdf; */
/*         avey[i] = ytmp * invdf; */
/*         avez[i] = ztmp * invdf; */
/*         aveo[i] = 1.0; */
/*         aveb[i] = btmp * invdf; */
/*     } */
/* } */


void
AveCoordsOcc(CoordsArray *cdsA)
{
    int             i, j;
    double         *avex = cdsA->avecoords->x,
                   *avey = cdsA->avecoords->y,
                   *avez = cdsA->avecoords->z,
                   *aveo = cdsA->avecoords->o,
                   *aveb = cdsA->avecoords->b;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsj;
    double          occ, occsum, invocc;

    memset(avex, 0, vlen * sizeof(double));
    memset(avey, 0, vlen * sizeof(double));
    memset(avez, 0, vlen * sizeof(double));
    memset(aveb, 0, vlen * sizeof(double));

    for (i = 0; i < vlen; ++i)
    {
        occsum = 0.0;
        for (j = 0; j < cnum; ++j)
        {
            coordsj = (Coords *) coords[j];
            occ = coordsj->o[i];
            occsum += occ;
			avex[i] += occ * coordsj->x[i];
			avey[i] += occ * coordsj->y[i];
			avez[i] += occ * coordsj->z[i];
			aveb[i] += occ * coordsj->b[i];
        }

        invocc = 1.0 / occsum;

		avex[i] *= invocc;
		avey[i] *= invocc;
		avez[i] *= invocc;
        aveo[i] = 1.0;
		aveb[i] *= invocc;
    }
}


void
CalcCoordsPrincAxes(Coords *coords, double **rotmat)
{
    double         *evals = (double *) malloc(3 * sizeof(double));
    double          det;
    int             i, j;

    CoordsInnerProd2(coords);
    //eigensym((const double **) coords->innerprod2, evals, rotmat, 3);
    //Mat3TransposeIp(rotmat);
    //printf("\nCalcCoordsPrincAxes B:");
    //Mat3Print(rotmat);
    //CoordsInnerProd2(coords);
    jacobi3_cyc(coords->innerprod2, evals, rotmat, 1e-8);
    //Mat3Print(rotmat);
    //printf("\nCalcCoordsPrincAxes A:");
    det = Mat3Det((const double **) rotmat);

    if (det < 0)
    {
        //printf("\nNEGATIVE DETERMINANT\n");
        for (i = 0; i < 3; ++i)
        {
            if (rotmat[i][i] < 0)
            {
                for (j = 0; j < 3; ++j)
                    rotmat[i][j] *= -1.0;
                    
                break;
            }
        }

        //Mat3Print(rotmat);
    }

/*     Mat3Print(rotmat); */
/*     printf("\n evals %f %f %f  det = %f %f", */
/*            evals[0], evals[1], evals[2], det, Mat3Det((const double **)rotmat)); */

    free(evals);
}


void
SumCoordsTB(CoordsArray *cdsA, const int exclude)
{
    int             i, j;
    double          xtmp, ytmp, ztmp, otmp, btmp;
    const Coords  **coords = (const Coords **) cdsA->coords;

    for (i = 0; i < cdsA->vlen; ++i)
    {
        xtmp = ytmp = ztmp = otmp = btmp = 0.0;
        for (j = 0; j < exclude; ++j)
        {
            xtmp += coords[j]->x[i];
            ytmp += coords[j]->y[i];
            ztmp += coords[j]->z[i];
            otmp += coords[j]->o[i];
            btmp += coords[j]->b[i];
        }

        for (j = exclude + 1; j < cdsA->cnum; ++j)
        {
            xtmp += coords[j]->x[i];
            ytmp += coords[j]->y[i];
            ztmp += coords[j]->z[i];
            otmp += coords[j]->o[i];
            btmp += coords[j]->b[i];
        }
    }
}


/* multiplies first quaternion by second, puts result in second */
/* when concatenating quats, you multiply the second rotation by the first,
   in that noncommunative order */
double
*ConcatQuatsIp(const double *quat1, double *quat2)
{
    double          ww, xx, yy, zz,
                    wx, xw, yz, zy,
                    wy, xz, yw, zx,
                    wz, xy, yx, zw;

    ww = quat1[0] * quat2[0];
    xx = quat1[1] * quat2[1];
    yy = quat1[2] * quat2[2];
    zz = quat1[3] * quat2[3];
    wx = quat1[0] * quat2[1];
    xw = quat1[1] * quat2[0];
    yz = quat1[2] * quat2[3];
    zy = quat1[3] * quat2[2];
    wy = quat1[0] * quat2[2];
    xz = quat1[1] * quat2[3];
    yw = quat1[2] * quat2[0];
    zx = quat1[3] * quat2[1];
    wz = quat1[0] * quat2[3];
    xy = quat1[1] * quat2[2];
    yx = quat1[2] * quat2[1];
    zw = quat1[3] * quat2[0];

    quat2[0] = ww - xx - yy - zz;
    quat2[1] = wx + xw + yz - zy;
    quat2[2] = wy - xz - yw - zx;
    quat2[3] = wz + xy - yx + zw;

    return(quat2);
}
