/*
    Theseus - maximum likelihood superpositioning of macromolecular structures

    Copyright (C) 2004-2008 Douglas L. Theobald

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the:

    Free Software Foundation, Inc.,
    59 Temple Place, Suite 330,
    Boston, MA  02111-1307  USA

    -/_|:|_|_\-
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <float.h>
#include "pdbStats.h"
#include "pdbIO.h"
#include "Embed.h"
#include "FragCoords.h"
//#include "quartic2.h"
#include "QuarticHornFrag.h"


/* evaluates the horn quartic for coefficients c and given x */
static double
eval_horn_quart(double *c, double x)
{
    return(((x*x+c[2])*x+c[1])*x+c[0]);
}


/* evaluates the derivative of the horn quartic for coefficients c and given x */
static double
eval_horn_quart_deriv(double *c, double x)
{
    return(2.0*(2.0*x*x + c[2])*x + c[1]);
}


/* evaluates the quartic for coefficients c and given x */
double
eval_quart(double *c, double x)
{
    return((((c[4]*x+c[3])*x+c[2])*x+c[1])*x+c[0]);
}


/* evaluates the derivative of the quartic for coefficients c and given x */
double
eval_quart_deriv(double *c, double x)
{
    return(((4.0*c[4]*x + 3.0*c[3])*x + 2.0*c[2])*x + c[1]);
}


/* Newton-Raphson root finding */
static double
horn_root(double *coeff, double guess, int n, double delta)
{
    int             i;
    double          x, xold;
    
    i = 0;
    x = guess;

    while (i < n)
    {
       xold = x;
       x -= (eval_horn_quart(coeff, x) / eval_horn_quart_deriv(coeff, x));
    
       if (fabs(x - xold) < delta)
          return(x);
    
       ++i;
    }

    fprintf(stderr, "\n\n ERROR21: newton-raphson root finding in \'root\' did not converge \n");
    return(x);
}


void
ComputeQuartHornFrag(const FragCoords *frag1, const FragCoords *frag2, double *coeff)
{
    double          Sxx, Sxy, Sxz,
                    Syx, Syy, Syz,
                    Szx, Szy, Szz,
                    Szz2, Syy2, Sxx2,
                    Sxy2, Syz2, Sxz2,
                    Syx2, Szy2, Szx2;
    double          x1, x2, y1, y2, z1, z2;
    const double   *fx1 = frag1->x, *fy1 = frag1->y, *fz1 = frag1->z,
                   *fx2 = frag2->x, *fy2 = frag2->y, *fz2 = frag2->z;
    int             i;

    Sxx = Sxy = Sxz = Syx = Syy = Syz = Szx = Szy = Szz = 0.0;
    for (i = 0; i < frag1->fraglen; ++i)
    {
        x1 = fx1[i];
        y1 = fy1[i];
        z1 = fz1[i];
        x2 = fx2[i];
        y2 = fy2[i];
        z2 = fz2[i];
   
    
        Sxx += (x1 * x2);
        Sxy += (x1 * y2);
        Sxz += (x1 * z2);

        Syx += (y1 * x2);
        Syy += (y1 * y2);
        Syz += (y1 * z2);

        Szx += (z1 * x2);
        Szy += (z1 * y2);
        Szz += (z1 * z2);  
    }

    Sxx2 = Sxx * Sxx;
    Syy2 = Syy * Syy;
    Szz2 = Szz * Szz;

    Sxy2 = Sxy * Sxy;
    Syz2 = Syz * Syz;
    Sxz2 = Sxz * Sxz;

    Syx2 = Syx * Syx;
    Szy2 = Szy * Szy;
    Szx2 = Szx * Szx;

    coeff[4] = 1.0;
    coeff[3] = 0.0;
    coeff[2] = -2.0 * (Sxx2 + Syy2 + Szz2 + Sxy2 + Syx2 + Sxz2 + Szx2 + Syz2 + Szy2);
    coeff[1] = 8.0 * (Sxx*Syz*Szy + Syy*Szx*Sxz + Szz*Sxy*Syx - Sxx*Syy*Szz - Syz*Szx*Sxy - Szy*Syx*Sxz);

    coeff[0] = mysquare(Sxy2 + Sxz2 - Syx2 - Szx2);
    coeff[0] += (+(Sxy+Syx)*(Syz-Szy)+(Sxz-Szx)*(Sxx-Syy-Szz)) * (-(Sxy-Syx)*(Syz+Szy)+(Sxz-Szx)*(Sxx+Syy-Szz));
    coeff[0] +=  ((-Sxx2 + Syy2 + Syz2 + 2.0*Syz*Szy + Szy2 - 2.0*Syy*Szz + Szz2)
                * (-Sxx2 + Syy2 + Syz2 - 2.0*Syz*Szy + Szy2 + 2.0*Syy*Szz + Szz2));
    coeff[0] += (-(Sxz+Szx)*(Syz-Szy)+(Sxy-Syx)*(Sxx-Syy-Szz)) * ((-Sxz+Szx)*(Syz+Szy)+(Sxy-Syx)*(Sxx-Syy+Szz));
    coeff[0] += (-(Sxz+Szx)*(Syz+Szy)-(Sxy+Syx)*(Sxx+Syy-Szz)) * ((-Sxz+Szx)*(Syz-Szy)-(Sxy+Syx)*(Sxx+Syy+Szz));
    coeff[0] += (+(Sxy+Syx)*(Syz+Szy)+(Sxz+Szx)*(Sxx-Syy+Szz)) * (-(Sxy-Syx)*(Syz-Szy)+(Sxz+Szx)*(Sxx+Syy+Szz));
}


/* returns the sum of the deviations
   rmsd = sqrt(sum_dev/atom_num) */
double
QuarticHornFrag(const FragCoords *frag1, const FragCoords *frag2, double *coeff)
{
    double          frag1_radgyr;
    double          frag2_radgyr;
    double          rmsd2;
    int             n = 1000;

    frag1_radgyr = RadGyrSqrFrag(frag1);
    frag2_radgyr = RadGyrSqrFrag(frag2);

    ComputeQuartHornFrag(frag1, frag2, coeff);

    rmsd2 = horn_root(coeff, ((frag1_radgyr + frag2_radgyr) / 2.0), n, 1e-3);
    rmsd2 = (frag1_radgyr + frag2_radgyr) - (2.0 * rmsd2);

    return (rmsd2);
}

