#!/bin/bash
# Copyright 2006 Kapil Hari Paranjape <kapil@imsc.res.in>
# bootstrap modifications are
# Copyright 2007 Patrice Dumas <pertusus@free.fr>
# This program is released into the public domain.
# You may do whatever you please with it.

set -e

# This script takes the literate sources for Eitan Gurari's TeX4HT
# system and produces a directory tree like the one that is
# distributed as a tar.gz archive. It is gleaned from the 3-4 line
# headers that is put on each of the literate source files indicating
# how that particular file is to be handled.

# This script is meant to be idempotent in case the existing
# Debian tex4ht system is old and the literate sources need
# to be rebuilt using the new tex4ht tree.

# The script depends on the following:
# (a) A complete Debian installation of tex4ht! 
#     This is literate source after all :-)
# (b) A complete Debian installation of tex
# (c) A Java compiler is required for some steps.

# The script is meant to be run from the top level
# of the Debian source for TeX4HT which contains:
# (a) the literate source in the subdirectory lit/
# (b) the top-level tree of TeX4HT in src/ texmf/ bin/ 
# (c) the Debian stuff in the subdirectory debian/
# The new tree is built in the subdirectory dest/

bootstrap=yes
[ z"$1" = 'z-k' ] && bootstrap=no

TOP=${PWD}
LITDIR=${TOP}/lit
DESTDIR=${TOP}/dest
DEBDIR=${TOP}/debian

unset  TEXINPUTS TEXMFCNF TEX4HTINPUTS T4HTINPUTS \
	TEX4HTFONTSET TEXINPUTS_tex4ht TEXINPUTS_t4ht

# We need a different texmf.cnf that expands
# TeX's memory usage. We will create this in 
# the following directory:
TEXMFCNF=${DESTDIR}

# The order of directories searched for input
# is: first the newly created files, then the
# distributed tree and finally the existing installation.
if [ "$bootstrap" = "yes" ]; then
	TEXINPUTS=${LITDIR}:${DESTDIR}/texmf/tex//:
	TEX4HTINPUTS=${DESTDIR}/texmf/tex4ht//:
	T4HTINPUTS=${DESTDIR}/texmf/tex4ht//:
	TEX4HTFONTSET=${DESTDIR}/texmf/tex4ht/ht-fonts//:
else
	TEXINPUTS=${LITDIR}:${DESTDIR}/texmf/tex//:${TOP}/texmf/tex//:
	TEX4HTINPUTS=${DESTDIR}/texmf/tex4ht//:${TOP}/texmf/tex4ht//:
	T4HTINPUTS=${DESTDIR}/texmf/tex4ht//:${TOP}/texmf/tex4ht//:
	TEX4HTFONTSET=${DESTDIR}/texmf/tex4ht/ht-fonts//:${TOP}/texmf/tex4ht/ht-fonts//:
fi
TEXINPUTS_tex4ht="$TEX4HTINPUTS"
TEXINPUTS_t4ht="$T4HTINPUTS"

# Sanitise the build environment
LC_CTYPE=C
LANG=C

# The installation will run some scripts like xhlatex 
# which are installed in /usr/share/tex4ht in Debian.
PATH=$PATH:/usr/share/tex4ht:$DESTDIR/bin/linux:$DESTDIR/bin/unix

export LANG LC_CTYPE PATH

# A function to clean up all the stuff we do not use
function texclean()
{

  rm -f *.{aux,dvi,idv,ps,4tc,4ct,log,lg,xref,tmp,trc} mktex4ht.cnf

}

# This is the directory heirarchy that we need to have in
# place so that installations can go smoothly.
function mkdirheir()
{
  mkdir -p ${DESTDIR}/texmf/tex/generic/tex4ht
  mkdir -p ${DESTDIR}/bin/{unix,win32,linux}
  mkdir -p ${DESTDIR}/texmf/tex4ht/{xttl,xtpipes,bin,ht-fonts}
  mkdir -p ${DESTDIR}/texmf/tex4ht/base/{unix,win32}
  mkdir -p ${DESTDIR}/src/java
  mkdir -p ${DESTDIR}/litdoc
}

# Clean up the directory heirarchy
function cleandirheir()
{
  rm -rf ${DESTDIR}
  mkdir ${DESTDIR}
}

# The patched version of texmf.cnf
# It expands TeX's memory usage and
# puts the current directory at the *tail*
# of TeX's search path.
function makesomefiles()
{	
  if [ ! -f ${DESTDIR}/texmf.cnf ]
  then
    cat ${DEBDIR}/lit/add_texmf.cnf $(kpsewhich texmf.cnf) >> ${DESTDIR}/texmf.cnf
  fi
  if [ $bootstrap = 'yes' ]
  then
    # these files are required for bootstrap, see below
    cp ${TOP}/texmf/tex/generic/tex4ht/{tex4ht,html4-math,html4,info4ht,infoht4,infomml}.4ht\
       ${DESTDIR}/texmf/tex/generic/tex4ht
  fi
}

# Function to setup the new TeX environment
function setupenv()
{
  export TEXINPUTS TEXMFCNF TEX4HTINPUTS T4HTINPUTS \
  TEX4HTFONTSET TEXINPUTS_tex4ht TEXINPUTS_t4ht 
}

# Installation function that moves the files into the directory
# heirarchy
function movefiles()
{
  # Run within ${DESTDIR}
  set +e
  mv -f *.sty *.4ht texmf/tex/generic/tex4ht/
  mv -f m-tex4ht.tex texmf/tex/generic/tex4ht/
  [ -f tex4ht.env-unix ] && mv -f tex4ht.env-unix texmf/tex4ht/base/unix/tex4ht.env
  [ -f tex4ht.env-win32 ] && mv -f tex4ht.env-win32 texmf/tex4ht/base/win32/tex4ht.env
  mv -f *.c src
  mv -f *.{html,png,css} litdoc/
  cp -p mk4ht.perl bin/win32/mk4ht
  mv -f mk4ht.perl bin/unix/mk4ht
  for i in *.unix
  do 
    j=$(basename $i .unix)
    mv -f $i bin/unix/$j
  done
  mv -f *.bat bin/win32
  cp -p *.4xt texmf/tex4ht/xttl/
  mv -f *.dtd *.4xt texmf/tex4ht/xtpipes/
# .class are not generated, and they are scattered at different places
#  mv -f *.class src/class/
  mv -f *.java work.dir/* src/java/
  set -e
}

# Sanity check
[ -d ${LITDIR} -a -d ${DEBDIR} ] || \
(echo "Must be in top-level source directory to run this"; exit 0)

# the trick is to avoid argument list too long for rm!
[ $bootstrap = 'yes' ] && cleandirheir

# Initialise the working environment
mkdirheir
makesomefiles
setupenv

cd $DESTDIR

if [ $bootstrap = 'yes' ]; then
	# make some 4ht files. This creates th4.4ht.
	tex tex4ht-4ht
	# Now make tex4ht.sty. Indeed tex4ht-4ht and other tex files require 
	# tex4ht.sty to do the other 4ht files
	tex tex4ht-sty
	# Make cond4ht.4ht, required for next tex4ht-4ht run.
	tex tex4ht-cond4ht
	# now that th4.4ht is present, mktex4ht.4ht files is created, by using
	# tex4ht.sty. It is during that run that {tex4ht,html4-math,html4}.4ht
	# are required.
	tex tex4ht-4ht 
	# creates tex4ht.4ht. Needs {info4ht,infoht4,infomml}.4ht 
	latex tex4ht-options
	# remaining 4ht files required for bootstrap
	for htype in html4 info info-html4 info-mml; do
		latex tex4ht-${htype}.tex
		latex tex4ht-${htype}.tex
		latex tex4ht-${htype}.tex
	done
	# now that bootstrap is done remove the bootstrap files, and replace them
	# with the generated files
	rm texmf/tex/generic/tex4ht/*
	movefiles
	texclean

	# restart everything with the bootstrapped files
	tex tex4ht-sty
	tex tex4ht-cond4ht
	tex tex4ht-4ht
	latex tex4ht-options
	for htype in html4 info info-html4 info-mml; do
	  latex tex4ht-${htype}.tex
	  latex tex4ht-${htype}.tex
	  latex tex4ht-${htype}.tex
	done
	# install bootstrapped 4ht files
	movefiles
	texclean
fi

# now start the build from the beginning

# Make tex4ht.c
tex tex4ht-c

# Make t4ht.c
tex tex4ht-t4ht

# Make scripts
latex tex4ht-mkht
perl -c mk4ht.perl
latex ./mkht-scripts.4ht

# Make tex4ht.sty
tex tex4ht-sty

# Make tex4ht.env
latex tex4ht-env
if [ "bootstrap" = "no" ]
then
	# move away generated tex4ht.env
	# as it will override existing settings
	mv tex4ht.env tex4ht.env.gen
fi

# Make basic 4ht files
tex tex4ht-4ht
tex tex4ht-cond4ht
tex ./cond4ht.4ht
latex tex4ht-options

tex tex4ht-htcmd

# Make first move and clean
movefiles
texclean

# set the script executables
chmod a+x bin/unix/*

if [ "$bootstrap" = "yes" ]; then
# compile the executables
  for file in src/*.c; do
    basefile=`basename $file .c`
    #cc -O2 -g \
    #  -DHAVE_STRING_H -DHAVE_DIRENT_H -DHAVE_UNISTD_H -DHAVE_SYS_DIR_H \
    #  -DENVFILE="\"/usr/share/texmf/tex4ht/base/unix/tex4ht.env\"" \
    #  $file -o bin/linux/$basefile 
    cc -O2 -g \
       -DHAVE_STRING_H -DHAVE_DIRENT_H -DHAVE_UNISTD_H \
       -DHAVE_SYS_DIR_H -DKPATHSEA \
       $file -o bin/linux/$basefile -lkpathsea
  done
fi

# For 23 html ht styles. It is important that
# The current directory be the last in the search
# path for some of these builds to work. ?? Is it still true?
# Seems that html4 needs to be done first

for htype in html4 html-speech info info-html4 html0 html32 \
  docbook jsmath jsml javahelp mathml mathltx mathplayer moz \
  tei svg word unicode
do
   xhlatex tex4ht-${htype} "html,3,sections+"
done

htlatex tex4ht-ooffice "xhtml,3,sections+"

for htype in info-mml info-svg info-javahelp info-ooffice
do
   htlatex tex4ht-${htype} "html,sections+"
done

latex tex4ht-bibtex2

# The xttl files are for the experimental speech support.
# use javac
# and xttl/jsml.4xt, and base/*/xv4ht.cat and the xtpipes dtd
htlatex xtpipes "xhtml,3,next" "" "-d./"
#xhlatex tex4ht-docbook-xttl "html,next,3" "" "-d./"
htlatex tex4ht-docbook-xtpipes "xhtml,next,3" "" "-d./"
#xhlatex tex4ht-oo-xttl "html,next,3" "" "-d./"
htlatex tex4ht-oo-xtpipes "xhtml,next,3" "" "-d./"
#xhlatex tex4ht-html-speech-xtpipes "html,next,3" "" "-d./"
htlatex tex4ht-html-speech-xtpipes "html,next,3" "" "-d./"
#htlatex tex4ht-xhtml-xttl "xhtml,next,3" "" "-d./"
htlatex tex4ht-xhtml-xtpipes "xhtml,next,3" "" "-d./"
htlatex tex4ht-jsml-xtpipes "xhtml,next,3" "" "-d./"

movefiles
texclean

# Make font files. The Needs directives in this file cause "t4ht" to
# move the files into their correct locations automatically.
for font in noncjk cjk; do
  ht tex tex4ht-fonts-${font}
done

htlatex tex4ht-fonts-4hf "xhtml,3"

movefiles
texclean
# also remove the htf files that are already copied as stated above
rm -f *.htf

# That's it.

exit 0
# Remarks about other sources in lit/
# 1. tex4ht-auto-script is a contributed script. Its results are not
# used in the Debian package. Its build fails, and the scripts don't
# seem to be shipped.
# 2. tex4ht-fonts-modern.tex is input by tex4ht-fonts-noncjk.tex and is
# not built separately
# 3. wripro.tex seems to be something specific for Windows/Mathematica?

