/*
 * xgdvi -  TeX DVI previewer with GTK 
 *
 *  Copyright (C) 1999-2007  Hirotsugu Kakugawa. All rights reserved. 
 *  See "COPYING" for distribution of this software. 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
  
/*
 * History
 *
 *  9 Sep 1999  First implementation with basic features: file select,
 *              magnify, shrink, paper select, reload, and exit.
 *              Tested with GTK+ 1.2.3.
 * 16 Sep 1999  Added -stdin-ctl feature.
 * 19 Sep 1999  Deleted -stdin-ctl feature.
 * 22 Sep 1999  Added PrevPage, Nextpage, FirstPage, LastPage buttons.
 * 30 Sep 1999  Added Lisence button.
 *  1 Oct 1999  Added cursor changes depending on running status.
 *  2 Oct 1999  Added Printing panel. Added License panel.
 *  2 Oct 1999  Added printer message window and abort button
 *              in printing panel.
 *  4 Oct 1999  Changed Paper Size panel.
 *  9 Oct 1999  Added page table. Added scrolling preview image by 
 *              mouse button-1. 
 * 12 Oct 1999  Added multiple buffer and automatic reload on file update. 
 * 14 Oct 1999  Added handling code for multiple file name on command line.
 *              Tested with GTK+ 1.2.6. 
 * 18 Oct 1999  Internationalized with GNU gettext.
 * 19 Oct 1999  Added saving & restoring buffer state.
 * 26 Oct 1999  Changed to use paper and device mode list services of DVIlib.
 * 27 Oct 1999  Support for color EPS figures for True/Direct color visuals.
 * 29 Oct 1999  Added -stdin-ctl feature again.
 * 13 Nov 1999  Added path.c and path.h for canonical path representation.
 *  1 Feb 2000  Enhanced lprspec.c.
 *  1 Nov 2001  Improved file update checking. Support for color text.
 *  5 Nov 2001  Changed not to use frame buffer and incremental eps display
 *              feature. Tested with Gtk+ 1.2.10.
 * 12 Nov 2001  Addewd EPS image caching feature.
 * 18 Nov 2002  Improved EPS image caching and EPS clipping.
 * 15 Jun 2005  Antialiasing text display with colored box in background.
 *  6 Oct 2005  Cmdline option -fast-aa for antialiasing mode selection.
 *
 */


#include "../config.h"
#include "../with.h"

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRINGS_H
#  include <strings.h>
#endif
#if HAVE_STDARG_H
#  include <stdarg.h>
#else
#  include <vararg.h>
#endif
#if HAVE_SYS_TYPES_H
#  include <sys/types.h>
#endif
#if HAVE_SYS_STAT_H
#  include <sys/stat.h>
#endif
#if HAVE_SIGNAL_H
#  include <signal.h>
#endif
#if WITH_GETTEXT
#  include <libintl.h>
#endif
#include <sys/time.h>

#include <glib.h>
#include <gtk/gtk.h>

#include "libdvi29.h"
#include "defs.h"
#include "xgdvi.h"
#include "files.h"
#include "gui.h"
#include "buff.h"
#include "state.h"
#include "path.h"



#define ARG_FILES_SIZE   (MAX_BUFFS+1)
char            *arg_files[ARG_FILES_SIZE+1];

char            *dvi_file  = NULL; 
DVI              dvi       = NULL;
int              page      = 0;
double           shrink    = 0;
int              paper     = 0;
int              orient    = 0;

char           **paper_list = NULL;
char           **mode_list  = NULL;

double           on_buffer_switch_shrink = -1;
int              on_buffer_switch_paper  = -1;
int              on_buffer_switch_orient = -1;
double           on_buffer_switch_view_x = -1;
double           on_buffer_switch_view_y = -1;

DVI_DEVICE       dev       = NULL;

char            *param_kpathsea_mode  = NULL;
int              param_dpi            = -1;
char            *param_vflibcap       = VFLIBCAP;
double           param_shrink         = (double)SHRINK; 
char            *param_paper_name     = PAPER_SIZE; 
int              param_paper          = -1; 
int              param_orient         = ORI_PORT;
int              param_rvideo         = 0;
int              param_aa_perfect     = 1;
char            *param_foregr         = COLOR_FG;
char            *param_backgr         = COLOR_BG;
int              param_draw_eps       = 1;
int              param_gui_type       = -1;
int              param_no_buttons     = 0;
int              param_centering      = 0;
int              param_no_eps_cache   = 0;
int              param_no_initial_file_panel = -1;
int              param_delayed_font_open     = 1;
char            *param_win_title      = "xgdvi";
unsigned int     param_win_size_w     = WIN_WIDTH;
unsigned int     param_win_size_h     = WIN_HEIGHT;
int              param_win_pos_x      = -1;
int              param_win_pos_y      = -1;
int              param_win_pos_flag   = 0;
int              param_stdin_ctl      = 0;
int              param_no_state       = 0;
int              param_force_mono     = 0;
int              param_force_color    = 0;
int              param_scroll_delta   = SCROLL_DELTA;
int              param_window_delta   = WINDOW_DELTA;

int              paper_aa_w, paper_aa_h;

int              ofs_x, ofs_y;
int              ui_cmd, page_state, status;

double            default_color_fg_r;
double            default_color_fg_g;
double            default_color_fg_b;
double            default_color_bg_r;
double            default_color_bg_g;
double            default_color_bg_b;

double            current_color_fg_r;
double            current_color_fg_g;
double            current_color_fg_b;
double            current_color_bg_r;
double            current_color_bg_g;
double            current_color_bg_b;
 
GtkWidget                *xgdvi           = NULL;
guint                     ctl_chan    = -1;
guint                     timeout_tag = -1; 

static GtkWidget         *DrawingArea     = NULL;
static GtkScrolledWindow *ScrolledWindow  = NULL;
static GtkWidget         *StatusBar       = NULL;
static GdkPixmap         *BackPixmap      = NULL;
static GdkGC            **color_gcs       = NULL;
static GdkVisual         *Visual          = NULL; 
static GdkVisualType      VisualType      = -1; 

static GdkCursor *cursor_reading;
static GdkCursor *cursor_drawing;
static GdkCursor *cursor_eps;
static GdkCursor *cursor_done;
static GdkCursor *cursor_scroll;
static GdkCursor *cursor_current;


static void   put_rgb(long pos_x, long pos_y,
		      unsigned char *bm, int raster, long w, long h, 
		      int do_cache, long dvipos);
static void   open_dvi(char*,int,int);
static void   show_device_list(void);
static void   show_paper_list(void);
static void   customize(void);
static gint   file_update_check(gpointer data);
static int    parse_args(int argc, char **argv);
static void   usage(void);
static int    x_parse_geometry(char *str, int *xp, int *yp,
			       unsigned int *wp, unsigned int *hp);
static void   change_window_size(int pid, int ori);
static void   resize_window_delta(int dx, int dy);
static void   init_colors(void);
static guint32 pixel_value(guint16 r, guint16 g, guint16 b);
static void    change_window_colors(double,double,double,double,double,double);
static DVI_DEVICE  create_dev(void);
void          x_usleep(int t);
void          clean_on_exit(void);
#if 0
static GdkCursor *get_cursor_done(void);
#endif

#if HAVE_SIGNAL_H
static RETSIGTYPE  sig_usr1(int);
static RETSIGTYPE  sig_cleanup(int);
#endif




int
main (int argc, char *argv[])
{
  int    file_nargs, b, i;
  char   params[3*256], wd[1024], *p;
#if WITH_GETTEXT
  char  *catname = I18NCATALOG;
#endif

  status = STAT_STARTUP;

  if ((argc > 0) 
      && ((strcmp(*argv, "--help") == 0)
	  || (strcmp(*argv, "-help") == 0)
	  || (strcmp(*argv, "-h") == 0)))
    usage();
    
  DVI_SETUP();
  paper_list = DVI_get_paper_size_list();
  mode_list  = DVI_get_device_mode_list();

#if HAVE_UNSETENV
  unsetenv("XMODIFIERS"); 
#endif
  gtk_set_locale();

#if WITH_GETTEXT
  bindtextdomain(catname, LOCALE_DIR);
  textdomain(catname);
#endif

  gtk_init(&argc, &argv);
  gdk_rgb_init();

  customize();

  file_nargs = parse_args(argc, argv);

  sprintf(params, "TeX_DPI=%d, TeX_KPATHSEA_MODE=%s, TeX_KPATHSEA_PROGRAM=%s",
	  param_dpi, param_kpathsea_mode, "xgdvi");

  if (DVI_INIT(param_vflibcap, params) < 0) 
    EXIT_MSG1(_("Failed to initialize DVI reader.\n"));

  xgdvi = create_xgdvi();
  DrawingArea    = gtk_object_get_data (GTK_OBJECT (xgdvi), "DrawingArea");
  ScrolledWindow = gtk_object_get_data (GTK_OBJECT (xgdvi), "ScrolledWindow");
  StatusBar      = gtk_object_get_data (GTK_OBJECT (xgdvi), "StatusBar");

  gtk_window_set_default_size(GTK_WINDOW(xgdvi), 
			      param_win_size_w, param_win_size_h);
  gtk_widget_show (xgdvi);
  gdk_window_set_hints(xgdvi->window, param_win_pos_x, param_win_pos_y,
		       0,0,0,0, param_win_pos_flag);
  if (param_win_pos_flag != 0)
    gdk_window_move(xgdvi->window, param_win_pos_x, param_win_pos_y);

  if (param_no_buttons == 1)
    change_button_mode(0);

  create_FileSelection();

  create_BufferSelection();
  create_PaperSelection();
  create_Printer();
  create_License();

  cursor_reading = gdk_cursor_new(GDK_SHUTTLE);
  cursor_drawing = gdk_cursor_new(GDK_WATCH);
  cursor_eps     = gdk_cursor_new(GDK_COFFEE_MUG);
  cursor_scroll  = gdk_cursor_new(GDK_PLUS);
#if 0
  cursor_done    = get_cursor_done();
#else
  cursor_done    = gdk_cursor_new(GDK_ARROW);
#endif

  while (gtk_events_pending()){
    gtk_main_iteration();
  }

  if ((dev = create_dev()) == NULL)
    EXIT_MSG1(_("Failed to initialize DVI reader.\n"));
  
  buffer_init();

  for (i = 0; arg_files[i] != NULL; i++){
    find_file_no_select(arg_files[i], 1, paper, orient);
  }

  if (param_no_state == 0){
    state_restore();
  }

#if 0
  if (dvi != NULL){
    ui_cmd     = CMD_DRAW_PAGE;
    page_state = PAGE_EMPTY;
  } else {
    ui_cmd     = CMD_NONE;
    page_state = PAGE_DRAW_FINISH;
  }
  switched_buffer();
#endif

  if (file_nargs == 0)
    b = 0;
  else 
    b = 1;

  find_file(arg_files[b]);

  if (file_nargs == 0){
    p = g_get_current_dir();
    if (strcmp(p, "/") == 0){
      sprintf(wd, "%s", p);
    } else {
      sprintf(wd, "%s/", p);
    }
    g_free(p);
    set_fname_FileSelection(wd);
    message(_("Please select a DVI file."));
    if (param_no_initial_file_panel == 0){
      show_FileSelection();
    }
  }

  if (param_centering == 1)
    preview_window_set_pos_center(0.5, 0.5);

#if 0
  printf("* buffers. before mainloop\n");
  buffer_dump();
#endif

  if (UPDATE_CHECK_INTERVAL > 0){
    timeout_tag = gtk_timeout_add(UPDATE_CHECK_INTERVAL * 1000, 
				  file_update_check, NULL);
  }

  if (param_stdin_ctl == 1){
    ctl_chan = gtk_input_add_full(0, GDK_INPUT_READ, 
				  event_command_input, NULL, NULL, NULL);
  }

#if HAVE_SIGNAL_H
  signal(SIGUSR1, sig_usr1);     /* for debugging */
  signal(SIGHUP,  sig_cleanup);  /* for cleaning up cache files */
  signal(SIGINT,  sig_cleanup);
  signal(SIGTERM, sig_cleanup);
#if 0
  signal(SIGABRT, sig_cleanup);
  signal(SIGQUIT, sig_cleanup);
  signal(SIGKILL, sig_cleanup);
  signal(SIGSEGV, sig_cleanup);
#endif
#endif

  status = STAT_MAINLOOP;

#if HAVE_ATEXIT
  atexit(clean_on_exit);
#endif

  gtk_main ();

  return 0;
}

#if 0
static GdkCursor *get_cursor_done(void)
{
  static guchar cursor1_bits[] = {
    0x80, 0x01, 0x40, 0x02, 0x20, 0x04, 0x10, 0x08, 0x08, 0x10, 0x04, 0x20,
    0x82, 0x41, 0x41, 0x82, 0x41, 0x82, 0x82, 0x41, 0x04, 0x20, 0x08, 0x10,
    0x10, 0x08, 0x20, 0x04, 0x40, 0x02, 0x80, 0x01};
  static guchar cursor1mask_bits[] = {
    0x80, 0x01, 0xc0, 0x03, 0x60, 0x06, 0x30, 0x0c, 0x18, 0x18, 0x8c, 0x31,
    0xc6, 0x63, 0x63, 0xc6, 0x63, 0xc6, 0xc6, 0x63, 0x8c, 0x31, 0x18, 0x18,
    0x30, 0x0c, 0x60, 0x06, 0xc0, 0x03, 0x80, 0x01};
  GdkCursor *cursor;
  GdkPixmap *source, *mask;
  GdkColor fg = { 0, 0xffff, 0x0000, 0x0000 };  /* Red. */
  GdkColor bg = { 0, 0xffff, 0xffff, 0x0000 };  /* White */

  source = gdk_bitmap_create_from_data (NULL, cursor1_bits,     16, 16);
  mask   = gdk_bitmap_create_from_data (NULL, cursor1mask_bits, 16, 16);
  cursor = gdk_cursor_new_from_pixmap (source, source, &fg, &bg, 8, 8);
  gdk_pixmap_unref (source);
  gdk_pixmap_unref (mask);

  return cursor;
}
#endif

static void
customize(void)
{
  int          isdir;
  struct stat  st;
  char        *f;

  f = g_strdup_printf("%s%s%s", XGDVI_CONFIG_DIR, "/", XGDVI_RCFILE);
  gtk_rc_add_default_file(f);
  g_free(f);
  f = NULL;

#if HAVE_STAT
  f = g_strdup_printf("%s%s%s", g_get_home_dir(), "/", XGDVI_PERSONAL_DIR);
  if (stat(f, &st) >= 0){
#if !defined(__linux__) && !defined(__GLIBC__)
    isdir = ((st.st_mode & S_IFDIR) != 0);
#else
    isdir = S_ISDIR(st.st_mode);
#endif
    if (!isdir){
      fprintf(stderr, _("xgdvi: %s should be a directory.\n"), f);
      return;
    }
  } else {
    if (mkdir(f, 0755) == 0)
      fprintf(stderr, _("Created %s\n"), f);
    else
      fprintf(stderr, _("Failed to create %s\n"), f);
  }
  g_free(f);
  f = NULL;
#endif

  f = g_strdup_printf("%s%s%s%s%s", 
		      g_get_home_dir(), "/", XGDVI_PERSONAL_DIR,
		      "/", XGDVI_RCFILE);
  gtk_rc_add_default_file(f);
  g_free(f);
  f = NULL;
}


static int
parse_args(int argc, char **argv)
{
  int     nfiles, i;
  char    init_dvi[1024], *p;
  double  d;
  double  mag = 1.0;

  for (i = 0; i < ARG_FILES_SIZE+1; i++)
    arg_files[i] = NULL;
  nfiles = 0;
  sprintf(init_dvi, "%s%s%s", XGDVI_CONFIG_DIR, "/", _("Startup16.dvi"));
  arg_files[nfiles++] = g_strdup(init_dvi);

  param_gui_type = 1;
  param_no_initial_file_panel = 0;

  if ((p = getenv("XGDVI_SCROLL_DELTA")) != NULL){
    param_scroll_delta = atoi(p);
  }
  if ((p = getenv("XGDVI_WINDOW_DELTA")) != NULL){
    param_window_delta = atoi(p);
  }

  for (--argc, argv++; argc > 0;  argc--, argv++){
    if ((strcmp(*argv, "-m") == 0) && (argc > 1)){
      argc--; argv++;
      mag = atof(*argv);
    } else if ((strcmp(*argv, "-g") == 0)  && (argc > 2)){
      argc--;  argv++; 
      if ((param_win_size_w = atoi(*argv)) < 10)
	param_win_size_w = WIN_WIDTH;
      argc--;  argv++; 
      if ((param_win_size_h = atoi(*argv)) < 10)
	param_win_size_h = WIN_HEIGHT;
    } else if ((strcmp(*argv, "-geometry") == 0) && (argc > 2)){
      argc--;  argv++; 
      if (x_parse_geometry(*argv, &param_win_pos_x, &param_win_pos_y, 
			   &param_win_size_w, &param_win_size_h) < 0){
	EXIT_MSG2(_("xgdvi: Illegal geometry %s"), *argv);
	exit(1);
      }
    } else if (strcmp(*argv, "-gui1") == 0){
      param_gui_type = 1;
    } else if (strcmp(*argv, "-gui2") == 0){
      param_gui_type = 2;
    } else if (strcmp(*argv, "-gui3") == 0){
      param_gui_type = 3;
    } else if (strcmp(*argv, "-gui4") == 0){
      param_gui_type = 4;
    } else if (strcmp(*argv, "-nb") == 0){
      param_no_buttons = 1;
    } else if (strcmp(*argv, "-c") == 0){
      param_centering = 1;
    } else if ((strcmp(*argv, "-v") == 0)  && (argc > 1)){
      argc--;  argv++;
      param_vflibcap = *argv;
    } else if ((strcmp(*argv, "-dpi") == 0) && (argc > 1)){
      argc--;  argv++;
      param_dpi = atoi(*argv);
    } else if ((strcmp(*argv, "-mode") == 0) && (argc > 1)){
      argc--;  argv++;
      param_kpathsea_mode = *argv;
    } else if ((strcmp(*argv, "-fg") == 0) && (argc > 1)){
      argc--;  argv++;
      param_foregr = *argv;
    } else if ((strcmp(*argv, "-bg") == 0) && (argc > 1)){
      argc--;  argv++;
      param_backgr = *argv;
    } else if (strcmp(*argv, "-eps") == 0){
      param_draw_eps = 1;
    } else if (strcmp(*argv, "-no-eps") == 0){
      param_draw_eps = 0;
    } else if (strcmp(*argv, "-nec") == 0){
      param_no_eps_cache = 1;
    } else if (strcmp(*argv, "-ndf") == 0){
      param_delayed_font_open = 0;
    } else if (strcmp(*argv, "-fast-aa") == 0){
      param_aa_perfect = 0;
    } else if (strcmp(*argv, "-no-file-panel") == 0){
      param_no_initial_file_panel = 1;
    } else if ((strcmp(*argv, "-l") == 0)
	       || (strcmp(*argv, "-landscape") == 0)){
      param_orient = ORI_LAND;
    } else if (strcmp(*argv, "-wt") == 0){
      argc--;  argv++;
      param_win_title = *argv;
    } else if (DVI_parse_device_mode(&argv[0][1], NULL, NULL, NULL) >= 0){
      param_kpathsea_mode = &argv[0][1];
    } else if (DVI_parse_paper_size(&argv[0][1], NULL, NULL) >= 0){
      param_paper_name = &argv[0][1];
    } else if ((strcmp(*argv, "-h") == 0) || (strcmp(*argv, "--help") == 0)){
      usage();
    } else if (**argv != '-'){
      if (nfiles < ARG_FILES_SIZE-1){
	arg_files[nfiles] = path_canonical(*argv); 
	if (arg_files[nfiles] != NULL)
	  nfiles++;
      }
    } else if (strcmp(*argv, "-stdin-ctl") == 0){
      param_stdin_ctl = 1;
    } else if (strcmp(*argv, "-paper-list") == 0){
      show_paper_list();
      exit(0);
    } else if (strcmp(*argv, "-mode-list") == 0){
      show_device_list();
      exit(0);
    } else if (strcmp(*argv, "-no-state") == 0){
      param_no_state = 1;
    } else if (strcmp(*argv, "-mono") == 0){
      param_force_mono = 1;
    } else if (strcmp(*argv, "-color") == 0){
      param_force_color = 1;
    } else if (strcmp(*argv, "-rv") == 0){
      param_rvideo = 1;
    } else if ((strcmp(*argv, "-sd") == 0) && (argc > 1)){
      argc--;  argv++;
      param_scroll_delta = atoi(*argv);
    } else if ((strcmp(*argv, "-wd") == 0) && (argc > 1)){
      argc--;  argv++;
      param_window_delta = atoi(*argv);
    } else 
      EXIT_MSG2(_("Unknow option: %s\n"), *argv);
  }

  if (mag < MAG_MIN)
    EXIT_MSG1(_("Maginification is too small."));
  if (MAG_MAX < mag)
    EXIT_MSG1(_("Maginification is too large."));

  if (param_kpathsea_mode == NULL){
    param_kpathsea_mode = KPATHSEA_MODE;
    if (param_dpi <= 0)
      param_dpi = DPI;
  } else {
    if (DVI_parse_device_mode(param_kpathsea_mode, &d, NULL, NULL) >= 0){
      param_dpi = d;
    } else {
      if (param_dpi <= 0)
	param_dpi = DPI;
    }
  }

  param_shrink = ((double)SHRINK * param_dpi) / (mag * DPI);
  shrink = param_shrink;
  
  param_paper = paper_id(param_paper_name);
  paper  = param_paper;
  orient = param_orient;
  if (param_paper < 0)
    EXIT_MSG2(_("Unknown paper size: %s\n"), param_paper_name);

  change_window_size(param_paper, param_orient);

  if ((param_scroll_delta < 1) || (100 < param_scroll_delta)){
    fprintf(stderr,
	    _("xgdvi: Scroll delta is out of range. [1 ... 100]\n"));
    exit(1);
  }
  if ((param_window_delta < 1) || (500 < param_window_delta)){
    fprintf(stderr,
	    _("xgdvi: Window delta is out of range. [1 ... 500]\n"));
    exit(1);
  }

  return  nfiles-1;
}

static void
usage(void)
{
  printf(_("XGDVI %s is a previewer for TeX DVI files on X Window System.\n"),
	 VERSION);
  printf(_("Usage: xgdvi [PaperSize] [Device] [Options] DVI-FILE\n"));
  printf(_("PaperSize:\n"));
  printf(_("  -a4, -a5, -a6, -a7, -b4, -b5, -letter, -us, -legal, etc.\n"));
  printf(_("  -l           Landscape mode\n"));
  printf(_("  -paper-list  Print list of supported paper size names\n"));
  printf(_("Device: select device mode name for font search by kpathsea\n"));
  printf(_("  -cx, -sparcptr, -ljfour, ..., -MODE\n"));
  printf(_("  -mode MODE   Same as -MODE\n"));
  printf(_("  -dpi DPI     Device resolution (in case above fails) [%d]\n"), 
	 DPI);
  printf(_("  -mode-list   Print list of supported device mode names\n"));
  printf(_("Options:  (Each value enclosed by [ ] is the default.)\n"));
  printf(_("  -m M         Magnification (%.2f<=M<=%.2f) [1.0]\n"),
	 (double)MAG_MIN, (double)MAG_MAX);
  printf(_("  -mono        Gray-scale display\n"));
  printf(_("  -color       Color display\n"));
  printf(_("  -g W H       Window size [%d %d]\n"), WIN_WIDTH, WIN_HEIGHT);
  printf(_("  -v VFLIBCAP  A path name of vflibcap [%s]\n"), VFLIBCAP);
  printf(_("  -fg COLOR    Foreground color [%s]\n"), COLOR_FG);
  printf(_("  -bg COLOR    Background color [%s]\n"), COLOR_BG);
  printf(_("  -gui1, ..., -gui4   Select button arrangement style [-gui1]\n"));
  printf(_("Key operations:"));
  printf(_("\
  C-f, C-b   Move view port forward, backward, resp.\n\
  C-p, C-n   Move view port up, down, resp.\n\
  C-a, C-e   Go to beginning, end of line, resp.\n\
  C-v   Next page                     v     Previous page \n\
  <     Go to the first page          >     Go to the last page \n\
  ,     Go to next 10 page            .     Go to previous 10 page \n\
  ]     Magnify document              [     Shrink document\n\
  }     Enlarge window                {     Shrink window \n\
  0     Enlarge window vertically     9     Shrink window vertically \n\
  )     Enlarge window horizontally   (     Shrink window horizontally \n\
  q     Quit                          o     Swap paper orientation\n"));
  exit(0);
}


static int
x_parse_geometry(char *str, 
		 int *xp, int *yp,
		 unsigned int *wp, unsigned int *hp)
{
  char  *p = str;
  unsigned int  p1, p2;
  int           q1, q2, n;
  char          c0, c1, c2;

  if (*p == '=')
    p++;

  if (!isdigit((int)*p) && (*p != '+') && (*p != '-')){
    return -1;
  }

  if (isdigit((int)*p)){
    n = sscanf(p, "%u%[xX]%u%c%d%c%d", &p1, &c0, &p2, &c1, &q1, &c2, &q2);
    switch (n){
    case 7: 
      *xp = q1; 
      *yp = q2; 
      *wp = p1; 
      *hp = p2; 
      param_win_pos_flag = GDK_HINT_POS;
      if (c1 == '-')
	*xp = gdk_screen_width()  - *wp - *xp;
      if (c2 == '-')
	*yp = gdk_screen_height() - *hp - *yp;
      break;
    case 3: 
      *wp = p1; 
      *hp = p2; 
      break;
    default:
      return -1;
    }
  } else {
    n = sscanf(p, "%c%d%c%d", &c1, &q1, &c2, &q2);
    switch (n){
    case 4: 
      *xp = q1; 
      *yp = q2; 
      param_win_pos_flag = GDK_HINT_POS;
      if (c1 == '-')
	*xp = gdk_screen_width()  - *wp - *xp;
      if (c2 == '-')
	*yp = gdk_screen_height() - *hp - *yp;
      break;
    default:
      return -1;
    }
  }

#if 0
  printf("** %d %d %d %d %d %d\n", *xp, *yp, *wp, *hp,
	 gdk_screen_width(), gdk_screen_height());
#endif

  return 0;
}




static void
show_paper_list(void)
{
  char  **p;
  int   i;
  
  if ((p = paper_list) == NULL){
    printf(_("Paper size list is not found.\n"));
  } else {
    i = 0;
    printf(_("Paper Size: \n"));
    while (*p != NULL){
      printf("%s%s", *p, (*(p+1) == NULL) ? "\n" : ", ");
      i++;
      p++;
    }
  }
}

static void
show_device_list(void)
{
  char **p, *desc;
  int  i;
  
  if ((p = mode_list) == NULL){
    printf(_("Device mode list is not found.\n"));
  } else {
    i = 0;
    printf(_("Device Mode Names: \n"));
    printf(_("   Mode Name     Description\n"));
    while (*p != NULL){
      DVI_parse_device_mode(*p, NULL, NULL, &desc);
      printf("%12s     %s\n", *p, desc);
      i++;
      p++;
    }
  }
}


int
paper_id(char *paper_name)
{
  char  *p1, *p2;
  int   i;

  for (i = 0; paper_list[i] != NULL; i++){
    p1 = paper_list[i];
    p2 = paper_name;
    while (toupper((int) *p1) == toupper((int)*p2)){
      if ((*p1 == '\0') || (*p2 == '\0'))
	break;
      p1++;
      p2++;
    }
    if ((*p1 == '\0') && (*p2 == '\0'))
      return i;
  }
  return -1;
}

char*
paper_name(int pid)
{
  int  i;

  for (i = 0; paper_list[i] != NULL; i++){
    if (i == pid)
      return  paper_list[i];
  }

  return NULL;
}




int 
find_file_no_select(char *file, int pg, int pid, int oid)
{
  char   *dvi_file_new, *s;
  DVI_PROPERTY  dvi_props;

  if (buffer_full()){
    message(_("Error: Too many DVI files."));
    return -1;
  }

#if 0
  printf("** %s\n", file);
#endif

  s = DVI_find_dvi_path(file); 
  if (s == NULL)
    return -1;
  dvi_file_new = path_canonical(s);
  free(s);
  s = NULL;
  if (dvi_file_new == NULL)
    return -1;
  
  if (buffer_check_in_buffer(dvi_file_new) > 0){
    return 0;
  }

  dvi_props = make_dvi_property();
  buffer_add(dev, dvi_file_new, NULL, dvi_props, pg, pid, oid, 0);
  
  g_free(dvi_file_new);
  dvi_file_new = NULL;

  return 0;
}

int 
find_file(char *file)
{
  open_dvi(file, 0, 0);
  return 1;
}

int 
find_file_gui(char *file)
{
  open_dvi(file, 0, 1);
  return 1;
}

int 
find_file_kill_old(char *file)
{
  open_dvi(file, 0, 0);
  return 1;
}

int 
reload_file(void)
{
  open_dvi(NULL, 1, 0); 
  return 1;
}

static void
open_dvi(char *file, int reload, int gpos)
{
  char   *dvi_file_new, *s;
  DVI     dvi_new;
  DVI_PROPERTY  dvi_props;

  ui_cmd       = CMD_NONE;
  page_state   = PAGE_DRAW_FINISH;
  dvi_file_new = NULL;

  if (reload == 1){
    if (dvi_file == NULL)
      return;
    dvi_file_new = g_strdup(dvi_file);
  } else {
    if (file != NULL){
      s = DVI_find_dvi_path(file); 
      dvi_file_new = path_canonical(s);
      free(s);
      s = NULL;
    }
    if (dvi_file_new == NULL){
      cursor_current = cursor_done;
      gdk_window_set_cursor(xgdvi->window, cursor_current);
      message(_("Error: Cannot read: %s"), file);
      gdk_flush();
      g_free(dvi_file_new);
      dvi_file_new = NULL;
      return;
    }
    if (buffer_full()){
      cursor_current = cursor_done;
      gdk_window_set_cursor(xgdvi->window, cursor_current);
      message(_("Error: Too many DVI files."));
      gdk_flush();
      g_free(dvi_file_new);
      dvi_file_new = NULL;
      return;
    }
  }

  buffer_save_current_state();

  dvi_props = make_dvi_property();
  dvi_new = load_dvi(dvi_file_new, dev, dvi_props, 0);

  if (dvi_new == NULL){
    g_free(dvi_file_new);
    dvi_file_new = NULL;
    return;
  }

  if (reload == 1){
    if (page > dvi_new->pages)
      page = dvi_new->pages;
  } else {
    page = 1;
  }

  if (buffer_check_in_buffer(dvi_file_new) > 0){
    buffer_update_dvi(dvi_file_new, dvi_new); 
    buffer_select(dvi_file_new);
    switched_buffer();
  } else {
    if (buffer_add(dev, dvi_file_new, dvi_new, dvi_props, 
		   1, -1, -1, gpos) >= 0){
      buffer_select(dvi_file_new);
      switched_buffer();
    }
  }
  update_page_display();


  g_free(dvi_file_new);
  dvi_file_new = NULL;
}


DVI
load_dvi(char *dvi_file_new, DVI_DEVICE dvi_dev, 
	 DVI_PROPERTY dvi_props, int retry)
{
  DVI     dvi_new;
  int     r;

#if 0
  printf("** %s\n", dvi_file_new);
#endif

  cursor_current = cursor_reading;
  gdk_window_set_cursor(xgdvi->window, cursor_current);
  gdk_flush();

  r = retry;
  dvi_new = NULL;

  do {
    r--;
    message(_("Reading a DVI file: %s ..."), dvi_file_new);
    if ((dvi_new = DVI_CREATE(dvi_dev, dvi_file_new, dvi_props)) == NULL){
      message(_("DVI file is broken: %s"), dvi_file_new);
      x_usleep(1000);
      continue;
    } 
    x_usleep(1000);
    if (DVI_FILE_MODIFIED(dvi_new, dvi_file_new) == 1){
      DVI_DISPOSE(dvi_new, dvi_dev);
      dvi_new = NULL;
      x_usleep(1000);
      continue;
    }
    message(_("Opening fonts ..."));
    if (DVI_OPEN_FONT(dvi_new, dvi_dev) < 0){
      DVI_DISPOSE(dvi_new, dvi_dev);
      dvi_new = NULL;
      message(_("Error: Cannot open all fonts."));
      continue;
    }
    x_usleep(1000);

    if (DVI_FILE_MODIFIED(dvi_new, dvi_file_new) == 0)
      break;

    DVI_DISPOSE(dvi_new, dvi_dev);
    dvi_new = NULL;
  } while (r >= 0);

  if (dvi_new != NULL)
    dvi_new->udp1 = imgcache_new();
  cursor_current = cursor_done;
  gdk_window_set_cursor(xgdvi->window, cursor_current);
  gdk_flush();

  return  dvi_new;
}


void
x_usleep(int t)
{
#ifdef HAVE_USLEEP
  usleep(t);
#else
  sleep((999+t)/1000);
#endif
}



void
switched_buffer(void)
{
  int resize;

  resize = 0;

  if (on_buffer_switch_paper >= 0){
    resize = 1;
    paper = on_buffer_switch_paper;
    on_buffer_switch_paper = -1;
  }
  if (on_buffer_switch_orient >= 0){
    resize = 1;
    orient = on_buffer_switch_orient;
    on_buffer_switch_orient = -1;
  }
  if (on_buffer_switch_shrink > 0){
    resize = 1;
    shrink = on_buffer_switch_shrink;
    on_buffer_switch_shrink = -1;
  }
  if (resize == 1)
    resize_window();
  if (on_buffer_switch_view_x < 0)
    on_buffer_switch_view_x = 0.0;
  if (on_buffer_switch_view_y < 0)
    on_buffer_switch_view_y = 0.0;
  preview_window_set_pos(on_buffer_switch_view_x, on_buffer_switch_view_y);

  if (dvi_file != NULL)
    set_fname_FileSelection(dvi_file);
  
  if (dvi != NULL){
    message(_("%s (%d pages)"), dvi_file, dvi->pages);
    cursor_current = cursor_drawing;
    gdk_window_set_cursor(xgdvi->window, cursor_current);
    gdk_flush();
    ui_cmd     = CMD_DRAW_PAGE;
    page_state = PAGE_EMPTY;
    update_page_list();
    update_page_display();
    clear_page();
    draw_page(); 
  } else {
    if (dvi_file == NULL)
      message(_("Please select a DVI file."));
    cursor_current = cursor_done;
    gdk_window_set_cursor(xgdvi->window, cursor_current);
    gdk_flush();
    ui_cmd     = CMD_NONE;
    page_state = PAGE_DRAW_FINISH;
    update_page_list();
    clear_page();
  }
}

static gint
file_update_check(gpointer data)
{
#if 0
  printf("** tick\n");
#endif

  if (dvi_file == NULL)
    return TRUE;

  if ((dvi == NULL)
	 || ((dvi != NULL) && (DVI_FILE_MODIFIED(dvi, dvi_file) == 1))){
#if 0
    printf("** Reload %s\n", dvi_file);
#endif
    reload_file();
#if 0
    if (dvi != NULL)
      draw_page();
#endif
  }

  return TRUE;
}





static void
DEV_color_rgb(DVI_DEVICE dev, DVI dvi, int page_resume_flag,
	      double fg_r, double fg_g, double fg_b)
{
#if 0
  printf("FOREGROUND COLOR %.3f %.3f %.3f\n", fg_r, fg_g, fg_b);
#endif

  change_window_colors(fg_r, fg_g, fg_b, 
		       current_color_bg_r,
		       current_color_bg_g,
		       current_color_bg_b);
}

static void
DEV_bgcolor_rgb(DVI_DEVICE dev, DVI dvi, int page_resume_flag,
		double bg_r, double bg_g, double bg_b)
{
#if 0
  printf("BACKGROUND COLOR %.3f %.3f %.3f\n", bg_r, bg_g, bg_b);
#endif

  change_window_colors(current_color_fg_r,
		       current_color_fg_g,
		       current_color_fg_b,
		       bg_r, bg_g, bg_b);
  clear_page();
  return; 
}




static void
put_rgb_2(long pos_x, long pos_y, 
	  unsigned char *bm, int raster, long w, long h, 
	  int do_cache, long dvipos)
{
  long      xx, xx2, yy, yy2, dots;
  long      yyb, xxb, yye, xxe;
  unsigned char  d;
#define PBUFF_SIZE  256
  GdkPoint    pb_points[N_AA][PBUFF_SIZE];
  int         pb_n[N_AA];
  static unsigned char  bits[] = { 
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01 };
  
  if ((pos_x+ofs_x)/AA >= paper_aa_w)
    return;
  if ((pos_y+ofs_y)/AA >= paper_aa_h)
    return;

  for (dots = 0; dots < N_AA; dots++){
    pb_n[dots] = 0;
  }

  yyb = AA * (long) ((pos_y + ofs_y) / AA);
  xxb = AA * (long) ((pos_x + ofs_x) / AA);
  yye = AA * (long) ((pos_y + ofs_y + h + AA-1) / AA);
  xxe = AA * (long) ((pos_x + ofs_x + w + AA-1) / AA);

  for (yy = yyb; yy < yye; yy += AA){
    if ((yy/AA < 0) || (yy/AA >= paper_aa_h))
      continue;
    for (xx = xxb; xx < xxe; xx += AA){
      if ((xx < 0) || (xx/AA >= paper_aa_w))
	continue;
      dots = 0;
      for (yy2 = 0; yy2 < AA; yy2++){
	if (((yy + yy2) < (pos_y + ofs_y)) ||
	    ((pos_y + ofs_y + h) <= (yy + yy2)))
	  continue;
	for (xx2 = 0; xx2 < AA; xx2++){
	  if (((xx + xx2) < (pos_x + ofs_x)) ||
	      ((pos_x + ofs_x + w) <= (xx + xx2)))
	    continue;
	  if (bm != NULL){
	    d = bm[(yy + yy2 - (pos_y + ofs_y)) * raster 
		  + (xx + xx2 - (pos_x + ofs_x))/8];
	    if ((d & bits[(xx + xx2 - (pos_x + ofs_x)) % 8]) != 0)
	      dots++;
	  } else {
	    dots++;
	  }
	}
      }
      if (dots == 0)
	continue;
      if (pb_n[dots] == PBUFF_SIZE){
	gdk_draw_points(BackPixmap, color_gcs[dots],
			pb_points[dots], pb_n[dots]);
	pb_n[dots] = 0;
      }
      pb_points[dots][pb_n[dots]].x = xx/AA;
      pb_points[dots][pb_n[dots]].y = yy/AA;
      ++pb_n[dots];
    }
  }

  for (dots = 0; dots < N_AA; dots++){
    if (pb_n[dots] == 0)
      continue;
    gdk_draw_points(BackPixmap, 
		    color_gcs[dots], pb_points[dots], pb_n[dots]);
    pb_n[dots] = 0;
  }

  { 
    long dx0, dy0, w0, h0;

    if ((dx0 = (pos_x+ofs_x)/AA) < 0)
      dx0 = 0;
    if ((dy0 = (pos_y+ofs_y)/AA) < 0)
      dy0 = 0; 
    if ((w0 = (xxe-xxb)/AA) > paper_aa_w)
      w0 = paper_aa_w;
    if ((h0 = (yye-yyb)/AA) > paper_aa_h)
      h0 = paper_aa_h;

    gdk_draw_pixmap(DrawingArea->window, color_gcs[N_AA-1], BackPixmap, 
		    dx0, dy0, dx0, dy0, w0, h0);
  }

}


static void
put_rgb_1(long pos_x, long pos_y, 
	  unsigned char *bm, int raster, long w, long h, 
	  int do_cache, long dvipos)
{
  /* FEATURE: antialias on non-default background color, 
     e.g., \colorbox{}{}.
  */

  long      xx, xx2, yy, yy2;
  long      yyb, xxb, yye, xxe;
  long      dx0, dy0, w0, h0;
  int       dots;
  GdkImage *img  = NULL;
  GdkGC    *gc   = NULL;
  guint     rx, gx, bx;
  unsigned char  d;
  static unsigned char  bits[] = { 
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01 };
  
  if ((pos_x+ofs_x)/AA >= paper_aa_w)
    return;
  if ((pos_y+ofs_y)/AA >= paper_aa_h)
    return;

  yyb = AA * (long) ((pos_y + ofs_y) / AA);
  xxb = AA * (long) ((pos_x + ofs_x) / AA);
  yye = AA * (long) ((pos_y + ofs_y + h + AA-1) / AA);
  xxe = AA * (long) ((pos_x + ofs_x + w + AA-1) / AA);

  if ((dx0 = (pos_x+ofs_x)/AA) < 0)
    dx0 = 0;
  if ((dy0 = (pos_y+ofs_y)/AA) < 0)
    dy0 = 0; 
  if ((w0 = (xxe-xxb)/AA) > (paper_aa_w-dx0))
    w0 = paper_aa_w - dx0;
  if ((h0 = (yye-yyb)/AA) > (paper_aa_h-dy0))
    h0 = paper_aa_h - dy0;
  img = gdk_image_get(BackPixmap, dx0, dy0, w0, h0);
  gc = gdk_gc_new(DrawingArea->window);
  rx = 1 << (16 - Visual->red_prec);
  gx = 1 << (16 - Visual->green_prec);
  bx = 1 << (16 - Visual->blue_prec);

  for (yy = yyb; yy < yye; yy += AA){
    if ((yy/AA < 0) || (yy/AA >= paper_aa_h))
      continue;
    for (xx = xxb; xx < xxe; xx += AA){
      if ((xx < 0) || (xx/AA >= paper_aa_w))
	continue;
      dots = 0;
      for (yy2 = 0; yy2 < AA; yy2++){
	if (((yy + yy2) < (pos_y + ofs_y)) ||
	    ((pos_y + ofs_y + h) <= (yy + yy2)))
	  continue;
	for (xx2 = 0; xx2 < AA; xx2++){
	  if (((xx + xx2) < (pos_x + ofs_x)) ||
	      ((pos_x + ofs_x + w) <= (xx + xx2)))
	    continue;
	  if (bm != NULL){
	    d = bm[(yy + yy2 - (pos_y + ofs_y)) * raster 
		  + (xx + xx2 - (pos_x + ofs_x))/8];
	    if ((d & bits[(xx + xx2 - (pos_x + ofs_x)) % 8]) != 0)
	      dots++;
	  } else {
	    dots++;
	  }
	}
      }
      if (dots == 0){
	continue;
      } else if (dots == AA*AA){
	gdk_draw_point(BackPixmap, color_gcs[dots], xx/AA, yy/AA);
      } else {
 	GdkColor     c;
	guint32       p0;
	guint32       r0, g0, b0;

	if ((Visual == NULL) || (img == NULL)){
	  gdk_draw_point(BackPixmap, color_gcs[dots], xx/AA, yy/AA);
	} else {
	  p0 = gdk_image_get_pixel(img, (xx-xxb)/AA, (yy-yyb)/AA);
	  r0 = ((p0 & Visual->red_mask)   >> Visual->red_shift)   * rx;
	  g0 = ((p0 & Visual->green_mask) >> Visual->green_shift) * gx;
	  b0 = ((p0 & Visual->blue_mask)  >> Visual->blue_shift)  * bx;
          c.red
	    = (dots*(0x10000*current_color_fg_r-r0))/(AA*AA) + r0;
	  c.green
	    = (dots*(0x10000*current_color_fg_g-g0))/(AA*AA) + g0;
	  c.blue
	    = (dots*(0x10000*current_color_fg_b-b0))/(AA*AA) + b0;
          c.pixel = pixel_value(c.red, c.green, c.blue);
	  gdk_gc_set_foreground(gc, &c);
	  gdk_draw_point(BackPixmap, gc, xx/AA, yy/AA);	  
	}
      }
    }
  }

  gdk_draw_pixmap(DrawingArea->window, color_gcs[N_AA-1], BackPixmap, 
		  dx0, dy0, dx0, dy0, w0, h0);
  gdk_image_destroy(img);
}


static void
put_rgb(long pos_x, long pos_y, 
	unsigned char *bm, int raster, long w, long h, 
	int do_cache, long dvipos)
{
  if (param_aa_perfect == 1){
    /* Perfect Antialias (but very slow on remote X display) 
       If \colorbox command is not used, choise for param_aa_perfect == 0
       is better because of its rendering speed.   */
    put_rgb_1(pos_x, pos_y, bm, raster, w, h, do_cache, dvipos);
  } else {
    /* Imperfect Antialias (but fast on remote X display) */
    put_rgb_2(pos_x, pos_y, bm, raster, w, h, do_cache, dvipos);
  }
}



static void
DEV_put_bitmap_rgb(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bm, int font_id, 
		   long dvipos, long code_point, long pos_x, long pos_y)
{
  if ((font_id < 0) && (dvipos > 0) && (code_point < 0)){
    put_rgb(pos_x, pos_y, bm->bitmap, bm->raster, bm->width, bm->height, 
	    1, dvipos);
  } else {
    put_rgb(pos_x, pos_y, bm->bitmap, bm->raster, bm->width, bm->height, 
	    0, -1);
  }
}

static void
DEV_put_rectangle_rgb(DVI_DEVICE dev, DVI dvi, 
		      long pos_x, long pos_y, long w, long h)
{
  put_rgb(pos_x, pos_y, NULL, 0, w, h, 0, -1);
}




static void
put_pixmap_rgb(DVI_PIXMAP_RGB pm, long pos_x, long pos_y, 
	       int do_cache, long dvipos)
{
  guchar  *img, *q;
  int     *imgx, *qx;
  guint32  r, g, b;
  long     xx, yy, xx2, yy2, yyb, xxb, yye, xxe, w, h;
  unsigned char  *p;

#if 0
  printf("PIXMAP (W,H)=(%ld,%ld) at (%ld,%ld), raster %ld\n", 
	 pm->width, pm->height, pos_x, pos_y, pm->raster);
#endif

  if ((pos_x+ofs_x)/AA >= paper_aa_w)
    return;
  if ((pos_y+ofs_y)/AA >= paper_aa_h)
    return;

  h = pm->height;
  w = pm->width;
  if ((w == 0) || (h == 0))
    return;

  yyb = AA * (long) ((pos_y + ofs_y) / AA);
  xxb = AA * (long) ((pos_x + ofs_x) / AA);
  yye = AA * (long) ((pos_y + ofs_y + h + AA-1) / AA);
  xxe = AA * (long) ((pos_x + ofs_x + w + AA-1) / AA);

  if (yyb < 0)
    yyb = 0;
  if (xxb < 0)
    xxb = 0;
  if (yye >= paper_aa_h * AA)
    yye = paper_aa_h * AA;
  if (xxe >= paper_aa_w * AA)
    xxe = paper_aa_w * AA;

  if ((imgx = calloc(4*((xxe-xxb)/AA)*((yye-yyb)/AA), sizeof(int))) == NULL)
    return;

  for (yy = yyb; yy < yye; yy += AA){
    for (xx = xxb; xx < xxe; xx += AA){
      for (yy2 = 0; yy2 < AA; yy2++){
	if (((yy+yy2)/AA < 0) || ((yy+yy2)/AA >= paper_aa_h))
	  continue;
	if (((yy + yy2) < (pos_y + ofs_y)) ||
	    ((pos_y + ofs_y + h) <= (yy + yy2)))
	  continue;
	for (xx2 = 0; xx2 < AA; xx2++){
	  if (((xx+xx2) < 0) || ((xx+xx2)/AA >= paper_aa_w))
	    continue;
	  if (((xx+xx2) < (pos_x + ofs_x)) ||
	      ((pos_x + ofs_x + w) <= (xx+xx2)))
	    continue;
	  p = &pm->bitmap[(yy+yy2-(pos_y+ofs_y))*pm->raster 
			 + 3*(xx+xx2-(pos_x+ofs_x))];
	  qx = &imgx[(((yy-yyb)/AA) * ((xxe-xxb)/AA) + (xx-xxb)/AA) * 4 ];
	  *(qx++) += 1;
	  *(qx++) += *(p++);
	  *(qx++) += *(p++);
	  *(qx++) += *(p++);
	}
      }
    }
  }

  if ((img = malloc(3*((xxe-xxb)/AA)*((yye-yyb)/AA))) == NULL)
    return;

  q = img;
  qx = imgx;
  for (yy = yyb ; yy < yye; yy += AA){
    for (xx = xxb ; xx < xxe; xx += AA){
      if (qx[0] == 0)
	continue;
      r = 255 * qx[1] / (pm->max * qx[0]);
      g = 255 * qx[2] / (pm->max * qx[0]); 
      b = 255 * qx[3] / (pm->max * qx[0]);
      qx += 4;
      *(q++) = r;
      *(q++) = g;
      *(q++) = b;
    }
  }

  if (do_cache != 0){
    imgcache_do_cache((IMGCACHE)dvi->udp1, dvipos, shrink, IMGTYPE_PIXMAP_RGB,
		      img, (xxe-xxb)/AA, (yye-yyb)/AA, xxb/AA, yyb/AA);
#if 0
    printf(">>> pos_x,pos_y=(%ld,%ld), x,y=(%ld,%ld), w,h=(%ld,%ld)\n", 
	 pos_x, pos_y, xxb/AA, yyb/AA, (xxe-xxb)/AA, (yye-yyb)/AA);
#endif
  }

  gdk_draw_rgb_image (BackPixmap, 
		      DrawingArea->style->fg_gc[GTK_STATE_NORMAL],
		      xxb/AA, yyb/AA, (xxe-xxb)/AA, (yye-yyb)/AA,
		      GDK_RGB_DITHER_MAX, img, 3*(xxe-xxb)/AA);
  gdk_draw_rgb_image (DrawingArea->window, 
		      DrawingArea->style->fg_gc[GTK_STATE_NORMAL],
		      xxb/AA, yyb/AA, (xxe-xxb)/AA, (yye-yyb)/AA,
		      GDK_RGB_DITHER_MAX, img, 3*(xxe-xxb)/AA);
  gdk_flush();
}


static void
DEV_put_pixmap_rgb(DVI_DEVICE dev, DVI dvi, DVI_PIXMAP_RGB pm, 
		   int font_id, long dvipos, long code_point, 
		   long pos_x, long pos_y)
{
  if ((font_id < 0) && (dvipos > 0) && (code_point < 0)){
    put_pixmap_rgb(pm, pos_x, pos_y, 1, dvipos);
  } else {
    put_pixmap_rgb(pm, pos_x, pos_y, 0, dvipos);
  }
}




static int
DEV_before_ps_figure(DVI_DEVICE dev, DVI dvi, char *cmd, 
		     long pos_x, long pos_y, long dvipos)
{
  guchar    *img;
  int        type;
  long       x, y, w, h;

  cursor_current = cursor_eps;
  gdk_window_set_cursor(xgdvi->window, cursor_current);
  gdk_flush();

  if (param_no_eps_cache == 1)
    return 1;

  img = imgcache_is_cached((IMGCACHE)dvi->udp1, 
			   dvipos, shrink, &type, &w, &h, &x, &y);
  if (img == NULL)
    return 1; 

  gdk_draw_rgb_image (BackPixmap, 
		      DrawingArea->style->fg_gc[GTK_STATE_NORMAL],
		      x, y, w, h, GDK_RGB_DITHER_MAX, img, 3*w);
  gdk_draw_rgb_image (DrawingArea->window, 
		      DrawingArea->style->fg_gc[GTK_STATE_NORMAL],
		      x, y, w, h, GDK_RGB_DITHER_MAX, img, 3*w);
  imgcache_free_img(img, type);

  return 0; 
}

static void
DEV_after_ps_figure(DVI_DEVICE dev, DVI dvi, char *cmd, 
		    long x, long y, long dvipos)
{
  cursor_current = cursor_drawing;
  gdk_window_set_cursor(xgdvi->window, cursor_current);
  gdk_flush();
}

static int
DEV_draw_ps_figures(DVI_DEVICE dev, DVI dvi)
{
  if (param_draw_eps == 1)
    return  1;
  return  0;
}

#if 0
static int
DEV_ps_figure_display_step(DVI_DEVICE dev, DVI dvi, long w, long h)
{
  return EPS_INCREMENTAIL_STEP;
}
#endif






static int
DEV_poll(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  static int  t = 0;

  if (poll_type != DVI_POLL_PAGE)
    return 0;
  if ((--t) > 0)
    return 0;
  t = POLLING_INTERVAL;

  while (gtk_events_pending()){
    gtk_main_iteration();
    if (ui_cmd != CMD_NONE)
      return 1;
  }

  return 0;
}






DVI_PROPERTY
make_dvi_property(void)
{
  DVI_PROPERTY  dvi_props;

  dvi_props = DVI_PROPERTY_ALLOC_DEFAULT();
  DVI_PROPERTY_SET(dvi_props,   DVI_PROP_ASYNC_GS_INVOCATION);
  if (param_delayed_font_open == 1)
    DVI_PROPERTY_SET(dvi_props, DVI_PROP_DELAYED_FONT_OPEN);
#if 0
  DVI_PROPERTY_SET(dvi_props, DVI_PROP_INCREMENTAL_EPS_DISPLAY);
#endif

  return dvi_props;
}

static DVI_DEVICE
create_dev(void)
{
  DVI_DEVICE  d;

  if ((d = DVI_DEVICE_ALLOC()) == NULL)
    return NULL;

  d->h_dpi = d->v_dpi        = param_dpi;
  d->device_polling          = DEV_poll;
  d->put_bitmap            = DEV_put_bitmap_rgb;
  d->put_rectangle         = DEV_put_rectangle_rgb;
  if ((param_force_mono == 0) &&
      ((VisualType == GDK_VISUAL_TRUE_COLOR) || 
       (VisualType == GDK_VISUAL_DIRECT_COLOR))){
    d->put_pixmap_rgb        = DEV_put_pixmap_rgb;
    d->color_rgb             = DEV_color_rgb;
    d->bgcolor_rgb           = DEV_bgcolor_rgb;
  }
  d->message_advice          = NULL;

  d->draw_ps_figures         = DEV_draw_ps_figures;
  d->before_ps_figure        = DEV_before_ps_figure;
  d->after_ps_figure         = DEV_after_ps_figure;
#if 0
  d->ps_figure_display_step  = DEV_ps_figure_display_step;
#endif

  return  d;
}




gint
event_configure_win(GtkWidget *w, GdkEventConfigure *ev)
{
#if 0
  printf("** configure event: window <%d,%d> %dx%d\n",
	 ev->x, ev->y, ev->width, ev->height);
#endif
  param_win_size_w = ev->width;
  param_win_size_h = ev->height;
  return  FALSE;
}


gint
event_configure(GtkWidget *w, GdkEventConfigure *ev)
{
#if 0
  printf("CONFIGURE EVENT\n");
#endif

  if (BackPixmap != NULL)
    return FALSE;

  init_colors();

  BackPixmap = gdk_pixmap_new(DrawingArea->window, 
			      paper_aa_w, paper_aa_h, -1);
  gdk_draw_rectangle(BackPixmap, color_gcs[0], TRUE, 0, 0, 
		     paper_aa_w, paper_aa_h);

  page_state = PAGE_EMPTY;
  ui_cmd = CMD_DRAW_PAGE;
  return  FALSE;
}

gint
event_expose(GtkWidget *w, GdkEventExpose *ev)
{
  if (BackPixmap == NULL)
    return FALSE;

  gdk_draw_pixmap(DrawingArea->window, 
		  DrawingArea->style->fg_gc[GTK_WIDGET_STATE(DrawingArea)], 
		  BackPixmap, ev->area.x, ev->area.y, 
		  ev->area.x, ev->area.y, ev->area.width, ev->area.height);

  if (page_state == PAGE_DRAW_FINISH){
    if ((dvi != NULL) && (dvi_file != NULL)
	&& (DVI_FILE_MODIFIED(dvi, dvi_file) == 1)){
      page_state = PAGE_EMPTY;
      ui_cmd = CMD_DRAW_PAGE;
    }
    return FALSE;
  }

  if (status == STAT_MAINLOOP)
    draw_page();

  return FALSE;
}

static void
init_colors()
{
  GdkColormap *cmap;
  GdkColor     color_fg, color_bg;
  int          i;
  
  VisualType = GDK_VISUAL_DIRECT_COLOR;
  Visual = gdk_visual_get_best_with_type(VisualType);
  if (Visual == NULL){
    VisualType = GDK_VISUAL_TRUE_COLOR;
    Visual = gdk_visual_get_best_with_type(VisualType);
  }
  if ((Visual == NULL) || (Visual->depth < 15)){
    VisualType = GDK_VISUAL_STATIC_COLOR;
    Visual = gdk_visual_get_best_with_type(VisualType);
  }
  if (Visual == NULL){
    VisualType = GDK_VISUAL_PSEUDO_COLOR;
    Visual = gdk_visual_get_best_with_type(VisualType);
  }
  if (Visual == NULL){
    VisualType = -1;
  }

  if ((param_force_color == 1) &&
      (VisualType != GDK_VISUAL_TRUE_COLOR) &&
      (VisualType != GDK_VISUAL_DIRECT_COLOR)){
    printf(_("No support for TrueColor/DirectColor Visuals. Use grayscale display.\n"));
  }

  if (color_gcs == NULL){
    if (    !gdk_color_parse(param_foregr, &color_fg)
	 || !gdk_color_parse(param_backgr, &color_bg) ){
      fprintf(stderr, "Unknown color names. Use default colors.\n");
      if (  !gdk_color_parse(COLOR_FG, &color_fg)
	 || !gdk_color_parse(COLOR_BG, &color_bg) ){
	color_fg.red = color_fg.green = color_fg.blue = (guint16)0;
	color_bg.red = color_bg.green = color_bg.blue = (guint16)65535;
      }
    }
    cmap = gdk_window_get_colormap(DrawingArea->window);
    gdk_colormap_alloc_color(cmap, &color_fg, FALSE, TRUE);
    gdk_colormap_alloc_color(cmap, &color_bg, FALSE, TRUE);

    default_color_fg_r = current_color_fg_r
      = (double) color_fg.red   / 65535.0;
    default_color_fg_g = current_color_fg_g 
      = (double) color_fg.green / 65535.0;
    default_color_fg_b = current_color_fg_b = 
      (double) color_fg.blue  / 65535.0;
    default_color_bg_r = current_color_bg_r = 
      (double) color_bg.red   / 65535.0;
    default_color_bg_g = current_color_bg_g = 
      (double) color_bg.green / 65535.0;
    default_color_bg_b = current_color_bg_b = 
      (double) color_bg.blue  / 65535.0;

    color_gcs = g_new(GdkGC*, N_AA);
    for (i = 0; i < N_AA; i++)
      color_gcs[i] = gdk_gc_new(DrawingArea->window);

    change_window_colors(default_color_fg_r, 
			 default_color_fg_g,
			 default_color_fg_b,
			 default_color_bg_r,
			 default_color_bg_g,
			 default_color_bg_b);
  }
}


static void
change_window_colors(double fg_r, double fg_g, double fg_b,
		     double bg_r, double bg_g, double bg_b)
{
  GdkColor     cols;
  double    r, g, b;
  int       i;

#if 0
  printf("* %.2f %.2f %.2f   %.2f %.2f %.2f\n", 
	 fg_r, fg_g, fg_b, bg_r, bg_g, bg_b);
#endif

  if (fg_r > 1.0) fg_r = 1.0;
  if (fg_g > 1.0) fg_g = 1.0;
  if (fg_b > 1.0) fg_b = 1.0;
  if (bg_r > 1.0) bg_r = 1.0;
  if (bg_g > 1.0) bg_g = 1.0;
  if (bg_b > 1.0) bg_b = 1.0;

  current_color_fg_r = (fg_r >= 0) ? fg_r : default_color_fg_r;
  current_color_fg_g = (fg_g >= 0) ? fg_g : default_color_fg_g;
  current_color_fg_b = (fg_b >= 0) ? fg_b : default_color_fg_b;
  current_color_bg_r = (bg_r >= 0) ? bg_r : default_color_bg_r;
  current_color_bg_g = (bg_g >= 0) ? bg_g : default_color_bg_g;
  current_color_bg_b = (bg_b >= 0) ? bg_b : default_color_bg_b;

  r = (double) (current_color_fg_r - current_color_bg_r) / (double)(N_AA-1);
  g = (double) (current_color_fg_g - current_color_bg_g) / (double)(N_AA-1);
  b = (double) (current_color_fg_b - current_color_bg_b) / (double)(N_AA-1);
  for (i = 0; i < N_AA; i++){
    cols.red   = (guint16) ceil((current_color_bg_r + r * i) * 65535.0);
    cols.green = (guint16) ceil((current_color_bg_g + g * i) * 65535.0);
    cols.blue  = (guint16) ceil((current_color_bg_b + b * i) * 65535.0);
    cols.pixel = pixel_value(cols.red, cols.green, cols.blue);
    gdk_gc_set_foreground(color_gcs[i], &cols);
  }
} 


static guint32
pixel_value(guint16 r, guint16 g, guint16 b)
{
  guint32      pixel;
  GdkColormap *cmap;
  GdkColor     cols;
  
  if (param_rvideo == 1){
    r = 65535 - r;
    g = 65535 - g;
    b = 65535 - b;
  }

  if ((VisualType == GDK_VISUAL_TRUE_COLOR) ||
      (VisualType == GDK_VISUAL_DIRECT_COLOR)){
    pixel = ( ((r >> (16-Visual->red_prec))   << Visual->red_shift)   |
	      ((g >> (16-Visual->green_prec)) << Visual->green_shift) |
	      ((b >> (16-Visual->blue_prec))  << Visual->blue_shift));
  } else {
    cmap = gdk_window_get_colormap(DrawingArea->window);
    cols.red   = r;
    cols.green = g;
    cols.blue  = b;
    gdk_colormap_alloc_color(cmap, &cols, FALSE, TRUE);
    pixel = cols.pixel;
  }
  return pixel;
}


void
draw_page(void)
{
  int  dp;

#if 1
  if (status == STAT_RUNNING)
    return;
#else
  if (status != STAT_MAINLOOP)
    return;
#endif

  if (dvi == NULL)
    return;

  file_update_check(NULL);

  if (dvi == NULL)
    return;

  if (timeout_tag >= 0){
    gtk_timeout_remove(timeout_tag);
    timeout_tag = -1;
  }

  status = STAT_RUNNING;
  while (ui_cmd == CMD_DRAW_PAGE){
    if (dvi != NULL)
      message(_("%s (%d pages)"), dvi_file, dvi->pages);
    else
      message(_("%s"), dvi_file);
    if (dvi == NULL)
      break;
    cursor_current = cursor_drawing;
    gdk_window_set_cursor(xgdvi->window, cursor_current);
    gdk_flush();
    clear_page();
    ui_cmd = CMD_NONE;
    page_state = PAGE_DRAWING;
    dp = DVI_DRAW_PAGE(dvi, dev, page, shrink / AA);
    if (dp != DVI_DRAW_INTERRUPTED){
      page_state = PAGE_DRAW_FINISH;
      ui_cmd = CMD_NONE;
    }
  }
  status = STAT_MAINLOOP;

  cursor_current = cursor_done;
  gdk_window_set_cursor(xgdvi->window, cursor_current);
  gdk_flush();

  if (UPDATE_CHECK_INTERVAL > 0){
    timeout_tag = gtk_timeout_add(UPDATE_CHECK_INTERVAL * 1000,
				  file_update_check, NULL);
  }
}

void
clear_page(void)
{
  gdk_draw_rectangle(BackPixmap, color_gcs[0], TRUE, 0, 0,
		     paper_aa_w, paper_aa_h);
  gdk_draw_pixmap(DrawingArea->window, color_gcs[N_AA-1], 
		  BackPixmap, 0, 0, 0, 0, paper_aa_w, paper_aa_h);
  gdk_flush();
  page_state = PAGE_EMPTY;
}

void
resize_window(void)
{
  if (dvi == NULL)
    return;
  change_window_size(paper, orient);
  gtk_drawing_area_size(GTK_DRAWING_AREA(DrawingArea), paper_aa_w, paper_aa_h);
  gtk_container_check_resize (GTK_CONTAINER(ScrolledWindow));
  if (BackPixmap != NULL)
    gdk_pixmap_unref(BackPixmap);
  BackPixmap = gdk_pixmap_new(DrawingArea->window, paper_aa_w, paper_aa_h, -1);
}


static void
change_window_size(int pid, int orientation)
{
  char   *p;
  double  pw, ph;
  int     w, h;

  if ((p = paper_name(pid)) == NULL)
    return;
  DVI_parse_paper_size(p, &pw, &ph);

  if (orientation == ORI_PORT){
    w = pw * AA * param_dpi / shrink;
    h = ph * AA * param_dpi / shrink;
  } else {
    w = ph * AA * param_dpi / shrink;
    h = pw * AA * param_dpi / shrink;
  }
  paper_aa_w = (w + AA - 1) / AA;
  paper_aa_h = (h + AA - 1) / AA;
  ofs_x = OFFSET_X * AA * param_dpi / shrink;
  ofs_y = OFFSET_Y * AA * param_dpi / shrink;
}


void
resize_window_delta(int dx, int dy)
{
  int   xx, yy;
  
  xx = dx * param_window_delta;
  yy = dy * (param_window_delta * paper_aa_h / paper_aa_w);
  param_win_size_w += xx;
  param_win_size_h += yy;
  if (param_win_size_w < 100)
    param_win_size_w = 100;
  if (param_win_size_h < 100)
    param_win_size_h = 100;
  gtk_window_set_default_size(GTK_WINDOW(xgdvi),
			      param_win_size_w, param_win_size_h);
  gtk_widget_queue_resize(GTK_WIDGET(xgdvi));
}




void
message(char *fmt,...)
{
  va_list   ap;
  char      msg[512];
  guint     mid, cid;
  static int  empty = 1;

  va_start(ap, fmt);
  vsprintf(msg, fmt, ap);
  cid = gtk_statusbar_get_context_id(GTK_STATUSBAR(StatusBar),
				     "xgdvi_message");
  if (empty == 0)
    gtk_statusbar_pop(GTK_STATUSBAR(StatusBar), cid);
  mid = gtk_statusbar_push(GTK_STATUSBAR(StatusBar), cid, msg);
  empty = 0;
  va_end(ap);
  gdk_flush();
}




void 
run_command(int cmd)
{
  int  pf;

  switch (cmd){
  default:
  case -1:
    return;
  case 'q':
    state_save();
#if 0
    gtk_main_quit();
#endif
    xgdvi_cleanup();
    break;
  case ' ':
  case CTL('V'):
    goto_page(page + 1); 
    break;
  case 'b':
  case 'v': 
  case 0x7f:
    goto_page(page - 1); 
    break;
  case '<':
    goto_page(1); 
    break;
  case '>':
    if (dvi != NULL)
      goto_page(dvi->pages);
    break;
  case ',':
    if (dvi == NULL)
      break;
    if (1 <= page-10)
      goto_page(page-10);
    else  
      goto_page(1);
    break;
  case '.':  
    if (dvi == NULL)
      break;
    if (page+10 <= dvi->pages) 
      goto_page(page+10);
    else  
      goto_page(dvi->pages);
    break;
  case ';':
    buffer_select_next(1);
    switched_buffer();
    update_page_display();
    break;
  case ':':
    buffer_select_next(0);
    switched_buffer();
    update_page_display();
    break;
  case CTL('P'): case 'k': 
    pf = preview_window_get_pos_type();
    if ((pf & POS_TYPE_TOP) == 0){
      preview_window_scroll_rdelta(0, -param_scroll_delta);
    }
    break;
  case CTL('N'): case 'j': 
    if (dvi == NULL)
      break;
    pf = preview_window_get_pos_type();
    if ((pf & POS_TYPE_BOTTOM) == 0){
      preview_window_scroll_rdelta(0, param_scroll_delta);
    }
    break;
  case CTL('B'): case 'h':
    pf = preview_window_get_pos_type();
    if ((pf & POS_TYPE_LEFT) != 0){
      if ((pf & POS_TYPE_TOP) != 0){
	if (page-1 >= 1){
	  goto_page(page-1);
	  preview_window_set_pos_type(POS_TYPE_BOTTOM|POS_TYPE_RIGHT);
	}
      } else {
	preview_window_scroll_rdelta(SCROLL_MAX, -param_scroll_delta);
      }
    } else {
      preview_window_scroll_rdelta(-param_scroll_delta, 0);
    }
    break;
  case CTL('F'): case 'l':
    if (dvi == NULL)
      break;
    pf = preview_window_get_pos_type();
    if ((pf & POS_TYPE_RIGHT) != 0){
      if ((pf & POS_TYPE_BOTTOM) != 0){
	if (page+1 <= dvi->pages){
	  goto_page(page+1);
	  preview_window_set_pos_type(POS_TYPE_TOP|POS_TYPE_LEFT);
	}
      } else {
	preview_window_scroll_rdelta(-SCROLL_MAX, param_scroll_delta);
      }
    } else {
      preview_window_scroll_rdelta(param_scroll_delta, 0);
    }
    break;
  case CTL('A'): 
    preview_window_scroll_rdelta(-SCROLL_MAX, 0);
    break;
  case CTL('E'): 
    preview_window_scroll_rdelta(SCROLL_MAX, 0);
    break;
  case '^': 
    preview_window_scroll_rdelta(0, -SCROLL_MAX);
    break;
  case '_': 
    preview_window_scroll_rdelta(0, SCROLL_MAX);
    break;
  case '{': 
    if (dvi != NULL)
      resize_window_delta(-1,-1);
    break;
  case '}':  
    if (dvi != NULL)
      resize_window_delta(+1,+1);
    break;
  case '9': 
    if (dvi != NULL)
      resize_window_delta( 0,-1);
    break;
  case '0': 
    if (dvi != NULL)
      resize_window_delta( 0,+1);
    break;
  case ')': 
    if (dvi != NULL)
      resize_window_delta(+1, 0);
    break;
  case '(': 
    if (dvi != NULL)
      resize_window_delta(-1, 0);
    break;
  case '[': 
    if (dvi != NULL)
      change_window_params(1, shrink * 10/9, 0);
    break;
  case ']': 
    if (dvi != NULL)
      change_window_params(1, shrink * 9/10, 0);
    break;
  case 'o':
    if (dvi != NULL) {
      change_window_params(0, shrink, 1);
      imgcache_purge((IMGCACHE)dvi->udp1); 
    }
    break;
  case '+':
    preview_window_set_pos_center(0.5, 0.5);
    break;
  case 'O': case CTL('O'):  
    on_File_clicked(NULL, NULL);
    break;
  case 'P': 
    on_Print_clicked(NULL, NULL);
    break;
  case 'B': 
    on_Buffer_clicked(NULL, NULL);
    break;
  case CTL('W'): 
    on_buffButtonClose_clicked(NULL, NULL);
    break;
  }

  if (ui_cmd == CMD_DRAW_PAGE){
    draw_page();
  }
}

void
change_button_mode(int mode)
{
  int  i;
  GtkWidget  *w, *ws;
  static char *name[] = { "CmdBoxTop", "CmdBoxLeft", "StatusBar", NULL};

  ws = gtk_object_get_data (GTK_OBJECT (xgdvi), "ScrolledWindow");
  if (mode == 1){
    for (i = 0; name[i] != NULL; i++){
      w = gtk_object_get_data (GTK_OBJECT (xgdvi), name[i]); 
      if (w != NULL)
	gtk_widget_show (w);
    }
    if (ws != NULL)
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (ws),
				      GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
    update_page_display();
  } else {
    for (i = 0; name[i] != NULL; i++){
      w = gtk_object_get_data (GTK_OBJECT (xgdvi), name[i]); 
      if (w != NULL)
	gtk_widget_hide (w);
    }
    if (ws != NULL)
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (ws),
				      GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  }
}




static int  button_pressed_x = 0;
static int  button_pressed_y = 0;


gint
event_button_press(GtkWidget *w, GdkEventButton *ev)
{
  if (ev->button == 1){
    button_pressed_x = ev->x;
    button_pressed_y = ev->y;
    gdk_window_set_cursor(xgdvi->window, cursor_scroll);
  } else if (ev->button == 4){
    run_command(CTL('V'));
  } else if (ev->button == 5){
    run_command('v');
  }
  return TRUE;
}

gint
event_motion_notify(GtkWidget *w, GdkEventMotion *ev)
{
  int  dx, dy, x, y;
  GdkModifierType state;

  if (ev->is_hint != 0){
    gdk_window_get_pointer (ev->window, &x, &y, &state);
  } else {
    x = ev->x;
    y = ev->y;
    state = ev->state;
  }

  if ((state & GDK_BUTTON1_MASK) == 0)
    return TRUE;

  dx = button_pressed_x - x;
  dy = button_pressed_y - y;

  if ((dx*dx + dy*dy) < 10){   /* Movement is too small. */
    return TRUE;
  }

  preview_window_scroll_delta(dx, dy);

  gdk_window_get_pointer (ev->window, &x, &y, &state);  /* important */
  button_pressed_x = x;
  button_pressed_y = y;

  return TRUE;
}

gint
event_button_release(GtkWidget *w, GdkEventButton *ev)
{
  int  dx, dy;

  if (ev->button != 1)
    return TRUE;

  dx = button_pressed_x - ev->x;
  dy = button_pressed_y - ev->y;
  preview_window_scroll_delta(dx, dy);
  button_pressed_x = ev->x;
  button_pressed_y = ev->y;

  gdk_window_set_cursor(xgdvi->window, cursor_current);
  gdk_flush();

  buffer_save_current_state();

  return TRUE;
}



void
preview_window_scroll_rdelta(int rx, int ry)
{
  int  dx, dy;
  GtkAdjustment *hadj, *vadj;

  hadj = gtk_scrolled_window_get_hadjustment(ScrolledWindow);
  vadj = gtk_scrolled_window_get_vadjustment(ScrolledWindow);

  dx = (hadj->upper - hadj->lower) * rx / 100;
  dy = (vadj->upper - hadj->lower) * ry / 100;

#if 0
  printf("** %.2f %.2f  %d %d\n", 
	 (hadj->upper - hadj->lower), (vadj->upper - vadj->lower),
	 dx, dy);
#endif

  preview_window_scroll_delta(dx, dy);
}

void
preview_window_scroll_delta(int dx, int dy)
{
  int  new_x, new_y;
  GtkAdjustment *hadj, *vadj;

  hadj = gtk_scrolled_window_get_hadjustment(ScrolledWindow);
  vadj = gtk_scrolled_window_get_vadjustment(ScrolledWindow);

  new_x = hadj->value + dx;
  if (new_x < hadj->lower)
    new_x = hadj->lower;
  if (new_x > hadj->upper - hadj->page_size)
    new_x = hadj->upper - hadj->page_size;

  new_y = vadj->value + dy;
  if (new_y < vadj->lower)
    new_y = vadj->lower;
  if (new_y > vadj->upper - vadj->page_size)
    new_y = vadj->upper - vadj->page_size;

#if 0
  printf("Move: %d %d  %d %d\n", dx, dy, new_x, new_y);
  printf("  HAdj: %.2f %.2f %.2f %.2f\n",
	 hadj->lower, hadj->value, hadj->upper,	hadj->page_size);
  printf("  VAdj: %.2f %.2f %.2f %.2f\n", 
	 vadj->lower, vadj->value, vadj->upper, vadj->page_size);
#endif

  hadj->value = new_x;
  vadj->value = new_y;
  gtk_adjustment_value_changed(hadj);
  gtk_adjustment_value_changed(vadj);
}

void
preview_window_set_pos_type(int type)
{
  int  new_x, new_y;
  GtkAdjustment *hadj, *vadj;

  hadj = gtk_scrolled_window_get_hadjustment(ScrolledWindow);
  vadj = gtk_scrolled_window_get_vadjustment(ScrolledWindow);

  new_x = hadj->value;
  new_y = vadj->value;

  if ((type & POS_TYPE_LEFT) != 0)
    new_x = hadj->lower;
  if ((type & POS_TYPE_RIGHT) != 0)
    new_x = hadj->upper - hadj->page_size;

  if ((type & POS_TYPE_TOP) != 0)
    new_y = vadj->lower;
  if ((type & POS_TYPE_BOTTOM) != 0)
    new_y = vadj->upper - vadj->page_size;

  hadj->value = new_x;
  vadj->value = new_y;
  gtk_adjustment_value_changed(hadj);
  gtk_adjustment_value_changed(vadj);
}

void
preview_window_get_pos(double *rx, double *ry)
{
  GtkAdjustment *hadj, *vadj;

  hadj = gtk_scrolled_window_get_hadjustment(ScrolledWindow);
  vadj = gtk_scrolled_window_get_vadjustment(ScrolledWindow);

  *rx = hadj->value / (hadj->upper - hadj->lower);
  *ry = vadj->value / (vadj->upper - vadj->lower);
}

int
preview_window_get_pos_type(void)
{
  int  pos_type;
  GtkAdjustment *hadj, *vadj;

  hadj = gtk_scrolled_window_get_hadjustment(ScrolledWindow);
  vadj = gtk_scrolled_window_get_vadjustment(ScrolledWindow);

  pos_type = 0;
  if (vadj->value == 0)
    pos_type |= POS_TYPE_TOP; 
  if (vadj->value == vadj->upper - vadj->page_size)
    pos_type |= POS_TYPE_BOTTOM; 
  if (hadj->value == 0)
    pos_type |= POS_TYPE_LEFT; 
  if (hadj->value == hadj->upper - hadj->page_size)
    pos_type |= POS_TYPE_RIGHT; 

  return pos_type;
}


void
preview_window_set_pos(double rx, double ry)
{
  double new_x, new_y;
  GtkAdjustment *hadj, *vadj;

  if (rx >= 0){
    hadj = gtk_scrolled_window_get_hadjustment(ScrolledWindow);
    new_x = rx * (hadj->upper - hadj->lower) + hadj->lower;
    if (new_x < hadj->lower)
      new_x = hadj->lower;
    if (hadj->upper - hadj->page_size < new_x)
      new_x = hadj->upper - hadj->page_size;
    hadj->value = new_x;
    gtk_adjustment_value_changed(hadj);
  }

  if (ry >= 0){
    vadj = gtk_scrolled_window_get_vadjustment(ScrolledWindow);
    new_y = ry * (vadj->upper - vadj->lower) + vadj->lower;
    if (new_y < vadj->lower)
      new_y = vadj->lower;
    if (vadj->upper - vadj->page_size < new_y)
      new_y = vadj->upper - vadj->page_size;
    vadj->value = new_y;
    gtk_adjustment_value_changed(vadj);
  }
}

void
preview_window_set_pos_center(double rx, double ry)
{
  double new_x, new_y;
  GtkAdjustment *hadj, *vadj;

  if (rx >= 0){
    hadj = gtk_scrolled_window_get_hadjustment(ScrolledWindow);
    new_x = rx * hadj->upper + (1 - rx) * hadj->lower - hadj->page_size/2.0;
    if (new_x < hadj->lower)
      new_x = hadj->lower;
    if (hadj->upper - hadj->page_size < new_x)
      new_x = hadj->upper - hadj->page_size;
    hadj->value = new_x;
    gtk_adjustment_value_changed(hadj);
  }

  if (ry >= 0){
    vadj = gtk_scrolled_window_get_vadjustment(ScrolledWindow);
    new_y = ry * vadj->upper + (1 - ry) * vadj->lower - vadj->page_size/2.0;
    if (new_y < vadj->lower)
      new_y = vadj->lower;
    if (vadj->upper - vadj->page_size < new_y)
      new_y = vadj->upper - vadj->page_size;
    vadj->value = new_y;
    gtk_adjustment_value_changed(vadj);
  }
}



int
change_window_params(int change_shrink, double sh, int swap_orientation)
{
  int  v = 0;

  if (dvi == NULL)
    return 0;

  if (swap_orientation == 1){
#if 0
    imgcache_purge((IMGCACHE)dvi->udp1); 
#endif
    if (orient == ORI_PORT)
      orient = ORI_LAND;
    else
      orient = ORI_PORT;
    v = 1;
  }
  if ((change_shrink == 1) 
      && ((double)SHRINK/MAG_MAX <= sh)
      && (sh <= (double)SHRINK/MAG_MIN) ){
    shrink = sh;
    v = 1;
  }

  if (v == 1){
    buffer_save_current_state();
    resize_window();
    clear_page();
    ui_cmd = CMD_DRAW_PAGE;
  }
  
  return v;
}

int
goto_page(int p)
{ 
  int  v = 0;

  if (dvi == NULL){
    ui_cmd = CMD_NONE;
  } else {
    if ((1 <= p) && (p <= dvi->pages) && (p != page)){
      page = p;
      ui_cmd = CMD_DRAW_PAGE;
      clear_page();
      v = 1;
    }
    buffer_save_current_state();
    update_page_display();
    gbuff_change_page(dvi_file, page, dvi->pages);
  }

  return  v;
}



#if HAVE_SIGNAL_H

static RETSIGTYPE  
sig_usr1(int sig)
{
  buffer_dump();  
}

static RETSIGTYPE  
sig_cleanup(int sig)
{
  xgdvi_cleanup();
}

#endif


void
xgdvi_cleanup(void)
{
  fprintf(stderr, "\nExiting xgdvi..."); 
  fprintf(stderr, "\nbye.\n");
  fflush(stderr);
#if !HAVE_ATEXIT
  buffer_cleanup();
#endif
  exit(0);
}

void 
clean_on_exit(void)
{
  buffer_cleanup();
}
