/*
 * lprspec.c
 *
 *  Copyright (C) 1999-2001  Hirotsugu Kakugawa. All rights reserved. 
 *  See "COPYING" for distribution of this software. 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
  

#include "../config.h"

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/param.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRINGS_H
#  include <strings.h>
#endif
#if HAVE_PWD_H
#  include <pwd.h>
#endif

#include <gtk/gtk.h>
#include <glib.h>

#include "libdvi29.h"
#include "xgdvi.h"
#include "strstream.h"

extern char  *paper_name(int); 


char*
lprspec(DVI dvi,
	char *fmt, 
	char *dvifile, 
	int printall, int curr_page, 
	int region_beg, int region_end,
	int page, int npages, 
	int paper, int orient)
{
  int          i, nest, esc;
  str_stream  *z;
  str_stream  *tmp1, *tmp2;
  char         ch, cmd, *p, *q, *p1, *p2; 
  char         str[MAXPATHLEN], str2[512];
  char         tmpfile[MAXPATHLEN];

  strcpy(tmpfile, "");
  if ((z = str_stream_open()) == NULL)
    return NULL;
  tmp1 = str_stream_open();
  tmp2 = str_stream_open();
  if ((tmp1 == NULL) || (tmp2 == NULL)){
    str_stream_close(z);
    if (tmp1 == NULL)
      str_stream_close(tmp1);
    if (tmp2 == NULL)
      str_stream_close(tmp2);
    return NULL;
  }

  while (*fmt != '\0'){

    if (*fmt != '%'){
      ch = *fmt;
      fmt++;
      if (ch != '\\'){
	if ((ch == '\n') || (ch == '\r')){
	  ;   /* ignore CRLF */
	} else {
	  str_stream_add_char(z, ch);
	}
      } else {
	str_stream_add_char(z, *fmt);
	fmt++;
      }
      
    } else if (*(fmt+1) == '\0'){
      fmt++;

    } else {

      fmt++;
      cmd = *fmt;
      fmt++;

      switch (cmd){

      default:  
	break;

      case '%':
	str_stream_add(z, "%");
	break;

      case 'f': /* %f: DVI file name */
	str_stream_add(z, dvifile);
	break;

      case 'b': /* %b: Base name of DVI file */
	str_stream_add(z, g_basename(dvifile));
	break;

      case 'B': /* %B: Base name of DVI file without extension */
	for (q = g_basename(dvifile); *q != '.'; q++){
	  str_stream_add_char(z, *q);
	}
	break;

      case 'F': /* %F: Full path name of DVI file */
      case 'd': /* %d: Directory part of full path name of DVI file */
	if (g_path_is_absolute(dvifile)){
	  strcpy(str, dvifile);
	} else {
	  p = g_get_current_dir();
	  strcpy(str, p);
	  strcat(str, "/"); 
	  strcat(str, dvifile); 
	  g_free(p);
	}
	if (cmd == 'F'){
	  str_stream_add(z, str);
	} else if (cmd == 'd'){
	  p = g_dirname(str);
	  str_stream_add(z, p);
	  g_free(p);
	}
	break;

      case 'c': /* %c: Current (absolute) page number of the DVI file  
		   (1 <= %c <= %n) */
	str_stream_add_int(z, curr_page);
	break;

      case 'C': /* %C: Current page number of the DVI file */
	str_stream_add_int(z, (int)dvi->page_no[curr_page-1].c[0]);
	break;

      case 'n': /* %n: The total number of pages in the DVI file */
	str_stream_add_int(z, npages);
	break;

      case 'R':
	cmd = *fmt;
	switch (*fmt){
	default:  
	  break;
	case 'b':
	  str_stream_add_int(z, (int)region_beg);
	  break;
	case 'e':
	  str_stream_add_int(z, (int)region_end);
	  break;
	case 'B':  
	  str_stream_add_int(z, (int)dvi->page_no[region_beg-1].c[0]);
	  break;
	case 'E':
	  str_stream_add_int(z, (int)dvi->page_no[region_end-1].c[0]);
	  break;
	case 's':
	  str_stream_add_int(z, region_end - region_beg + 1);
	  break;
	case 'r': /* %Rr{HEADER}{DELIM} */
	case 'R': /* %RR{HEADER}{DELIM} */
	  if (printall == 1)
	    break;
	  strcpy(str, "-");
	  strcpy(str2, "-");
	  if (*fmt == '{'){  	  /* get HEADER */
	    for (fmt++, i = 0;
		 (*fmt != '}') && (*fmt != '\0') && (i < sizeof(str)-1); 
		 fmt++, i++){
	      str[i] = *fmt;
	    }
	    str[i] = '\0';
	    if (*fmt == '\0')
	      break;
	    fmt++;
	    if (*fmt == '{'){	  /* get DELIM */
	      for (fmt++, i = 0; 
		   (*fmt != '}') && (*fmt != '\0') && (i < sizeof(str2)-1); 
		   fmt++, i++){
		str2[i] = *fmt;
	      }
	      str2[i] = '\0';
	      if (*fmt != '\0')
		fmt++;
	    }
	  }
	  str_stream_add(z, str);
	  if (cmd == 'r'){
	    if (region_beg != region_end){
	      str_stream_add_int(z, (int)region_beg);
	      str_stream_add(z, str2);
	      str_stream_add_int(z, (int)region_end);
	    } else {
	      str_stream_add_int(z, (int)region_beg);
	    }
	  } else {
	    if (region_beg != region_end){
	      str_stream_add_int(z, (int)dvi->page_no[region_beg-1].c[0]);
	      str_stream_add(z, str2);
	      str_stream_add_int(z, (int)dvi->page_no[region_end-1].c[0]);
	    } else {
	      str_stream_add_int(z, (int)dvi->page_no[region_beg-1].c[0]);
	    }
	  }
	  break;
	}
	fmt++; 
	break;

      case 'p':
	str_stream_add_int(z, page);
	break;

      case 'P':
	str_stream_add_int(z, (int)dvi->page_no[page-1].c[0]);
	break;

      case 'a': /* %a{FMT}{SEP}: Expand FMT for each page.
		   `FMT' is format string for each page.
		      * %p is expanded to absolute page number, and
		      * %P is expanded to page number string
		   `SEP' is a separator between expanded FMTs. 
		      * When `SEP' is omitted, a space is used instead. */
	if (*fmt == '{'){    /* {FMT} */
	  fmt++;
	  nest = 0;
	  esc = 0;
	  while ((nest == 0) && (esc == 0)
		 && (*fmt != '}') && (*fmt != '\0')){
	    str_stream_add_char(tmp1, *fmt);
	    if (esc == 1){
	      esc = 0;
	    } else {
	      if (*fmt == '{')
		nest++;
	      else if (*fmt == '}')
		nest--;
	      else if (*fmt == '\\')
		esc = 1;
	    }
	    fmt++;
	  }
	  if (*fmt != '\0')
	    fmt++;
	} else {
	  str_stream_add(tmp1, "%p");
	}
	p1 = str_stream_compose(tmp1);
	str_stream_clear(tmp1);
	if (*fmt == '{'){    /* {SEP} */
	  fmt++;
	  while ((*fmt != '}') && (*fmt != '\0')){
	    str_stream_add_char(tmp2, *fmt);
	    fmt++;
	  }
	  if (*fmt != '\0')
	    fmt++;
	} else {
	  str_stream_add(tmp2, " ");
	}
	p2 = str_stream_compose(tmp2);
	str_stream_clear(tmp2);
	for (i = region_beg; i <= region_end; i++){
	    q = lprspec(dvi, p1, dvifile, printall, curr_page, 
			region_beg, region_end,
			i, npages, paper, orient);
	    str_stream_add(z, q); 
	    free(q);
	    if (i < npages)
	      str_stream_add(z, p2); 
	}
	if (p1 != NULL)
	  free(p1);
	if (p2 != NULL)
	  free(p2);
	break;

      case 's': /* %p: Paper size name (in lower case) */
	for (p = paper_name(paper); *p != '\0'; p++)
	  str_stream_add_char(z, tolower((int)*p));
	break;

      case 'S': /* %p: Paper size name (in upper case) */
	for (p = paper_name(paper); *p != '\0'; p++)
	  str_stream_add_char(z, toupper((int)*p));
	break;

      case 'o': /* %o{STR}: STR if paper orientation is Portrait  */
	if (*fmt == '{'){
	  for (fmt++; (*fmt != '}') && (*fmt != '\0'); fmt++)
	    if (orient == 0)
	      str_stream_add_char(z, *fmt);
	  if (*fmt != '\0')
	    fmt++;
	}
	break;

      case 'O': /* %O{STR}: STR if paper orientation is landscape  */
	if (*fmt == '{'){
	  for (fmt++; (*fmt != '}') && (*fmt != '\0'); fmt++)
	    if (orient == 1)
	      str_stream_add_char(z, *fmt);
	  if (*fmt != '\0')
	    fmt++;
	}
	break;

      case 'w': /* %c: Current working directory */
	p = g_get_current_dir();
	str_stream_add(z, p);
	g_free(p);
	break;

      case 't': /* %t: Make new temporary file name (with temp directory) */
	if (strcmp(tmpfile, "") == 0){
	  sprintf(tmpfile, "%s/xgdvi-XXXXXXXX", g_get_tmp_dir());
#ifdef HAVE_MKSTEMP
	  mkstemp(tmpfile);
#else
	  mktemp(tmpfile);
#endif
	}
	str_stream_add(z, tmpfile);
	break;

      case 'T': /* %T: Temporary dir */
	str_stream_add(z, g_get_tmp_dir());
	break;

      case 'u': /* %u: Login name (user name) */
	str_stream_add(z, g_get_user_name());
	break;

      case 'r': /* %r: Real user name (GECOS filed) */
	str_stream_add(z, g_get_real_name());
	break;

      case 'h': /* %h: Home directory */
	str_stream_add(z, g_get_home_dir());
	break;

      case 'i': /* %i: User id */
#if HAVE_GETUID
	str_stream_add_int(z, (int)getuid());
#else
	str_stream_add_int(z, 1);
#endif
	break;

      case 'g': /* %g: Group id */
#if HAVE_GETGID
	str_stream_add_int(z, (int)getgid());
#else
	str_stream_add_int(z, 1);
#endif
	break;

      case 'H': /* %H: Host name */
#if HAVE_GETHOSTNAME
	if (gethostname(str, sizeof(str)) != 0)
	  strcpy(str, "localhost");    /* ERROR */
#else
	strcpy(str, "localhost");
#endif
	str_stream_add(z, str);
	break;

      case 'e': /* %e{ENV}: Value of an env variable ENV if defined */
	if (*fmt == '{'){
	  for (fmt++, i = 0; 
	       (*fmt != '}') && (*fmt != '\0') && (i < sizeof(str)); 
	       fmt++, i++)
	    str[i] = *fmt;
	  str[i] = '\0';
	  if (*fmt != '\0')
	    fmt++;
	  if ((p = g_getenv(str)) != NULL)
	    str_stream_add(z, p);
	}
	break;

      case 'E': /* %E{ENV}{ALT-STR}: Value of an env variable ENV if defined */
	if (*fmt == '{'){
	  for (fmt++, i = 0; (*fmt != '}') && (*fmt != '\0'); fmt++, i++)
	    str[i] = *fmt;
	  str[i] = '\0';
	  if (*fmt != '\0'){
	    fmt++;
	    if (*fmt == '{'){
	      for (fmt++, i = 0; (*fmt != '}') && (*fmt != '\0'); fmt++, i++)
		str2[i] = *fmt;
	      str2[i] = '\0';
	      if (*fmt != '\0')
		fmt++;
	      if ((p = g_getenv(str)) != NULL)
		str_stream_add(z, p);
	      else
		str_stream_add(z, str2);
	    }
	  }
	}
	break;

      case 'x':
	cmd = *fmt;
	fmt++;
	switch (cmd){
	case 'A':
	  sprintf(str, "%d", AA);
	  str_stream_add(z, str);
	  break;
	case 'S':
	  sprintf(str, "%.4f", shrink);
	  str_stream_add(z, str);
	  break;
	case 's':
	  sprintf(str, "%.4f", shrink / (double)AA);
	  str_stream_add(z, str);
	  break;
	case 'K':
	  sprintf(str, "%s", param_kpathsea_mode);
	  str_stream_add(z, str);
	  break;
	case 'D':
	  sprintf(str, "%d", param_dpi);
	  str_stream_add(z, str);
	  break;
	case 'V':
	  sprintf(str, "%s", param_vflibcap);
	  str_stream_add(z, str);
	  break;
	}
	break;
      }
    }
  }

  str_stream_close(tmp1);
  str_stream_close(tmp2);

  return  str_stream_compose_and_close(z);
}

