/*
 * resource.c - a module for retrieving resource values
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1996-1997 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/param.h>
#if STDC_HEADERS
#  include <string.h>
#else
#  include <strings.h>
#endif
#if STDC_HEADERS
#  include <unistd.h>
#endif

#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>

#include "dvi-2_6.h"
#include "defs.h"
#include "cf-xldvi.h"
#include "resource.h"
#include "window.h"
#include "paper.h"
#include "../X11/x11-defs.h"


Public struct s_resource  Resource;

Private XrmDatabase  xrdb; 
Private XrmDatabase init_application_resources(void);
Private void  get_resource(char**,char*);
Private void  bogus_resource_int(char*,int);
Private void  bogus_resource_str(char*,char*);
Private int   visual_type(char *name);

extern double atof();


Public void
get_default_application_resources(void)
{
  Resource.vflibcap      = DEFAULT_VFLIBCAP;
  Resource.s_dpi         = DEFAULT_DPI;
  Resource.kpathsea_mode = DEFAULT_KPATHSEA_MODE;
  Resource.s_paper       = DEFAULT_PAPER;
  Resource.s_orient      = DEFAULT_ORIENTATION;
  Resource.s_shrink      = DEFAULT_SHRINK_FACTOR;
  Resource.s_offset_x    = DEFAULT_OFFSET_X;
  Resource.s_offset_y    = DEFAULT_OFFSET_Y;
  Resource.color_char    = DEFAULT_PAPER_CHAR_COLOR;
  Resource.color_paper   = DEFAULT_PAPER_PAPER_COLOR;
  Resource.color_back    = DEFAULT_PAPER_BACKGROUND_COLOR;
  Resource.color_frame   = DEFAULT_PAPER_FRAME_COLOR;
  Resource.s_visual_text = DEFAULT_VISUAL_TEXT;
  Resource.s_page_cache  = DEFAULT_PAGE_CACHE_SIZE;
  Resource.s_poll        = DEFAULT_POLLING_INTERVAL;
  Resource.s_eps_style   = DEFAULT_EPS_FIGURE_STYLE;
  Resource.s_draw_eps    = DEFAULT_DRAW_EPS_FIGS;
  Resource.s_visual_eps  = DEFAULT_VISUAL_EPS;
  Resource.gs_path       = DEFAULT_GS_PATH;
  Resource.s_gs_timeout  = DEFAULT_GS_TIMEOUT;
  Resource.temp_dir      = DEFAULT_TEMP_DIRECTORY;
  xrdb = init_application_resources();
}

Public void
get_application_resources(void)
{
  get_resource(&Resource.vflibcap,         "vflibcap");
  get_resource(&Resource.s_dpi,            "dpi");
  get_resource(&Resource.kpathsea_mode,    "kpathsea_mode");
  get_resource(&Resource.s_paper,          "paperSize");
  get_resource(&Resource.s_orient,         "orientation");
  get_resource(&Resource.s_shrink,         "shrinkFactor");
  get_resource(&Resource.s_offset_x,       "offsetX");
  get_resource(&Resource.s_offset_y,       "offsetY");
  get_resource(&Resource.color_char,       "colorCharacter");
  get_resource(&Resource.color_paper,      "colorPaper");
  get_resource(&Resource.color_back,       "colorBackground");
  get_resource(&Resource.color_frame,      "colorPaper");
  get_resource(&Resource.s_visual_text,    "visualText");
  get_resource(&Resource.s_page_cache,     "pageCacheSize");
  get_resource(&Resource.s_poll,           "pollingInterval");
  get_resource(&Resource.s_eps_style,      "epsStyle");
  get_resource(&Resource.s_draw_eps,       "drawEPSFigures");
  get_resource(&Resource.s_visual_eps,     "visualEPS");
  get_resource(&Resource.temp_dir,         "tempDirectory");
}

Private XrmDatabase
init_application_resources(void)
{
  XrmDatabase  xrdb, xrdb1; 
  char         *xrm, *home, *xenv, *p;
  char         path[MAXPATHLEN], host[256];

  XrmInitialize();
  xrdb = NULL;

  sprintf(path, "%s/lib/X11/app-defaults/%s", XDIR, PROG_NAME_RESOURCE);
  xrdb = XrmGetFileDatabase(path);

  if ((home = (char*)getenv("HOME")) == NULL)
    home = ".";

  if ((xenv = getenv("XAPPLRESDIR")) != NULL)
    sprintf(path, "%s/%s", xenv, PROG_NAME_RESOURCE);
  else {
    sprintf(path, "%s/.app-defaults/%s", home, PROG_NAME_RESOURCE);
  }
  xrdb1 = XrmGetFileDatabase(path); 
  if (xrdb1 != NULL)
    XrmMergeDatabases(xrdb1, &xrdb);

  if ((xrm = XResourceManagerString(x_display)) != NULL){
    xrdb1 = XrmGetStringDatabase(xrm);
  } else {
    sprintf(path, "%s/.Xdefaults", home);
    xrdb1 = XrmGetFileDatabase(path);
  }
  if (xrdb1 != NULL)
    XrmMergeDatabases(xrdb1, &xrdb);

  if ((xenv = getenv("XENVIRONMENT")) != NULL){
    xrdb1 = XrmGetFileDatabase(xenv);
#if HAVE_GETHOSTNAME
  } else {
    gethostname(host, sizeof(host));
    host[sizeof(host)-1] = '\0';
    if ((p = strchr(host, '.')) != NULL)
      *p = '\0';
    sprintf(path, "%s/.Xdefaults-%s", home, host);
    xrdb1 = XrmGetFileDatabase(path);
#endif
  }
  if (xrdb1 != NULL)
    XrmMergeDatabases(xrdb1, &xrdb);

  return xrdb;
}

Private void 
get_resource(char **val, char *res)
{
  char     res_name[1024], res_class[1024];
  char     *type;
  XrmValue result;

  sprintf(res_name,  "%s.%s", PROG_NAME_RESOURCE, res);
  sprintf(res_class, "%s.%s", "Program", "");
  XrmGetResource(xrdb, res_name, res_class, &type, &result);
  if (result.addr != NULL)
    *val = result.addr;
  /**printf("**RESOURCE: %s = %s\n", res, *val); **/
}


Public void
resource_conv(void)
{
  if (Resource.s_dpi != NULL)
    Resource.dpi      = atoi(Resource.s_dpi);
  if (Resource.s_paper != NULL){
    Resource.orient   = PAPER_IS_PORTRAIT(Resource.paper);
    if (Resource.s_orient != NULL)
    Resource.paper    = paper_sym2id(Resource.s_paper, Resource.s_orient);
  }
  if (Resource.s_offset_x != NULL)
    Resource.offset_x = atof(Resource.s_offset_x);
  if (Resource.s_offset_y != NULL)
    Resource.offset_y = atof(Resource.s_offset_y);
  if (Resource.s_shrink != NULL)
    Resource.shrink   = atof(Resource.s_shrink);
  if (Resource.s_page_cache != NULL)
    Resource.page_cache = atoi(Resource.s_page_cache);
  if (Resource.s_poll != NULL)
    Resource.poll     = atoi(Resource.s_poll);

  Resource.eps_style = EPS_STYLE_LATEX2E_GRAPHICS_STY;
  if (Resource.s_eps_style != NULL){
    if (strcmp(Resource.s_eps_style, EPS_STYLE_LATEX209_EPSF_STY_STR) == 0)
      Resource.eps_style = EPS_STYLE_LATEX209_EPSF_STY;
  }
  if (Resource.s_draw_eps != NULL)
    Resource.draw_eps  = atob(Resource.s_draw_eps);
  if (Resource.s_visual_eps != NULL)
    Resource.visual_eps  = visual_type(Resource.s_visual_eps);
  if (Resource.s_visual_text != NULL)
    Resource.visual_text  = visual_type(Resource.s_visual_text);

  if (Resource.shrink <= SHRINK_MIN)
    Resource.shrink   = SHRINK_MIN;
  if (Resource.page_cache <= 0)
    Resource.page_cache = atoi(DEFAULT_PAGE_CACHE_SIZE);
  if (Resource.page_cache <= 0)
    Resource.page_cache = 1;
}

Private int 
visual_type(char *name)
{
  int  t = VISUAL_MONO;
  
  if (tolower(*name) == tolower(VISUAL_GRAY_STR[0]))
    t = VISUAL_GRAY;
  else if (tolower(*name) == tolower(VISUAL_RGB_STR[0]))
    t = VISUAL_RGB;
  return t;
}


Public int
resource_check(void)
{
  int  err;

  err = 0;
  if (Resource.dpi <= 0){
    --err;
    bogus_resource_int("DPI", Resource.dpi);
  }
  if (Resource.paper <= 0){
    --err;
    bogus_resource_str("paper size", Resource.s_paper);
  }
  if (Resource.shrink <= 0){
    --err;
    bogus_resource_int("shrink", Resource.shrink);
  }
  if (((Resource.gs_path == NULL)
       || (strcmp(Resource.gs_path, "") == 0))){
    --err;
    bogus_resource_str("No Ghostscript path", Resource.gs_path);
  }
  return err;
}

Private void
bogus_resource_int(char *reason, int val)
{
  DEV_message_error(dev, dviobj, "Illegal value for %s: %d\n", reason, val);
}
Private void
bogus_resource_str(char *reason, char *val)
{
  DEV_message_error(dev, dviobj, "Illegal value for %s: %s\n", reason, val);
}

/*EOF*/
