/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-page-presence.c:
 * A GtkNotebook page exposing
 * org.freedesktop.Telepathy.Connection.Interface.SimplePresence functionality.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Originally by Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "page-simple-presence.h"

#include <glade/glade.h>
#include <telepathy-glib/enums.h>
#include <telepathy-glib/interfaces.h>

#include "constants.h"
#include "dlg-handles-list.h"
#include "dlg-user-presence.h"
#include "page-priv.h"
#include "preferences.h"
#include "util.h"

struct _TIPageSimplePresenceClass {
    TIPageClass parent;
};

G_DEFINE_TYPE (TIPageSimplePresence, ti_page_simple_presence, TI_TYPE_PAGE);

enum {
    TI_COLUMN_SIMPLE_PRESENCE_LIST_NAME = 0,
    TI_COLUMN_SIMPLE_PRESENCE_LIST_HANDLE,
    TI_COLUMN_SIMPLE_PRESENCE_LIST_STATUS_TYPE,
    TI_COLUMN_SIMPLE_PRESENCE_LIST_STATUS,
    TI_COLUMN_SIMPLE_PRESENCE_LIST_STATUS_MESSAGE,
    TI_COLUMN_SIMPLE_PRESENCE_LIST_N_COLUMNS
};

/* Function prototypes */
void _ti_page_simple_presence_setup_page (TIPage *page, GladeXML *glade_xml);
void _ti_page_simple_presence_restart_page (TIPage *page);
static void _ti_page_simple_presence_refresh_valid_statuses (TIPageSimplePresence *self);
static void _ti_page_simple_presence_build_valid_statuses_treeview (
    TIPageSimplePresence *self, GladeXML *glade_xml);
static void _ti_page_simple_presence_build_presence_treeview (TIPageSimplePresence *self,
    GladeXML *glade_xml);
static void _add_valid_status_to_list (gpointer key, gpointer value,
    gpointer user_data);
static gchar *_get_presence_status_type_string (guint presence_status_type);

static void _ti_page_simple_presence_presence_update (TpConnection *connection,
    GHashTable *presences, gpointer unused, GObject *object);
static void _ti_page_simple_presence_remove_presence (TIPageSimplePresence *page_simple_presence,
    guint contact_handle);
static void _ti_page_simple_presence_add_presence (
    TIPageSimplePresence *page_simple_presence, guint contact_handle,
    guint status_type, const gchar *status, const gchar *message);

static void _ti_page_simple_presence_get_presence (TIPageSimplePresence *self);
static void _ti_page_simple_presence_clear_presence_list (TIPageSimplePresence *self);
static void _ti_page_simple_presence_set_presence (TIPageSimplePresence *self);
static void _ti_page_simple_presence_handle_display_mode_changed (
    TIPageSimplePresence *self, guint handle_display_mode);

struct _TIPageSimplePresencePrivate {
    TIPreferences *preferences;
    TIHandleMapper *handle_mapper;

    TpConnection *connection;

    TIDlgUserPresence *dlg_user_presence;
    TIDlgHandlesList *dlg_handles_list;

    GtkWindow *parent_wnd;

    GtkListStore *valid_statuses_list;

    GtkTreeView *presence_treeview;
    GtkTreeStore *presence_tree;
};
typedef struct _TIPageSimplePresencePrivate TIPageSimplePresencePrivate;

#define TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE(object) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_PAGE_SIMPLE_PRESENCE, \
                                  TIPageSimplePresencePrivate))

static void
ti_page_simple_presence_dispose (GObject *object)
{
  TIPageSimplePresence *self = TI_PAGE_SIMPLE_PRESENCE (object);
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);

  if (priv->preferences != NULL)
    {
      g_signal_handlers_disconnect_by_func (priv->preferences,
          G_CALLBACK (_ti_page_simple_presence_handle_display_mode_changed), self);
      g_object_unref (priv->preferences);
      priv->preferences = NULL;
    }

  TI_OBJ_UNREF (priv->handle_mapper);
  TI_OBJ_UNREF (priv->connection);
  TI_OBJ_UNREF (priv->dlg_user_presence);
  TI_OBJ_UNREF (priv->dlg_handles_list);

  G_OBJECT_CLASS (ti_page_simple_presence_parent_class)->dispose (object);
}


static void
ti_page_simple_presence_constructed (GObject *object)
{
  void (*chain_up) (GObject *) =
    G_OBJECT_CLASS (ti_page_simple_presence_parent_class)->constructed;
  TIPageSimplePresence *self = TI_PAGE_SIMPLE_PRESENCE (object);

  if (chain_up != NULL)
    chain_up (object);

  (void) self;
}


static void
ti_page_simple_presence_class_init (TIPageSimplePresenceClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  TIPageClass *page_class = TI_PAGE_CLASS (klass);

  gobject_class->dispose = ti_page_simple_presence_dispose;
  gobject_class->constructed = ti_page_simple_presence_constructed;

  page_class->setup_page = _ti_page_simple_presence_setup_page;
  page_class->restart_page = _ti_page_simple_presence_restart_page;

  g_type_class_add_private (klass, sizeof (TIPageSimplePresencePrivate));
}

static void
ti_page_simple_presence_init (TIPageSimplePresence *self)
{
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);

  priv->preferences = ti_preferences_new ();
  priv->handle_mapper = NULL;
  priv->connection = NULL;
  priv->dlg_user_presence = NULL;
  priv->dlg_handles_list = NULL;
}

/**
 * Returns a new instance.
 */
TIPageSimplePresence *
ti_page_simple_presence_new (GtkWindow *parent_wnd,
                      GtkNotebook *parent_notebook,
                      TpConnection *connection,
                      TIHandleMapper *handle_mapper)
{
  TIPage *page;
  TIPageSimplePresencePrivate *priv;

  g_assert (parent_wnd != NULL);

  page = g_object_new (TI_TYPE_PAGE_SIMPLE_PRESENCE, NULL);

  priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (page);
  priv->parent_wnd = parent_wnd;

  priv->connection = connection;
  g_object_ref (connection);

  priv->handle_mapper = handle_mapper;
  g_object_ref (handle_mapper);

  priv->dlg_handles_list = ti_dlg_handles_list_new (parent_wnd, handle_mapper,
      "Get Presence");
  g_assert (priv->dlg_handles_list != NULL);

  g_signal_connect_swapped (priv->preferences, "handle-display-mode-changed",
      G_CALLBACK (_ti_page_simple_presence_handle_display_mode_changed),
      page);

  _ti_page_new (&page, parent_notebook, "page-simple-presence.xml");

  return (TIPageSimplePresence *) page;
}

void
_ti_page_simple_presence_setup_page (TIPage *page,
                              GladeXML *glade_xml)
{
  TIPageSimplePresence *page_simple_presence = TI_PAGE_SIMPLE_PRESENCE (page);
  TIPageSimplePresencePrivate *priv =
      TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (page_simple_presence);
  GtkWidget *widget = NULL;

  /* "Refresh" button in "Valid statuses" frame. */
  widget = glade_xml_get_widget (glade_xml, "button_refresh_valid_statuses");
  g_assert (GTK_IS_BUTTON (widget));
  g_signal_connect_swapped (widget, "clicked",
      G_CALLBACK (_ti_page_simple_presence_refresh_valid_statuses),
      page_simple_presence);

  /* Valid statuses list */
  priv->valid_statuses_list = gtk_list_store_new (4,
      G_TYPE_STRING, G_TYPE_STRING, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN);
  _ti_page_simple_presence_build_valid_statuses_treeview (page_simple_presence,
    glade_xml);

  /* "Set Presence" dialog */
  priv->dlg_user_presence = ti_dlg_user_presence_new (priv->parent_wnd,
      GTK_TREE_MODEL (priv->valid_statuses_list), 0);
  g_assert (priv->dlg_user_presence != NULL);

  /* Presence Tree */
  priv->presence_tree = gtk_tree_store_new (5,
      G_TYPE_STRING /* name */ ,
      G_TYPE_STRING /* handle */ ,
      G_TYPE_STRING /* status type */ ,
      G_TYPE_STRING /* status */ ,
      G_TYPE_STRING /* status message */ );
  _ti_page_simple_presence_build_presence_treeview (page_simple_presence,
      glade_xml);

  tp_cli_connection_interface_simple_presence_connect_to_presences_changed
      (priv->connection, _ti_page_simple_presence_presence_update, NULL, NULL,
      (GObject *) page_simple_presence, NULL);

  /* "Get Presence" button */
  widget = glade_xml_get_widget (glade_xml, "button_get_presence");
  g_assert (GTK_IS_BUTTON (widget));
  g_signal_connect_swapped (widget, "clicked",
      G_CALLBACK (_ti_page_simple_presence_get_presence), page_simple_presence);

  /* "Clear Presence List" button */
  widget = glade_xml_get_widget (glade_xml, "button_presence_clear");
  g_assert (GTK_IS_BUTTON (widget));
  g_signal_connect_swapped (widget, "clicked",
      G_CALLBACK (_ti_page_simple_presence_clear_presence_list), page_simple_presence);

  /* "Set Presence" button */
  widget = glade_xml_get_widget (glade_xml, "button_set_presence");
  g_assert (GTK_IS_BUTTON (widget));
  g_signal_connect_swapped (widget, "clicked",
      G_CALLBACK (_ti_page_simple_presence_set_presence),
      page_simple_presence);
}

void
_ti_page_simple_presence_restart_page (TIPage *page)
{
  TIPageSimplePresence *self = TI_PAGE_SIMPLE_PRESENCE (page);

  _ti_page_simple_presence_refresh_valid_statuses (self);
}

static void
got_statuses (TpProxy *proxy,
              const GValue *statuses,
              const GError *error,
              gpointer user_data,
              GObject *object)
{
  TIPageSimplePresence *self = TI_PAGE_SIMPLE_PRESENCE (object);
  GHashTable *specs;

  if (error != NULL)
    {
      g_printerr ("Getting Statuses property failed: %s\n", error->message);
      return;
    }

  specs = g_value_get_boxed(statuses);

  g_hash_table_foreach (specs, _add_valid_status_to_list, self);
}

static void
_ti_page_simple_presence_refresh_valid_statuses (TIPageSimplePresence *self)
{
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);

  gtk_list_store_clear (priv->valid_statuses_list);

  tp_cli_dbus_properties_call_get (
      priv->connection, -1,
      TP_IFACE_CONNECTION_INTERFACE_SIMPLE_PRESENCE, "Statuses",
      got_statuses, NULL, NULL, (GObject *) self);
}

static void
_ti_page_simple_presence_build_valid_statuses_treeview (
    TIPageSimplePresence *self,
    GladeXML *glade_xml)
{
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);
  GtkWidget *treeview;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  treeview = glade_xml_get_widget (glade_xml, "treeview_valid_statuses");
  gtk_tree_view_set_model (GTK_TREE_VIEW (treeview),
      GTK_TREE_MODEL (priv->valid_statuses_list));

  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes ("Name", renderer,
      "text", 0,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

  column = gtk_tree_view_column_new_with_attributes ("Type", renderer,
      "text", 1,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

  column = gtk_tree_view_column_new_with_attributes ("Self", renderer,
      "text", 2,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

  column = gtk_tree_view_column_new_with_attributes ("Message", renderer,
      "text", 3,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
}

static void
_ti_page_simple_presence_build_presence_treeview (TIPageSimplePresence *self,
                                           GladeXML *glade_xml)
{
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  priv->presence_treeview = GTK_TREE_VIEW (glade_xml_get_widget (glade_xml,
        "treeview_presence"));
  g_assert (priv->presence_treeview != NULL);

  gtk_tree_view_set_model (priv->presence_treeview,
      GTK_TREE_MODEL (priv->presence_tree));

  renderer = gtk_cell_renderer_text_new ();

  if (ti_preferences_get_handle_display_mode (priv->preferences) ==
      TI_PREFERENCES_HANDLE_DISPLAY_HANDLE)
    {
      column = gtk_tree_view_column_new_with_attributes ("Handle", renderer,
          "text", TI_COLUMN_SIMPLE_PRESENCE_LIST_HANDLE,
          NULL);
      gtk_tree_view_append_column (priv->presence_treeview, column);
    }
  else
    {
      column = gtk_tree_view_column_new_with_attributes ("Name", renderer,
          "text", TI_COLUMN_SIMPLE_PRESENCE_LIST_NAME,
          NULL);
      gtk_tree_view_append_column (priv->presence_treeview, column);
    }

  column = gtk_tree_view_column_new_with_attributes ("Status type", renderer,
      "text", TI_COLUMN_SIMPLE_PRESENCE_LIST_STATUS_TYPE,
      NULL);
  gtk_tree_view_append_column (priv->presence_treeview, column);

  column = gtk_tree_view_column_new_with_attributes ("Status", renderer,
      "text", TI_COLUMN_SIMPLE_PRESENCE_LIST_STATUS,
      NULL);
  gtk_tree_view_append_column (priv->presence_treeview, column);

  column = gtk_tree_view_column_new_with_attributes ("Status Message", renderer,
      "text", TI_COLUMN_SIMPLE_PRESENCE_LIST_STATUS_MESSAGE,
      NULL);
  gtk_tree_view_append_column (priv->presence_treeview, column);
}

/**
 * Add Valid Status to List - Helper Function, called by
 * _ti_page_simple_presence_refresh_valid_statuses()
 */
static void
_add_valid_status_to_list (gpointer key,
                           gpointer value,
                           gpointer user_data)
{
  TIPageSimplePresence *self = TI_PAGE_SIMPLE_PRESENCE (user_data);
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);
  const gchar *status_name = key;
  GValueArray *status_info = value;
  GtkTreeIter iter;
  gchar *type_str = NULL;
  gboolean can_set_on_self;
  gboolean has_message;
  GValue *gvalue = NULL;

  gvalue = g_value_array_get_nth (status_info, 0);
  type_str = _get_presence_status_type_string (g_value_get_uint (gvalue));

  gvalue = g_value_array_get_nth (status_info, 1);
  can_set_on_self = g_value_get_boolean (gvalue);

  gvalue = g_value_array_get_nth (status_info, 2);
  has_message = g_value_get_boolean (gvalue);

  gtk_list_store_append (priv->valid_statuses_list, &iter);
  gtk_list_store_set (priv->valid_statuses_list, &iter,
      0, status_name,
      1, type_str,
      2, can_set_on_self,
      3, has_message,
      -1);

  g_free (type_str);
}

/**
 * Get Presence Status Type String - Helper Function
 *
 * @return Presence status type string representation. Must be freed after use.
 */
static gchar *
_get_presence_status_type_string (guint presence_status_type)
{
  switch (presence_status_type)
    {
    case TP_CONNECTION_PRESENCE_TYPE_OFFLINE:
      return g_strdup_printf ("%u - Offline", presence_status_type);

    case TP_CONNECTION_PRESENCE_TYPE_AVAILABLE:
      return g_strdup_printf ("%u - Available", presence_status_type);

    case TP_CONNECTION_PRESENCE_TYPE_AWAY:
      return g_strdup_printf ("%u - Away", presence_status_type);

    case TP_CONNECTION_PRESENCE_TYPE_EXTENDED_AWAY:
      return g_strdup_printf ("%u - Ext. Away", presence_status_type);

    case TP_CONNECTION_PRESENCE_TYPE_HIDDEN:
      return g_strdup_printf ("%u - Hidden", presence_status_type);

    case TP_CONNECTION_PRESENCE_TYPE_UNKNOWN:
      return g_strdup_printf ("%u - Unknown", presence_status_type);

    case TP_CONNECTION_PRESENCE_TYPE_ERROR:
      return g_strdup_printf ("%u - Error", presence_status_type);

    default:
      return g_strdup_printf ("%u - Invalid", presence_status_type);
    }
}

static void
_ti_page_simple_presence_presence_update (TpConnection *connection,
                                   GHashTable *presences,
                                   gpointer unused,
                                   GObject *object)
{
  TIPageSimplePresence *self = TI_PAGE_SIMPLE_PRESENCE (object);
  GArray *presence_update_array;
  guint i;
  TIHashEntry hash_entry;
  guint contact_handle;
  GValueArray *contact_presence;
  GValue *gvalue;
  guint status_type;
  const gchar *status;
  const gchar *message;

  presence_update_array = ti_hash_table_to_array (presences);

  for (i = 0; i < presence_update_array->len; i++)
    {
      hash_entry = g_array_index (presence_update_array, TIHashEntry, i);

      contact_handle = GPOINTER_TO_UINT (hash_entry.key);
      contact_presence = (GValueArray *) (hash_entry.value);

      g_assert (contact_presence->n_values == 3);

      /* First, remove current presence status for that contact (if any). */
      _ti_page_simple_presence_remove_presence (self, contact_handle);

      gvalue = g_value_array_get_nth (contact_presence, 0);
      status_type = g_value_get_uint (gvalue);

      gvalue = g_value_array_get_nth (contact_presence, 1);
      status = g_value_get_string (gvalue);

      gvalue = g_value_array_get_nth (contact_presence, 2);
      message = g_value_get_string (gvalue);

      _ti_page_simple_presence_add_presence (self, contact_handle, status_type,
          status, message);
    }

  g_array_free (presence_update_array, TRUE);
}

static void
_ti_page_simple_presence_remove_presence (TIPageSimplePresence *self,
                                   guint contact_handle)
{
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);
  gboolean keep_going;
  GtkTreeIter iter;
  gchar *curr_handle_str = NULL;
  gchar *contact_handle_str = NULL;

  keep_going = gtk_tree_model_get_iter_first (
      GTK_TREE_MODEL (priv->presence_tree), &iter);

  if (!keep_going)
    return; /* The tree is probably empty. */

  contact_handle_str = g_strdup_printf ("%u", contact_handle);

  while (keep_going)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (priv->presence_tree), &iter,
          TI_COLUMN_SIMPLE_PRESENCE_LIST_HANDLE, &curr_handle_str,
          -1);

      if (g_str_equal (curr_handle_str, contact_handle_str))
        {
          gtk_tree_store_remove (priv->presence_tree, &iter);
          keep_going = FALSE;
        }

      g_free (curr_handle_str);

      if (keep_going)
        keep_going = gtk_tree_model_iter_next (
            GTK_TREE_MODEL (priv->presence_tree), &iter);
    }

  g_free (contact_handle_str);
}

static void
_ti_page_simple_presence_add_presence (TIPageSimplePresence *self,
                                guint contact_handle,
                                guint status_type,
                                const gchar *status,
                                const gchar *message)
{
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);
  GtkTreeIter top_iter;
  gchar *contact_handle_str = NULL;
  gchar *name;

  name = ti_handle_mapper_get_contact_handle_name (priv->handle_mapper,
      contact_handle);
  if (name == NULL)
    {
      /* Fallback to its handle number */
      name = g_strdup_printf ("%u", contact_handle);
    }

  contact_handle_str = g_strdup_printf ("%u", contact_handle);

  gtk_tree_store_append (priv->presence_tree, &top_iter, NULL);
  gtk_tree_store_set (priv->presence_tree, &top_iter,
      TI_COLUMN_SIMPLE_PRESENCE_LIST_NAME, name,
      TI_COLUMN_SIMPLE_PRESENCE_LIST_HANDLE, contact_handle_str,
      TI_COLUMN_SIMPLE_PRESENCE_LIST_STATUS_TYPE,
        _get_presence_status_type_string (status_type),
      TI_COLUMN_SIMPLE_PRESENCE_LIST_STATUS, status,
      TI_COLUMN_SIMPLE_PRESENCE_LIST_STATUS_MESSAGE, message,
      -1);

  g_free (name);
}

static void
_ti_page_simple_presence_get_presence (TIPageSimplePresence *self)
{
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);
  GArray *handles;
  GError *error = NULL;
  GHashTable *presences = NULL;

  if (!ti_dlg_handles_list_run (priv->dlg_handles_list, &handles))
    return;

  if (tp_cli_connection_interface_simple_presence_run_get_presences
        (priv->connection, -1, handles, &presences, &error, NULL))
    {
      _ti_page_simple_presence_presence_update (priv->connection, presences,
          NULL, G_OBJECT(self));
    }
  else
    {
      g_printerr ("GetPresence(): %s\n", error->message);
      g_error_free (error);
    }

  g_hash_table_destroy (presences);
  g_array_free (handles, TRUE);
}

static void
_ti_page_simple_presence_clear_presence_list (TIPageSimplePresence *self)
{
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);

  gtk_tree_store_clear (priv->presence_tree);
}

static void
_ti_page_simple_presence_set_presence (TIPageSimplePresence *self)
{
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);
  GError *error = NULL;
  gchar *status = NULL;
  gchar *message = NULL;

  ti_dlg_user_presence_run (priv->dlg_user_presence, &status, &message);

  if (!tp_cli_connection_interface_simple_presence_run_set_presence
        (priv->connection, -1, status, message, &error, NULL))
    {
      g_printerr ("SetPresence(): %s\n", error->message);
      g_error_free (error);
    }

  /* Clean up */
  g_free (status);
  g_free (message);
}

static void
_ti_page_simple_presence_handle_display_mode_changed (
  TIPageSimplePresence *self, guint handle_display_mode)
{
  TIPageSimplePresencePrivate *priv = TI_PAGE_SIMPLE_PRESENCE_GET_PRIVATE (self);
  GtkTreeViewColumn *contact_handle_column;
  GtkCellRenderer *renderer;
  GList *renderers_list = NULL;

  contact_handle_column = gtk_tree_view_get_column (priv->presence_treeview,
      0);

  renderers_list = gtk_tree_view_column_get_cell_renderers (
      contact_handle_column);
  g_assert (g_list_length (renderers_list) == 1);

  renderer = GTK_CELL_RENDERER (renderers_list->data);

  if (handle_display_mode == TI_PREFERENCES_HANDLE_DISPLAY_HANDLE)
    {
      gtk_tree_view_column_set_title (contact_handle_column, "Handle");

      gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
          "text", TI_COLUMN_SIMPLE_PRESENCE_LIST_HANDLE,
          NULL);
    }
  else
    {
      g_assert (handle_display_mode == TI_PREFERENCES_HANDLE_DISPLAY_NAME);

      gtk_tree_view_column_set_title (contact_handle_column, "Name");

      gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
          "text", TI_COLUMN_SIMPLE_PRESENCE_LIST_NAME,
          NULL);
    }

  g_list_free (renderers_list);
}
