/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-page-presence.c:
 * A GtkNotebook page exposing
 * org.freedesktop.Telepathy.Connection.Interface.Contacts functionality.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Originally by Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "page-contacts.h"

#include <glade/glade.h>
#include <telepathy-glib/enums.h>
#include <telepathy-glib/interfaces.h>

#include "constants.h"
#include "dlg-handles-list.h"
#include "page-priv.h"
#include "preferences.h"
#include "util.h"

struct _TIPageContactsClass {
    TIPageClass parent;
};

G_DEFINE_TYPE (TIPageContacts, ti_page_contacts, TI_TYPE_PAGE);

enum {
    TI_COLUMN_CONTACTS_LIST_NAME = 0,
    TI_COLUMN_CONTACTS_LIST_HANDLE,
    TI_COLUMN_CONTACTS_LIST_ATTRIBUTE,
    TI_COLUMN_CONTACTS_LIST_VALUE,
    TI_COLUMN_CONTACTS_LIST_N_COLUMNS
};

/* Function prototypes */
void _ti_page_contacts_setup_page (TIPage *page, GladeXML *glade_xml);
void _ti_page_contacts_restart_page (TIPage *page);
static void _ti_page_contacts_refresh_contact_attribute_interfaces (
   TIPageContacts *self);
static void _ti_page_contacts_build_contact_attribute_interfaces_treeview (
    TIPageContacts *self, GladeXML *glade_xml);
static void _ti_page_contacts_build_contact_attributes_treeview (
    TIPageContacts *self, GladeXML *glade_xml);

static void _ti_page_contacts_clear_contact_attributes_list (
  TIPageContacts *self);
static void _ti_page_contacts_handle_display_mode_changed (
    TIPageContacts *self, guint handle_display_mode);
static void _ti_page_contacts_remove_contact_attribute_data (
    TIPageContacts *self, guint contact_handle);
static void _ti_page_contacts_get_contact_attributes (TIPageContacts *self);

struct _TIPageContactsPrivate {
    TIPreferences *preferences;
    TIHandleMapper *handle_mapper;

    TpConnection *connection;

    TIDlgHandlesList *dlg_handles_list;

    GtkWindow *parent_wnd;

    GtkListStore *contact_attribute_interfaces;
    gchar **contact_attribute_interfaces_strv;

    GtkTreeView *contact_attributes_treeview;
    GtkTreeStore *contact_attributes_treestore;
};
typedef struct _TIPageContactsPrivate TIPageContactsPrivate;

#define TI_PAGE_CONTACTS_GET_PRIVATE(object) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_PAGE_CONTACTS, \
                                  TIPageContactsPrivate))

static void
ti_page_contacts_dispose (GObject *object)
{
  TIPageContacts *self = TI_PAGE_CONTACTS (object);
  TIPageContactsPrivate *priv = TI_PAGE_CONTACTS_GET_PRIVATE (self);

  if (priv->preferences != NULL)
    {
      g_signal_handlers_disconnect_by_func (priv->preferences,
          G_CALLBACK (_ti_page_contacts_handle_display_mode_changed), self);
      g_object_unref (priv->preferences);
      priv->preferences = NULL;
    }

  TI_OBJ_UNREF (priv->handle_mapper);
  TI_OBJ_UNREF (priv->connection);
  TI_OBJ_UNREF (priv->dlg_handles_list);

  G_OBJECT_CLASS (ti_page_contacts_parent_class)->dispose (object);
}


static void
ti_page_contacts_constructed (GObject *object)
{
  void (*chain_up) (GObject *) =
    G_OBJECT_CLASS (ti_page_contacts_parent_class)->constructed;
  TIPageContacts *self = TI_PAGE_CONTACTS (object);

  if (chain_up != NULL)
    chain_up (object);

  (void) self;
}


static void
ti_page_contacts_class_init (TIPageContactsClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  TIPageClass *page_class = TI_PAGE_CLASS (klass);

  gobject_class->dispose = ti_page_contacts_dispose;
  gobject_class->constructed = ti_page_contacts_constructed;

  page_class->setup_page = _ti_page_contacts_setup_page;
  page_class->restart_page = _ti_page_contacts_restart_page;

  g_type_class_add_private (klass, sizeof (TIPageContactsPrivate));
}

static void
ti_page_contacts_init (TIPageContacts *self)
{
  TIPageContactsPrivate *priv = TI_PAGE_CONTACTS_GET_PRIVATE (self);

  priv->preferences = ti_preferences_new ();
  priv->handle_mapper = NULL;
  priv->connection = NULL;
  priv->dlg_handles_list = NULL;
  priv->contact_attribute_interfaces_strv = NULL;
}

/**
 * Returns a new instance.
 */
TIPageContacts *
ti_page_contacts_new (GtkWindow *parent_wnd,
                      GtkNotebook *parent_notebook,
                      TpConnection *connection,
                      TIHandleMapper *handle_mapper)
{
  TIPage *page;
  TIPageContactsPrivate *priv;

  g_assert (parent_wnd != NULL);

  page = g_object_new (TI_TYPE_PAGE_CONTACTS, NULL);

  priv = TI_PAGE_CONTACTS_GET_PRIVATE (page);
  priv->parent_wnd = parent_wnd;

  priv->connection = connection;
  g_object_ref (connection);

  priv->handle_mapper = handle_mapper;
  g_object_ref (handle_mapper);

  priv->dlg_handles_list = ti_dlg_handles_list_new (parent_wnd, handle_mapper,
      "Get Presence");
  g_assert (priv->dlg_handles_list != NULL);

  g_signal_connect_swapped (priv->preferences, "handle-display-mode-changed",
      G_CALLBACK (_ti_page_contacts_handle_display_mode_changed),
      page);

  _ti_page_new (&page, parent_notebook, "page-contacts.xml");

  return (TIPageContacts *) page;
}

void
_ti_page_contacts_setup_page (TIPage *page,
                              GladeXML *glade_xml)
{
  TIPageContacts *page_contacts = TI_PAGE_CONTACTS (page);
  TIPageContactsPrivate *priv =
      TI_PAGE_CONTACTS_GET_PRIVATE (page_contacts);
  GtkWidget *widget = NULL;

  /* "Refresh" button in "Valid statuses" frame. */
  widget = glade_xml_get_widget (glade_xml,
    "button_refresh_contact_attribute_interfaces");
  g_assert (GTK_IS_BUTTON (widget));
  g_signal_connect_swapped (widget, "clicked",
      G_CALLBACK (_ti_page_contacts_refresh_contact_attribute_interfaces),
      page_contacts);

  /* Inspectable interfaces list */
  priv->contact_attribute_interfaces = gtk_list_store_new (1, G_TYPE_STRING);
  _ti_page_contacts_build_contact_attribute_interfaces_treeview (
      page_contacts, glade_xml);

  /* Inspect Tree */
  priv->contact_attributes_treestore = gtk_tree_store_new (4,
      G_TYPE_STRING /* name */ ,
      G_TYPE_STRING /* handle */ ,
      G_TYPE_STRING /* attribute name */ ,
      G_TYPE_STRING /* attribute value */ );
  _ti_page_contacts_build_contact_attributes_treeview (page_contacts,
      glade_xml);

  /* "inspect button */
  widget = glade_xml_get_widget (glade_xml, "button_get_contact_attributes");
  g_assert (GTK_IS_BUTTON (widget));
  g_signal_connect_swapped (widget, "clicked",
      G_CALLBACK (_ti_page_contacts_get_contact_attributes), page_contacts);

  /* "Clear inspect List" button */
  widget = glade_xml_get_widget (glade_xml, "button_contact_attributes_clear");
  g_assert (GTK_IS_BUTTON (widget));
  g_signal_connect_swapped (widget, "clicked",
      G_CALLBACK (_ti_page_contacts_clear_contact_attributes_list),
      page_contacts);
}

void
_ti_page_contacts_restart_page (TIPage *page)
{
  TIPageContacts *self = TI_PAGE_CONTACTS (page);

  _ti_page_contacts_refresh_contact_attribute_interfaces (self);
}

static void
got_contact_attribute_interfaces (TpProxy *proxy,
                                  const GValue *interfaces,
                                  const GError *error,
                                  gpointer user_data,
                                  GObject *object)
{
  TIPageContacts *self = TI_PAGE_CONTACTS (object);
  TIPageContactsPrivate *priv =
      TI_PAGE_CONTACTS_GET_PRIVATE (self);
  gchar **arr;
  guint i;

  if (error != NULL)
    {
      g_printerr ("Getting ContactAttributeInterfaces property failed: %s\n",
        error->message);
      return;
    }

  /* clear again to prevent races */
  gtk_list_store_clear (priv->contact_attribute_interfaces);
  arr = g_value_get_boxed(interfaces);

  for (i = 0; arr[i] != NULL; i++)
    {
      GtkTreeIter iter;

      gtk_list_store_append (priv->contact_attribute_interfaces, &iter);
      gtk_list_store_set (priv->contact_attribute_interfaces, &iter,
        0, arr[i], -1);
    }

  g_strfreev (priv->contact_attribute_interfaces_strv);
  priv->contact_attribute_interfaces_strv = g_strdupv (arr);
}

static void
_ti_page_contacts_refresh_contact_attribute_interfaces (TIPageContacts *self)
{
  TIPageContactsPrivate *priv = TI_PAGE_CONTACTS_GET_PRIVATE (self);

  gtk_list_store_clear (priv->contact_attribute_interfaces);

  tp_cli_dbus_properties_call_get (
      priv->connection, -1,
      TP_IFACE_CONNECTION_INTERFACE_CONTACTS, "ContactAttributeInterfaces",
      got_contact_attribute_interfaces, NULL, NULL, (GObject *) self);
}

static void
_ti_page_contacts_build_contact_attribute_interfaces_treeview (
    TIPageContacts *self,
    GladeXML *glade_xml)
{
  TIPageContactsPrivate *priv = TI_PAGE_CONTACTS_GET_PRIVATE (self);
  GtkWidget *treeview;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  treeview = glade_xml_get_widget (glade_xml,
    "treeview_contact_attribute_interfaces");
  gtk_tree_view_set_model (GTK_TREE_VIEW (treeview),
      GTK_TREE_MODEL (priv->contact_attribute_interfaces));

  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes ("Interface", renderer,
      "text", 0,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
}

static void
_ti_page_contacts_build_contact_attributes_treeview (TIPageContacts *self,
                                    GladeXML *glade_xml)
{
  TIPageContactsPrivate *priv = TI_PAGE_CONTACTS_GET_PRIVATE (self);
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  priv->contact_attributes_treeview = GTK_TREE_VIEW (
      glade_xml_get_widget (glade_xml, "treeview_contact_attributes"));
  g_assert (priv->contact_attributes_treeview != NULL);

  gtk_tree_view_set_model (priv->contact_attributes_treeview,
      GTK_TREE_MODEL (priv->contact_attributes_treestore));

  renderer = gtk_cell_renderer_text_new ();

  if (ti_preferences_get_handle_display_mode (priv->preferences) ==
      TI_PREFERENCES_HANDLE_DISPLAY_HANDLE)
    {
      column = gtk_tree_view_column_new_with_attributes ("Handle", renderer,
          "text", TI_COLUMN_CONTACTS_LIST_HANDLE,
          NULL);
      gtk_tree_view_append_column (priv->contact_attributes_treeview, column);
    }
  else
    {
      column = gtk_tree_view_column_new_with_attributes ("Name", renderer,
          "text", TI_COLUMN_CONTACTS_LIST_NAME,
          NULL);
      gtk_tree_view_append_column (priv->contact_attributes_treeview, column);
    }

  column = gtk_tree_view_column_new_with_attributes ("Attribute", renderer,
      "text", TI_COLUMN_CONTACTS_LIST_ATTRIBUTE,
      NULL);
  gtk_tree_view_append_column (priv->contact_attributes_treeview, column);

  column = gtk_tree_view_column_new_with_attributes ("Value", renderer,
      "text", TI_COLUMN_CONTACTS_LIST_VALUE,
      NULL);
  gtk_tree_view_append_column (priv->contact_attributes_treeview, column);
}

static void
_ti_page_contacts_get_contact_attributes_update (TpConnection *connection,
  GHashTable *contact_attributes, gpointer unused, GObject *object)
{
  TIPageContacts *self = TI_PAGE_CONTACTS (object);
  TIPageContactsPrivate *priv = TI_PAGE_CONTACTS_GET_PRIVATE (self);
  GHashTableIter iter;
  gpointer key;
  gpointer value;

  g_hash_table_iter_init (&iter, contact_attributes);

  while (g_hash_table_iter_next (&iter, &key, &value))
    {
      GtkTreeIter iter, child_iter;
      GHashTableIter attr_iter;
      GHashTable *contact_attribute_data;
      gpointer attr_key, attr_value;
      guint contact_handle;
      gchar *name, *contact_handle_str;

      contact_handle = GPOINTER_TO_UINT (key);
      contact_attribute_data = (GHashTable *) (value);

      /* First, remove current presence status for that contact (if any). */
      _ti_page_contacts_remove_contact_attribute_data (self, contact_handle);

      name = ti_handle_mapper_get_contact_handle_name (priv->handle_mapper,
          contact_handle);
      if (name == NULL)
        {
          /* Fallback to its handle number */
          name = g_strdup_printf ("%u", contact_handle);
        }

      contact_handle_str = g_strdup_printf ("%u", contact_handle);

      gtk_tree_store_append (priv->contact_attributes_treestore, &iter, NULL);
      gtk_tree_store_set (priv->contact_attributes_treestore, &iter,
        TI_COLUMN_CONTACTS_LIST_NAME, name,
        TI_COLUMN_CONTACTS_LIST_HANDLE, contact_handle_str,
        -1);

      g_free (name);
      g_free (contact_handle_str);

      g_hash_table_iter_init (&attr_iter, contact_attribute_data);
      while (g_hash_table_iter_next (&attr_iter, &attr_key, &attr_value))
        {
          gchar *str, *attr_name;
          GValue *val;

          attr_name = (gchar *)attr_key;
          val = (GValue *)attr_value;

          str = ti_value_to_string (val);

          gtk_tree_store_insert_before (priv->contact_attributes_treestore,
            &child_iter, &iter, NULL);
          gtk_tree_store_set (priv->contact_attributes_treestore, &child_iter,
            TI_COLUMN_CONTACTS_LIST_ATTRIBUTE, attr_name,
            TI_COLUMN_CONTACTS_LIST_VALUE, str,
            -1);

          g_free (str);

        }
    }
}

static void
_ti_page_contacts_remove_contact_attribute_data (TIPageContacts *self,
  guint contact_handle)
{
  TIPageContactsPrivate *priv = TI_PAGE_CONTACTS_GET_PRIVATE (self);
  gboolean keep_going;
  GtkTreeIter iter;
  gchar *curr_handle_str = NULL;
  gchar *contact_handle_str = NULL;

  keep_going = gtk_tree_model_get_iter_first (
      GTK_TREE_MODEL (priv->contact_attributes_treestore), &iter);

  if (!keep_going)
    return; /* The tree is probably empty. */

  contact_handle_str = g_strdup_printf ("%u", contact_handle);

  while (keep_going)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (priv->contact_attributes_treestore),
          &iter, TI_COLUMN_CONTACTS_LIST_HANDLE, &curr_handle_str, -1);

      if (g_str_equal (curr_handle_str, contact_handle_str))
        {
          gtk_tree_store_remove (priv->contact_attributes_treestore, &iter);
          keep_going = FALSE;
        }

      g_free (curr_handle_str);

      if (keep_going)
        keep_going = gtk_tree_model_iter_next (
            GTK_TREE_MODEL (priv->contact_attributes_treestore), &iter);
    }

  g_free (contact_handle_str);
}

static void
_ti_page_contacts_get_contact_attributes (TIPageContacts *self)
{
  TIPageContactsPrivate *priv = TI_PAGE_CONTACTS_GET_PRIVATE (self);
  GArray *handles;
  GError *error = NULL;
  GHashTable *result = NULL;

  if (!ti_dlg_handles_list_run (priv->dlg_handles_list, &handles))
    return;

  if (tp_cli_connection_interface_contacts_run_get_contact_attributes (
      priv->connection, -1, handles,
      (const gchar **)priv->contact_attribute_interfaces_strv,
      FALSE, &result, &error, NULL))
    {
      _ti_page_contacts_get_contact_attributes_update (priv->connection,
          result, NULL, G_OBJECT(self));
      g_hash_table_destroy (result);
    }
  else
    {
      g_printerr ("InspectContacts(): %s\n", error->message);
      g_error_free (error);
    }

  g_array_free (handles, TRUE);
}

static void
_ti_page_contacts_clear_contact_attributes_list (TIPageContacts *self)
{
  TIPageContactsPrivate *priv = TI_PAGE_CONTACTS_GET_PRIVATE (self);

  gtk_tree_store_clear (priv->contact_attributes_treestore);
}

static void
_ti_page_contacts_handle_display_mode_changed (
  TIPageContacts *self, guint handle_display_mode)
{
  TIPageContactsPrivate *priv = TI_PAGE_CONTACTS_GET_PRIVATE (self);
  GtkTreeViewColumn *contact_handle_column;
  GtkCellRenderer *renderer;
  GList *renderers_list = NULL;

  contact_handle_column = gtk_tree_view_get_column (
      priv->contact_attributes_treeview, 0);

  renderers_list = gtk_tree_view_column_get_cell_renderers (
      contact_handle_column);
  g_assert (g_list_length (renderers_list) == 1);

  renderer = GTK_CELL_RENDERER (renderers_list->data);

  if (handle_display_mode == TI_PREFERENCES_HANDLE_DISPLAY_HANDLE)
    {
      gtk_tree_view_column_set_title (contact_handle_column, "Handle");

      gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
          "text", TI_COLUMN_CONTACTS_LIST_HANDLE,
          NULL);
    }
  else
    {
      g_assert (handle_display_mode == TI_PREFERENCES_HANDLE_DISPLAY_NAME);

      gtk_tree_view_column_set_title (contact_handle_column, "Name");

      gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
          "text", TI_COLUMN_CONTACTS_LIST_NAME,
          NULL);
    }

  g_list_free (renderers_list);
}
