#include "config.h"
#include <string.h>
#include <panel-applet.h>

#include "image-loader.h"
#include "properties.h"

void
teatime_load_properties (Teatime *teatime)
{
  gint i;
  PanelApplet *applet = teatime->applet;
  GSList *time_list = NULL, *name_list = NULL;
  gint num_teas;
  
  teatime->properties.sound = NULL;

  teatime->properties.teas = panel_applet_gconf_get_list (applet,
                                                          TEATIME_PREFS_TEAS,
                                                          GCONF_VALUE_STRING,
                                                          NULL);
  teatime->properties.times = panel_applet_gconf_get_list (applet,
                                                           TEATIME_PREFS_TIMES,
                                                           GCONF_VALUE_INT,
                                                           NULL);
  teatime->properties.num_teas = MIN (g_slist_length (teatime->properties.teas),
                                      g_slist_length (teatime->properties.times));
  num_teas = teatime->properties.num_teas;
  
  teatime->properties.drawing_times = g_new (gint, num_teas);
  teatime->properties.tea_names =     g_new (gchar*, num_teas);
  
  name_list = teatime->properties.teas;
  time_list = teatime->properties.times;
  for (i=0; i<num_teas; i++)
  {
    teatime->properties.tea_names[i] = g_strdup (name_list->data);
    teatime->properties.drawing_times[i] = GPOINTER_TO_INT (time_list->data);
    name_list = name_list->next;
    time_list = time_list->next;
  }
  
  teatime->properties.last_tea =   panel_applet_gconf_get_int    (applet, TEATIME_PREFS_LAST_TEA, NULL);
  teatime->properties.cup =        panel_applet_gconf_get_int    (applet, TEATIME_PREFS_CUP, NULL);
  teatime->properties.popup =      panel_applet_gconf_get_bool   (applet, TEATIME_PREFS_POPUP, NULL);
  teatime->properties.play_sound = panel_applet_gconf_get_bool   (applet, TEATIME_PREFS_PLAY_SOUND, NULL);
  teatime->properties.sound =      panel_applet_gconf_get_string (applet, TEATIME_PREFS_SOUND, NULL );
  teatime->properties.show_clock = panel_applet_gconf_get_bool   (applet, TEATIME_PREFS_SHOW_CLOCK, NULL); 
}

static void
disable_sound_chooser (GtkToggleButton *button, GtkWidget *chooser)
{
  gtk_widget_set_sensitive (chooser, button->active);
}


static void
properties_cup_changed (GtkWidget *toggle_button,
                        Teatime *teatime)
{
  gint i;

  for (i=0; i<TEACUPS; i++)
  {
    if (GTK_TOGGLE_BUTTON (teatime->pref_cup_button[i])->active)
    {
      panel_applet_gconf_set_int (PANEL_APPLET (teatime->applet),
                                  TEATIME_PREFS_CUP,
                                  i,
                                  NULL);
      if (teatime->properties.cup != i)
      {
        teatime->properties.cup = i;
        teatime_image_loader_load_applet_images (teatime);
        if (GTK_WIDGET_REALIZED (teatime->area))
            teatime_applet_redraw (teatime);
      }
      break;
    }
  }
}

static void
properties_sound_changed (GtkWidget *widget,
                          Teatime *teatime)
{
  if (teatime->properties.sound)
      g_free (teatime->properties.sound);
  teatime->properties.sound = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (widget));
  
  panel_applet_gconf_set_string (PANEL_APPLET (teatime->applet),
                                 TEATIME_PREFS_SOUND,
                                 teatime->properties.sound,
                                 NULL);
}

static void
properties_alarm_changed (GtkWidget *widget,
                          Teatime *teatime)
{
  GtkWidget *popup;
  GtkWidget *play_sound;

  popup = gtk_object_get_data (GTK_OBJECT (teatime->pref_dialog), TEATIME_PREFS_POPUP);
  play_sound = gtk_object_get_data (GTK_OBJECT (teatime->pref_dialog), TEATIME_PREFS_PLAY_SOUND);

  if (teatime->properties.popup != GTK_TOGGLE_BUTTON (popup)->active)
  {
    panel_applet_gconf_set_bool (PANEL_APPLET (teatime->applet),
                                 TEATIME_PREFS_POPUP,
                                 GTK_TOGGLE_BUTTON (popup)->active,
                                 NULL);
    teatime->properties.popup = GTK_TOGGLE_BUTTON (popup)->active;
  }

  if (teatime->properties.play_sound != GTK_TOGGLE_BUTTON (play_sound)->active)
  {  
    panel_applet_gconf_set_bool (PANEL_APPLET (teatime->applet),
                                 TEATIME_PREFS_PLAY_SOUND,
                                 GTK_TOGGLE_BUTTON (play_sound)->active,
                                 NULL);
    teatime->properties.play_sound = GTK_TOGGLE_BUTTON (play_sound)->active;
  }
}

static void
properties_clock_changed (GtkWidget *widget,
			  Teatime *teatime)
{
  GtkWidget *clock;

  clock = gtk_object_get_data (GTK_OBJECT (teatime->pref_dialog), TEATIME_PREFS_SHOW_CLOCK);

  if (teatime->properties.show_clock != GTK_TOGGLE_BUTTON (clock)->active)
  {
    panel_applet_gconf_set_bool (PANEL_APPLET (teatime->applet),
                                 TEATIME_PREFS_SHOW_CLOCK,
                                 GTK_TOGGLE_BUTTON (clock)->active,
                                 NULL);
    teatime->properties.show_clock = GTK_TOGGLE_BUTTON (clock)->active;
  }
}

static void
properties_drawingtime_changed (GtkWidget *widget,
				Teatime *teatime)
{
  gint i, v;
  GSList *list = NULL;
  gint tea_nr = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (widget), "tea_nr"));
  TimeProperties *prop = g_object_get_data (G_OBJECT (widget), "prop");

  for (i=0; i<teatime->properties.num_teas; i++)
  {
    if (i == tea_nr)
    {
      GtkAdjustment *adj = gtk_range_get_adjustment (GTK_RANGE (widget));
      gdouble val = gtk_adjustment_get_value (adj);
      gchar *text = NULL;
      gint s = 0, m = 0, h = 0;

      if (val < 20)
      {
        s = (gint)(val * 3);
      }
      else if (val < 74)
      {
        val = (val - 20) * 10 + 60;
        m = val / 60;
        s = ((gint)(val / 10)) * 10 % 60; 
      }
      else
      {
        val = (val - 74) * 60 + 600;
        m = (gint)(val / 60);
        if (m >= 60)
        {
          h = (gint)(m / 60);
          m = m % 60;
        }
      }
      if (h)
        text = g_strdup_printf (_("%i std %-2.2i min"), h, m);
      else if (m)
        text = g_strdup_printf (_("%i min %-2.2i sec"), m, s);
      else
        text = g_strdup_printf (_("%i sec"), s);
      gtk_label_set_text (GTK_LABEL (prop->time_label[i]), text);
      g_free (text);

      v = h * 3600 + m * 60 + s;

      teatime->properties.drawing_times[i] = v;
      teatime_set_menu (teatime);
    }
    else
      v = teatime->properties.drawing_times[i];
    list = g_slist_append (list, GINT_TO_POINTER(v));
  }
  panel_applet_gconf_set_list (teatime->applet,
                               TEATIME_PREFS_TIMES,
                               GCONF_VALUE_INT,
                               list,
                               NULL);
}

static gdouble
adjustment_from_drawingtime (gint time)
{
  gdouble adj;
  
  if (time < 60)
    adj = time / 3;
  else if (time < 600)
    adj = (time - 60) / 10 + 20;
  else
    adj = (time - 600) / 60 + 74;
  return adj;
}

static void
update_gconf_tea_and_time_list (Teatime *teatime)
{
  GSList *time_list = NULL, *name_list = NULL;
  gint i;
  
  for (i=0; i<teatime->properties.num_teas; i++)
  { 
    time_list = g_slist_append (time_list, GINT_TO_POINTER(teatime->properties.drawing_times[i]));
    name_list = g_slist_append (name_list, teatime->properties.tea_names[i]);
  }
  panel_applet_gconf_set_list (teatime->applet,
                               TEATIME_PREFS_TIMES,
                               GCONF_VALUE_INT,
                               time_list,
                               NULL);
  panel_applet_gconf_set_list (teatime->applet,
                               TEATIME_PREFS_TEAS,
                               GCONF_VALUE_STRING,
                               name_list,
                               NULL);

}

static void
remove_tea (GtkWidget *widget,
            Teatime *teatime)
{
  TimeProperties *prop = g_object_get_data (G_OBJECT (widget), "prop");
  gint tea = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (widget), "remove_nr"));
  gint i;

  /* sanity check */ 
  if (tea >= teatime->properties.num_teas)
      return;
  /* that should not be possible */
  if (teatime->properties.num_teas == 1)
      return;

  gtk_widget_destroy (prop->time_table[tea]);
 
  /* move all data one line up */
  if (tea < teatime->properties.num_teas - 1)
  {
    for (i = tea; i < teatime->properties.num_teas - 1; i++)
    {
      prop->time_table[i] =    prop->time_table[i+1];
      prop->tea_label[i] =     prop->tea_label[i+1];
      prop->time_label[i] =    prop->time_label[i+1];
      prop->time_slider[i] =   prop->time_slider[i+1];
      prop->remove_button[i] = prop->remove_button[i+1];
      
      g_object_set_data (G_OBJECT (prop->remove_button[i]), "remove_nr", GINT_TO_POINTER (i));
      g_object_set_data (G_OBJECT (prop->time_slider[i]), "tea_nr", GINT_TO_POINTER (i));
      
      teatime->properties.drawing_times[i] = teatime->properties.drawing_times[i+1];
      teatime->properties.tea_names[i] = teatime->properties.tea_names[i+1];
    }
  }
  
  teatime->properties.num_teas--;

  /* it should not be possible to remove all teas */
  if (teatime->properties.num_teas == 1)
      gtk_widget_set_sensitive (prop->remove_button[0], FALSE);
  
  update_gconf_tea_and_time_list (teatime);
  teatime_set_menu (teatime);
}

static void
create_time_block (Teatime *teatime,
                   TimeProperties *prop,
                   gint pos)
{
  gint time;
  gdouble time_adj;
  gchar *val_label;
  gchar *label;
  GtkWidget *image;
  gchar *tip_text;
  
  prop->time_table[pos] = gtk_table_new (2, 3, FALSE);
  gtk_box_pack_start (GTK_BOX (prop->time_vbox), prop->time_table[pos], FALSE, FALSE, 4);
  
  label = g_strdup_printf (_("%s drawing time"), _(teatime->properties.tea_names[pos]));
  prop->tea_label[pos] = gtk_label_new (label);
  g_free (label);
  gtk_table_attach_defaults (GTK_TABLE (prop->time_table[pos]),
                             prop->tea_label[pos],
                             0, 1,
                             0, 1);

  time_adj = adjustment_from_drawingtime (teatime->properties.drawing_times[pos]);
  
  prop->time_label[pos] = gtk_label_new ("");
  gtk_table_attach (GTK_TABLE (prop->time_table[pos]),
                    prop->time_label[pos],
                    1, 2,
                    0, 1,
                    GTK_EXPAND | GTK_FILL,
                    GTK_SHRINK,
                    0, 0); 
  
  prop->time_adj[pos] = (GtkAdjustment *)gtk_adjustment_new (time_adj, 3.334, 304.9999, 0.1, 3.3333, 0.1);
  prop->time_slider[pos] = gtk_hscale_new (prop->time_adj[pos]);
  gtk_scale_set_draw_value (GTK_SCALE (prop->time_slider[pos]), FALSE);
  gtk_range_set_update_policy (GTK_RANGE (prop->time_slider[pos]), GTK_UPDATE_CONTINUOUS);
  
  gtk_table_attach (GTK_TABLE (prop->time_table[pos]),
                    prop->time_slider[pos],
                    0, 2,
                    1, 2,
                    GTK_EXPAND | GTK_FILL,
                    GTK_SHRINK,
                    0, 0);

  g_object_set_data (G_OBJECT (prop->time_slider[pos]), "tea_nr", GINT_TO_POINTER (pos));
  g_object_set_data (G_OBJECT (prop->time_slider[pos]), "prop", prop);
  g_signal_connect (G_OBJECT (prop->time_slider[pos]), "value_changed",
                    G_CALLBACK (properties_drawingtime_changed), teatime);
  properties_drawingtime_changed (prop->time_slider[pos], teatime);
  
  image = gtk_image_new_from_stock (GTK_STOCK_DELETE,
                                    GTK_ICON_SIZE_BUTTON);
  prop->remove_button[pos] = gtk_button_new ();
  
  tip_text = g_strconcat (_("Remove "), _(teatime->properties.tea_names[pos]), NULL);
  gtk_tooltips_set_tip (prop->tips,
                        prop->remove_button[pos],
                        tip_text,
                        NULL);
  g_free (tip_text);
  
  gtk_container_add (GTK_CONTAINER (prop->remove_button[pos]),
                     image);
  gtk_table_attach (GTK_TABLE (prop->time_table[pos]),
                    prop->remove_button[pos],
                    2, 3,
                    0, 2,
                    GTK_SHRINK, GTK_SHRINK, 0, 0);
  g_object_set_data (G_OBJECT (prop->remove_button[pos]), "remove_nr", GINT_TO_POINTER (pos));
  g_object_set_data (G_OBJECT (prop->remove_button[pos]), "prop", prop);
  g_signal_connect(G_OBJECT (prop->remove_button[pos]), "clicked",
                   G_CALLBACK (remove_tea), teatime);
}     

static void
add_tea (GtkWidget *entry,
         Teatime *teatime)
{
  TimeProperties *prop = g_object_get_data (G_OBJECT (entry), "prop");
  const gchar *text = gtk_entry_get_text (GTK_ENTRY (entry));
  gint num_teas;
 
  if (teatime->properties.num_teas == 1)
      gtk_widget_set_sensitive (prop->remove_button[0], TRUE);
  
  teatime->properties.num_teas++;
  num_teas = teatime->properties.num_teas;
  
  prop->time_table =    g_renew (GtkWidget *, prop->time_table, num_teas);
  prop->tea_label =     g_renew (GtkWidget *, prop->tea_label, num_teas);
  prop->time_label =    g_renew (GtkWidget *, prop->time_label, num_teas);
  prop->time_slider =   g_renew (GtkWidget *, prop->time_slider, num_teas);
  prop->remove_button = g_renew (GtkWidget *, prop->remove_button, num_teas);
  prop->time_adj =      g_renew (GtkAdjustment *, prop->time_adj, num_teas);
  
  teatime->properties.tea_names =     g_renew (gchar *, teatime->properties.tea_names, num_teas);
  teatime->properties.drawing_times = g_renew (gint, teatime->properties.drawing_times, num_teas);

  teatime->properties.tea_names[num_teas - 1] = g_strdup (text);
  teatime->properties.drawing_times[num_teas - 1] = 10;

  update_gconf_tea_and_time_list (teatime);
  
  create_time_block (teatime, prop, num_teas - 1);
  gtk_widget_show_all (prop->time_table[num_teas - 1]);
  gtk_entry_set_text (GTK_ENTRY (entry), _("New tea"));
}

static void
add_tea2 (GtkWidget *button,
          Teatime *teatime)
{
  GtkWidget *entry = g_object_get_data (G_OBJECT (button), "entry");
  add_tea (entry, teatime);
}

void
teatime_display_properties_dialog(BonoboUIComponent *uic,
                          Teatime *teatime,
                          const gchar *verbname)
{
  PanelApplet *applet;

  GtkWidget *notebook;
  GtkWidget *tab_label[3];
  gint i;

  TimeProperties *prop;
  gint num_teas;
  GtkWidget *time_vbox;
  GtkWidget *time_reset;
  GtkWidget *time_outer_vbox;
  GtkWidget *time_add_tea_button;
  GtkWidget *time_add_tea_entry;
  GtkWidget *time_add_tea_label;
  GtkWidget *time_add_tea_hbox;
  
  GtkWidget *cup_vbox;
  GtkWidget *cup_hbox;
  GtkWidget *cup_label;
  GSList *group = NULL;
  
  GtkWidget *alarm_box;
  GtkWidget *alarm_popup_button;
  GtkWidget *alarm_play_sound_button;
  GtkWidget *alarm_sound_chooser;
  GValue alarm_val = {0,};
  
  GtkWidget *show_clock_button;

  if (teatime->pref_dialog)
  {
    gtk_widget_show (teatime->pref_dialog);
    gdk_window_raise (teatime->pref_dialog->window);
    return;
  }

  
  prop = g_new (TimeProperties, 1);
  prop->tips = gtk_tooltips_new ();
  num_teas = teatime->properties.num_teas;
  prop->drawing_times = g_new (gint, num_teas);
  prop->time_table =    g_new (GtkWidget *, num_teas);
  prop->tea_label =     g_new (GtkWidget *, num_teas);
  prop->time_label =    g_new (GtkWidget *, num_teas);
  prop->time_slider =   g_new (GtkWidget *, num_teas);
  prop->remove_button = g_new (GtkWidget *, num_teas);
  prop->time_adj =      g_new (GtkAdjustment *, num_teas);
  
  applet = PANEL_APPLET (teatime->applet);
    
  teatime->pref_dialog = gtk_dialog_new_with_buttons (_("Teatime Properties"),
                                             NULL,
                                             0,
                                             GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                                             NULL);
  
  gtk_window_set_icon_from_file (GTK_WINDOW (teatime->pref_dialog),
                                 teatime->icon_path, 
                                 NULL);
    
  tab_label[0] = gtk_label_new (_("drawing time"));
  tab_label[1] = gtk_label_new (_("tea cup"));
  tab_label[2] = gtk_label_new (_("alarm"));

  notebook = gtk_notebook_new ();
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (teatime->pref_dialog)->vbox), notebook, FALSE, FALSE, 6);

  /* drawing times page */
  time_outer_vbox = gtk_vbox_new (FALSE, 0);
  prop->time_vbox = gtk_vbox_new (FALSE, 6);
  
  gtk_box_pack_start (GTK_BOX (time_outer_vbox), prop->time_vbox, TRUE, FALSE, 0);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook),
                            time_outer_vbox,
                            tab_label[0]); 

  for (i=0; i<num_teas; i++)
  {
    create_time_block (teatime, prop, i);
  }
  time_add_tea_hbox = gtk_hbox_new (FALSE, 0);
  time_add_tea_button = gtk_button_new_with_label (_("Add tea"));
  time_add_tea_entry = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (time_add_tea_entry), _("New tea"));
  
  g_object_set_data (G_OBJECT (time_add_tea_button), "entry", time_add_tea_entry);
  g_object_set_data (G_OBJECT (time_add_tea_entry), "prop", prop);
  
  g_signal_connect (G_OBJECT (time_add_tea_entry), "activate",
                    G_CALLBACK (add_tea), teatime);
  g_signal_connect (G_OBJECT (time_add_tea_button), "clicked",
                    G_CALLBACK (add_tea2), teatime);
  
  gtk_box_pack_start (GTK_BOX (time_add_tea_hbox), time_add_tea_button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (time_add_tea_hbox), time_add_tea_entry, TRUE, TRUE, 0);
  
  gtk_box_pack_start (GTK_BOX (time_outer_vbox), time_add_tea_hbox, FALSE, FALSE, 6);


  
  /* cup selection page */

  cup_vbox = gtk_vbox_new (FALSE, 6);
  cup_hbox = gtk_hbox_new (TRUE, 2);
  cup_label = gtk_label_new (_("Chose your teacup"));
  
  gtk_box_pack_start (GTK_BOX (cup_vbox), cup_label, FALSE, FALSE, 2);
  gtk_box_pack_start (GTK_BOX (cup_vbox), cup_hbox, FALSE, FALSE, 2);

  for (i=0; i<TEACUPS; i++)
  {
    gchar *string;
      
    teatime->pref_cup[i] = gtk_image_new_from_pixbuf (teatime->properties.teacups[i]);
    teatime->pref_cup_button[i] = gtk_radio_button_new(group);
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (teatime->pref_cup_button[i]));
    gtk_container_add (GTK_CONTAINER (teatime->pref_cup_button[i]),
                       teatime->pref_cup[i]);
    gtk_container_set_border_width (GTK_CONTAINER ( teatime->pref_cup_button[i]), 0);
    gtk_box_pack_start (GTK_BOX (cup_hbox), teatime->pref_cup_button[i],
                        FALSE, FALSE, 2);
    g_signal_connect (G_OBJECT (teatime->pref_cup_button[i]), "toggled",
                      G_CALLBACK (properties_cup_changed), teatime);
  }

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
                                teatime->pref_cup_button[teatime->properties.cup]), TRUE);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook),
                            cup_vbox,
                            tab_label[1]);
  
  /* actions page */
    
  alarm_box = gtk_vbox_new (FALSE, 0);
  alarm_popup_button = gtk_check_button_new_with_label (_("Open window if tea is ready"));
  alarm_play_sound_button = gtk_check_button_new_with_label (_("Play sound if tea is ready"));

  alarm_sound_chooser = gtk_file_chooser_button_new (_("Sound to play"), GTK_FILE_CHOOSER_ACTION_OPEN);

  show_clock_button = gtk_check_button_new_with_label (_("Show clock while drawing"));

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (alarm_popup_button),
                                teatime->properties.popup);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (alarm_play_sound_button),
                                teatime->properties.play_sound);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (show_clock_button),
                                teatime->properties.show_clock); 

  if (!teatime->properties.play_sound)
      disable_sound_chooser (GTK_TOGGLE_BUTTON (alarm_play_sound_button), alarm_sound_chooser);
  if (teatime->properties.sound && strlen(teatime->properties.sound))
      gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (alarm_sound_chooser), teatime->properties.sound);

  g_object_set_data (G_OBJECT (teatime->pref_dialog), TEATIME_PREFS_POPUP, alarm_popup_button);
  g_object_set_data (G_OBJECT (teatime->pref_dialog), TEATIME_PREFS_PLAY_SOUND, alarm_play_sound_button);
  g_object_set_data (G_OBJECT (teatime->pref_dialog), TEATIME_PREFS_SOUND_CHOOSER, alarm_sound_chooser);
  g_object_set_data (G_OBJECT (teatime->pref_dialog), TEATIME_PREFS_SHOW_CLOCK, show_clock_button);
  
  gtk_box_pack_start (GTK_BOX (alarm_box), alarm_popup_button, FALSE, FALSE, 6);
  gtk_box_pack_start (GTK_BOX (alarm_box), alarm_play_sound_button, FALSE, FALSE, 6);
  gtk_box_pack_start (GTK_BOX (alarm_box), alarm_sound_chooser, FALSE, FALSE, 2);
  gtk_box_pack_start (GTK_BOX (alarm_box), show_clock_button, FALSE, FALSE, 12);
    
  g_signal_connect (G_OBJECT (alarm_play_sound_button), "toggled",
                    G_CALLBACK (disable_sound_chooser), alarm_sound_chooser);
  g_signal_connect (G_OBJECT (alarm_play_sound_button), "toggled",
                    G_CALLBACK (properties_alarm_changed), teatime);
  g_signal_connect (G_OBJECT (alarm_sound_chooser), "selection-changed",
                    G_CALLBACK (properties_sound_changed), teatime);
  
  g_signal_connect (G_OBJECT (alarm_popup_button), "toggled",
                    G_CALLBACK (properties_alarm_changed), teatime);
  
  g_signal_connect (G_OBJECT (show_clock_button), "toggled",
                    G_CALLBACK (properties_clock_changed), teatime);
  
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook),
                            alarm_box,
                            tab_label[2]);
  
  g_signal_connect (G_OBJECT (teatime->pref_dialog), "destroy",
                    G_CALLBACK (gtk_widget_destroyed), &teatime->pref_dialog);
		  
  gtk_widget_set_usize (GTK_WIDGET (teatime->pref_dialog), 600, -1);
  gtk_widget_show_all (teatime->pref_dialog);
  gtk_dialog_run (GTK_DIALOG (teatime->pref_dialog));
  gtk_widget_destroy (teatime->pref_dialog);
}

