[comment {-*- tcl -*- doctools manpage}]
[manpage_begin math::optimize n 0.2]
[copyright {2004 Arjen Markus <arjenmarkus@users.sourceforge.net>}]
[copyright {2004 Kevn B. Kenny <kennykb@users.sourceforge.net>}]
[moddesc   {Math}]
[titledesc {Optimisation routines}]
[require Tcl 8.2]
[require math::optimize [opt 0.2]]
[description]
[para]
This package implements several optimisation algorithms:

[list_begin bullet]
[bullet]
Minimize or maximize a function over a given interval

[bullet]
Solve a linear program (maximize a linear function subject to linear
constraints)

[list_end]

[para]
The package is fully implemented in Tcl. No particular attention has
been paid to the accuracy of the calculations. Instead, the
algorithms have been used in a straightforward manner.
[para]
This document describes the procedures and explains their usage.
[para]
[emph Note:] The linear programming algorithm is described but not yet
operational.

[section "PROCEDURES"]
[para]
This package defines the following public procedures:
[list_begin definitions]

[call [cmd ::math::optimize::minimize] [arg begin] [arg end] [arg func] [arg maxerr]]
Minimize the given (continuous) function by examining the values in the
given interval. The procedure determines the values at both ends and in the
centre of the interval and then constructs a new interval of 1/2 length
that includes the minimum. No guarantee is made that the [emph global]
minimum is found.
[nl]
The procedure returns the "x" value for which the function is minimal.
[nl]
[emph {This procedure has been deprecated - use min_bound_1d instead}]
[nl]
[arg begin] - Start of the interval
[nl]
[arg end] - End of the interval
[nl]
[arg func] - Name of the function to be minimized (a procedure taking
one argument).
[nl]
[arg maxerr] - Maximum relative error (defaults to 1.0e-4)

[call [cmd ::math::optimize::maximize] [arg begin] [arg end] [arg func] [arg maxerr]]
Maximize the given (continuous) function by examining the values in the
given interval. The procedure determines the values at both ends and in the
centre of the interval and then constructs a new interval of 1/2 length
that includes the maximum. No guarantee is made that the [emph global]
maximum is found.
[nl]
The procedure returns the "x" value for which the function is maximal.
[nl]
[emph {This procedure has been deprecated - use max_bound_1d instead}]
[nl]
[arg begin] - Start of the interval
[nl]
[arg end] - End of the interval
[nl]
[arg func] - Name of the function to be maximized (a procedure taking
one argument).
[nl]
[arg maxerr] - Maximum relative error (defaults to 1.0e-4)

[call [cmd ::math::optimize::min_bound_1d] [arg func] [arg begin] [arg end] [opt "[option -relerror] [arg reltol]"] [opt "[option -abserror] [arg abstol]"] [opt "[option -maxiter] [arg maxiter]"] [opt "[option -trace] [arg traceflag]"]]

Miminizes a function of one variable in the given interval.  The procedure
uses Brent's method of parabolic interpolation, protected by golden-section
subdivisions if the interpolation is not converging.  No guarantee is made
that a [emph global] minimum is found.  The function to evaluate, [arg func],
must be a single Tcl command; it will be evaluated with an abscissa appended
as the last argument.
[nl]
[arg x1] and [arg x2] are the two bounds of
the interval in which the minimum is to be found.  They need not be in
increasing order.
[nl]
[arg reltol], if specified, is the desired upper bound
on the relative error of the result; default is 1.0e-7.  The given value
should never be smaller than the square root of the machine's floating point
precision, or else convergence is not guaranteed.  [arg abstol], if specified,
is the desired upper bound on the absolute error of the result; default
is 1.0e-10.  Caution must be used with small values of [arg abstol] to
avoid overflow/underflow conditions; if the minimum is expected to lie
about a small but non-zero abscissa, you consider either shifting the
function or changing its length scale.
[nl]
[arg maxiter] may be used to constrain the number of function evaluations
to be performed; default is 100.  If the command evaluates the function
more than [arg maxiter] times, it returns an error to the caller.
[nl]
[arg traceFlag] is a Boolean value. If true, it causes the command to
print a message on the standard output giving the abscissa and ordinate
at each function evaluation, together with an indication of what type
of interpolation was chosen.  Default is 0 (no trace).

[call [cmd ::math::optimize::min_unbound_1d] [arg func] [arg begin] [arg end] [opt "[option -relerror] [arg reltol]"] [opt "[option -abserror] [arg abstol]"] [opt "[option -maxiter] [arg maxiter]"] [opt "[option -trace] [arg traceflag]"]]

Miminizes a function of one variable over the entire real number line.
The procedure uses parabolic extrapolation combined with golden-section
dilatation to search for a region where a minimum exists, followed by
Brent's method of parabolic interpolation, protected by golden-section
subdivisions if the interpolation is not converging.  No guarantee is made
that a [emph global] minimum is found.  The function to evaluate, [arg func],
must be a single Tcl command; it will be evaluated with an abscissa appended
as the last argument.
[nl]
[arg x1] and [arg x2] are two initial guesses at where the minimum
may lie.  [arg x1] is the starting point for the minimization, and
the difference between [arg x2] and [arg x1] is used as a hint at the
characteristic length scale of the problem.
[nl]
[arg reltol], if specified, is the desired upper bound
on the relative error of the result; default is 1.0e-7.  The given value
should never be smaller than the square root of the machine's floating point
precision, or else convergence is not guaranteed.  [arg abstol], if specified,
is the desired upper bound on the absolute error of the result; default
is 1.0e-10.  Caution must be used with small values of [arg abstol] to
avoid overflow/underflow conditions; if the minimum is expected to lie
about a small but non-zero abscissa, you consider either shifting the
function or changing its length scale.
[nl]
[arg maxiter] may be used to constrain the number of function evaluations
to be performed; default is 100.  If the command evaluates the function
more than [arg maxiter] times, it returns an error to the caller.
[nl]
[arg traceFlag] is a Boolean value. If true, it causes the command to
print a message on the standard output giving the abscissa and ordinate
at each function evaluation, together with an indication of what type
of interpolation was chosen.  Default is 0 (no trace).

[call [cmd ::math::optimize::solveLinearProgram] [arg constraints] [arg objective]]
Solve a [emph "linear program"] in standard form using a straightforward
implementation of the Simplex algorithm. (In the explanation below: The
linear program has N constraints and M variables).
[nl]
The procedure returns a list of M values, the values for which the
objective function is maximal or a single keyword if the linear program
is not feasible or unbounded (either "unfeasible" or "unbounded")
[nl]
[arg constraints] - Matrix of coefficients plus maximum values that
implement the linear constraints. It is expected to be a list of N lists
of M+1 numbers each, M coefficients and the maximum value.
[nl]
[arg objective] - The M coefficients of the objective function
[list_end]

[section NOTES]
[para]
Several of the above procedures take the [emph names] of procedures as
arguments. To avoid problems with the [emph visibility] of these
procedures, the fully-qualified name of these procedures is determined
inside the optimize routines. For the user this has only one
consequence: the named procedure must be visible in the calling
procedure. For instance:
[example {
    namespace eval ::mySpace {
       namespace export calcfunc
       proc calcfunc { x } { return $x }
    }
    #
    # Use a fully-qualified name
    #
    namespace eval ::myCalc {
       puts [min_bound_1d ::myCalc::calcfunc $begin $end]
    }
    #
    # Import the name
    #
    namespace eval ::myCalc {
       namespace import ::mySpace::calcfunc
       puts [min_bound_1d calcfunc $begin $end]
    }
}]

The simple procedures [emph minimum] and [emph maximum] have been
deprecated: the alternatives are much more flexible, robust and
require less function evaluations.

[section EXAMPLES]
[para]
Let us take a few simple examples:
[para]
Determine the maximum of f(x) = x^3 exp(-3x), on the interval (0,10):
[example {
proc efunc { x } { expr {$x*$x*$x * exp(-3.0*$x)} }
puts "Maximum at: [::math::optimize::max_bound_1d efunc 0.0 10.0]"
}]
[para]
The maximum allowed error determines the number of steps taken (with
each step in the iteration the interval is reduced with a factor 1/2).
Hence, a maximum error of 0.0001 is achieved in approximately 14 steps.
[para]
An example of a [emph "linear program"] is:
[para]
Optimise the expression 3x+2y, where:
[example {
   x >= 0 and y >= 0 (implicit constraints, part of the
                     definition of linear programs)

   x + y   <= 1      (constraints specific to the problem)
   2x + 5y <= 10
}]
[para]
This problem can be solved as follows:
[example {

   set solution [::math::optimize::solveLinearProgram \
      { { 1.0   1.0   1.0 }
        { 2.0   5.0  10.0 } } \
        { 3.0   2.0 }]
}]
[para]
Note, that a constraint like:
[example {
   x + y >= 1
}]
can be turned into standard form using:
[example {
   -x  -y <= -1
}]

[para]
The theory of linear programming is the subject of many a text book and
the Simplex algorithm that is implemented here is the best-known
method to solve this type of problems, but it is not the only one.

[keywords math optimization minimum maximum "linear program"]

[manpage_end]
