'''
:copyright: Copyright 2006-2009 by Oliver Schoenborn, all rights reserved.
:license: BSD, see LICENSE.txt for details.
'''

from weakref import ref as WeakRef

from pubsub import setupv1
from pubsub import pub

pub.setupForTesting()

#
# Code for a simple command-line test
#
def testParam():
    def testFunc00(): pass
    def testFunc21(a,b,c=1): pass
    def testFuncA(*args): pass
    def testFuncAK(*args,**kwds): pass
    def testFuncK(**kwds): pass
    
    class Foo:
        def testMeth(self,a,b): pass
        def __call__(self, a): pass
    class Foo2:
        def __call__(self, *args): pass
    
    assert pub.paramMinCount(testFunc00)==(0,0)
    assert pub.paramMinCount(testFunc21)==(2,1)
    assert pub.paramMinCount(testFuncA) ==(1,0)
    assert pub.paramMinCount(testFuncAK)==(1,0)
    assert pub.paramMinCount(testFuncK) ==(0,0)
    foo = Foo()
    assert pub.paramMinCount(Foo.testMeth)==(2,0)
    assert pub.paramMinCount(foo.testMeth)==(2,0)
    assert pub.paramMinCount(foo)==(1,0)
    assert pub.paramMinCount(Foo2())==(1,0)

#------------------------

global deathNotified
deathNotified = 0
def preNotifyNode(self, dead):
    global deathNotified
    deathNotified += 1
    print 'testPreNotifyNODE heard notification of', `dead`
pub.setDeadCallback(preNotifyNode)

def testTreeNode():

    class WS:
        def __init__(self, s):
            self.s = s
        def __call__(self, msg):
            print 'WS#', self.s, ' received msg ', msg
        def __str__(self):
            return self.s
        
    def testPreNotifyRoot(dead):
        print 'testPreNotifyROOT heard notification of', `dead`

    node = pub.TopicTreeNode((pub.ALL_TOPICS,), WeakRef(testPreNotifyRoot))
    boo, baz, bid = WS('boo'), WS('baz'), WS('bid')
    node.addCallable(boo)
    node.addCallable(baz)
    node.addCallable(boo)
    assert node.getCallables() == [boo,baz]
    assert node.hasCallable(boo)
    
    node.removeCallable(bid) # no-op
    assert node.hasCallable(baz)
    assert node.getCallables() == [boo,baz]
    
    node.removeCallable(boo)
    assert node.getCallables() == [baz]
    assert node.hasCallable(baz)
    assert not node.hasCallable(boo)
    
    node.removeCallable(baz)
    assert node.getCallables() == []
    assert not node.hasCallable(baz)

    node2 = node.createSubtopic('st1', ('st1',))
    node3 = node.createSubtopic('st2', ('st2',))
    cb1, cb2, cb = WS('st1_cb1'), WS('st1_cb2'), WS('st2_cb')
    node2.addCallable(cb1)
    node2.addCallable(cb2)
    node3.addCallable(cb)
    node2.createSubtopic('st3', ('st1','st3'))
    node2.createSubtopic('st4', ('st1','st4'))
   
    print str(node)
    assert str(node) == ' (st1: st1_cb1 st1_cb2  (st4: ) (st3: )) (st2: st2_cb )'

    # verify send message, and that a dead listener does not get sent one
    delivered = node2.sendMessage('hello')
    assert delivered == 2
    del cb1
    delivered = node2.sendMessage('hello')
    assert delivered == 1
    assert deathNotified == 1
    
#------------------------

def testValidate():
    class Foo:
        def __call__(self, a):   pass
        def fun(self, b):        pass
        def fun2(self, b=1):     pass
        def fun3(self, a, b=2):  pass
        def badFun(self):        pass
        def badFun2():           pass
        def badFun3(self, a, b): pass
        
    server = pub.Publisher()
    foo = Foo()
    server.validate(foo)
    server.validate(foo.fun)
    server.validate(foo.fun2)
    server.validate(foo.fun3)
    assert not server.isValid(foo.badFun)
    assert not server.isValid(foo.badFun2)
    assert not server.isValid(foo.badFun3)
    
#------------------------

class SimpleListener:
    def __init__(self, number):
        self.number = number
    def __call__(self, message = ''): 
        print 'Callable #%s got the message "%s"' %(self.number, message)
    def notify(self, message):
        print '%s.notify() got the message "%s"' %(self.number, message)
    def __str__(self):
        return "SimpleListener_%s" % self.number

def testSubscribe():
    publisher = pub.Publisher()
    
    topic1 = 'politics'
    topic2 = ('history','middle age')
    topic3 = ('politics','UN')
    topic4 = ('politics','NATO')
    topic5 = ('politics','NATO','US')
    
    lisnr1 = SimpleListener(1)
    lisnr2 = SimpleListener(2)
    def func(message, a=1): 
        print 'Func received message "%s"' % message
    lisnr3 = func
    lisnr4 = lambda x: 'Lambda received message "%s"' % x

    assert not publisher.isSubscribed(lisnr1)
    assert not publisher.isSubscribed(lisnr2)
    assert not publisher.isSubscribed(lisnr3)
    assert not publisher.isSubscribed(lisnr4)
    
    publisher.subscribe(lisnr1, topic1)
    assert publisher.getAssociatedTopics(lisnr1) == [(topic1,)]
    publisher.subscribe(lisnr1, topic2)
    publisher.subscribe(lisnr1, topic1) # do it again, should be no-op
    assert publisher.getAssociatedTopics(lisnr1) == [(topic1,),topic2]
    publisher.subscribe(lisnr2.notify, topic3)
    assert publisher.getAssociatedTopics(lisnr2.notify) == [topic3]
    assert publisher.getAssociatedTopics(lisnr1) == [(topic1,),topic2]
    publisher.subscribe(lisnr3, topic5)
    assert publisher.getAssociatedTopics(lisnr3) == [topic5]
    assert publisher.getAssociatedTopics(lisnr2.notify) == [topic3]
    assert publisher.getAssociatedTopics(lisnr1) == [(topic1,),topic2]
    publisher.subscribe(lisnr4)
    
    print "Publisher tree: ", publisher
    assert publisher.isSubscribed(lisnr1)
    assert publisher.isSubscribed(lisnr1, topic1)
    assert publisher.isSubscribed(lisnr1, topic2)
    assert publisher.isSubscribed(lisnr2.notify)
    assert publisher.isSubscribed(lisnr3, topic5)
    assert publisher.isSubscribed(lisnr4, pub.ALL_TOPICS)
    expectTopicTree = 'all: <lambda>  (politics: SimpleListener_1  (UN: SimpleListener_2.notify ) (NATO:  (US: func ))) (history:  (middle age: SimpleListener_1 ))'
    print "Publisher tree: ", publisher
    assert str(publisher) == expectTopicTree
    
    publisher.unsubscribe(lisnr1, 'booboo') # should do nothing
    assert publisher.getAssociatedTopics(lisnr1) == [(topic1,),topic2]
    assert publisher.getAssociatedTopics(lisnr2.notify) == [topic3]
    assert publisher.getAssociatedTopics(lisnr3) == [topic5]
    publisher.unsubscribe(lisnr1, topic1)
    assert publisher.getAssociatedTopics(lisnr1) == [topic2]
    assert publisher.getAssociatedTopics(lisnr2.notify) == [topic3]
    assert publisher.getAssociatedTopics(lisnr3) == [topic5]
    publisher.unsubscribe(lisnr1, topic2)
    publisher.unsubscribe(lisnr1, topic2)
    publisher.unsubscribe(lisnr2.notify, topic3)
    publisher.unsubscribe(lisnr3, topic5)
    assert publisher.getAssociatedTopics(lisnr1) == []
    assert publisher.getAssociatedTopics(lisnr2.notify) == []
    assert publisher.getAssociatedTopics(lisnr3) == []
    publisher.unsubscribe(lisnr4)
    
    expectTopicTree = 'all:  (politics:  (UN: ) (NATO:  (US: ))) (history:  (middle age: ))'
    print "Publisher tree: ", publisher
    assert str(publisher) == expectTopicTree
    assert publisher.getDeliveryCount() == 0
    assert publisher.getMessageCount() == 0
    
    publisher.unsubAll()
    assert str(publisher) == 'all: '
    
#------------------------
    
def testUnsubAll():
    publisher = pub.Publisher()
    
    topic1 = 'politics'
    topic2 = ('history','middle age')
    topic3 = ('politics','UN')
    topic4 = ('politics','NATO')
    topic5 = ('politics','NATO','US')
    
    lisnr1 = SimpleListener(1)
    lisnr2 = SimpleListener(2)
    def func(message, a=1): 
        print 'Func received message "%s"' % message
    lisnr3 = func
    lisnr4 = lambda x: 'Lambda received message "%s"' % x

    publisher.subscribe(lisnr1, topic1)
    publisher.subscribe(lisnr1, topic2)
    publisher.subscribe(lisnr2.notify, topic3)
    publisher.subscribe(lisnr3, topic2)
    publisher.subscribe(lisnr3, topic5)
    publisher.subscribe(lisnr4)
    
    expectTopicTree = 'all: <lambda>  (politics: SimpleListener_1  (UN: SimpleListener_2.notify ) (NATO:  (US: func ))) (history:  (middle age: SimpleListener_1 func ))'
    print "Publisher tree: ", publisher
    assert str(publisher) == expectTopicTree

    publisher.unsubAll(topic1)
    assert publisher.getAssociatedTopics(lisnr1) == [topic2]
    assert not publisher.isSubscribed(lisnr1, topic1)
    
    publisher.unsubAll(topic2)
    print publisher
    assert publisher.getAssociatedTopics(lisnr1) == []
    assert publisher.getAssociatedTopics(lisnr3) == [topic5]
    assert not publisher.isSubscribed(lisnr1)
    assert publisher.isSubscribed(lisnr3, topic5)
    
    #print "Publisher tree: ", publisher
    expectTopicTree = 'all: <lambda>  (politics:  (UN: SimpleListener_2.notify ) (NATO:  (US: func ))) (history:  (middle age: ))'
    assert str(publisher) == expectTopicTree
    publisher.unsubAll(pub.ALL_TOPICS)
    #print "Publisher tree: ", publisher
    expectTopicTree = 'all:  (politics:  (UN: SimpleListener_2.notify ) (NATO:  (US: func ))) (history:  (middle age: ))'
    assert str(publisher) == expectTopicTree
    
    publisher.unsubAll()

#------------------------

def testSend():
    publisher = pub.Publisher()
    called = []
    
    class TestListener:
        def __init__(self, num):
            self.number = num
        def __call__(self, b): 
            called.append( 'TL%scb' % self.number )
        def notify(self, b):
            called.append( 'TL%sm' % self.number )
    def funcListener(b):
        called.append('func')
        
    lisnr1 = TestListener(1)
    lisnr2 = TestListener(2)
    lisnr3 = funcListener
    lisnr4 = lambda x: called.append('lambda')

    topic1 = 'politics'
    topic2 = 'history'
    topic3 = ('politics','UN')
    topic4 = ('politics','NATO','US')
    topic5 = ('politics','NATO')
    
    publisher.subscribe(lisnr1, topic1)
    publisher.subscribe(lisnr2, topic2)
    publisher.subscribe(lisnr2.notify, topic2)
    publisher.subscribe(lisnr3, topic4)
    publisher.subscribe(lisnr4)
    
    print publisher
    
    # setup ok, now test send/receipt
    publisher.sendMessage(topic1)
    assert called == ['lambda','TL1cb']
    called = []
    publisher.sendMessage(topic2)
    assert called == ['lambda','TL2cb','TL2m']
    called = []
    publisher.sendMessage(topic3)
    assert called == ['lambda','TL1cb']
    called = []
    publisher.sendMessage(topic4)
    assert called == ['lambda','TL1cb','func']
    called = []
    publisher.sendMessage(topic5)
    assert called == ['lambda','TL1cb']
    assert publisher.getDeliveryCount() == 12
    assert publisher.getMessageCount() == 5

    # test weak referencing works:
    global deathNotified
    deathNotified = 0
    del lisnr2
    called = []
    publisher.sendMessage(topic2)
    assert called == ['lambda']
    assert deathNotified == 2
        
# testSend()
#assert deathNotified == 5
    
def testDead():
    # verify if weak references work as expected
    print '------ Starting testDead ----------'
    node = pub.TopicTreeNode('t1', None)
    lisnr1 = SimpleListener(1)
    lisnr2 = SimpleListener(2)
    lisnr3 = SimpleListener(3)
    lisnr4 = SimpleListener(4)

    node.addCallable(lisnr1)
    node.addCallable(lisnr2)
    node.addCallable(lisnr3)
    node.addCallable(lisnr4)
    
    print 'Deleting listeners first'
    global deathNotified
    deathNotified = 0
    del lisnr1
    del lisnr2
    assert deathNotified == 2
    
    print 'Deleting node first'
    deathNotified = 0
    del node
    del lisnr3
    del lisnr4
    assert deathNotified == 0
    
    lisnr1 = SimpleListener(1)
    lisnr2 = SimpleListener(2)
    lisnr3 = SimpleListener(3)
    lisnr4 = SimpleListener(4)
    
    # try same with root of tree
    node = pub.TopicTreeRoot()
    node.addTopic(('',), lisnr1)
    node.addTopic(('',), lisnr2)
    node.addTopic(('',), lisnr3)
    node.addTopic(('',), lisnr4)
    # add objects that will die immediately to see if cleanup occurs
    # this must be done visually as it is a low-level detail
    deathNotified = 0
    pub.TopicTreeRoot.callbackDeadLimit = 3
    node.addTopic(('',), SimpleListener(5))
    node.addTopic(('',), SimpleListener(6))
    node.addTopic(('',), SimpleListener(7))
    print node.numListeners()
    assert node.numListeners() == (4, 3)
    node.addTopic(('',), SimpleListener(8))
    assert node.numListeners() == (4, 0)
    assert deathNotified == 4
    
    print 'Deleting listeners first'
    deathNotified = 0
    del lisnr1
    del lisnr2
    assert deathNotified == 2
    print 'Deleting node first'
    deathNotified = 0
    del node
    del lisnr3
    del lisnr4
    assert deathNotified == 0
    
