'''
Measure performance of pubsub, so that impact of proposed performance enhancing
algorithms can be proven. Measure with 

    python c:\python24\lib\timeit.py -n1 "import perf; perf.pubsubVer=N; perf.runTest()"
    
with N=version of pubsub ie 1, 2 or 3.

Results as of 3.1.0: 7.9 s (pubsub kwargs protocol)

Caution about comparing between API version 1 and 3: this test uses
listeners that do NO processing whatsoever, whereas in most
applications, the handling of messages within listeners will be more
compute intensive than the dispatching of the messages via either
version of pubsub. So the conclusion from the results, for instance that
"version 3 is xx times slower than version 1", is unlikely to be
correct: the numbers don't reflect the impact of pubsub on performance
of your application, just the individual "raw" pubsub performance, so
they can only be used to improve pubsub within one API version, not to
compare between API's.

In fact, it is very likely that you won't notice any performance difference
between using either versions of pubsub (but you don't know an API's
bottlenecks until you measure hence this test).

:copyright: Copyright 2006-2009 by Oliver Schoenborn, all rights reserved.
:license: BSD, see LICENSE.txt for details.

'''


testNum = 0 # keeps track of how many times the module is run


def runTest(testVersion=3):
    # ---------------- init from command line ----------

    import sys
    #if len(sys.argv) > 1:
    #    testVersion = int(sys.argv[1])

    numRuns = 10**5
    global testNum
    testNum += 1
    
    # --------- create arrays for random sampling -------

    # ss strings
    # nn numbers
    # ff functions
    # mm modules
    # cc classes
    expectArgNames = {
       't1': ('ss',), 
       't1.s2': ('ss', 'nn'), 
       't1.st2.st3': ('ss', 'nn', 'ff'), 
       't1.st2.st4': ('ss', 'nn', 'ff', 'mm'), 
       't1.st2.st3.st5': ('ss', 'nn', 'ff', 'mm', 'cc'),
       } 
    
    from math import sin, cos, asin, acos
    import sys, os, glob, platform
    
    class Foo: pass
    class Bar: pass
    class Baz: pass
    class Fim: pass

    # different types of arguments that will be passed to listeners
    argVals = dict(
       nn = (1, 2, 3, 4, None),
       ss = ('a', 'b', 'c', 'd', None),
       ff = (sin, cos, asin, acos, None),
       mm = (sys, os, glob, platform, None),
       cc = (Foo, Bar, Baz, Fim, None),
       )
    
    # ---------------- Adapter classes: addapt listeners and senders ----------

    class TestVersion1:

        def __init__(self):
            self.called = [False]*6

        def listener1(self, msg=None): self.called[1] = True
        def listener2(self, msg): self.called[2] = True
        def listener3(self, msg): self.called[3] = True
        def listener4(self, msg): self.called[4] = True
        def listener5(self, msg): self.called[5] = True
    
        def sendMsg(self, topic, args):
            pub.sendMessage(topic, args)
        
        def getArgNames(self, listeners):
            'No intro available for getting args, just return from expectArgNames'
            return dict( (topic, expectArgNames[topic]) for topic in listeners.keys() )
        
    class TestVersion2(TestVersion1):
        pass

    class TestVersion3:

        def __init__(self):
            self.called = [False]*6

        def listener1(self, ss): self.called[1] = True
        def listener2(self, ss, nn): self.called[2] = True
        def listener3(self, ss, nn, ff): self.called[3] = True
        def listener4(self, ss, nn, ff, mm): self.called[4] = True
        def listener5(self, ss, nn, ff, mm, cc): self.called[5] = True

        def sendMsg(self, topic, args):
            pub.sendMessage(topic, **args)

        def getArgNames(self, listeners):
            'Get arg names for topic, from list of listeners. Should be same as expectArgNames'
            def argKeys(topic):
                topicObj = pub.getTopic(topic)
                reqd, opt = topicObj.getArgs()
                assert topicObj.isSendable()
                return reqd + opt
            names = dict( (topic, argKeys(topic)) for topic in listeners.keys() )
            assert [set(expectArgNames[topic]) == set(argKeys(topic)) for topic in listeners.keys()] == [True]*5
            return names


    # ------------------------------------
    
    if testVersion == 1:
        from pubsub import setupv1
        from pubsub import pub
        tester = TestVersion1()
        if testNum == 1:
            print 'Will test pubsub version 1'
        
    elif testVersion == 2:
        from pubsub import setuparg1
        from pubsub import pub
        tester = TestVersion2()
        if testNum == 1:
            print 'Will test pubsub version 3, arg1 protocol'
        
    else:
        from pubsub import pub
        tester = TestVersion3()
        if testNum == 1:
            print 'Will test pubsub version 3, kwargs protocol'
        
    listeners = {
       't1': tester.listener1, 
       't1.s2': tester.listener2, 
       't1.st2.st3': tester.listener3, 
       't1.st2.st4': tester.listener4, 
       't1.st2.st3.st5': tester.listener5, 
       }
    
    for (topic, ll) in listeners.iteritems():
        pub.subscribe(ll, topic)
        
    topicArgNames = tester.getArgNames(listeners)
    topics = listeners.keys()
    
    
    # ---------------RUN the test! ------
    
    assert tester.called == [False]*6
    
    from random import randrange as randint
     
    def randVal(k):
        vals = argVals[k]
        return vals[ randint(0, len(vals)) ]
    
    for run in xrange(0, numRuns):
        #if run % 1000 == 0: print '.',
        topic = topics[ randint(0, len(topics)) ]
        args = dict( (argName, randVal(argName)) for argName in topicArgNames[topic] )
        tester.sendMsg(topic, args)
        
    assert tester.called == [False]+[True]*5
    print 'Test run %s completed' % testNum
    

if __name__ == '__main__':
    runTest()
    