/*
 *	tardy - a tar post-processor
 *	Copyright (C) 1998, 1999, 2003, 2004 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: interface definition for common/tar/output.cc
 */

#ifndef COMMON_TAR_OUTPUT_H
#define COMMON_TAR_OUTPUT_H

#include <tar/header.h>

/**
  * The tar_output class is used to represent an abstract output stream
  * containing a sequence of file headers and file data.
  *
  * The expected sequence of operations is
  *									<pre>
  *                 |
  *     +-----------------------+
  *     | write_archive_begin   |
  *     +-----------------------+
  *                 | 
  *                 | ,------------------<---------------.
  *                 |/                                    \
  *                 V                                      |
  *                 |                                      |
  *     +-----------------------+                          |
  *     |     write_header      |                          |
  *     +-----------------------+                          |
  *                 |                                      |
  *     +-----------------------+                          |
  *     | write_header_padding  |                          |
  *     +-----------------------+                          |
  *                 |                                      |
  *                 *------->-------.                      ^
  *                 |                \                     |
  *                 |                 |                    |
  *                 |                 | ,------<-----.     |
  *                 |                 |/              \    |
  *                 |                 V                |   |
  *                 |                 |                |   |
  *                 |     +-----------------------+    |   |
  *                 |     |      write_data       |    ^   |
  *                 |     +-----------------------+    |   |
  *                 |                 |               /    |
  *                 |                 *-------->-----'     |
  *                 |                 |                    |
  *                 |                /                     |
  *                 | ,-----<-------'                      ^
  *                 |/                                     |
  *                 V                                      |
  *                 |                                      |
  *                 |                                      |
  *     +-----------------------+                          |
  *     |  write_data_padding   |                          |
  *     +-----------------------+                          |
  *                 |                                     /
  *                 *-------------------->---------------'
  *                 |
  *     +-----------------------+
  *     |   write_archive_end   |
  *     +-----------------------+
  *                 |							</pre>
  */
class tar_output
{
public:
    /**
      * The destructor.
      */
    virtual ~tar_output();

    /**
      * The fatal method is used to report fatal errors.  The name of
      * the output file will be prepended.
      *
      * \param fmt
      *     The format controlling the string to be printed.  See
      *     printf(3) for a description of the formats and their
      *     arguments.
      */
    void fatal(char *fmt, ...) const ATTR_PRINTF(2, 3);

    /**
      * The fatal method is used to report fatal errors.  The name of
      * the output file will be prepended, the string equivalkent of
      * errno will be appended.
      */
    void nfatal(char *, ...) const ATTR_PRINTF(2, 3);

    /**
      * The write_header method is used to write file meta-data to
      * the output stream.  Must be called exactly once for each file.
      * The size in the header shall exactly match the amount of data
      * written with the write_data methods which follow.
      */
    virtual void write_header(const tar_header &) = 0;

    /**
      * The write_header_padding is used to write padding to bring
      * the output stream to a suitable byte multiple.  Must be called
      * exactly once for each file, after the header has been written.
      */
    virtual void write_header_padding();

    /**
      * The write_data method is used to write file data to the output
      * stream.  The file header must be written first.  Must be called
      * zero or more times for each file.
      */
    virtual void write_data(const void *, int) = 0;

    /**
      * The write_data_padding is used to write padding to bring the
      * output stream to a suitable byte multiple.  Must be called
      * exactly once for each file, after all the data has been written.
      */
    virtual void write_data_padding();

    /**
      * The filename method is used to retrieve the name of the output
      * file, which the archive is being written to.
      */
    virtual const char *filename() const = 0;

    /**
      * The write_archive_begin method is used to begin an archive.
      * It shall be called before the first file header is written.
      */
    virtual void write_archive_begin();

    /**
      * The write_archive_begin method is used to end an archive.
      * It shall be called after the last file data padding has been
      * written.
      */
    virtual void write_archive_end();

    /**
      * The set_block_size method is used to set the output block size
      * in bytes, the multiple of which will always be written to the
      * output.  The largely controlls the number of bytes of padding at
      * the end of the archive.  The default is no padding.
      *
      * \param nbytes
      *     The number of bytes per block.
      */
    virtual void set_block_size(long nbytes);

protected:
    /**
      * The default constructor.
      * Only derived classes may use it.
      */
    tar_output();

private:
    /**
      * The copy constructor.  Do not use.
      */
    tar_output(const tar_output &);

    /**
      * The assignment operator.  Do not use.
      */
    tar_output &operator = (const tar_output &);
};

#endif /* COMMON_TAR_OUTPUT_H */
