/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <tapioca/tpa-base.h>
#include <tapioca/tpa-client.h>

#include <glib.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <signal.h>
#include <sys/types.h>

/* variables */
static GMainLoop *mainloop;
static TpaManagerFactory *factory;
static TpaManager *manager;
static TpaProfile *profile;
static TpaParameter *parameter;
static TpaConnection *connection;

static void status_changed_cb(TpaConnection *conn, guint status, guint reason);
static void channel_created_cb(TpaConnection *conn, TpaChannel *channel);

void exit_program()
{
    tpa_connection_disconnect (connection);
	g_object_unref (factory);
    g_main_loop_quit (mainloop);
    g_print ("Disconnected. Finishing test...\n");
}

int main (int argc, char **argv)
{
    const gchar *account;
    const gchar *password;

    signal(SIGINT, exit_program);
    g_type_init_with_debug_flags (3);
    mainloop = g_main_loop_new (NULL, FALSE);

    tpa_debug_set_flags_from_env ();
    tpa_thread_init (FALSE);

    factory = tpa_manager_factory_new ();

    if (!factory)
    {
        g_print("failed to create Connection Manager Factory!\n");
        exit(1);
    }

    manager = tpa_manager_factory_get_manager(factory, "jabber");

    if (!manager)
    {
        g_print("failed to create Connection Manager!\n");
        exit(1);
    }

    if (!(account = g_getenv ("ACCOUNT"))) {
        g_print ("please set environment ACCOUNT!\n");
        g_object_unref (manager);
        exit(1);
    }
    else
        g_print ("account:%s\n", account);

    if (!(password = g_getenv ("PASSWORD"))) {
        g_print ("please set environment PASSWORD!\n");
        g_object_unref (manager);
        exit(1);
    }
    else
        g_print ("password set\n");

    /* setup the profile for connection */
    profile = tpa_manager_get_profile (manager, "jabber");

    parameter = tpa_profile_get_parameter (profile, "account");
    tpa_parameter_set_value_as_string (parameter, account);
    parameter = tpa_profile_get_parameter (profile, "password");
    tpa_parameter_set_value_as_string (parameter, password);
    parameter = tpa_profile_get_parameter (profile, "server");
    tpa_parameter_set_value_as_string (parameter, "talk.google.com");
    parameter = tpa_profile_get_parameter (profile, "resource");
    tpa_parameter_set_value_as_string (parameter, "tapioca");
    parameter = tpa_profile_get_parameter (profile, "port");
    tpa_parameter_set_value_as_uint (parameter, 5223);
    parameter = tpa_profile_get_parameter (profile, "old-ssl");
    tpa_parameter_set_value_as_boolean (parameter, TRUE);
    parameter = tpa_profile_get_parameter (profile, "ignore-ssl-errors");
    tpa_parameter_set_value_as_boolean (parameter, TRUE);

    /* Get a Connection */
    connection = tpa_manager_request_connection (manager, profile);

    if(!connection)
    {
        g_print("failed to create Connection!\n");
        exit(1);
    }

    /* Set up connection callbacks */
    g_signal_connect(G_OBJECT(connection), "status-changed", G_CALLBACK(status_changed_cb), NULL);
    g_signal_connect(G_OBJECT(connection), "channel-created", G_CALLBACK(channel_created_cb), NULL);

    /* Connect!  */
    tpa_connection_connect(connection);

    g_main_loop_run(mainloop);
    tpa_thread_shutdown (FALSE);
    return 0;
}

static void
status_changed_cb(TpaConnection *conn, guint status, guint reason)
{
    g_print("Got state-changed signal from TpaConnection\nstatus: %d, reason: %d\n", status, reason);
    if(status == TPA_CONNECTION_STATUS_CONNECTED)
    {
        TpaUserContact *user;
        TpaContactList *list;

        g_print("Connected!\nProtocol: %s\n", tpa_connection_get_protocol(conn));
        g_print("Status: %d\n", tpa_connection_get_status(conn));

        tpa_connection_get_open_channels (conn);
        user = tpa_connection_get_user_contact (conn);
        list = tpa_connection_get_contactlist (conn);
        tpa_contact_list_get_known (list);
        tpa_user_contact_set_capabilities (user, TPA_CAPABILITY_AUDIO);
    }
    if(status == TPA_CONNECTION_STATUS_DISCONNECTED)
    {
        g_print("Disconnected. Finninshing test...\n");
        g_main_loop_quit(mainloop);
    }
}

static gboolean
channel_play_cb (gpointer data)
{
    TpaStreamChannel *stream = TPA_STREAM_CHANNEL (data);

    tpa_stream_channel_join (stream);
    return FALSE;
}

static void
channel_created_cb(TpaConnection *conn, TpaChannel *channel)
{
    g_print("Got channel-created signal from TpaConnection\nchannel: %p\n", channel);

    if (tpa_channel_get_channel_type (channel) == TPA_CHANNEL_TYPE_STREAM) {
        g_timeout_add (10000, channel_play_cb, channel);
    }
}
