/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-room-list.h"

#include "tpa-room-list-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

/* signal enum */
enum
{
    GOT_ROOMS,
    LISTING_ROOMS,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaRoomListPrivate {
    GObject *object;
};

/* we need to define the get_type function */
GType
tpa_room_list_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIRoomList),
            NULL,   /* base init */
            NULL,   /* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIRoomList",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_room_list_init (TpaIRoomList *iface,
                    gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->get_listing_rooms = NULL;
    iface->list_rooms = NULL;

    /* Interface signals */
    signals[GOT_ROOMS] =
        g_signal_new ("got-rooms",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOXED,
                      G_TYPE_NONE, 1, (dbus_g_type_get_collection ("GPtrArray", (dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_STRING, (dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE)), G_TYPE_INVALID)))));

    signals[LISTING_ROOMS] =
        g_signal_new ("listing-rooms",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOOLEAN,
                      G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
        iface->priv = g_new0 (TpaIRoomListPrivate, 1);
        VERBOSE ("private members %p", iface->priv);
}

void
tpa_room_list_finalize (GObject *obj)
{
    TpaIRoomList *iface = TPA_IROOM_LIST (obj);
    VERBOSE ("(%p)", obj);

    if (iface->priv)
        g_free (iface->priv);
}

/**
 * tpa_room_list_get_listing_rooms
 *
 * Implements DBus method ListingRooms
 * on interface org.freedesktop.Telepathy.Channel.Type.StreamedMedia
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_room_list_get_listing_rooms (GObject *obj,
                                 gboolean *ret,
                                 GError **error)
{
    TpaIRoomList *iface = TPA_IROOM_LIST (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    g_return_error_val_if_fail (iface->get_listing_rooms != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_listing_rooms (obj, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_room_list_list_rooms
 *
 * Implements DBus method ListRooms
 * on interface org.freedesktop.Telepathy.Channel.Type.StreamedMedia
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_room_list_list_rooms (GObject *obj,
                          GError **error)
{
    TpaIRoomList *iface = TPA_IROOM_LIST (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->list_rooms != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->list_rooms (obj);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_room_list_signal_got_rooms
 *
 * Implements DBus signal GotRooms
 * on interface org.freedesktop.Telepathy.Channel.Type.RoomList
 */
void
tpa_room_list_signal_got_rooms (GObject *obj,
                                GPtrArray *rooms)
{
    g_assert (TPA_IS_IROOM_LIST (obj));

    VERBOSE ("(%p, %p)", obj, rooms);

    g_signal_emit (obj, signals[GOT_ROOMS], 0, rooms);
}

/**
 * tpa_room_list_signal_listing_rooms
 *
 * Implements DBus signal ListingRooms
 * on interface org.freedesktop.Telepathy.Channel.Type.RoomList
 */
void
tpa_room_list_signal_listing_rooms (GObject *obj,
                                    gboolean listing)
{
    g_assert (TPA_IS_IROOM_LIST (obj));

    VERBOSE ("(%p, %d)", obj, listing);

    g_signal_emit (obj, signals[LISTING_ROOMS], 0, listing);
}
