/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-group.h"
#include "tpa-account.h"

#include "tpa-group-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

/* signal enum */
enum
{
    GROUP_FLAGS_CHANGED,
    MEMBERS_CHANGED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaIGroupPrivate {
    GObject *object;
    GArray *members;
    guint handle;
};

/* we need to define the get_type function */
GType
tpa_group_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIGroup),
            NULL,   /* base init */
            NULL,   /* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIGroup",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_group_init (TpaIGroup *iface,
                gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->add_members = NULL;
    iface->remove_members = NULL;
    iface->get_all_members = NULL;
    iface->get_group_flags = NULL;
    iface->get_local_pending_members = NULL;
    iface->get_remote_pending_members = NULL;
    iface->get_members = NULL;

    /* Interface signals */
    signals[GROUP_FLAGS_CHANGED] =
        g_signal_new ("group-flags-changed",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      tpa_marshal_VOID__UINT_UINT,
                      G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);

    signals[MEMBERS_CHANGED] =
        g_signal_new ("members-changed",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      tpa_marshal_VOID__STRING_BOXED_BOXED_BOXED_BOXED_UINT_UINT,
                      G_TYPE_NONE, 7, G_TYPE_STRING, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, G_TYPE_UINT, G_TYPE_UINT);

    iface->priv = g_new0 (TpaIGroupPrivate, 1);
    iface->priv->members = g_array_new (FALSE, TRUE, sizeof (guint));
    VERBOSE ("private members %p", iface->priv);
}

void
tpa_group_finalize (GObject *obj)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    VERBOSE ("(%p)", obj);

    if (iface->priv) {
        g_array_free (iface->priv->members, TRUE);
        g_free (iface->priv);
    }
}

/**
 * tpa_group_add_members
 *
 * Implements DBus method AddMembers
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_group_add_members (GObject *obj,
                       const GArray *constants,
                       const gchar *message,
                       GError **error)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->remove_members != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->add_members (obj, constants, message);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return !(error_code);
}

/**
 * tpa_group_remove_members
 *
 * Implements DBus method RemoveMembers
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_group_remove_members (GObject *obj,
                          const GArray *constants,
                          const gchar *message,
                          GError **error)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->remove_members != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->remove_members (obj, constants, message);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return !(error_code);
}

/**
 * tpa_group_get_all_members
 *
 * Implements DBus method GetAllMembers
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_group_get_all_members (GObject *obj,
                           GArray **ret,
                           GArray **ret1,
                           GArray **ret2,
                           GError **error)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->get_all_members != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_all_members (obj, ret, ret1, ret2);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return !(error_code);
}

/**
 * tpa_group_get_group_flags
 *
 * Implements DBus method GetGroupFlags
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_group_get_group_flags (GObject *obj,
                           guint *ret,
                           GError **error)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->get_group_flags != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_group_flags (obj, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return !(error_code);
}

/**
 * tpa_group_get_handle_owners
 *
 * Implements DBus method GetHandleOwners
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_group_get_handle_owners (GObject *obj,
                             GArray *handles,
                             GArray **ret,
                             GError **error)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    // TODO: Parse handles

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return !(error_code);
}

/**
 * tpa_group_get_local_pending_members
 *
 * Implements DBus method GetLocalPendingMembers
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_group_get_local_pending_members (GObject *obj,
                                     GArray **ret,
                                     GError **error)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->get_local_pending_members != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_local_pending_members (obj, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return !(error_code);
}

/**
 * tpa_group_get_members
 *
 * Implements DBus method GetMembers
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_group_get_members (GObject *obj,
                       GArray **ret,
                       GError **error)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    TpaError error_code = TPA_ERROR_NONE;
    guint c = 0;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    if (iface->get_members)
        error_code = iface->get_members (obj, ret);
    else if (ret) {
        *ret = g_array_new (FALSE, TRUE, sizeof (guint));
        for (c = 0; c < iface->priv->members->len ; c++)
            g_array_append_val (*ret , g_array_index (iface->priv->members, guint, c));
    }
    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return !(error_code);
}

/**
 * tpa_group_get_remote_pending_members
 *
 * Implements DBus method GetRemotePendingMembers
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_group_get_remote_pending_members (GObject *obj,
                                      GArray **ret,
                                      GError **error)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->get_remote_pending_members != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_remote_pending_members (obj, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return !(error_code);
}

/**
 * tpa_group_get_self_handle
 *
 * Implements DBus method GetSelfHandle
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_group_get_self_handle (GObject *obj,
                           guint *ret,
                           GError **error)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    if (iface->priv->handle != -1)
        *ret = iface->priv->handle;
    else
        error_code = TPA_ERROR_DISCONNECTED;

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_group_signal_group_flags_changed
 *
 * Implements DBus signal GroupFlagsChanged
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 */
void
tpa_group_signal_group_flags_changed (GObject *obj,
                                      guint added,
                                      guint removed)
{
    TpaIGroup *iface = TPA_IGROUP (obj);

    if (!iface)
        return;

    VERBOSE ("(%p)", iface);

    g_signal_emit (iface, signals[GROUP_FLAGS_CHANGED], 0, added, removed);
}

/**
 * tpa_group_signal_members_changed
 *
 * Implements DBus signal MembersChanged
 * on interface org.freedesktop.Telepathy.Channel.Interface.Group
 */
void
tpa_group_signal_members_changed (GObject *obj,
                                  const gchar *message,
                                  GArray *added,
                                  GArray *removed,
                                  GArray *local_pending,
                                  GArray *remote_pending,
                                  const gchar *actor,
                                  guint reason)
{
    TpaIGroup *iface = TPA_IGROUP (obj);
    guint handle = 0;

    if (!iface)
        return;

    VERBOSE ("(%p)", iface);

    g_signal_emit (iface, signals[MEMBERS_CHANGED], 0, message, added, removed,
        local_pending, remote_pending, handle, reason);
}
