/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-contact-info.h"

#include "tpa-contact-info-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONTACT_INFO

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

/* signal enum */
enum
{
    GOT_CONTACT_INFO,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaIContactInfoPrivate {
    GObject *object;
};

/* we need to define the get_type function */
GType
tpa_contact_info_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIContactInfo),
            NULL,	/* base init */
            NULL,	/* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIContactInfo",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_contact_info_init (TpaIContactInfo *iface,
                             gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->request_contact_info = NULL;

    /* Interface signals */
    signals[GOT_CONTACT_INFO] =
        g_signal_new ("got-contact-info",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      tpa_marshal_VOID__UINT_STRING,
                      G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_STRING);

    iface->priv = g_new0 (TpaIContactInfoPrivate, 1);
}

void
tpa_contact_info_finalize (GObject *obj)
{
    TpaIContactInfo *iface = TPA_ICONTACT_INFO (obj);
    VERBOSE ("(%p)", obj);

    if (iface->priv)
        g_free (iface->priv);
}

/**
 * tpa_contact_info_request_contact_info
 *
 * Implements DBus method RequestContactInfo
 * on interface org.freedesktop.Telepathy.Connection.Interface.ContactInfo
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_contact_info_request_contact_info (GObject *obj,
                                       guint contact,
                                       GError **error)
{
    TpaIContactInfo *iface = TPA_ICONTACT_INFO (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %d)", obj, contact);

    g_return_error_val_if_fail (iface->request_contact_info != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->request_contact_info (obj, contact);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_contact_info_signal_got_contact_info
 *
 * Implements DBus signal GotContactInfo
 * on interface org.freedesktop.Telepathy.Connection.Interface.ContactInfo
 */
void
tpa_contact_info_signal_got_contact_info (GObject *obj,
                                          guint contact,
                                          const gchar *vcard)
{
    g_assert (TPA_IS_ICONTACT_INFO (obj));

    VERBOSE ("(%p)", obj);

    g_signal_emit (obj, signals[GOT_CONTACT_INFO], 0, contact, vcard);
}
