/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with self library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION_MANAGER

#include <tapioca/base/tpa-debug.h>

#include "tpa-thread.h"

static DBusGConnection *bus;
static gboolean initialized = FALSE;
static GThreadPool *pool;

static void
tpa_thread_func (TpaThread *thread,
                 GThreadPool *thread_pool)
{
    gpointer *args;
    guint i;

    args = g_new0 (gpointer, thread->args->n_values);

    for (i = 0; i < thread->args->n_values; i++)
        args[i] = g_value_get_pointer (g_value_array_get_nth (thread->args, i));

    switch (i) {
        case 0:
            thread->invoke (thread->proxy, thread->data);
            break;
        case 1:
            thread->invoke (thread->proxy, args[0], thread->data);
            break;
         case 2:
            thread->invoke (thread->proxy, args[0], args[1], thread->data);
            break;
         case 3:
            thread->invoke (thread->proxy, args[0], args[1], args[2], thread->data);
            break;
         case 4:
            thread->invoke (thread->proxy, args[0], args[1], args[2], args[3], thread->data);
            break;
         case 5:
            thread->invoke (thread->proxy, args[0], args[1], args[2], args[3], args[4], thread->data);
            break;
         case 6:
            thread->invoke (thread->proxy, args[0], args[1], args[2], args[3], args[4], args[5], thread->data);
            break;
         case 7:
            thread->invoke (thread->proxy, args[0], args[1], args[2], args[3], args[4], args[5], args[6], thread->data);
            break;
    }

    g_free (args);
    g_value_array_free (thread->args);
    if (thread_pool)
        INFO ("thread %p exited", thread);
    g_free (thread);
}

gboolean
tpa_thread_init (gboolean threaded)
{
    GError *error = NULL;
    DBusConnection *conn;

    if (initialized)
        return TRUE;

    VERBOSE ("");
    g_type_init ();
    if (!g_thread_supported ())
        g_thread_init (NULL);
    dbus_g_thread_init ();
    tpa_debug_set_flags_from_env ();

    /* init thread pool */
    if (threaded)
       pool = g_thread_pool_new ((GFunc) tpa_thread_func, pool, 1, FALSE, NULL);

    bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (!bus) {
        ERROR ("failed to open connection to dbus");
        goto fail;
    }
    conn = dbus_g_connection_get_connection(bus);
    if (!conn) {
        ERROR ("failed to setup dbus mainloop");
        goto fail;
    }

    initialized = TRUE;
    VERBOSE ("return TRUE");
    return TRUE;
fail:
    VERBOSE ("return FALSE");
    return FALSE;
}

void
tpa_thread_push (TpaThread *thread)
{
    VERBOSE ("(%p)", thread);

    if (!pool)
        tpa_thread_func (thread, NULL);
    else {
        g_thread_pool_push (pool, thread, NULL);
        INFO ("thread %p (%d) pushed", thread, g_thread_pool_unprocessed (pool));
    }
    VERBOSE ("return");
}

void
tpa_thread_shutdown (gboolean wait)
{
    VERBOSE ("");

    if (pool) {
        g_thread_pool_free (pool, TRUE, wait);
        pool = NULL;
    }

    VERBOSE ("return");
}

DBusGConnection *
tpa_thread_get_bus (void)
{
    GError *error = NULL;

    VERBOSE ("");
    if (!bus) {
        bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
        if (!bus) {
            VERBOSE ("return NULL");
            return NULL;
        }
    }

    VERBOSE ("return %p", bus);
    return bus;
}
