/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <time.h>

#include "tpa-stream-channel-priv.h"
#include "tpa-channel-target-priv.h"
#include "tpa-channel-priv.h"
#include "tpa-audio-stream-priv.h"
#include "tpa-video-stream-priv.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-channel-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

struct _TpaStreamChannelPrivate {
    DBusGProxy *group_proxy;
    DBusGProxy *stream_proxy;
    DBusGProxy *channel_handler_proxy;
    GHashTable *streams;
    gboolean disposed;
};

G_DEFINE_TYPE(TpaStreamChannel, tpa_stream_channel, TPA_TYPE_CHANNEL)

/* Signals */
enum {
    STREAM_ADDED,
    STREAM_REMOVED,
    LAST_SIGNAL
};

static guint tpa_stream_channel_signals[LAST_SIGNAL] = { 0 };

static void
proxy_stream_added (gpointer proxy,
                    guint id,
                    guint contact_handle,
                    guint type,
                    gpointer user_data)
{
    TpaStreamChannel* self = TPA_STREAM_CHANNEL (user_data);
    TpaStream *stream = NULL;

    VERBOSE ("(%d, %d, %d, %p)", id, contact_handle, type, user_data);
    g_assert (self);

    if (!g_hash_table_lookup (self->priv->streams, GUINT_TO_POINTER (id))) {
        if (type == TPA_STREAM_TYPE_AUDIO)
            stream = TPA_STREAM (tpa_audio_stream_new_incoming (TPA_CHANNEL (self),
                                                                id,
                                                                contact_handle,
                                                                type));
        else if (type == TPA_STREAM_TYPE_VIDEO)
            stream = TPA_STREAM (tpa_video_stream_new_incoming (TPA_CHANNEL (self),
                                                                id,
                                                                contact_handle,
                                                                type));

        if (stream) {
            g_hash_table_insert (self->priv->streams, GUINT_TO_POINTER (id), stream);
            g_signal_emit (self, tpa_stream_channel_signals[STREAM_ADDED], 0, stream);
        }
    }

    VERBOSE ("return");
}

static void
proxy_stream_removed (DBusGProxy *proxy,
                      guint id,
                      gpointer user_data)
{
    TpaStreamChannel* self = TPA_STREAM_CHANNEL (user_data);
    TpaStream *stream = NULL;

    VERBOSE ("(%p, %d, %p)", proxy, id, user_data);
    g_assert (self);

    if ((stream = g_hash_table_lookup (self->priv->streams, GUINT_TO_POINTER (id)))) {
        if (g_hash_table_remove (self->priv->streams, GUINT_TO_POINTER (id))) {
            g_object_unref (stream);
            g_signal_emit (self, tpa_stream_channel_signals[STREAM_REMOVED], 0, id);
        }
    }
    VERBOSE ("return");
}

static gboolean
tpa_stream_channel_handle_channel (TpaStreamChannel *self)
{
    TpaUserContact *user;
    DBusGProxy *proxy;
    DBusGProxy *conn_proxy;
    const gchar *service_name;
    const gchar *conn_path;
    const gchar *channel_path;
    GError *error = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    g_object_get (TPA_CHANNEL (self), "user", &user, NULL);
    proxy = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_CHANNEL);
    conn_proxy = tpa_object_get_proxy (TPA_OBJECT (user), TPA_INTERFACE_CONNECTION);

    service_name = dbus_g_proxy_get_bus_name (conn_proxy);
    conn_path = dbus_g_proxy_get_path (conn_proxy);
    channel_path = dbus_g_proxy_get_path (proxy);
    dbus_g_proxy_call (self->priv->channel_handler_proxy, "HandleChannel", &error,
                       G_TYPE_STRING, service_name,
                       DBUS_TYPE_G_OBJECT_PATH, conn_path,
                       G_TYPE_STRING, TPA_INTERFACE_STREAMED_MEDIA,
                       DBUS_TYPE_G_OBJECT_PATH, channel_path,
                       G_TYPE_UINT, 0,
                       G_TYPE_UINT, 0,
                       G_TYPE_INVALID, G_TYPE_INVALID);

    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
        VERBOSE ("return false");
        return FALSE;
    }

    INFO ("media stream %p handling channel %p", self, TPA_CHANNEL (self))

    g_object_unref (proxy);
    g_object_unref (conn_proxy);

    VERBOSE ("return true");
    return TRUE;
}

void
tpa_stream_channel_update_stream_list (TpaStreamChannel *self)
{
    guint i;
    GError *error = NULL;
    GPtrArray *streams = NULL;
    TpaStream *new_stream = NULL;

    if (!org_freedesktop_Telepathy_Channel_Type_StreamedMedia_list_streams (self->priv->stream_proxy, &streams, &error)
        || error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }
    else {
        for (i = 0; i < streams->len; i++) {
            GValueArray *stream = g_ptr_array_index (streams, i);
            guint id = g_value_get_uint (g_value_array_get_nth (stream, 0));
            guint contact = g_value_get_uint (g_value_array_get_nth (stream, 1));
            guint type = g_value_get_uint (g_value_array_get_nth (stream, 2));
            guint state = g_value_get_uint (g_value_array_get_nth (stream, 3));
            guint direction = g_value_get_uint (g_value_array_get_nth (stream, 4));
            guint pending = g_value_get_uint (g_value_array_get_nth (stream, 5));

            INFO ("stream id:%d contact:%d type:%d state:%d direction:%d pending:%d",
                  id, contact, type, state, direction, pending);

            if (g_hash_table_lookup (self->priv->streams, GUINT_TO_POINTER (id))) {
                //TODO - update some stream's attributes.
            }
            else {
                if (type == TPA_STREAM_TYPE_AUDIO)
                    new_stream = TPA_STREAM (tpa_audio_stream_new (TPA_CHANNEL (self),
                                                                   id,
                                                                   contact,
                                                                   type,
                                                                   state,
                                                                   direction,
                                                                   pending));
                else if (type == TPA_STREAM_TYPE_VIDEO)
                    new_stream = TPA_STREAM (tpa_video_stream_new (TPA_CHANNEL (self),
                                                                   id,
                                                                   contact,
                                                                   type,
                                                                   state,
                                                                   direction,
                                                                   pending));

                if (new_stream) {
                    g_hash_table_insert (self->priv->streams, GUINT_TO_POINTER (id), new_stream);
                    new_stream = NULL;
                }
            }
        }
    }
}

static GObject*
tpa_stream_channel_constructor (GType type,
                                guint n_construct_params,
                                GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaStreamChannel *self;
    DBusGConnection *bus;
    GError *error = NULL;

    object = G_OBJECT_CLASS (tpa_stream_channel_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_STREAM_CHANNEL (object);

    bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);

    if (!bus) {
        ERROR ("failed to open connection to dbus");
        g_error_free (error);
        return NULL;
    }

    self->priv->stream_proxy = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_STREAMED_MEDIA);
    self->priv->group_proxy = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_GROUP);

    self->priv->channel_handler_proxy = dbus_g_proxy_new_for_name (bus,
                                        "org.freedesktop.Telepathy.StreamEngine",
                                        "/org/freedesktop/Telepathy/StreamEngine",
                                        "org.freedesktop.Telepathy.ChannelHandler");
    tpa_object_add_proxy (TPA_OBJECT (self), self->priv->channel_handler_proxy);

    self->priv->streams = g_hash_table_new_full (g_direct_hash,
                                                 g_direct_equal,
                                                 NULL,
                                                 g_object_unref);

    tpa_object_connect_signal (TPA_OBJECT (self),
                               TPA_INTERFACE_STREAMED_MEDIA,
                               "StreamAdded",
                               G_CALLBACK (proxy_stream_added),
                               self);

    tpa_object_connect_signal (TPA_OBJECT (self),
                               TPA_INTERFACE_STREAMED_MEDIA,
                               "StreamRemoved",
                               G_CALLBACK (proxy_stream_removed),
                               self);

    if (!tpa_stream_channel_handle_channel (self))
        ERROR ("failed to handle channel.");

    tpa_stream_channel_update_stream_list (self);

    return object;
}

static void
tpa_stream_channel_dispose (GObject *object)
{
    TpaStreamChannel *self = TPA_STREAM_CHANNEL (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    if (self->priv->channel_handler_proxy)
        g_object_unref (self->priv->channel_handler_proxy);
    if (self->priv->group_proxy)
        g_object_unref (self->priv->group_proxy);
    if (self->priv->stream_proxy)
        g_object_unref (self->priv->stream_proxy);
    if (self->priv->streams)
        g_hash_table_destroy (self->priv->streams);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_stream_channel_parent_class)->dispose (object);
}

static void
tpa_stream_channel_class_init (TpaStreamChannelClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_stream_channel_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaStreamChannelPrivate));

    gobject_class->dispose = tpa_stream_channel_dispose;
    gobject_class->constructor = tpa_stream_channel_constructor;

    tpa_stream_channel_signals[STREAM_ADDED] = g_signal_new ("stream-added",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__OBJECT,
        G_TYPE_NONE,
        1,
        TPA_TYPE_STREAM);

    tpa_stream_channel_signals[STREAM_REMOVED] = g_signal_new ("stream-removed",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__UINT,
        G_TYPE_NONE,
        1,
        G_TYPE_UINT);
}

static void
tpa_stream_channel_init (TpaStreamChannel *self)
{
    self->priv = TPA_STREAM_CHANNEL_GET_PRIVATE (self);
    self->priv->group_proxy = NULL;
    self->priv->stream_proxy = NULL;
    self->priv->channel_handler_proxy = NULL;
    self->priv->streams = NULL;
    self->priv->disposed = FALSE;
}

TpaStreamChannel *
tpa_stream_channel_new (DBusGProxy *proxy,
                        TpaUserContact *user,
                        TpaChannelTarget *target)
{
    TpaStreamChannel *self = NULL;

    g_return_val_if_fail (proxy != NULL, NULL);
    g_return_val_if_fail (user != NULL, NULL);
    g_return_val_if_fail (target != NULL, NULL);

    /* Pass givin arguments to constructor */
    self = TPA_STREAM_CHANNEL (g_object_new (TPA_TYPE_STREAM_CHANNEL,
                                             "proxy", proxy,
                                             "target", target,
                                             "user", user,
                                             "type", TPA_CHANNEL_TYPE_STREAM,
                                             NULL));

    return self;
}

static void
tpa_stream_channel_stream_foreach (gpointer key,
                                   gpointer value,
                                   gpointer user_data)
{
    GList **list = (GList **) user_data;
    *list = g_list_append (*list, value);
}

G_CONST_RETURN GList*
tpa_stream_channel_get_streams (TpaStreamChannel *self)
{
    GList *list = NULL;
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return");

    g_hash_table_foreach (self->priv->streams, tpa_stream_channel_stream_foreach, &list);
    return list;
}

void
tpa_stream_channel_join (TpaStreamChannel *self)
{
    GArray *handles = NULL;
    GError *error = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!org_freedesktop_Telepathy_Channel_Interface_Group_get_local_pending_members (self->priv->group_proxy, &handles, &error)
        || error)
        goto err;
    else if (handles->len) {
        guint id;
        TpaUserContact *user = tpa_channel_get_owner (TPA_CHANNEL (self));
        TpaHandle *handle = tpa_channel_target_get_handle (TPA_CHANNEL_TARGET (user));

        id = tpa_handle_get_id (handle);
        g_array_append_val (handles, id);

        if (!org_freedesktop_Telepathy_Channel_Interface_Group_add_members (self->priv->group_proxy, handles, "", &error)
            || error)
            goto err;
    }

err:
    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

    if (handles)
        g_array_free (handles, TRUE);

    VERBOSE ("return");
}

void
tpa_stream_channel_request_streams (TpaStreamChannel *self,
                                    const TpaContact *contact,
                                    GList *type_list)
{
    guint contact_id = 0;
    GError *error = NULL;
    GPtrArray *streams = NULL;
    GArray *types = NULL;
    GList *walk  = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    g_return_if_fail (contact != NULL);
    g_return_if_fail (type_list != NULL);

    contact_id = tpa_handle_get_id (tpa_channel_target_get_handle (TPA_CHANNEL_TARGET (contact)));

    types = g_array_new (FALSE, FALSE, sizeof (guint));

    walk = type_list;

    while (walk) {
        g_array_append_val (types, walk->data);
        walk = g_list_next (walk);
    }

    if (!org_freedesktop_Telepathy_Channel_Type_StreamedMedia_request_streams (self->priv->stream_proxy,
                                                                               contact_id,
                                                                               types,
                                                                               &streams,
                                                                               &error) ||
        error) {
// TODO:        ERROR ("%s", error->message);
        if (error)
            g_error_free (error);
    }

    g_array_free (types, TRUE);

    VERBOSE ("return");
}

void
tpa_stream_channel_remove_stream (TpaStreamChannel *self,
                                  TpaStream *stream)
{
    GError *error = NULL;
    GArray *streams = NULL;
    guint id = 0;

    VERBOSE ("(%p)", self);
    g_assert (self);
    g_assert (stream);

    streams = g_array_new (FALSE, FALSE, sizeof (guint));
    id = tpa_stream_get_id (stream);
    g_array_append_val (streams, id);

    if (!org_freedesktop_Telepathy_Channel_Type_StreamedMedia_remove_streams (self->priv->stream_proxy, streams, &error) ||
        error) {
        if (error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
    }

    g_array_free (streams, TRUE);
    VERBOSE ("return");
}
