/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <stdlib.h>

#include "tpa-parameter.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION_MANAGER

#include <tapioca/base/tpa-debug.h>

struct _TpaParameterPrivate {
    gchar *name;
    gchar *friendly_name;
    TpaParameterFlags flags;
    GType type;
    GValue *value;
    GValue *default_value;
    void (* convert_func) (GValue *value, gchar **string_value);

    gboolean disposed;
};

enum
{
    ARG_0,
    ARG_NAME,
    ARG_TYPE
};

G_DEFINE_TYPE(TpaParameter, tpa_parameter, G_TYPE_OBJECT)

static void
tpa_parameter_convert_string (GValue *value,
                              gchar **string_value)
{
    if (value && !*string_value)
        *string_value = g_value_dup_string (value);
    else if (value && *string_value) {
        g_value_set_string (value, g_strdup (*string_value));
    }
}

static void
tpa_parameter_convert_boolean (GValue *value,
                               gchar **string_value)
{
    if (value && !*string_value)
        *string_value = g_value_get_boolean (value) ? g_strdup ("true") :
            g_strdup ("false");
    else if (value && *string_value)
        g_value_set_boolean (value, (g_str_equal (*string_value, "true")));
}

static void
tpa_parameter_convert_int (GValue *value,
                           gchar **string_value)
{
    if (value && !*string_value) {
        *string_value = g_new (gchar, 256);
        *string_value = g_ascii_dtostr (*string_value, 256, g_value_get_int (value));
    }
    else if (value && *string_value)
        g_value_set_int (value, g_ascii_strtod (*string_value, NULL));
}

static void
tpa_parameter_convert_uint (GValue *value,
                            gchar **string_value)
{
    if (value && !*string_value) {
        *string_value = g_new (gchar, 256);
        *string_value = g_ascii_dtostr (*string_value, 256, g_value_get_uint (value));
    }
    else if (value && *string_value)
        g_value_set_uint (value, g_ascii_strtod (*string_value, NULL));
}

static void
tpa_parameter_dispose (GObject *object)
{
    TpaParameter *self = TPA_PARAMETER (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    /* Free private data */
    if (self->priv->name)
        g_free (self->priv->name);

    if (self->priv->friendly_name)
        g_free (self->priv->friendly_name);

    if (self->priv->value)
        g_free (self->priv->value);

    if (self->priv->default_value)
        g_free (self->priv->default_value);

    G_OBJECT_CLASS (tpa_parameter_parent_class)->dispose (object);
}

static void
tpa_parameter_set_property (GObject *object,
                            guint prop_id,
                            const GValue *value,
                            GParamSpec *pspec)
{
    TpaParameter *self = TPA_PARAMETER (object);

    switch (prop_id) {
        case ARG_NAME:
            self->priv->name = g_value_dup_string (value);
            break;
        case ARG_TYPE:
            self->priv->type = g_value_get_gtype (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
tpa_parameter_get_property (GObject *object,
                            guint prop_id,
                            GValue *value,
                            GParamSpec *pspec)
{
    TpaParameter *self = TPA_PARAMETER (object);

    switch (prop_id) {
        case ARG_NAME:
            g_value_set_string (value, self->priv->name);
            break;
        case ARG_TYPE:
            g_value_set_gtype (value, self->priv->type);
            break;
        default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
          break;
    }
}

static GObject*
tpa_parameter_constructor (GType type,
                           guint n_construct_params,
                           GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaParameter *self;

    object = G_OBJECT_CLASS (tpa_parameter_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_PARAMETER (object);

    g_value_init (self->priv->default_value, self->priv->type);
    g_value_init (self->priv->value, self->priv->type);

    switch (self->priv->type) {
        case G_TYPE_STRING:
            self->priv->convert_func = tpa_parameter_convert_string;
            break;
        case G_TYPE_BOOLEAN:
            self->priv->convert_func = tpa_parameter_convert_boolean;
            break;
        case G_TYPE_UINT:
            self->priv->convert_func = tpa_parameter_convert_uint;
            break;
        case G_TYPE_INT:
            self->priv->convert_func = tpa_parameter_convert_int;
            break;
    }

    return object;
}

static void
tpa_parameter_class_init (TpaParameterClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_parameter_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaParameterPrivate));

    gobject_class->dispose = tpa_parameter_dispose;
    gobject_class->constructor = tpa_parameter_constructor;
    gobject_class->set_property = tpa_parameter_set_property;
    gobject_class->get_property = tpa_parameter_get_property;

    g_object_class_install_property (gobject_class,
                                     ARG_NAME,
                                     g_param_spec_string ("name",
                                     "name",
                                     "name",
                                     NULL,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_TYPE,
                                     g_param_spec_gtype ("type",
                                     "type",
                                     "type",
                                     G_TYPE_NONE,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));
}

static void
tpa_parameter_init (TpaParameter *self)
{
    self->priv = TPA_PARAMETER_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->friendly_name = NULL;
    self->priv->flags = TPA_PARAMETER_FLAGS_NONE;
    self->priv->default_value = g_new0 (GValue, 1);
    self->priv->value = g_new0 (GValue, 1);
}

/**
 * tpa_parameter_new:
 * @returns: #TpaParametes instance.
 *
 * Create new instance of #TpaParameter.
 */
TpaParameter *
tpa_parameter_new (const gchar *name,
                   GType type)
{
    TpaParameter *self = NULL;

    self = TPA_PARAMETER (g_object_new (TPA_TYPE_PARAMETER,
                          "name", name,
                          "type", type,
                          NULL));

    return self;
}

/**
 * tpa_parameter_get_name:
 * @self: #TpaParameter instance
 * @returns: NULL if fail.
 *
 * Get parameter's name.
 *
 */
const gchar *
tpa_parameter_get_name (TpaParameter *self)
{
    VERBOSE ("(%p)", self);

    return self->priv->name;
}


/**
 * tpa_parameter_get_default_value:
 * @self: #TpaParameter instance
 * @returns: NULL if fail.
 *
 * Get copy of parameter default value that must be free.
 *
 */
GValue *
tpa_parameter_get_default_value (TpaParameter *self)
{
    GValue *value = NULL;

    VERBOSE ("(%p)", self);

    value = g_new0 (GValue, 1);
    g_value_init (value, G_VALUE_TYPE (self->priv->default_value));
    g_value_copy (self->priv->default_value, value);

    return value;
}

/**
 * tpa_parameter_get_default_value_as_string:
 * @self: #TpaParameter instance
 * @returns: NULL if fail.
 *
 * Function to get any parameter default value to string.
 * Note that self function convert parameter value to string.
 *
 * Supports convertions to:
 *   boolean:
 *     "true" for TRUE value.
 *     "false" for FALSE value.
 *   double
 *   integer
 *   string
 *
 */
gchar *
tpa_parameter_get_default_value_as_string (TpaParameter *self)
{
    gchar *value_as_string = NULL;

    VERBOSE ("(%p)", self);

    self->priv->convert_func (self->priv->default_value, &value_as_string);

    VERBOSE ("(return %s)", value_as_string);
    return value_as_string;
}

/**
 * tpa_parameter_get_default_value_as_uint:
 * @self: #TpaParameter instance
 * @returns: 0 if fail.
 *
 * Fast function to get parameter default value that has known type as unsigned
 * integer.
 * DO NOT USE IT FOR NON-UINTEGER PARAMETERS.
 *
 */
guint
tpa_parameter_get_default_value_as_uint (TpaParameter *self)
{
    VERBOSE ("(%p)", self);

    g_return_val_if_fail (G_VALUE_HOLDS_UINT (self->priv->default_value), 0);

    return g_value_get_uint (self->priv->default_value);
}

/**
 * tpa_parameter_get_default_value_as_int:
 * @self: #TpaParameter instance
 * @returns: 0 if fail.
 *
 * Fast function to get parameter default value that has known type as integer.
 * DO NOT USE IT FOR NON-INTEGER PARAMETERS.
 *
 */
gint
tpa_parameter_get_default_value_as_int (TpaParameter *self)
{
    VERBOSE ("(%p)", self);

    g_return_val_if_fail (G_VALUE_HOLDS_INT (self->priv->default_value), 0);

    return g_value_get_uint (self->priv->default_value);
}

/**
 * tpa_parameter_get_default_value_as_boolean:
 * @self: #TpaParameter instance
 * @returns: -1 if fail.
 *
 * Fast function to get parameter default value that has known type as boolean.
 * DO NOT USE IT FOR NON-BOOLEAN PARAMETERS.
 *
 */
gboolean
tpa_parameter_get_default_value_as_boolean (TpaParameter *self)
{
    VERBOSE ("(%p)", self);

    g_return_val_if_fail (G_VALUE_HOLDS_BOOLEAN (self->priv->default_value), FALSE);

    return g_value_get_uint (self->priv->default_value);
}

/**
 * tpa_parameter_set_default_value:
 * @self: #TpaParameter instance
 * @value: New GValue value.
 * @returns: FALSE if fail.
 *
 * Set parameter default value, givin value must hold the same type as parameter.
 *
 */
gboolean
tpa_parameter_set_default_value (TpaParameter *self,
                                  GValue *value)
{
    VERBOSE ("(%p, %p)", self, value);

    g_return_val_if_fail (G_VALUE_HOLDS (value, G_VALUE_TYPE (self->priv->default_value)),
         FALSE);

    g_value_copy (value, self->priv->default_value);
    g_value_copy (value, self->priv->value);

    return TRUE;
}

/**
 * tpa_parameter_set_default_value_as_string:
 * @self: #TpaParameter instance
 * @value_as_string: New string value or string to be converted.
 * @returns: FALSE if fail.
 *
 * Function to set any string default value to parameter value.
 * Note that self function convert the given value to
 * parameter type if it is not string type.
 *
 * Supports convertions to:
 *   boolean:
 *     "true" for TRUE value.
 *     "false" for FALSE value.
 *   double
 *   integer
 *   string
 */
gboolean
tpa_parameter_set_default_value_as_string (TpaParameter *self,
                                           const gchar *value_as_string)
{
    VERBOSE ("(%p, %s)", self, value_as_string);

    g_return_val_if_fail (value_as_string != NULL, FALSE);

    self->priv->convert_func (self->priv->default_value, (gchar **) &value_as_string);

    return TRUE;
}

/**
 * tpa_parameter_set_default_value_as_uint:
 * @self: #TpaParameter instance
 * @value_as_uint: New integer value.
 * @returns: FALSE if fail.
 *
 * Fast function to set parameter default value that has known type as unsigned integer.
 * DO NOT USE IT FOR NON-UINTEGER PARAMETERS.
 *
 */
gboolean
tpa_parameter_set_default_value_as_uint (TpaParameter *self,
                                          guint value_as_uint)
{
    VERBOSE ("(%p, %d)", self, value_as_uint);

    g_return_val_if_fail (G_VALUE_HOLDS_UINT (self->priv->default_value), FALSE);

    g_value_set_uint (self->priv->default_value, value_as_uint);

    return TRUE;
}

/**
 * tpa_parameter_set_default_value_as_int:
 * @self: #TpaParameter instance
 * @value_as_int: New integer value.
 * @returns: FALSE if fail.
 *
 * Fast function to set parameter default value that has known type as integer.
 * DO NOT USE IT FOR NON-INTEGER PARAMETERS.
 *
 */
gboolean
tpa_parameter_set_default_value_as_int (TpaParameter *self,
	                                    gint value_as_int)
{
    VERBOSE ("(%p, %d)", self, value_as_int);

    g_return_val_if_fail (G_VALUE_HOLDS_INT (self->priv->default_value), FALSE);

    g_value_set_int (self->priv->default_value, value_as_int);

    return TRUE;
}

/**
 * tpa_parameter_set_default_value_as_boolean:
 * @self: #TpaParameter instance
 * @value_as_boolean: New boolean value.
 * @returns: FALSE if fail.
 *
 * Fast function to set parameter dafault value that has known type as boolean.
 * DO NOT USE IT FOR NON-BOOLEAN PARAMETERS.
 *
 */
gboolean
tpa_parameter_set_default_value_as_boolean (TpaParameter *self,
                                            gboolean value_as_boolean)
{
    VERBOSE ("(%p, %s)", self, value_as_boolean ? "true" : "false");

    g_return_val_if_fail (G_VALUE_HOLDS_BOOLEAN (self->priv->default_value), FALSE);

    g_value_set_boolean (self->priv->default_value, value_as_boolean);

    return TRUE;
}

/**
 * tpa_parameter_get_value:
 * @self: #TpaParameter instance
 * @returns: NULL if fail.
 *
 * Get copy of parameter value that must be free.
 *
 */
GValue *
tpa_parameter_get_value (TpaParameter *self)
{
    GValue *value = NULL;

    VERBOSE ("(%p)", self);

    value = g_new0 (GValue, 1);
    g_value_init (value, G_VALUE_TYPE (self->priv->value));
    g_value_copy (self->priv->value, value);

    return value;
}

/**
 * tpa_parameter_get_value_as_string:
 * @self: #TpaParameter instance
 * @returns: NULL if fail.
 *
 * Function to get any parameter value to string.
 * Note that self function convert parameter value to string.
 *
 * Supports convertions to:
 *   boolean:
 *     "true" for TRUE value.
 *     "false" for FALSE value.
 *   double
 *   integer
 *   string
 *
 */
gchar *
tpa_parameter_get_value_as_string (TpaParameter *self)
{
    gchar *value_as_string = NULL;

    VERBOSE ("(%p)", self);

    /* Call callback to convert value to string */
    self->priv->convert_func (self->priv->value, &value_as_string);

    return value_as_string;
}

/**
 * tpa_parameter_get_value_as_uint:
 * @self: #TpaParameter instance
 * @returns: 0 if fail.
 *
 * Fast function to get parameter value that has known type as unsigned
 * integer.
 * DO NOT USE IT FOR NON-UINTEGER PARAMETERS.
 *
 */
guint
tpa_parameter_get_value_as_uint (TpaParameter *self)
{
    VERBOSE ("(%p)", self);

    g_return_val_if_fail (G_VALUE_HOLDS_UINT (self->priv->value), 0);

    return g_value_get_uint (self->priv->value);
}

/**
 * tpa_parameter_get_value_as_int:
 * @self: #TpaParameter instance
 * @returns: 0 if fail.
 *
 * Fast function to get parameter value that has known type as integer.
 * DO NOT USE IT FOR NON-INTEGER PARAMETERS.
 *
 */
gint
tpa_parameter_get_value_as_int (TpaParameter *self)
{
    VERBOSE ("(%p)", self);

    g_return_val_if_fail (G_VALUE_HOLDS_INT (self->priv->value), 0);

    return g_value_get_int (self->priv->value);
}

/**
 * tpa_parameter_get_value_as_boolean:
 * @self: #TpaParameter instance
 * @returns: -1 if fail.
 *
 * Fast function to get parameter value that has known type as boolean.
 * DO NOT USE IT FOR NON-BOOLEAN PARAMETERS.
 *
 */
gboolean
tpa_parameter_get_value_as_boolean (TpaParameter *self)
{
    VERBOSE ("(%p)", self);

    g_return_val_if_fail (G_VALUE_HOLDS_BOOLEAN (self->priv->value), -1);

    return g_value_get_boolean (self->priv->value);
}

/**
 * tpa_parameter_set_value:
 * @self: #TpaParameter instance
 * @value: New GValue value.
 * @returns: FALSE if fail.
 *
 * Set parameter value, givin value must hold the same type as parameter.
 *
 */
gboolean
tpa_parameter_set_value (TpaParameter *self,
                          GValue *value)
{
    VERBOSE ("(%p, %p)", self, value);

    g_return_val_if_fail (G_VALUE_HOLDS (value, G_VALUE_TYPE (self->priv->value)),
         FALSE);

    g_value_copy (value, self->priv->value);

    return TRUE;
}

/**
 * tpa_parameter_set_value_as_string:
 * @self: #TpaParameter instance
 * @value_as_string: New string value or string to be converted.
 * @returns: FALSE if fail.
 *
 * Function to set any string value to parameter value.
 * Note that this function convert the given value to
 * parameter type if it is not string type.
 *
 * Supports convertions to:
 *   boolean:
 *     "true" for TRUE value
 *     "false" for FALSE value
 *   double
 *   integer
 *   string
 */
gboolean
tpa_parameter_set_value_as_string (TpaParameter *self,
                                    const gchar *value_as_string)
{
    VERBOSE ("(%p, %s)", self, value_as_string);
    g_return_val_if_fail (value_as_string != NULL, FALSE);

    /* Call callback to convert string to value */
    self->priv->convert_func (self->priv->value, (gchar **) &value_as_string);

    return TRUE;
}

/**
 * tpa_parameter_set_value_as_uint:
 * @self: #TpaParameter instance
 * @value_as_uint: New unsigned integer value.
 * @returns: FALSE if fail.
 *
 * Fast function to set parameter value that has known type as unsigned
 * integer.
 * DO NOT USE IT FOR NON-UINTEGER PARAMETERS.
 *
 */
gboolean
tpa_parameter_set_value_as_uint (TpaParameter *self,
                                 guint value_as_uint)
{
    VERBOSE ("(%p, %d)", self, value_as_uint);

    g_return_val_if_fail (G_VALUE_HOLDS_UINT (self->priv->value), FALSE);

    g_value_set_uint (self->priv->value, value_as_uint);

    return TRUE;
}

/**
 * tpa_parameter_set_value_as_int:
 * @self: #TpaParameter instance
 * @value_as_int: New integer value.
 * @returns: FALSE if fail.
 *
 * Fast function to set parameter value that has known type as integer.
 * DO NOT USE IT FOR NON-INTEGER PARAMETERS.
 *
 */
gboolean
tpa_parameter_set_value_as_int (TpaParameter *self,
                                gint value_as_int)
{
    VERBOSE ("(%p, %d)", self, value_as_int);

    g_return_val_if_fail (G_VALUE_HOLDS_INT (self->priv->value), FALSE);

    g_value_set_int (self->priv->value, value_as_int);

    return TRUE;
}

/**
 * tpa_parameter_set_value_as_boolean:
 * @self: #TpaParameter instance
 * @value_as_boolean: New boolean value.
 * @returns: FALSE if fail.
 *
 * Fast function to set parameter value that has known type as boolean.
 * DO NOT USE IT FOR NON-BOOLEAN PARAMETERS.
 *
 */
gboolean
tpa_parameter_set_value_as_boolean (TpaParameter *self,
                                    gboolean value_as_boolean)
{
    VERBOSE ("(%p, %s)", self, value_as_boolean ? "true" : "false");

    g_return_val_if_fail (G_VALUE_HOLDS_BOOLEAN (self->priv->value), FALSE);

    g_value_set_boolean (self->priv->value, value_as_boolean);

    return TRUE;
}

/**
 * tpa_parameter_get_flags:
 * @self: #TpaParameter instance
 * @returns: #TpaParameterFlags
 *
 * Get parameter flags.
 *
 */
TpaParameterFlags
tpa_parameter_get_flags (TpaParameter *self)
{
    VERBOSE ("(%p)", self);

    return self->priv->flags;
}

/**
 * tpa_parameter_set_flags:
 * @self: #TpaParameter instance
 * @returns: FALSE if fail.
 *
 * Set parameter flags.
 *
 */
gboolean
tpa_parameter_set_flags (TpaParameter *self,
                         TpaParameterFlags flags)
{
    VERBOSE ("(%p, %d)", self, flags);

    self->priv->flags |= flags;

    return TRUE;
}
