#! /usr/bin/env python
# -*- coding: utf-8 -*-

# Allmydata Tahoe -- secure, distributed storage grid
#
# Copyright © 2008-2009 Allmydata, Inc.
#
# This file is part of Tahoe-LAFS.
#
# See the docs/about.html file for licensing information.

import os, shutil, stat, subprocess, sys, zipfile, re

##### sys.path management

def pylibdir(prefixdir):
    pyver = "python%d.%d" % (sys.version_info[:2])
    if sys.platform == "win32":
        return os.path.join(prefixdir, "Lib", "site-packages")
    else:
        return os.path.join(prefixdir, "lib", pyver, "site-packages")

basedir = os.path.dirname(os.path.abspath(__file__))
supportlib = pylibdir(os.path.join(basedir, "support"))

# locate our version number. This only works from a generated tarball, in
# which src/allmydata/_version.py was previously created by invoking
# "darcsver --count-all-patches".

verstrline = open("src/allmydata/_version.py", "rt").read()
VSRE = r"^verstr = ['\"]([^'\"]*)['\"]"
mo = re.search(VSRE, verstrline, re.M)
version = mo.group(1)

from setuptools import find_packages, setup
from setuptools.command import sdist
from setuptools import Command
from pkg_resources import require

# Make the dependency-version-requirement, which is used by the Makefile at
# build-time, also available to the app at runtime:
shutil.copyfile("_auto_deps.py",
                os.path.join("src", "allmydata", "_auto_deps.py"))

trove_classifiers=[
    "Development Status :: 5 - Production/Stable",
    "Environment :: Console",
    "Environment :: Web Environment",
    "License :: OSI Approved :: GNU General Public License (GPL)",
    "License :: DFSG approved",
    "License :: Other/Proprietary License",
    "Intended Audience :: Developers",
    "Intended Audience :: End Users/Desktop",
    "Intended Audience :: System Administrators",
    "Operating System :: Microsoft",
    "Operating System :: Microsoft :: Windows",
    "Operating System :: Microsoft :: Windows :: Windows NT/2000",
    "Operating System :: Unix",
    "Operating System :: POSIX :: Linux",
    "Operating System :: POSIX",
    "Operating System :: MacOS :: MacOS X",
    "Operating System :: OS Independent",
    "Natural Language :: English",
    "Programming Language :: C",
    "Programming Language :: Python",
    "Programming Language :: Python :: 2",
    "Programming Language :: Python :: 2.4",
    "Programming Language :: Python :: 2.5",
    "Programming Language :: Python :: 2.6",
    "Topic :: Utilities",
    "Topic :: System :: Systems Administration",
    "Topic :: System :: Filesystems",
    "Topic :: System :: Distributed Computing",
    "Topic :: Software Development :: Libraries",
    "Topic :: Communications :: Usenet News",
    "Topic :: System :: Archiving :: Backup",
    "Topic :: System :: Archiving :: Mirroring",
    "Topic :: System :: Archiving",
    ]


LONG_DESCRIPTION=\
"""Welcome to the Tahoe project, a secure, decentralized, fault-tolerant
filesystem.  All of the source code is available under a Free Software, Open
Source licence.

This filesystem is encrypted and spread over multiple peers in such a way that
it remains available even when some of the peers are unavailable,
malfunctioning, or malicious."""


class ShowSupportLib(Command):
    user_options = []
    def initialize_options(self):
        pass
    def finalize_options(self):
        pass
    def run(self):
        # TODO: --quiet suppresses the 'running show_supportlib' message.
        # Find a way to do this all the time.
        print supportlib # TODO windowsy

class ShowPythonPath(Command):
    user_options = []
    def initialize_options(self):
        pass
    def finalize_options(self):
        pass
    def run(self):
        # TODO: --quiet suppresses the 'running show_supportlib' message.
        # Find a way to do this all the time.
        print "PYTHONPATH=%s" % os.environ.get("PYTHONPATH", '')

class RunWithPythonPath(Command):
    description = "Run a subcommand with PYTHONPATH set appropriately"

    user_options = [ ("python", "p",
                      "Treat command string as arguments to a python executable"),
                     ("command=", "c", "Command to be run"),
                     ("directory=", "d", "Directory to run the command in"),
                     ]
    boolean_options = ["python"]

    def initialize_options(self):
        self.command = None
        self.python = False
        self.directory = None
    def finalize_options(self):
        pass
    def run(self):
        oldpp = os.environ.get("PYTHONPATH", "").split(os.pathsep)
        if oldpp == [""]:
            # grr silly split() behavior
            oldpp = []
        os.environ['PYTHONPATH'] = os.pathsep.join(oldpp + [supportlib,])

        # We must require the command to be safe to split on
        # whitespace, and have --python and --directory to make it
        # easier to achieve this.

        command = []
        if self.python:
            command.append(sys.executable)
        if self.command:
            command.extend(self.command.split())
        if not command:
            raise RuntimeError("The --command argument is mandatory")
        if self.directory:
            os.chdir(self.directory)
        if self.verbose:
            print "command =", " ".join(command)
        rc = subprocess.call(command)
        sys.exit(rc)

class TestMacDiskImage(Command):
    user_options = []
    def initialize_options(self):
        pass
    def finalize_options(self):
        pass
    def run(self):
        import sys
        sys.path.append('misc')
        import test_mac_diskimage
        return test_mac_diskimage.test_mac_diskimage('Allmydata', version=self.distribution.metadata.version)

class CheckAutoDeps(Command):
    user_options = []
    def initialize_options(self):
        pass
    def finalize_options(self):
        pass
    def run(self):
        import _auto_deps
        _auto_deps.require_auto_deps()


class MakeExecutable(Command):
    user_options = []
    def initialize_options(self):
        pass
    def finalize_options(self):
        pass
    def run(self):
        bin_tahoe_template = os.path.join("bin", "tahoe-script.template")

        # Create the 'tahoe-script.py' file under the 'bin' directory. The
        # 'tahoe-script.py' file is exactly the same as the
        # 'tahoe-script.template' script except that the shebang line is
        # rewritten to use our sys.executable for the interpreter. On
        # Windows, create a tahoe.exe will execute it. On non-Windows, make a
        # symlink to it from 'tahoe'. The tahoe.exe will be copied from the
        # setuptools egg's cli.exe and this will work from a zip-safe and
        # non-zip-safe setuptools egg.
        f = open(bin_tahoe_template, "rU")
        script_lines = f.readlines()
        f.close()
        script_lines[0] = "#!%s\n" % sys.executable
        tahoe_script = os.path.join("bin", "tahoe-script.py")
        f = open(tahoe_script, "w")
        for line in script_lines:
            f.write(line)
        f.close()
        if sys.platform == "win32":
            setuptools_egg = require("setuptools")[0].location
            if os.path.isfile(setuptools_egg):
                z = zipfile.ZipFile(setuptools_egg, 'r')
                for filename in z.namelist():
                    if 'cli.exe' in filename:
                        cli_exe = z.read(filename)
            else:
                cli_exe = os.path.join(setuptools_egg, 'setuptools', 'cli.exe')
            tahoe_exe = os.path.join("bin", "tahoe.exe")
            if os.path.isfile(setuptools_egg):
                f = open(tahoe_exe, 'wb')
                f.write(cli_exe)
                f.close()
            else:
                shutil.copy(cli_exe, tahoe_exe)
        else:
            try:
                os.remove(os.path.join('bin', 'tahoe'))
            except:
                # okay, probably it was already gone
                pass
            os.symlink('tahoe-script.py', os.path.join('bin', 'tahoe'))

        # chmod +x bin/tahoe-script.py
        old_mode = stat.S_IMODE(os.stat(tahoe_script)[stat.ST_MODE])
        new_mode = old_mode | (stat.S_IXUSR | stat.S_IRUSR |
                               stat.S_IXGRP | stat.S_IRGRP |
                               stat.S_IXOTH | stat.S_IROTH )
        os.chmod(tahoe_script, new_mode)

class MySdist(sdist.sdist):
    """ A hook in the sdist command so that we can determine whether this the
    tarball should be 'SUMO' or not, i.e. whether or not to include the
    external dependency tarballs. Note that we always include
    misc/dependencies/* in the tarball; --sumo controls whether tahoe-deps/*
    is included as well.
    """

    user_options = sdist.sdist.user_options + \
        [('sumo', 's',
          "create a 'sumo' sdist which includes the contents of tahoe-deps/*"),
         ]
    boolean_options = ['sumo']

    def initialize_options(self):
        sdist.sdist.initialize_options(self)
        self.sumo = False

    def make_distribution(self):
        # add our extra files to the list just before building the
        # tarball/zipfile. We override make_distribution() instead of run()
        # because setuptools.command.sdist.run() does not lend itself to
        # easy/robust subclassing (the code we need to add goes right smack
        # in the middle of a 12-line method). If this were the distutils
        # version, we'd override get_file_list().

        if self.sumo:
            # If '--sumo' was specified, include tahoe-deps/* in the sdist.
            # We assume that the user has fetched the tahoe-deps.tar.gz
            # tarball and unpacked it already.
            self.filelist.extend([os.path.join("tahoe-deps", fn)
                                  for fn in os.listdir("tahoe-deps")])
            # In addition, we want the tarball/zipfile to have -SUMO in the
            # name, and the unpacked directory to have -SUMO too. The easiest
            # way to do this is to patch self.distribution and override the
            # get_fullname() method. (an alternative is to modify
            # self.distribution.metadata.version, but that also affects the
            # contents of PKG-INFO).
            fullname = self.distribution.get_fullname()
            def get_fullname():
                return fullname + "-SUMO"
            self.distribution.get_fullname = get_fullname

        return sdist.sdist.make_distribution(self)


APPNAME='allmydata-tahoe'
APPNAMEFILE = os.path.join('src', 'allmydata', '_appname.py')
APPNAMEFILESTR = "__appname__ = '%s'" % (APPNAME,)
try:
    curappnamefilestr = open(APPNAMEFILE, 'rU').read()
except EnvironmentError:
    # No file, or unreadable or something, okay then let's try to write one.
    open(APPNAMEFILE, "w").write(APPNAMEFILESTR)
else:
    if curappnamefilestr.strip() != APPNAMEFILESTR:
        print "Error -- this setup.py file is configured with the 'application name' to be '%s', but there is already a file in place in '%s' which contains the contents '%s'.  If the file is wrong, please remove it and setup.py will regenerate it and write '%s' into it." % (APPNAME, APPNAMEFILE, curappnamefilestr, APPNAMEFILESTR)
        sys.exit(-1)

setup(name=APPNAME,
      version=version,
      description='secure, decentralized, fault-tolerant filesystem',
      long_description=LONG_DESCRIPTION,
      author='the allmydata.org Tahoe project',
      author_email='tahoe-dev@allmydata.org',
      url='http://allmydata.org/',
      license='GNU GPL',
      cmdclass={"show_supportlib": ShowSupportLib,
                "show_pythonpath": ShowPythonPath,
                "run_with_pythonpath": RunWithPythonPath,
                "check_auto_deps": CheckAutoDeps,
                "test_mac_diskimage": TestMacDiskImage,
                "make_executable": MakeExecutable,
                "sdist": MySdist,
                },
      package_dir = {'':'src'},
      packages=find_packages("src"),
      classifiers=trove_classifiers,
      test_suite="allmydata.test",
      include_package_data=True,
      entry_points = { 'console_scripts': [ 'tahoe = allmydata.scripts.runner:run' ] },
      zip_safe=False, # We prefer unzipped for easier access.
      )
