#include "Tag.h"
#include "TagSet.h"

using namespace std;

class FacetImpl
{
protected:
	int _ref;
	
	std::string _name;

	TagSet _tags;
	
public:
	FacetImpl(const std::string& name) throw ();

	/// Increment the reference count for this object
	void ref() throw () { ++_ref; }

	/// Decrement the reference count for this object, returning true when it
	/// reaches 0
	bool unref() throw () { return --_ref == 0; }

	bool hasTag(const std::string& name) const throw ();
	Tag obtainTag(const std::string& name) throw ();
	Tag getTag(const std::string& name) const throw ();

	friend class Facet;
	friend class Tag;
};

class TagImpl
{
protected:
	int _ref;
	
	FacetImpl* _facet;

	std::string _name;

public:
	TagImpl(FacetImpl* facet, const std::string& name) throw ();

	/// Increment the reference count for this object
	void ref() throw () { ++_ref; }

	/// Decrement the reference count for this object, returning true when it
	/// reaches 0
	bool unref() throw () { return --_ref == 0; }

	friend class Tag;
};

FacetImpl::FacetImpl(const std::string& name) throw ()
	: _name(name) {}
	
bool FacetImpl::hasTag(const std::string& name) const throw ()
{
	Tag sample(const_cast<FacetImpl*>(this), name);
	return _tags.find(sample) != _tags.end();
}

Tag FacetImpl::obtainTag(const std::string& name) throw ()
{
	Tag sample(this, name);
	TagSet::iterator i = _tags.find(sample);
	if (i == _tags.end())
	{
		_tags.insert(sample);
		return sample;
	} else
		return *i;
}

Tag FacetImpl::getTag(const std::string& name) const throw ()
{
	Tag sample(const_cast<FacetImpl*>(this), name);
	TagSet::iterator i = _tags.find(sample);
	if (i == _tags.end())
		return Tag();
	else
		return *i;
}


TagImpl::TagImpl(FacetImpl* facet, const std::string& name) throw ()
	: _facet(facet), _name(name) {}



Facet::Facet() throw ()
	: impl(0) {}

Facet::Facet(const std::string& name) throw ()
{
	impl = new FacetImpl(name);
	impl->ref();
}

Facet::Facet(FacetImpl* impl) throw ()
	: impl(impl)
{
	if (impl)
		impl->ref();
}

Facet::Facet(const Facet& ns) throw ()
{
	if (ns.impl)
		ns.impl->ref();
	impl = ns.impl;
}

Facet::~Facet() throw ()
{   
	if (impl && impl->unref())
		delete impl;
}

Facet& Facet::operator=(const Facet& ns) throw ()
{   
	if (ns.impl)
		ns.impl->ref();  // Do it early to correctly handle the case of x = x;
	if (impl && impl->unref())
		delete impl;
	impl = ns.impl;
	return *this;
}

bool Facet::operator==(const Facet& ns) const throw ()
{   
	return name() == ns.name();
}

bool Facet::operator<(const Facet& ns) const throw ()
{   
	return name() < ns.name();
}

Facet::operator bool() const throw ()
{
	return impl != 0;
}

const std::string& Facet::name() const throw () { return impl->_name; }

bool Facet::hasTag(const std::string& name) const throw ()
{
	return impl->hasTag(name);
}

Tag Facet::obtainTag(const std::string& name) throw ()
{
	return impl->obtainTag(name);
}

Tag Facet::getTag(const std::string& name) const throw ()
{
	return impl->getTag(name);
}


const TagSet& Facet::tags() const throw ()
{
	return impl->_tags;
}


	

Tag::Tag() throw ()
	: impl(0) {}

Tag::Tag(FacetImpl* facet, const std::string& name) throw ()
{
	impl = new TagImpl(facet, name);
	impl->ref();
}

Tag::Tag(const Tag& ns) throw ()
{
	if (ns.impl)
		ns.impl->ref();
	impl = ns.impl;
}

Tag::~Tag() throw ()
{   
	if (impl && impl->unref())
		delete impl;
}

Tag& Tag::operator=(const Tag& ns) throw ()
{   
	if (ns.impl)
		ns.impl->ref();  // Do it early to correctly handle the case of x = x;
	if (impl && impl->unref())
		delete impl;
	impl = ns.impl;
	return *this;
}

bool Tag::operator==(const Tag& ns) const throw ()
{   
	return impl->_facet == ns.impl->_facet && name() == ns.name();
}

bool Tag::operator<(const Tag& ns) const throw ()
{   
	return impl->_facet->_name < ns.impl->_facet->_name ||
		   (impl->_facet->_name == ns.impl->_facet->_name && name() < ns.name());
}

Tag::operator bool() const throw ()
{
	return impl != 0;
}

const std::string& Tag::name() const throw () { return impl->_name; }
std::string Tag::fullname() const throw ()
{
	if (impl->_facet->_name.empty())
		return impl->_name;
	else
		return impl->_facet->_name + "::" + impl->_name;
}

Facet Tag::facet() const throw ()
{
	return Facet(impl->_facet);
}

// vim:set ts=3 sw=3:
