/*-----------------------------------------------------------------------

                         SYRTHES version 4.1
                         -------------------

     This file is part of the SYRTHES Kernel, element of the
     thermal code SYRTHES.

     Copyright (C) 2009 EDF S.A., France

     contact: syrthes-support@edf.fr


     The SYRTHES Kernel is free software; you can redistribute it
     and/or modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2 of
     the License, or (at your option) any later version.

     The SYRTHES Kernel is distributed in the hope that it will be
     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.


     You should have received a copy of the GNU General Public License
     along with the SYRTHES Kernel; if not, write to the
     Free Software Foundation, Inc.,
     51 Franklin St, Fifth Floor,
     Boston, MA  02110-1301  USA

-----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "syr_usertype.h"
#include "syr_option.h"
#include "syr_tree.h"
#include "syr_bd.h"
#include "syr_const.h"
#include "syr_proto.h"

#ifdef _SYRTHES_MPI_
#include <mpi.h>
#endif

#ifdef _SYRTHES_CFD_
extern int nbCFD;
#endif

extern FILE *fflu;



/*|======================================================================|
  | SYRTHES 4.1                                       COPYRIGHT EDF 2009 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  |        Surfacic balance                                              |
  |======================================================================| */
void bilsurf(struct Maillage maillnodes, struct MaillageCL maillnodeus,  
	     struct Climcfd scoupf,struct Clim flux,struct Clim echang,
	     struct Contact rescon,struct Couple scoupr,struct Clim rayinf,
	     double *tmpsa,struct Bilan bilansurf,double tempss)
{
  int i,n;
  double bil1,bil2,bil3;
  double bil1tot,bil2tot,bil3tot;

  if (syrglob_nparts==1 || syrglob_rang==0){
    if (SYRTHES_LANG == FR)
      printf("\n ---Bilans de Flux surfaciques (W)---\n");
    else if (SYRTHES_LANG == EN)
      printf("\n ---Surface Flux Balance (W)---\n");
  }

  for (n=0;n<bilansurf.nb;n++)
    {
      bil1=bil2=bil3=0.;
#ifdef _SYRTHES_CFD_
      if (scoupf.exist)
	bil3 +=cfd_calbilsurf(n,scoupf.nelem,scoupf.nume,scoupf.tfluid,scoupf.hfluid,
			      maillnodes,maillnodeus,bilansurf,tmpsa);
#endif
      bil1 =calbilsurf(n,1,flux.nelem,flux.numf,flux.val1,flux.val1,
		       maillnodes,maillnodeus,bilansurf,tmpsa);
      bil1+=calbilsurf(n,2,echang.nelem,echang.numf,echang.val1,echang.val2,
		       maillnodes,maillnodeus,bilansurf,tmpsa);
      bil1+=calbilsurf(n,3,rayinf.nelem,rayinf.numf,rayinf.val1,rayinf.val2,
		       maillnodes,maillnodeus,bilansurf,tmpsa);
      bil2 =calbilsurf(n,3,scoupr.nelem,scoupr.numf,scoupr.t,scoupr.h,
		       maillnodes,maillnodeus,bilansurf,tmpsa);


      bil1tot=somme_double_parall(bil1);
      bil2tot=somme_double_parall(bil2);
      bil3tot=somme_double_parall(bil3);
      
      if (syrglob_nparts==1 || syrglob_rang==0) {
#ifdef _SYRTHES_CFD_
	if (scoupf.existglob){
	  if (SYRTHES_LANG == FR)
	    printf(" Temps= %14.8e Bilan %d * Cond_Lim= %12.5e  Radiatif= %12.5e  Convectif= %12.5e\n",
		   tempss,n+1,bil1tot,bil2tot,bil3tot);
	  else if (SYRTHES_LANG == EN)
	    printf(" Time= %14.8e Balance %d * Lim_Cond= %12.5e  Radiative= %12.5e  Convection= %12.5e\n",
		   tempss,n+1,bil1tot,bil2tot,bil3tot);
	  
	  fprintf(fflu,"SURF Time= %14.8e Balance %3d * Lim_Cond=    %12.5e  Radiative= %12.5e  Convection= %12.5e\n",
		  tempss,n+1,bil1tot,bil2tot,bil3tot);
	}
	else
	  {
	    if (SYRTHES_LANG == FR)
	      printf(" Temps= %14.8e Bilan %d * Cond_Lim= %12.5e  Radiatif= %12.5e\n",
		     tempss,n+1,bil1tot,bil2tot);
	    else if (SYRTHES_LANG == EN)
	      printf(" Time= %14.8e Balance %d * Lim_Cond= %12.5e  Radiative= %12.5e\n",
		     tempss,n+1,bil1tot,bil2tot);
	    
	    fprintf(fflu,"SURF Time= %14.8e Balance %3d * Lim_Cond=    %12.5e  Radiative= %12.5e\n",
		    tempss,n+1,bil1tot,bil2tot);
	  }
#else
	if (SYRTHES_LANG == FR)
	  printf(" Temps= %14.8e Bilan %d * Cond_Lim= %12.5e  Radiatif= %12.5e\n",
		 tempss,n+1,bil1tot,bil2tot);
	else if (SYRTHES_LANG == EN)
	  printf(" Time= %14.8e Balance %d * Lim_Cond= %12.5e  Radiative= %12.5e\n",
		 tempss,n+1,bil1tot,bil2tot);

	fprintf(fflu,"SURF Time= %14.8e Balance %3d * Lim_Cond=    %12.5e  Radiative= %12.5e\n",
		tempss,n+1,bil1tot,bil2tot);
#endif
	fflush(fflu);
      }
    }

}


/*|======================================================================|
  | SYRTHES 4.1                                       COPYRIGHT EDF 2009 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  |        Surfacic balance                                              |
  |======================================================================| */
double calbilsurf(int n,int type,int nelem,int *numf,double **v1,double **v2,
		  struct Maillage maillnodes,struct MaillageCL maillnodeus,
		  struct Bilan bilansurf,double *tmpsa)
{
  int i,j,nn,nf,nca,numref,ok,i1,i2,i3;
  double s,s12,s24,r1,r2,f1,f2,f3,phi,bil,hrayi[6];
  double spi3=Pi/3.;
  double s3=1./3.;

  if (maillnodes.iaxisy==1) nca=1; else nca=0;
  s24=1./24.;
  s12=1./12.;

  bil=0;
  for (i=0;i<nelem;i++)
    {
      nf=numf[i];
      numref=maillnodeus.nrefe[nf];
      ok=0;
      for (j=0;j<bilansurf.nbref[n];j++) if (numref==bilansurf.ref[n][j]) ok=1;
      if (ok)
	{
	  s=maillnodeus.volume[nf]; phi=0;
	  if (maillnodes.ndim==2 && maillnodes.iaxisy==0)
	    {
	      i1=maillnodeus.node[0][nf]; i2=maillnodeus.node[1][nf]; 
	      switch (type)
		{
		case 1:
		  f1=v1[0][i];
		  f2=v1[1][i];
		  break;
		case 2:
		  f1=v2[0][i]*(v1[0][i]-tmpsa[i1]);
		  f2=v2[1][i]*(v1[1][i]-tmpsa[i2]);
		  break;
		case 3:
		  for (j=0;j<2;j++)
		    {
		      nn=maillnodeus.node[j][nf];
		      hrayi[j] = v2[j][i]*sigma*(tmpsa[nn]+v1[j][i]+2*tkel)*
		                 ((tmpsa[nn]+tkel)*(tmpsa[nn]+tkel) + 
		                 (v1[j][i]+tkel)*(v1[j][i]+tkel) );
		    }
		  f1=hrayi[0]*(v1[0][i]-tmpsa[i1]);
		  f2=hrayi[1]*(v1[1][i]-tmpsa[i2]);
		  break;
		}
	      bil+=(f1+f2)*0.5*s;
	    }
	  else if (maillnodes.ndim==2 && maillnodes.iaxisy!=0)
	    {
	      r1=fabs(maillnodes.coord[nca][maillnodeus.node[0][nf]]);
	      r2=fabs(maillnodes.coord[nca][maillnodeus.node[1][nf]]);
	      i1=maillnodeus.node[0][nf]; i2=maillnodeus.node[1][nf]; 
	      switch (type)
		{
		case 1:
		  f1=v1[0][i];
		  f2=v1[1][i];
		  break;
		case 2:
		  f1=v2[0][i]*(v1[0][i]-tmpsa[i1]);
		  f2=v2[1][i]*(v1[1][i]-tmpsa[i2]);
		  break;
		case 3:
		  for (j=0;j<2;j++)
		    {
		      nn=maillnodeus.node[j][nf];
		      hrayi[j] = v2[j][i]*sigma*(tmpsa[nn]+v1[j][i]+2*tkel)*
		                 ((tmpsa[nn]+tkel)*(tmpsa[nn]+tkel) + 
		                 (v1[j][i]+tkel)*(v1[j][i]+tkel) );
		    }
		  f1=hrayi[0]*(v1[0][i]-tmpsa[i1]);
		  f2=hrayi[1]*(v1[1][i]-tmpsa[i2]);
		  break;
		}
	      phi=f1*(2*r1+r2) + f2*(r1+2*r2);
	      bil+=phi*spi3*s;
	    }
	  else
	    {
	      i1=maillnodeus.node[0][nf]; i2=maillnodeus.node[1][nf]; 
	      i3=maillnodeus.node[2][nf];
	      switch (type)
		{
		case 1:
		  f1=v1[0][i];
		  f2=v1[1][i];
		  f3=v1[2][i];
		  break;
		case 2:
		  f1=v2[0][i]*(v1[0][i]-tmpsa[i1]);
		  f2=v2[1][i]*(v1[1][i]-tmpsa[i2]);
		  f3=v2[2][i]*(v1[2][i]-tmpsa[i3]);
		  break;
		case 3:
		  for (j=0;j<3;j++)
		    {
		      nn=maillnodeus.node[j][nf];
		      hrayi[j] = v2[j][i]*sigma*(tmpsa[nn]+v1[j][i]+2*tkel)*
		                 ((tmpsa[nn]+tkel)*(tmpsa[nn]+tkel) + 
		                 (v1[j][i]+tkel)*(v1[j][i]+tkel) );
		    }
		  f1=hrayi[0]*(v1[0][i]-tmpsa[i1]);
		  f2=hrayi[1]*(v1[1][i]-tmpsa[i2]);
		  f3=hrayi[2]*(v1[2][i]-tmpsa[i3]);
		  break;
		}
	      phi=f1+f2+f3;
	      bil+=phi*s3*s;
	    }
	}
    }
  
  return bil;
}


#ifdef _SYRTHES_CFD_
/*|======================================================================|
  | SYRTHES 4.1                                       COPYRIGHT EDF 2009 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  |        Bilan surfacique                                              |
  |======================================================================| */
double cfd_calbilsurf(int n,int *nelem,int **numf,double **tf,double **hf,
		      struct Maillage maillnodes,struct MaillageCL maillnodeus,
		      struct Bilan bilansurf,double *tmpsa)
{
  int i,j,nn,nf,nca,numref,ok,i1,i2,i3,nb;
  double s,r1,r2,f1,f2,f3,phi,bil;
  double spi3=Pi/3.;
  double s3=1./3.;

  if (maillnodes.iaxisy==1) nca=1; else nca=0;

  bil=0;

  for (nb=0;nb<nbCFD;nb++)
    for (i=0;i<nelem[nb];i++)
    {
      nf=numf[nb][i];
      numref=maillnodeus.nrefe[nf];
      ok=0;
      for (j=0;j<bilansurf.nbref[n];j++) if (numref==bilansurf.ref[n][j]) ok=1;
      if (ok)
	{
	  s=maillnodeus.volume[nf]; phi=0;
	  if (maillnodes.ndim==2 && maillnodes.iaxisy==0)
	    {
	      i1=maillnodeus.node[0][nf]; i2=maillnodeus.node[1][nf]; 
	      f1=hf[nb][i]*(tf[nb][i]-tmpsa[i1]);
	      f2=hf[nb][i]*(tf[nb][i]-tmpsa[i2]);
	      bil+=(f1+f2)*0.5*s;
	    }
	  else if (maillnodes.ndim==2 && maillnodes.iaxisy!=0)
	    {
	      r1=fabs(maillnodes.coord[nca][maillnodeus.node[0][nf]]);
	      r2=fabs(maillnodes.coord[nca][maillnodeus.node[1][nf]]);
	      i1=maillnodeus.node[0][nf]; i2=maillnodeus.node[1][nf]; 
	      f1=hf[nb][i]*(tf[nb][i]-tmpsa[i1]);
	      f2=hf[nb][i]*(tf[nb][i]-tmpsa[i2]);
	      phi=f1*(2*r1+r2) + f2*(r1+2*r2);
	      bil+=phi*spi3*s;
	    }
	  else
	    {
	      i1=maillnodeus.node[0][nf]; i2=maillnodeus.node[1][nf]; 
	      i3=maillnodeus.node[2][nf];
	      f1=hf[nb][i]*(tf[nb][i]-tmpsa[i1]);
	      f2=hf[nb][i]*(tf[nb][i]-tmpsa[i2]);
	      f3=hf[nb][i]*(tf[nb][i]-tmpsa[i3]);
	      phi=f1+f2+f3;
	      bil+=phi*s3*s;
	    }
	}
    }
  
  return bil;
}
#endif


#ifdef _SYRTHES_CFD_
/*|======================================================================|
  | SYRTHES 4.1                                       COPYRIGHT EDF 2009 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  |        Surface balance (contribution of CFD codes)                   |
  |======================================================================| */
double cfd_bilsurf_conserv(int nelem,int *numf,double *tf,double *hf,
			   struct Maillage maillnodes,struct MaillageCL maillnodeus,
			   double *tmps)
{
  int i,j,nn,nf,nca,numref,ok,i1,i2,i3,nb;
  double s,r1,r2,f1,f2,f3,phi,bil,biltot;
  double spi3=Pi/3.;
  double s3=1./3.;

  if (maillnodes.iaxisy==1) nca=1; else nca=0;

  bil=0;


  for (i=0;i<nelem;i++)
    {
      nf=numf[i];
      numref=maillnodeus.nrefe[nf];
      if (maillnodeus.type[nf][POSCOUPF])
	{
	  s=maillnodeus.volume[nf]; phi=0;
	  if (maillnodes.ndim==2 && maillnodes.iaxisy==0)
	    {
	      i1=maillnodeus.node[0][nf]; i2=maillnodeus.node[1][nf]; 
	      f1=hf[i]*(tf[i]-tmps[i1]);
	      f2=hf[i]*(tf[i]-tmps[i2]);
	      bil+=(f1+f2)*0.5*s;
	    }
	  else if (maillnodes.ndim==2 && maillnodes.iaxisy!=0)
	    {
	      r1=fabs(maillnodes.coord[nca][maillnodeus.node[0][nf]]);
	      r2=fabs(maillnodes.coord[nca][maillnodeus.node[1][nf]]);
	      i1=maillnodeus.node[0][nf]; i2=maillnodeus.node[1][nf]; 
	      f1=hf[i]*(tf[i]-tmps[i1]);
	      f2=hf[i]*(tf[i]-tmps[i2]);
	      phi=f1*(2*r1+r2) + f2*(r1+2*r2);
	      bil+=phi*spi3*s;
	    }
	  else
	    {
	      i1=maillnodeus.node[0][nf]; i2=maillnodeus.node[1][nf]; 
	      i3=maillnodeus.node[2][nf];
	      f1=hf[i]*(tf[i]-tmps[i1]);
	      f2=hf[i]*(tf[i]-tmps[i2]);
	      f3=hf[i]*(tf[i]-tmps[i3]);
	      phi=f1+f2+f3;
	      bil+=phi*s3*s;
	    }
	}
    }
  
  biltot=bil;
  biltot=somme_double_parall(bil);

  return biltot;
}
#endif

/*|======================================================================|
  | SYRTHES 4.1                                       COPYRIGHT EDF 2009 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  |        Volumetric balance                                            |
  |======================================================================| */
void bilvol(struct Maillage maillnodes,struct Climcfd scouvf,struct Cvol fluxvol,
	    double *tmpsa,struct Bilan bilanvol,double tempss)
{
  int n;
  double bil1,bil2,bil1tot,bil2tot;


  if (syrglob_nparts==1 || syrglob_rang==0){
    if (SYRTHES_LANG == FR)
      printf("\n ---Bilans de Flux volumiques (W)---\n");
    else if (SYRTHES_LANG == EN)
      printf("\n ---Volume Heat Balance (W)---\n");
  }

  for (n=0;n<bilanvol.nb;n++)
    {
      
      bil1=0;bil2=0;

#ifdef _SYRTHES_CFD_
      if (scouvf.exist)
	bil2 +=cfd_calbilvol(n,scouvf.nelem,scouvf.nume,scouvf.tfluid,scouvf.hfluid,
			     maillnodes,bilanvol,tmpsa);
#endif
      bil1 =calbilvol(n,fluxvol.nelem,fluxvol.nume,fluxvol.val1,fluxvol.val2,
		       maillnodes,bilanvol,tmpsa);

   
      bil1tot=somme_double_parall(bil1);
      bil2tot=somme_double_parall(bil2);

      
      if (syrglob_nparts==1 || syrglob_rang==0)
	{
#ifdef _SYRTHES_CFD_
	  if (scouvf.existglob){
	    if (SYRTHES_LANG == FR )
	      printf(" Temps= %14.8e Bilan %d * flux_volumique= %12.5e * flux_volumique_cfd= %12.5e\n",tempss,n+1,bil1tot,bil2tot);
	    else if (SYRTHES_LANG == EN)
	      printf(" Time= %14.8e Balance %d * Volume_Flux= %12.5e * flux_volumique_cfd= %12.5e\n",tempss,n+1,bil1tot,bil2tot);
	  
	    fprintf(fflu,"VOL  Time= %14.8e Balance %3d * Volume_Flux= %12.5e* flux_volumique_cfd= %12.5e \n",tempss,n+1,bil1tot,bil2tot);
	  }
	  else
	    {
	      if (SYRTHES_LANG == FR)
		printf(" Temps= %14.8e Bilan %d * flux_volumique= %12.5e\n",tempss,n+1,bil1tot);
	      else if (SYRTHES_LANG == EN)
		printf(" Time= %14.8e Balance %d * Volume_Flux= %12.5e\n",tempss,n+1,bil1tot);
	      
	      fprintf(fflu,"VOL  Time= %14.8e Balance %3d * Volume_Flux= %12.5e\n",tempss,n+1,bil1tot);
	    }
#else
	  if (SYRTHES_LANG == FR)
	    printf(" Temps= %14.8e Bilan %d * flux_volumique= %12.5e\n",tempss,n+1,bil1tot);
	  else if (SYRTHES_LANG == EN)
	    printf(" Time= %14.8e Balance %d * Volume_Flux= %12.5e\n",tempss,n+1,bil1tot);
	  
	  fprintf(fflu,"VOL  Time= %14.8e Balance %3d * Volume_Flux= %12.5e\n",tempss,n+1,bil1tot);
#endif

	  fflush(fflu);
	}
    }

}


#ifdef _SYRTHES_CFD_
/*|======================================================================|
  | SYRTHES 4.1                                       COPYRIGHT EDF 2009 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  |        Surfacic balance (contribution of CFD codes)                  |
  |======================================================================| */
double cfd_calbilvol(int n,int *nelem,int **nume,double **tf,double **hf,
		     struct Maillage maillnodes,
		     struct Bilan bilanvol,double *tmpsa)
{
  int i,j,ne,nca,numref,ok,nb;
  double s,r1,r2,r3,bil;
  double s12f2pi;

  if (maillnodes.iaxisy==1) nca=1; else nca=0;
  s12f2pi=2.*Pi/3.;

  bil=0;

  for (nb=0;nb<nbCFD;nb++)
    for (i=0;i<nelem[nb];i++)
      {
	ne=nume[nb][i];
	numref=maillnodes.nrefe[ne];
	ok=0;
	for (j=0;j<bilanvol.nbref[n];j++) if (numref==bilanvol.ref[n][j]) ok=1;
	if (ok)
	  {
	    if (maillnodes.ndim==2 && maillnodes.iaxisy==0)
	      {
		bil+=tf[nb][i]*hf[nb][i]*maillnodes.volume[ne];
	      }
	    else if (maillnodes.ndim==2)
	      {
		r1=fabs(maillnodes.coord[nca][maillnodes.node[0][ne]]);
		r2=fabs(maillnodes.coord[nca][maillnodes.node[1][ne]]);
		r3=fabs(maillnodes.coord[nca][maillnodes.node[2][ne]]);
		bil+=tf[nb][i]*hf[nb][i]*(r1+r2+r3)*maillnodes.volume[ne]*s12f2pi;
	      }
	    else if (maillnodes.ndim==3)
	      {
		bil+=tf[nb][i]*hf[nb][i]*maillnodes.volume[ne];
	      }
	  }
      }
  return bil;
}
#endif







/*|======================================================================|
  | SYRTHES 4.1                                       COPYRIGHT EDF 2009 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  |        Volumetric balance                                            |
  |======================================================================| */
double calbilvol(int n,int nelem,int *nume,double *v1,double *v2,
		  struct Maillage maillnodes,
		  struct Bilan bilanvol,double *tmpsa)
{
  int i,j,nf,nca,numref,ok,nrefe;
  double bil,s12f2pi,r1,r2,r3;
  double x,y,z,tmoy;

  if (maillnodes.iaxisy==1) nca=1; else nca=0;
  s12f2pi=2.*Pi/12.;

  bil=0;
  for (i=0;i<nelem;i++)
    {
      nf=nume[i];
      numref=maillnodes.nrefe[nf];
      ok=0;
      for (j=0;j<bilanvol.nbref[n];j++) if (numref==bilanvol.ref[n][j]) ok=1;
      if (ok)
	{
	  data_element_moy(nf,maillnodes,tmpsa,&nrefe,&x,&y,&z,&tmoy);
  
	  if (maillnodes.ndim==2 && maillnodes.iaxisy==0)
	    {
	      bil+= (v1[i]*tmoy+v2[i]) *maillnodes.volume[nf];
	    }
	  else if (maillnodes.ndim==2)
	    {
	      r1=fabs(maillnodes.coord[nca][maillnodes.node[0][nf]]);
	      r2=fabs(maillnodes.coord[nca][maillnodes.node[1][nf]]);
	      r3=fabs(maillnodes.coord[nca][maillnodes.node[2][nf]]);
	      bil+= ((v1[i]*tmpsa[maillnodes.node[0][nf]]+v2[i]) *(2*r1+r2+r3)+
		      (v1[i]*tmpsa[maillnodes.node[1][nf]]+v2[i])*(r1+2*r2+r3)+
		     (v1[i]*tmpsa[maillnodes.node[2][nf]]+v2[i])*(r1+r2+2*r3))*
		      maillnodes.volume[nf]*s12f2pi;

	    }
	  else if (maillnodes.ndim==3)
	    {
	      bil+= (v1[i]*tmoy+v2[i]) *maillnodes.volume[nf];
	    }
	}
    }
  
  return bil;

}

/*|======================================================================|
  | SYRTHES 4.1                                       COPYRIGHT EDF 2009 |
  |======================================================================|
  | AUTEURS  : C. PENIGUEL, I. RUPP                                      |
  |======================================================================|
  |        Bilan solaire                                                 |
  |======================================================================| */
void bilsolaire(struct Maillage maillnodray,
		struct Soleil soleil,struct Bilan bilansolaire,
		struct Bande bandespec,double *fdfnp1,double tempss)
{
  int i,j,n,nn,nb,ok,numref;
  double *bilf,*bilr,*bilftot,*bilrtot;

  if (SYRTHES_LANG == FR)
    printf("\nBilans de Flux solaires (W) :\n");
  else if (SYRTHES_LANG == EN)
    printf("\nSolar Flux Balance (W) :\n");


  bilf=(double*)malloc(bandespec.nb*sizeof(double));
  bilr=(double*)malloc(bandespec.nb*sizeof(double));
  verif_alloue_double1d("bilsolaire",bilf);
  verif_alloue_double1d("bilsolaire",bilr);

  printf("---------------------------------------------------------------------------------\n");
  for (nn=0;nn<bilansolaire.nb;nn++)
    {
      for (nb=0;nb<bandespec.nb;nb++) bilf[nb]=bilr[nb]=0.;

      for (i=0;i<maillnodray.nelem;i++)
	{
	  numref=maillnodray.nrefe[i];
	  ok=0;
	  for (j=0;j<bilansolaire.nbref[nn];j++) if (numref==bilansolaire.ref[nn][j]) ok=1;
	  if (ok) 
	    for (nb=0;nb<bandespec.nb;nb++)
	      {
		bilf[nb]+=soleil.diffus[nb][i]*fdfnp1[i];
		bilr[nb]+=soleil.direct[nb][i]*maillnodray.volume[i];
	      }
	}

      if (syrglob_npartsray==1){
	bilftot=bilf;
	bilrtot=bilr;
      }

#ifdef _SYRTHES_MPI_
      else
	{
	  bilftot=(double*)malloc(bandespec.nb*sizeof(double));
	  bilrtot=(double*)malloc(bandespec.nb*sizeof(double));
	  for (nb=1;nb<bandespec.nb;nb++) {
	    MPI_Allreduce(bilf+nb,bilftot+nb,1,MPI_DOUBLE,MPI_SUM,syrglob_comm_world);
	    MPI_Allreduce(bilr+nb,bilrtot+nb,1,MPI_DOUBLE,MPI_SUM,syrglob_comm_world);
	  }
	}
#endif

      if (syrglob_nparts==1 || syrglob_rang==0)
	{
	  if (SYRTHES_LANG == FR)
	    {
	      for (nb=0;nb<bandespec.nb;nb++) {
		printf("Time= %14.8e Bilan %d  bande %d : direct=%12.5e diffu=%12.5e Total=%12.5e \n",
		       tempss,nn+1,nb+1,bilrtot[nb],bilftot[nb],bilftot[nb]+bilrtot[nb]);
		fprintf(fflu,"SOL  Time= %14.8e Bilan= %3d  bande= %d  direct= %12.5e diffu= %12.5e Total= %12.5e\n",
			tempss,nn+1,nb+1,bilrtot[nb],bilftot[nb],bilftot[nb]+bilrtot[nb]);
		fflush(fflu);
	      }
	    }
	  else if (SYRTHES_LANG == EN)
	    {
	      for (nb=0;nb<bandespec.nb;nb++) {
		printf("Time= %14.8e Balance %d  Spectral range %d : direct=%12.5e diffu=%12.5e Total=%12.5e\n",
		       tempss,nn+1,nb+1,bilrtot[nb],bilftot[nb],bilftot[nb]+bilrtot[nb]);
		fprintf(fflu,"SOL  Time= %14.8e Balance %d  Spectral_range %d  direct= %12.5e diffu= %12.5e Total= %12.5e\n",
			tempss,nn+1,nb+1,bilrtot[nb],bilftot[nb],bilftot[nb]+bilrtot[nb]);
		fflush(fflu);
	      }
	    }
	}
    }
  
  printf("---------------------------------------------------------------------------------\n");
  printf("\n");

  free(bilf);  free(bilr);  
  if (syrglob_npartsray>1){free(bilftot);  free(bilrtot);}
}

