#!/usr/bin/env python

__authors__ = "Martin Sandve Alnes"
__date__ = "2008-09-04 -- 2008-09-04"

import unittest
import os, sys, glob, shutil, commands

import ufl

from ufl import FiniteElement
from ufl import VectorElement
from ufl import TensorElement
from ufl import MixedElement

from ufl import BasisFunction
from ufl import TestFunction
from ufl import TrialFunction

from ufl import Function
from ufl import Constant

from ufl import dx, ds

import SyFi
import sfc as sfc

from cell_assembly import assemble_on_cell, cell2volume

import instant
instant.set_logging_level("warning")

def num_integrals(form):
    return (form.num_cell_integrals(), form.num_exterior_facet_integrals(), form.num_interior_facet_integrals())


_test_temp_dir = "volume_temp_dir"
_done_test_temp_dir = "done_volume_temp_dir"
class ElementIndexingTest(unittest.TestCase):
    def __init__(self, *args, **kwargs):
        unittest.TestCase.__init__(self, *args, **kwargs)
        shutil.rmtree(_done_test_temp_dir, ignore_errors=True)
        os.mkdir(_done_test_temp_dir)
    
    def setUp(self):
        #print "Running sfc jit test in testdir"
        #print "Imported SyFi from location", SyFi.__file__
        #print "Imported sfc  from location", sfc.__file__
        self.options = sfc.default_options()
        self.options.compilation.cache_dir = os.path.abspath("test_cache")
        # Generate code in a clean directory:
        shutil.rmtree(_test_temp_dir, ignore_errors=True)
        os.mkdir(_test_temp_dir)
        os.chdir(_test_temp_dir)
    
    def tearDown(self):
        dirs = glob.glob("*")
        os.chdir("..")
        for d in dirs:
            os.rename(os.path.join(_test_temp_dir, d), os.path.join(_done_test_temp_dir, d))
    
    def testSetup(self):
        pass

    def testScalarBasisFunction(self):
        polygon = "triangle"
        degree = 1
        element = FiniteElement("CG", polygon, degree)
        v = BasisFunction(element)
        a = v*dx
        form = sfc.jit(a, options = self.options)
        # Test form properties
        self.assertTrue(form.rank() == 1)
        self.assertTrue(form.num_coefficients() == 0)
        self.assertTrue(num_integrals(form) == (1,0,0))
        # TODO: Test values
        #const = 1.23
        #A = assemble_on_cell(form, polygon, coeffs=[const])
        #self.assertAlmostEqual(A, const*cell2volume[polygon])

    def testScalarFunction(self):
        polygon = "triangle"
        degree = 1
        element = FiniteElement("CG", polygon, degree)
        f = Function(element)
        a = f*dx
        form = sfc.jit(a, options = self.options)
        # Test form properties
        self.assertTrue(form.rank() == 0)
        self.assertTrue(form.num_coefficients() == 1)
        self.assertTrue(num_integrals(form) == (1,0,0))
        # Test values
        const = 1.23
        A = assemble_on_cell(form, polygon, coeffs=[const])
        self.assertAlmostEqual(A, const*cell2volume[polygon])

    def testVectorBasisFunction(self):
        polygon = "triangle"
        degree = 1
        element = VectorElement("CG", polygon, degree)
        v = BasisFunction(element)
        a = v[0]*dx
        form = sfc.jit(a, options = self.options)
        # Test form properties
        self.assertTrue(form.rank() == 1)
        self.assertTrue(form.num_coefficients() == 0)
        self.assertTrue(num_integrals(form) == (1,0,0))
        # TODO: Test values
        #const = ("1.23", "4.56")
        #A = assemble_on_cell(form, polygon, coeffs=[const])
        #self.assertAlmostEqual(A, (float(const[0]) + float(const[1]))*cell2volume[polygon])

    def testVectorFunction(self):
        polygon = "triangle"
        degree = 1
        element = VectorElement("CG", polygon, degree)
        f = Function(element)
        a = (f[0] + f[1])*dx
        form = sfc.jit(a, options = self.options)
        # Test form properties
        self.assertTrue(form.rank() == 0)
        self.assertTrue(form.num_coefficients() == 1)
        self.assertTrue(num_integrals(form) == (1,0,0))
        # Test values
        const = ("1.23", "4.56")
        A = assemble_on_cell(form, polygon, coeffs=[const])
        self.assertAlmostEqual(A, (float(const[0]) + float(const[1]))*cell2volume[polygon])


tests = [ElementIndexingTest]

if __name__ == "__main__":
    unittest.main()

