/*==================================================================
 * help.c - User interface help routines
 *
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 *==================================================================*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>

#include <libswami/SwamiConfig.h>

#include "SwamiUIObject.h"

#include "help.h"
#include "glade_interface.h"
#include "splash.h"
#include "i18n.h"
#include "util.h"

static void swamiui_help_swamitips_set_tip (GtkWidget * tips, gint tipnum);

static gchar *swamitips_msg[] = {
  N_("Welcome to Swami!\n\n"
    "Most operations are performed by Right clicking on the sound font tree."
    " The type of item clicked on determines the options that are available."),
  N_("To zoom in the sample viewer:\n"
    "SHIFT Left click and drag the mouse to the left or right in the sample"
    " viewer. A vertical line will appear to mark the position to zoom into,"
    " and the distance from the marker determines how fast the zoom is"
    " performed. Moving the mouse to the opposite side of"
    " the zoom marker will unzoom. Using the Right mouse button will have the"
    " opposite effect."),
  N_("To select a range of audio data in the sample viewer:\n"
    "Left click and drag to mark an area. An existing selection can be"
    " changed by holding down CTRL and Left or Right clicking to change the"
    " left or right ends of the selection respectively."),
  N_("To select multiple items in the sound font tree:\n"
    "hold down CTRL and/or SHIFT on the keyboard and click on multiple items."),
  N_("To add samples to instruments:\n"
    "select the samples and/or instrument zones you want to add and then Right"
    " click on the instrument you would like to add to and select"
    " \"Paste\". If an instrument zone was selected all its parameters"
    " will be copied into the newly created zone. The same procedure is used"
    " to add instruments to presets."),
  N_("To load samples from raw data files:\n"
    "Right click on an item under the Samples branch of a sound font in the"
    " tree and select \"New Sample\". In the sample load file dialog check"
    " the \"Load as RAW sample\" option. You will be prompted for sample"
    " format options. Fun to load non audio data, but keep that volume"
    " control down!"),
  N_("Items in the sound font tree (Presets, Instruments, Samples and Zones)"
    " can be pasted between two separate sound fonts or within the same"
    " sound font. Pasted items that don't make sense are ignored. If a"
    " duplicate item is found a dialog is displayed to allow you to"
    " Modify the new item or Keep the old one."),
  N_("The \"Undo History\" dialog can be used to \"Jump\" to arbitrary points"
    " in the undo history, browse \"Back\" and \"Forward\" through the last"
    " 10 positions and select alternate undo branches. The latter option"
    " enables you to navigate Swami's flexible tree based undo system,"
    " which won't lose any state information. Even after Undo-ing an action"
    " and performing another, you can still redo the first action. A new"
    " branch is created in the undo tree, making it easy to try different"
    " settings."),
  N_("No more tips!")
};

#define TIPCOUNT	(sizeof(swamitips_msg) / sizeof(swamitips_msg[0]))

static gint swamitip_current;	/* current swami tip # */

void
swamiui_help_about (void)
{
  GtkWidget *boutwin;
  GtkWidget *widg;

  if (swamiui_util_activate_unique_dialog ("about", 0)) return;

  boutwin = create_aboutwin ();

  swamiui_util_register_unique_dialog (boutwin, "about", 0);

  widg = gtk_object_get_data (GTK_OBJECT (boutwin), "LBLversion");
  gtk_label_set_text (GTK_LABEL (widg), VERSION);

/*
  widg = gtk_object_get_data (GTK_OBJECT (boutwin), "LBLcompops");
  gtk_label_set_text (GTK_LABEL (widg), COMPILE_OPTIONS);
*/

  /* if no splash image then hide Show Splash button */
#ifndef SPLASH
  widg = gtk_object_get_data (GTK_OBJECT (boutwin), "BTNsplash");
  gtk_widget_hide (widg);
#endif

  gtk_widget_show (boutwin);
}

void
swamiui_help_cb_show_splash (void)
{
  swamiui_splash_display (FALSE); /* Display splash with no timeout */
}

/* Create swami tips dialog and load it with current tip */
void
swamiui_help_swamitips_create (void)
{
  GtkWidget *tips;
  GtkWidget *widg;
  int tipnum, i;

  if (swamiui_util_activate_unique_dialog ("tips", 0)) return;

  tips = create_swamitips ();

  swamiui_util_register_unique_dialog (tips, "tips", 0);

  /* update check button to state of Tips Enabled on startup config var */
  i = swami_config_get_int ("gui", "tips", NULL);
  if (i)
    {
      widg = gtk_object_get_data (GTK_OBJECT (tips), "CHKagain");
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), TRUE);
    }

  widg = gtk_object_get_data (GTK_OBJECT (tips), "TXTview");
  gtk_text_set_word_wrap (GTK_TEXT (widg), TRUE);

  /* get current position in tips from config var */
  tipnum = swami_config_get_int ("gui_state", "tips_position", NULL);

  swamiui_help_swamitips_set_tip (tips, tipnum);

  gtk_widget_show (tips);
}

static void
swamiui_help_swamitips_set_tip (GtkWidget * tips, gint tipnum)
{
  GtkWidget *txtview;
  GtkWidget *btn;
  gchar *msg;
  gint pos;

  tipnum = CLAMP (tipnum, 0, TIPCOUNT - 1);

  btn = gtk_object_get_data (GTK_OBJECT (tips), "BTNprev");
  gtk_widget_set_sensitive (btn, (tipnum != 0));

  btn = gtk_object_get_data (GTK_OBJECT (tips), "BTNnext");
  gtk_widget_set_sensitive (btn, (tipnum != TIPCOUNT - 1));

  txtview = gtk_object_get_data (GTK_OBJECT (tips), "TXTview");

  /* delete all text in view */
  gtk_editable_delete_text (GTK_EDITABLE (txtview), 0, -1);

  msg = _(swamitips_msg[tipnum]);	/* get the tip */
  pos = 0;			/* add the new tip at position 0 */
  gtk_editable_insert_text (GTK_EDITABLE (txtview), msg, strlen (msg), &pos);

  swamitip_current = tipnum;
}

/* next tip callback */
void
swamiui_help_cb_swamitips_next (GtkWidget * btn, GtkWidget * tips)
{
  swamiui_help_swamitips_set_tip (tips, swamitip_current + 1);
}

/* previous tip callback */
void
swamiui_help_cb_swamitips_previous (GtkWidget * btn, GtkWidget * tips)
{
  swamiui_help_swamitips_set_tip (tips, swamitip_current - 1);
}

/* save tip position and "tips on startup" setting to config vars */
gboolean
swamiui_help_cb_swamitips_destroy (GtkWidget * tips)
{
  GtkWidget *btn;
  int i;

  i = swamitip_current + 1;
  if (i >= TIPCOUNT) i = TIPCOUNT;

  swami_config_set_int ("gui_state", "tips_position", i);

  btn = gtk_object_get_data (GTK_OBJECT (tips), "CHKagain");
  i = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (btn));

  swami_config_set_int ("gui", "tips", i);

  return (FALSE);
}
