/*
 *  Copyright (C) 2008 Nicolas Vion <nico@picapo.net>
 *
 *   This file is part of swac-get.
 *
 *   Swac-get is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   Swac-get is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with swac-get; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string>
#include <list>
#include <vector>
#include <iostream>

#include <libxml/uri.h>

#include "http.hh"
#include "posix.hh"
#include "basic.hh"
#include "global.hh"

#include "swac.hh"

#define IS_HTTP(str) (str.substr(0, 7) == "http://")
#define IS_FILE(str) (str.substr(0, 7) == "file://")
#define IS_DIR(str) (str.substr(str.length() - 1, 1)) == "/")


void Tswac::package_deselect_idx(std::string idx) {
	db.exec("DELETE FROM `sounds` WHERE `packages_idx` = '" + idx + "';");
	db.exec("DELETE FROM `alphaidx` WHERE `packages_idx` = '" + idx + "';");
	db.exec("DELETE FROM `packages` WHERE `idx` = '" + idx + "';");
}

bool Tswac::package_delete(std::string url) {
	SqlQueryResult result;
	db.exec("SELECT idx FROM `packages` WHERE `path` = " + db.str(url) + ";", result); 
	for (unsigned int i = 0; i < result.size(); i++)
		package_deselect_idx(result.at(i).at(0));
	return (db.changes() != 0);
}

void Tswac::package_deselect(std::string url) {
	if (IS_HTTP(url) or IS_FILE(url)) {
		print_action(std::string(_("DESELECT PACKAGE")) + " " + url);
		print_result(package_delete(url));
	}
	else 
		package_deselect_id(url);
}

void Tswac::package_select_url(std::string url) {
	print_action(std::string(_("LOAD PACKAGE")) + " " + url);

	package_delete(url);

	if (!Tswac_package::load("packages", url)) {
		print_result(false);
		return;
	}

	if (IS_FILE(url))
		print_result(read_file(db.lastInsertRowId(), url.substr(7, url.size() - 7) + "index.xml.bz2"));
	else if (IS_HTTP(url))
		print_result(read_url(db.lastInsertRowId(), url + "index.xml.bz2"));
	else 
		print_result(false);
}

void Tswac::package_select(std::string str) {
	if (IS_FILE(str) or IS_HTTP(str))
		package_select_url(str);
	else
		package_select_id(str);
}

std::string Tswac::package_get_available_version(std::string url) {
	SqlQueryResult result;
	db.exec("SELECT version FROM `packages_available` WHERE `path`=" + db.str(url) + ";", result);
	return (result.size() == 1) ? result.at(0).at(0) : "";
}

void Tswac::source_select(std::string url) {
	if (!((IS_FILE(url) or IS_HTTP(url)) and IS_DIR(url)) {
		print_error(_("repositroy adress must begin by \"file://\" or \"http://\" and finish by \"/\""));
		return;
	}

	print_action(std::string(_("SELECT SOURCE")) + " \"" + url + "\"");

	db.exec("DELETE FROM `sources` WHERE `path`=" + db.str(url) + ";");
	db.exec("INSERT INTO `sources` (path, selected) VALUES (" + db.str(url) + ", '1');");

	print_result(true);
}

void Tswac::source_deselect(std::string url) {
	print_action(std::string(_("DESELECT SOURCE")) + " \"" + url + "\"");

	db.exec("UPDATE `sources` SET `selected`='0' WHERE `path`=" + db.str(url) + ";");
	print_result(db.changes() != 0);
}

void Tswac::source_delete(std::string url) {
	print_action(std::string(_("DELETE SOURCE")) + " \"" + url + "\"");

	db.exec("DELETE FROM `sources` WHERE `path`=" + db.str(url) + ";");
	print_result(db.changes() != 0);

}

void Tswac::source_load(std::string url) {
	print_action(std::string(_("LOOKING ON ")) + url);
	if (IS_HTTP(url))
		print_result(Tswac_repository_http::load(url));
	
	else if (IS_FILE(url)) 
		print_result(Tswac_repository_local::load(url.substr(7, url.size() - 7)));

	else {
		print_error(std::string(_("Not a valid url:")) + " \"" + url + "\"");
		print_result(false);
	}
}

void Tswac::upgrade() {
	SqlQueryResult result;
	db.exec("SELECT path, version FROM `packages` ;", result); 
	for (unsigned int i = 0; i < result.size(); i++) {
		std::string url(result.at(i).at(0));
		if (result.at(i).at(1) == package_get_available_version(url))
			std::cout << url << ": " << _("Package is up to date") << std::endl;
		else
			package_select(url);
	}
}

void Tswac::update() {
	db.exec("DELETE FROM `packages_available`;");
	SqlQueryResult result;
	db.exec("SELECT path FROM `sources` WHERE selected='1';", result);
	for (unsigned int i = 0; i < result.size(); i++)
		source_load(result.at(i).at(0));
}

void Tswac::package_select_id(std::string packid) {
	print_action(std::string(_("Looking for package")) + " \"" + packid + "\"");
	std::cout << std::endl;

	SqlQueryResult result;
	db.exec("SELECT format, version, path FROM `packages_available` WHERE `packid`=" + db.str(packid) + ";", result);
	
	if (result.size() == 0) {
		print_error(std::string(_("Can not find package")) + " \"" + packid + "\"");
		return;
	}
	else if (result.size() == 1) {
		package_select(result.at(0).at(2));
		return;	
	}
	else {
		std::cout << _("Please, choose a package:") << std::endl;

		for (unsigned int i = 0; i < result.size(); i++) {
			std::cout << i + 1 << ")" << std::endl;
			std::cout << " " << _("format:")  << " " << result.at(i).at(0) << std::endl; 
			std::cout << " " << _("version:") << " " << result.at(i).at(1) << std::endl; 
			std::cout << " " << _("path:")    << " " << result.at(i).at(2) << std::endl; 
		}

		unsigned int selected;
		std::cin >> selected;
		if (selected > 0 and selected <= result.size())
			package_select(result.at(selected - 1).at(2));
	}
}

void Tswac::package_deselect_id(std::string packid) {
	SqlQueryResult result;
	db.exec("SELECT format, version, path FROM `packages` WHERE `packid`=" + db.str(packid) + ";", result);
	
	if (result.size() == 0) {
		print_error(std::string(_("Can not find package")) + " \"" + packid + "\"");
		return;
	}
	else if (result.size() == 1) {
		package_deselect(result.at(0).at(2));
		return;	
	}
	else {
		std::cout << _("Please, choose a package:") << std::endl;

		for (unsigned int i = 0; i < result.size(); i++) {
			std::cout << i + 1 << ")" << std::endl;
			std::cout << " " << _("format:")  << " " << result.at(i).at(0) << std::endl; 
			std::cout << " " << _("version:") << " " << result.at(i).at(1) << std::endl; 
			std::cout << " " << _("path:")    << " " << result.at(i).at(2) << std::endl; 
		}

		unsigned int selected;
		std::cin >> selected;
		if (selected > 0 and selected <= result.size())
			package_deselect(result.at(selected - 1).at(2));
	}
}

void Tswac::upgrade_structure() {
	SqlQueryResult sources, packages;
	db.exec("SELECT path, selected FROM sources;", sources);
	db.exec("SELECT path FROM packages;", packages);
	db.close();
	if (file_remove(db_path) and db_open(db_path)) {
		db_init();
		for (unsigned int i = 0; i < sources.size(); i++)
			db.exec("INSERT INTO sources (`path`, `selected`) VALUES (" + db.str(sources.at(i).at(0)) + ", " + db.str(sources.at(i).at(1)) + ");");
		update();
		for (unsigned int i=0; i < packages.size(); i++)
			package_select(packages.at(i).at(0));
	}
	else
		print_error(_("Can not upgrade database structure!"));	
}

void Tswac::sql(std::string sql) {
	std::cout << "SQL QUERY: \"" << sql << "\"" << std::endl;

	SqlQuery query(&db);
	if (query.prepare(sql) != SQLITE_OK)
		return;
	
	int nCols = query.columnCount();
	int nRows = 0;
	while (query.step() == SQLITE_ROW) {
		//show header
		if (nRows == 0) {
			for (int i = 0; i < query.columnCount(); i++) {
				if (i != 0) std::cout << "	";
				std::cout << query.columnName(i);
			}
			std::cout << std::endl << "--------------------------------------" << std::endl;
		}

		nRows++;		
		for (int i = 0; i < nCols; i++) {
			if (i != 0)
				std::cout << "	";
			std::cout << query.columnText(i);
		}
		std::cout << std::endl;
	}
	std::cout << _("Result(s): ") << nRows << std::endl;
}


