/* Copyright (C) 2007-2010 Open Information Security Foundation
 *
 * You can copy, redistribute or modify this Program under the terms of
 * the GNU General Public License version 2 as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * version 2 along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

/**
 * \file
 *
 * \author Kirby Kuehl <kkuehl@gmail.com>
 * \author Anoop Saldanha <anoopsaldanha@gmail.com>
 *
 * \file DCE/RPC parser and decoder
 *
 * \todo Remove all the unnecessary per byte incremental loops with a full one
 *       time jump, i.e.
 *
 *       input[0], input_len
 *       for (i = 0; i < x; i++)
 *           input++;
 *
 *       with
 *
 *       input += x;
 *
 *       You'll be surprised at how many such cases we have here.  We also need
 *       to do the same for htp parser.  Should speed up the engine drastically.
 */

#include "suricata-common.h"
#include "suricata.h"

#include "debug.h"
#include "decode.h"
#include "threads.h"

#include "util-print.h"
#include "util-pool.h"
#include "util-debug.h"

#include "flow-util.h"

#include "detect-engine-state.h"

#include "stream-tcp-private.h"
#include "stream-tcp-reassemble.h"
#include "stream-tcp.h"
#include "stream.h"

#include "app-layer-protos.h"
#include "app-layer-parser.h"
#include "app-layer.h"

#include "util-spm.h"
#include "util-unittest.h"

#include "app-layer-dcerpc.h"

enum {
    DCERPC_FIELD_NONE = 0,
    DCERPC_PARSE_DCERPC_HEADER,
    DCERPC_PARSE_DCERPC_BIND,
    DCERPC_PARSE_DCERPC_BIND_ACK,
    DCERPC_PARSE_DCERPC_REQUEST,
    /* must be last */
    DCERPC_FIELD_MAX,
};

void DCERPCUuidListFree(DCERPCUuidEntryList *list);

/* \brief hexdump function from libdnet, used for debugging only */
void hexdump(/*Flow *f,*/ const void *buf, size_t len)
{
    /* dumps len bytes of *buf to stdout. Looks like:
     * [0000] 75 6E 6B 6E 6F 77 6E 20
     *                  30 FF 00 00 00 00 39 00 unknown 0.....9.
     * (in a single line of course)
     */

    const unsigned char *p = buf;
    unsigned char c;
    size_t n;
    char bytestr[4] = {0};
    char addrstr[17] = {0};
    char hexstr[ 16*3 + 5] = {0};
    char charstr[16*1 + 5] = {0};
    for (n=1; n<=len; n++) {
        if (n%16 == 1) {
            /* store address for this line */
#if __WORDSIZE == 64
            snprintf(addrstr, sizeof(addrstr), "%.4"PRIx64,
            ((uint64_t)p-(uint64_t)buf) );
#else
            snprintf(addrstr, sizeof(addrstr), "%.4"PRIx32,
            ((uint32_t)p-(uint32_t)buf) );
#endif
        }

        c = *p;
        if (isalnum(c) == 0) {
            c = '.';
        }

        /* store hex str (for left side) */
        snprintf(bytestr, sizeof(bytestr), "%02X ", *p);
        strlcat(hexstr, bytestr, sizeof(hexstr)-strlen(hexstr)-1);

        /* store char str (for right side) */
        snprintf(bytestr, sizeof(bytestr), "%c", c);
        strlcat(charstr, bytestr, sizeof(charstr)-strlen(charstr)-1);

        if(n%16 == 0) {
            /* line completed */
            printf("[%4.4s]   %-50.50s  %s\n", addrstr, hexstr, charstr);
            hexstr[0] = 0;
            charstr[0] = 0;
        } else if(n%8 == 0) {
            /* half line: add whitespaces */
            strlcat(hexstr, "  ", sizeof(hexstr)-strlen(hexstr)-1);
            strlcat(charstr, " ", sizeof(charstr)-strlen(charstr)-1);
        }
        p++; /* next byte */
    }

    if (strlen(hexstr) > 0) {
        /* print rest of buffer if not empty */
        printf("[%4.4s]   %-50.50s  %s\n", addrstr, hexstr, charstr);
    }
}

/**
 * \brief printUUID function used to print UUID, Major and Minor Version Number
 * and if it was Accepted or Rejected in the BIND_ACK.
 */
void printUUID(const char *type, DCERPCUuidEntry *uuid)
{
    uint8_t i = 0;
    if (uuid == NULL) {
        return;
    }
    printf("%s UUID [%2u] %s ", type, uuid->ctxid,
            (uuid->result == 0) ? "Accepted" : "Rejected");
    for (i = 0; i < 16; i++) {
        printf("%02x", uuid->uuid[i]);
    }
    printf(" Major Version 0x%04x Minor Version 0x%04x\n", uuid->version,
            uuid->versionminor);
}

/**
 * \brief DCERPCParseSecondaryAddr reads secondaryaddrlen bytes from the BIND_ACK
 * DCERPC call.
 */
static uint32_t DCERPCParseSecondaryAddr(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();
    const uint8_t *p = input;
    while (dcerpc->dcerpcbindbindack.secondaryaddrlenleft-- && input_len--) {
        SCLogDebug("0x%02x ", *p);
        p++;
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t PaddingParser(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();
    const uint8_t *p = input;
    while (dcerpc->padleft-- && input_len--) {
        SCLogDebug("0x%02x ", *p);
        p++;
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t DCERPCGetCTXItems(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();
    const uint8_t *p = input;
    if (input_len) {
        switch (dcerpc->dcerpcbindbindack.ctxbytesprocessed) {
            case 0:
                if (input_len >= 4) {
                    dcerpc->dcerpcbindbindack.numctxitems = *p;
                    dcerpc->dcerpcbindbindack.numctxitemsleft = dcerpc->dcerpcbindbindack.numctxitems;
                    dcerpc->dcerpcbindbindack.ctxbytesprocessed += 4;
                    dcerpc->bytesprocessed += 4;
                    SCReturnUInt(4U);
                } else {
                    dcerpc->dcerpcbindbindack.numctxitems = *(p++);
                    dcerpc->dcerpcbindbindack.numctxitemsleft = dcerpc->dcerpcbindbindack.numctxitems;
                    if (!(--input_len))
                        break;
                }
                /* fall through */
            case 1:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 2:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 3:
                p++;
                input_len--;
                break;
        }
    }
    dcerpc->dcerpcbindbindack.ctxbytesprocessed += (p - input);
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

/**
 * \brief DCERPCParseBINDCTXItem is called for each CTXItem found the DCERPC BIND call.
 * each UUID is added to a TAILQ.
 */

static uint32_t DCERPCParseBINDCTXItem(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();
    const uint8_t *p = input;

    if (input_len) {
        switch (dcerpc->dcerpcbindbindack.ctxbytesprocessed) {
            case 0:
                if (input_len >= 44) {
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpcbindbindack.ctxid = *(p);
                        dcerpc->dcerpcbindbindack.ctxid |= *(p + 1) << 8;
                        dcerpc->dcerpcbindbindack.uuid[3] = *(p + 4);
                        dcerpc->dcerpcbindbindack.uuid[2] = *(p + 5);
                        dcerpc->dcerpcbindbindack.uuid[1] = *(p + 6);
                        dcerpc->dcerpcbindbindack.uuid[0] = *(p + 7);
                        dcerpc->dcerpcbindbindack.uuid[5] = *(p + 8);
                        dcerpc->dcerpcbindbindack.uuid[4] = *(p + 9);
                        dcerpc->dcerpcbindbindack.uuid[7] = *(p + 10);
                        dcerpc->dcerpcbindbindack.uuid[6] = *(p + 11);
                        dcerpc->dcerpcbindbindack.uuid[8] = *(p + 12);
                        dcerpc->dcerpcbindbindack.uuid[9] = *(p + 13);
                        dcerpc->dcerpcbindbindack.uuid[10] = *(p + 14);
                        dcerpc->dcerpcbindbindack.uuid[11] = *(p + 15);
                        dcerpc->dcerpcbindbindack.uuid[12] = *(p + 16);
                        dcerpc->dcerpcbindbindack.uuid[13] = *(p + 17);
                        dcerpc->dcerpcbindbindack.uuid[14] = *(p + 18);
                        dcerpc->dcerpcbindbindack.uuid[15] = *(p + 19);
                        dcerpc->dcerpcbindbindack.version = *(p + 20);
                        dcerpc->dcerpcbindbindack.version |= *(p + 21) << 8;
                        dcerpc->dcerpcbindbindack.versionminor = *(p + 22);
                        dcerpc->dcerpcbindbindack.versionminor |= *(p + 23) << 8;
                    } else { /* Big Endian */
                        dcerpc->dcerpcbindbindack.ctxid = *(p) << 8;
                        dcerpc->dcerpcbindbindack.ctxid |= *(p + 1);
                        dcerpc->dcerpcbindbindack.uuid[0] = *(p + 4);
                        dcerpc->dcerpcbindbindack.uuid[1] = *(p + 5);
                        dcerpc->dcerpcbindbindack.uuid[2] = *(p + 6);
                        dcerpc->dcerpcbindbindack.uuid[3] = *(p + 7);
                        dcerpc->dcerpcbindbindack.uuid[4] = *(p + 8);
                        dcerpc->dcerpcbindbindack.uuid[5] = *(p + 9);
                        dcerpc->dcerpcbindbindack.uuid[6] = *(p + 10);
                        dcerpc->dcerpcbindbindack.uuid[7] = *(p + 11);
                        dcerpc->dcerpcbindbindack.uuid[8] = *(p + 12);
                        dcerpc->dcerpcbindbindack.uuid[9] = *(p + 13);
                        dcerpc->dcerpcbindbindack.uuid[10] = *(p + 14);
                        dcerpc->dcerpcbindbindack.uuid[11] = *(p + 15);
                        dcerpc->dcerpcbindbindack.uuid[12] = *(p + 16);
                        dcerpc->dcerpcbindbindack.uuid[13] = *(p + 17);
                        dcerpc->dcerpcbindbindack.uuid[14] = *(p + 18);
                        dcerpc->dcerpcbindbindack.uuid[15] = *(p + 19);
                        dcerpc->dcerpcbindbindack.version = *(p + 20) << 8;
                        dcerpc->dcerpcbindbindack.version |= *(p + 21);
                        dcerpc->dcerpcbindbindack.versionminor = *(p + 22) << 8;
                        dcerpc->dcerpcbindbindack.versionminor |= *(p + 23);
                    }
                    //if (dcerpc->dcerpcbindbindack.ctxid == dcerpc->dcerpcbindbindack.numctxitems
                    //        - dcerpc->dcerpcbindbindack.numctxitemsleft) {

                    dcerpc->dcerpcbindbindack.uuid_entry = (DCERPCUuidEntry *)SCCalloc(1, sizeof(DCERPCUuidEntry));
                    if (dcerpc->dcerpcbindbindack.uuid_entry == NULL) {
                        SCLogDebug("UUID Entry is NULL");
                        SCReturnUInt(0);
                    }

                    dcerpc->dcerpcbindbindack.uuid_entry->internal_id = dcerpc->dcerpcbindbindack.uuid_internal_id++;

                    memcpy(dcerpc->dcerpcbindbindack.uuid_entry->uuid,
                            dcerpc->dcerpcbindbindack.uuid,
                            sizeof(dcerpc->dcerpcbindbindack.uuid));

                    dcerpc->dcerpcbindbindack.uuid_entry->ctxid = dcerpc->dcerpcbindbindack.ctxid;
                    dcerpc->dcerpcbindbindack.uuid_entry->version = dcerpc->dcerpcbindbindack.version;
                    dcerpc->dcerpcbindbindack.uuid_entry->versionminor = dcerpc->dcerpcbindbindack.versionminor;

                    /* store the first frag flag in the uuid as pfc_flags will
                     * be overwritten by new packets. */
                    if (dcerpc->dcerpchdr.pfc_flags & PFC_FIRST_FRAG) {
                        dcerpc->dcerpcbindbindack.uuid_entry->flags |= DCERPC_UUID_ENTRY_FLAG_FF;
                    }

                    TAILQ_INSERT_HEAD(&dcerpc->dcerpcbindbindack.uuid_list,
                            dcerpc->dcerpcbindbindack.uuid_entry,
                            next);

#ifdef UNITTESTS
                    if (RunmodeIsUnittests()) {
                        printUUID("BIND", dcerpc->dcerpcbindbindack.uuid_entry);
                    }
#endif
                    dcerpc->dcerpcbindbindack.numctxitemsleft--;
                    dcerpc->bytesprocessed += (44);
                    dcerpc->dcerpcbindbindack.ctxbytesprocessed += (44);
                    SCReturnUInt(44U);

                    //} else {
                    //    SCLogDebug("ctxitem %u, expected %u\n", dcerpc->dcerpcbindbindack.ctxid,
                    //            dcerpc->dcerpcbindbindack.numctxitems - dcerpc->dcerpcbindbindack.numctxitemsleft);
                    //    SCReturnUInt(0);
                    //}
                } else {
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpcbindbindack.ctxid = *(p++);
                    } else {
                        dcerpc->dcerpcbindbindack.ctxid = *(p++) << 8;
                    }
                    if (!(--input_len))
                        break;
                }
                /* fall through */
            case 1:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.ctxid |= *(p++) << 8;
                } else {
                    dcerpc->dcerpcbindbindack.ctxid |= *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 2:
                /* num transact items */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 3:
                /* reserved */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 4:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[3] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[0] = *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 5:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[2] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[1] = *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 6:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[1] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[2] = *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 7:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[0] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[3] = *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 8:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[5] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[4] = *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 9:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[4] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[5] = *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 10:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[7] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[6] = *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 11:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[6] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[7] = *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 12:
                /* The following bytes are in the same order for both big and little endian */
                dcerpc->dcerpcbindbindack.uuid[8] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 13:
                dcerpc->dcerpcbindbindack.uuid[9] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 14:
                dcerpc->dcerpcbindbindack.uuid[10] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 15:
                dcerpc->dcerpcbindbindack.uuid[11] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 16:
                dcerpc->dcerpcbindbindack.uuid[12] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 17:
                dcerpc->dcerpcbindbindack.uuid[13] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 18:
                dcerpc->dcerpcbindbindack.uuid[14] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 19:
                dcerpc->dcerpcbindbindack.uuid[15] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 20:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.version = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.version = *(p++) << 8;
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 21:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.version |= *(p++) << 8;
                } else {
                    dcerpc->dcerpcbindbindack.version |= *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 22:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.versionminor = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.versionminor = *(p++) << 8;
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 23:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.versionminor |= *(p++) << 8;
                } else {
                    dcerpc->dcerpcbindbindack.versionminor |= *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 24:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 25:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 26:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 27:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 28:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 29:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 30:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 31:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 32:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 33:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 34:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 35:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 36:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 37:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 38:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 39:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 40:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 41:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 42:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 43:
                p++;
                --input_len;
                //if (dcerpc->dcerpcbindbindack.ctxid == dcerpc->dcerpcbindbindack.numctxitems - dcerpc->dcerpcbindbindack.numctxitemsleft) {
                dcerpc->dcerpcbindbindack.uuid_entry = (DCERPCUuidEntry *)
                    SCCalloc(1, sizeof(DCERPCUuidEntry));
                if (dcerpc->dcerpcbindbindack.uuid_entry == NULL) {
                    SCLogDebug("UUID Entry is NULL\n");
                    SCReturnUInt(0);
                }

                dcerpc->dcerpcbindbindack.uuid_entry->internal_id =
                    dcerpc->dcerpcbindbindack.uuid_internal_id++;
                memcpy(dcerpc->dcerpcbindbindack.uuid_entry->uuid,
                        dcerpc->dcerpcbindbindack.uuid,
                        sizeof(dcerpc->dcerpcbindbindack.uuid));
                dcerpc->dcerpcbindbindack.uuid_entry->ctxid = dcerpc->dcerpcbindbindack.ctxid;
                dcerpc->dcerpcbindbindack.uuid_entry->version = dcerpc->dcerpcbindbindack.version;
                dcerpc->dcerpcbindbindack.uuid_entry->versionminor = dcerpc->dcerpcbindbindack.versionminor;

                /* store the first frag flag in the uuid as pfc_flags will
                 * be overwritten by new packets. */
                if (dcerpc->dcerpchdr.pfc_flags & PFC_FIRST_FRAG) {
                    dcerpc->dcerpcbindbindack.uuid_entry->flags |= DCERPC_UUID_ENTRY_FLAG_FF;
                }

                TAILQ_INSERT_HEAD(&dcerpc->dcerpcbindbindack.uuid_list,
                        dcerpc->dcerpcbindbindack.uuid_entry,
                        next);
#ifdef UNITTESTS
                if (RunmodeIsUnittests()) {
                    printUUID("BINDACK", dcerpc->dcerpcbindbindack.uuid_entry);
                }
#endif
                dcerpc->dcerpcbindbindack.numctxitemsleft--;
                dcerpc->bytesprocessed += (p - input);
                dcerpc->dcerpcbindbindack.ctxbytesprocessed += (p - input);
                SCReturnUInt((uint32_t)(p - input));

                //} else {
                //    SCLogDebug("ctxitem %u, expected %u\n", dcerpc->dcerpcbindbindack.ctxid,
                //            dcerpc->dcerpcbindbindack.numctxitems - dcerpc->dcerpcbindbindack.numctxitemsleft);
                //    SCReturnUInt(0);
                //}
                break;
        }
    }
    dcerpc->dcerpcbindbindack.ctxbytesprocessed += (p - input);
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

/**
 * \brief DCERPCParseBINDACKCTXItem is called for each CTXItem found in
 * the BIND_ACK call. The result (Accepted or Rejected) is added to the
 * correct UUID from the BIND call.
 */
static uint32_t DCERPCParseBINDACKCTXItem(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();
    const uint8_t *p = input;
    DCERPCUuidEntry *uuid_entry;

    if (input_len) {
        switch (dcerpc->dcerpcbindbindack.ctxbytesprocessed) {
            case 0:
                if (input_len >= 24) {
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpcbindbindack.result = *p;
                        dcerpc->dcerpcbindbindack.result |= *(p + 1) << 8;
                    } else {
                        dcerpc->dcerpcbindbindack.result = *p << 8;
                        dcerpc->dcerpcbindbindack.result |= *(p + 1);
                    }
                    TAILQ_FOREACH(uuid_entry, &dcerpc->dcerpcbindbindack.uuid_list, next) {
                        if (uuid_entry->internal_id == dcerpc->dcerpcbindbindack.uuid_internal_id) {
                            uuid_entry->result = dcerpc->dcerpcbindbindack.result;
#ifdef UNITTESTS
                            if (RunmodeIsUnittests()) {
                                printUUID("BIND_ACK", uuid_entry);
                            }
#endif
                            if (uuid_entry->result != 0)
                                break;

                            dcerpc->dcerpcbindbindack.uuid_entry = (DCERPCUuidEntry *)
                                SCCalloc(1, sizeof(DCERPCUuidEntry));
                            if (dcerpc->dcerpcbindbindack.uuid_entry != NULL) {
                                memcpy(dcerpc->dcerpcbindbindack.uuid_entry,
                                        uuid_entry,
                                        sizeof(DCERPCUuidEntry));
                                TAILQ_INSERT_HEAD(&dcerpc->dcerpcbindbindack.accepted_uuid_list,
                                        dcerpc->dcerpcbindbindack.uuid_entry,
                                        next);
                            }
                            break;
                        }
                    }
                    dcerpc->dcerpcbindbindack.uuid_internal_id++;
                    dcerpc->dcerpcbindbindack.numctxitemsleft--;
                    dcerpc->bytesprocessed += (24);
                    dcerpc->dcerpcbindbindack.ctxbytesprocessed += (24);
                    SCReturnUInt(24U);
                } else {
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpcbindbindack.result = *(p++);
                    } else {
                        dcerpc->dcerpcbindbindack.result = *(p++) << 8;
                    }
                    if (!(--input_len))
                        break;
                }
                /* fall through */
            case 1:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.result |= *(p++) << 8;
                } else {
                    dcerpc->dcerpcbindbindack.result |= *(p++);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 2:
                /* num transact items */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 3:
                /* reserved */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 4:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 5:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 6:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 7:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 8:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 9:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 10:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 11:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 12:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 13:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 14:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 15:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 16:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 17:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 18:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 19:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 20:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 21:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 22:
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 23:
                TAILQ_FOREACH(uuid_entry, &dcerpc->dcerpcbindbindack.uuid_list, next) {
                    if (uuid_entry->internal_id == dcerpc->dcerpcbindbindack.uuid_internal_id) {
                        uuid_entry->result = dcerpc->dcerpcbindbindack.result;
#ifdef UNITTESTS
                        if (RunmodeIsUnittests()) {
                            printUUID("BIND_ACK", uuid_entry);
                        }
#endif
                        if (uuid_entry->result != 0)
                            break;

                        dcerpc->dcerpcbindbindack.uuid_entry = (DCERPCUuidEntry *)
                            SCCalloc(1, sizeof(DCERPCUuidEntry));
                        if (dcerpc->dcerpcbindbindack.uuid_entry != NULL) {
                            memcpy(dcerpc->dcerpcbindbindack.uuid_entry,
                                    uuid_entry,
                                    sizeof(DCERPCUuidEntry));
                            TAILQ_INSERT_HEAD(&dcerpc->dcerpcbindbindack.accepted_uuid_list,
                                    dcerpc->dcerpcbindbindack.uuid_entry,
                                    next);
                        }
                        break;
                    }
                }
                dcerpc->dcerpcbindbindack.uuid_internal_id++;
                dcerpc->dcerpcbindbindack.numctxitemsleft--;
                p++;
                --input_len;
                break;

        }
    }
    dcerpc->dcerpcbindbindack.ctxbytesprocessed += (p - input);
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t DCERPCParseBIND(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();
    const uint8_t *p = input;
    if (input_len) {
        switch (dcerpc->bytesprocessed) {
            case 16:
                dcerpc->dcerpcbindbindack.numctxitems = 0;
                if (input_len >= 12) {
                    DCERPCUuidListFree(&dcerpc->dcerpcbindbindack.uuid_list);
                    if (dcerpc->dcerpchdr.type == BIND) {
                        DCERPCUuidListFree(&dcerpc->dcerpcbindbindack.accepted_uuid_list);
                    }
                    dcerpc->dcerpcbindbindack.uuid_internal_id = 0;
                    dcerpc->dcerpcbindbindack.numctxitems = *(p + 8);
                    dcerpc->dcerpcbindbindack.numctxitemsleft = dcerpc->dcerpcbindbindack.numctxitems;
                    dcerpc->bytesprocessed += 12;
                    SCReturnUInt(12U);
                } else {
                    /* max_xmit_frag */
                    p++;
                    if (!(--input_len))
                        break;
                }
                /* fall through */
            case 17:
                /* max_xmit_frag */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 18:
                /* max_recv_frag */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 19:
                /* max_recv_frag */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 20:
                /* assoc_group_id */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 21:
                /* assoc_group_id */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 22:
                /* assoc_group_id */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 23:
                /* assoc_group_id */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 24:
                DCERPCUuidListFree(&dcerpc->dcerpcbindbindack.uuid_list);
                if (dcerpc->dcerpchdr.type == BIND) {
                    DCERPCUuidListFree(&dcerpc->dcerpcbindbindack.accepted_uuid_list);
                }
                dcerpc->dcerpcbindbindack.uuid_internal_id = 0;
                dcerpc->dcerpcbindbindack.numctxitems = *(p++);
                dcerpc->dcerpcbindbindack.numctxitemsleft = dcerpc->dcerpcbindbindack.numctxitems;
                if (!(--input_len))
                    break;
                /* fall through */
            case 25:
                /* pad byte 1 */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 26:
                /* pad byte 2 */
                p++;
                if (!(--input_len))
                    break;
                /* fall through */
            case 27:
                /* pad byte 3 */
                p++;
                --input_len;
                break;
                /* fall through */
            default:
                dcerpc->bytesprocessed++;
                SCReturnUInt(1);
                break;
        }
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t DCERPCParseBINDACK(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();
    const uint8_t *p = input;

    switch (dcerpc->bytesprocessed) {
        case 16:
            dcerpc->dcerpcbindbindack.uuid_internal_id = 0;
            dcerpc->dcerpcbindbindack.numctxitems = 0;
            if (input_len >= 10) {
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.secondaryaddrlen = *(p + 8);
                    dcerpc->dcerpcbindbindack.secondaryaddrlen |= *(p + 9) << 8;
                } else {
                    dcerpc->dcerpcbindbindack.secondaryaddrlen = *(p + 8) << 8;
                    dcerpc->dcerpcbindbindack.secondaryaddrlen |= *(p + 9);
                }
                dcerpc->dcerpcbindbindack.secondaryaddrlenleft = dcerpc->dcerpcbindbindack.secondaryaddrlen;
                dcerpc->bytesprocessed += 10;
                SCReturnUInt(10U);
            } else {
                /* max_xmit_frag */
                p++;
                if (!(--input_len))
                    break;
            }
            /* fall through */
        case 17:
            /* max_xmit_frag */
            p++;
            if (!(--input_len))
                break;
            /* fall through */
        case 18:
            /* max_recv_frag */
            p++;
            if (!(--input_len))
                break;
            /* fall through */
        case 19:
            /* max_recv_frag */
            p++;
            if (!(--input_len))
                break;
            /* fall through */
        case 20:
            /* assoc_group_id */
            p++;
            if (!(--input_len))
                break;
            /* fall through */
        case 21:
            /* assoc_group_id */
            p++;
            if (!(--input_len))
                break;
            /* fall through */
        case 22:
            /* assoc_group_id */
            p++;
            if (!(--input_len))
                break;
            /* fall through */
        case 23:
            /* assoc_group_id */
            p++;
            if (!(--input_len))
                break;
            /* fall through */
        case 24:
            dcerpc->dcerpcbindbindack.secondaryaddrlen = *(p++) << 8;
            if (!(--input_len))
                break;
            /* fall through */
        case 25:
            dcerpc->dcerpcbindbindack.secondaryaddrlen |= *(p++);
            if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                dcerpc->dcerpcbindbindack.secondaryaddrlen = SCByteSwap16(dcerpc->dcerpcbindbindack.secondaryaddrlen);
            }
            dcerpc->dcerpcbindbindack.secondaryaddrlenleft = dcerpc->dcerpcbindbindack.secondaryaddrlen;
            SCLogDebug("secondaryaddrlen %u 0x%04x\n", dcerpc->dcerpcbindbindack.secondaryaddrlen,
                    dcerpc->dcerpcbindbindack.secondaryaddrlen);
            --input_len;
            break;
        default:
            dcerpc->bytesprocessed++;
            SCReturnUInt(1);
            break;
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t DCERPCParseREQUEST(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();
    const uint8_t *p = input;

    switch (dcerpc->bytesprocessed) {
        case 16:
            dcerpc->dcerpcbindbindack.numctxitems = 0;
            if (input_len >= 8) {
                if (dcerpc->dcerpchdr.type == REQUEST) {
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpcrequest.ctxid = *(p + 4);
                        dcerpc->dcerpcrequest.ctxid |= *(p + 5) << 8;
                        dcerpc->dcerpcrequest.opnum = *(p + 6);
                        dcerpc->dcerpcrequest.opnum |= *(p + 7) << 8;
                    } else {
                        dcerpc->dcerpcrequest.ctxid = *(p + 4) << 8;
                        dcerpc->dcerpcrequest.ctxid |= *(p + 5);
                        dcerpc->dcerpcrequest.opnum = *(p + 6) << 8;
                        dcerpc->dcerpcrequest.opnum |= *(p + 7);
                    }
                    dcerpc->dcerpcrequest.first_request_seen = 1;
                }
                dcerpc->bytesprocessed += 8;
                SCReturnUInt(8U);
            } else {
                /* alloc hint 1 */
                p++;
                if (!(--input_len))
                    break;
            }
            /* fall through */
        case 17:
            /* alloc hint 2 */
            p++;
            if (!(--input_len))
                break;
            /* fall through */
        case 18:
            /* alloc hint 3 */
            p++;
            if (!(--input_len))
                break;
            /* fall through */
        case 19:
            /* alloc hint 4 */
            p++;
            if (!(--input_len))
                break;
            /* fall through */
        case 20:
            /* context id 1 */
            dcerpc->dcerpcrequest.ctxid = *(p++);
            if (!(--input_len))
                break;
            /* fall through */
        case 21:
            /* context id 2 */
            dcerpc->dcerpcrequest.ctxid |= *(p++) << 8;
            if (!(dcerpc->dcerpchdr.packed_drep[0] & 0x10)) {
                dcerpc->dcerpcrequest.ctxid = SCByteSwap16(dcerpc->dcerpcrequest.ctxid);
            }
            dcerpc->dcerpcrequest.first_request_seen = 1;
            if (!(--input_len))
                break;
            /* fall through */
        case 22:
            if (dcerpc->dcerpchdr.type == REQUEST) {
                dcerpc->dcerpcrequest.opnum = *(p++);
            } else {
                p++;
            }
            if (!(--input_len))
                break;
            /* fall through */
        case 23:
            if (dcerpc->dcerpchdr.type == REQUEST) {
                dcerpc->dcerpcrequest.opnum |= *(p++) << 8;
                if (!(dcerpc->dcerpchdr.packed_drep[0] & 0x10)) {
                    dcerpc->dcerpcrequest.opnum = SCByteSwap16(dcerpc->dcerpcrequest.opnum);
                }
            } else {
                p++;
            }
            --input_len;
            break;
        default:
            dcerpc->bytesprocessed++;
            SCReturnUInt(1);
            break;
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

/** \internal
 *  \retval stub_len or 0 in case of error */
static uint32_t StubDataParser(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();
    uint8_t **stub_data_buffer = NULL;
    uint32_t *stub_data_buffer_len = NULL;

    SCLogDebug("input_len %u", input_len);

    /* request PDU.  Retrieve the request stub buffer */
    if (dcerpc->dcerpchdr.type == REQUEST) {
        stub_data_buffer = &dcerpc->dcerpcrequest.stub_data_buffer;
        stub_data_buffer_len = &dcerpc->dcerpcrequest.stub_data_buffer_len;

        SCLogDebug("REQUEST stub_data_buffer_len %u", *stub_data_buffer_len);

    /* response PDU.  Retrieve the response stub buffer */
    } else {
        stub_data_buffer = &dcerpc->dcerpcresponse.stub_data_buffer;
        stub_data_buffer_len = &dcerpc->dcerpcresponse.stub_data_buffer_len;

        SCLogDebug("RESPONSE stub_data_buffer_len %u", *stub_data_buffer_len);
    }
    if (*stub_data_buffer_len > (1024 * 1024)) {
        SCFree(*stub_data_buffer);
        *stub_data_buffer = NULL;
        SCReturnUInt(0);
    }

    uint32_t stub_len = MIN(dcerpc->padleft, input_len);
    if (stub_len == 0) {
        SCReturnInt(0);
    }
    SCLogDebug("stub_len %u input_len %u", stub_len, input_len);

    const uint8_t f = dcerpc->dcerpchdr.pfc_flags & (PFC_FIRST_FRAG|PFC_LAST_FRAG);
    SCLogDebug("f %02x, FIRST %s LAST %s", f,
            f & PFC_FIRST_FRAG ? "true" : "false",
            f & PFC_LAST_FRAG ? "true" : "false");
    if (stub_len == dcerpc->padleft && ((f == 0) || (f & PFC_LAST_FRAG))) {
        if (dcerpc->dcerpchdr.type == REQUEST) {
            dcerpc->dcerpcrequest.stub_data_buffer_reset = true;
            SCLogDebug("REQUEST reset stub");
        } else {
            dcerpc->dcerpcresponse.stub_data_buffer_reset = true;
            SCLogDebug("RESPONSE reset stub");
        }
    }

    void *ptmp = SCRealloc(*stub_data_buffer, *stub_data_buffer_len + stub_len);
    if (ptmp == NULL) {
        SCFree(*stub_data_buffer);
        *stub_data_buffer = NULL;
        SCReturnUInt(0);
    }
    *stub_data_buffer = ptmp;

    memcpy(*stub_data_buffer + *stub_data_buffer_len, input, stub_len);

    /* length of the buffered stub */
    *stub_data_buffer_len += stub_len;
    /* To see the total reassembled stubdata */
    //hexdump(*stub_data_buffer, *stub_data_buffer_len);

    dcerpc->padleft -= stub_len;
    dcerpc->bytesprocessed += stub_len;

    SCLogDebug("padleft %u bytesprocessed %u", dcerpc->padleft, dcerpc->bytesprocessed);

#ifdef DEBUG
    if (SCLogDebugEnabled()) {
        uint32_t i = 0;
        for (i = 0; i < stub_len; i++) {
            SCLogDebug("0x%02x ", input[i]);
        }
    }
#endif
    SCReturnUInt((uint32_t)stub_len);
}

/**
 * \brief DCERPCParseHeader parses the 16 byte DCERPC header
 * A fast path for normal decoding is used when there is enough bytes
 * present to parse the entire header. A slow path is used to parse
 * fragmented packets.
 * \retval -1 if DCERPC Header does not validate
 * \retval Number of bytes processed
 */
static int DCERPCParseHeader(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();
    const uint8_t *p = input;

    if (input_len) {
        SCLogDebug("dcerpc->bytesprocessed %u", dcerpc->bytesprocessed);
        switch (dcerpc->bytesprocessed) {
            case 0:
                if (input_len >= DCERPC_HDR_LEN) {
                    dcerpc->dcerpchdr.rpc_vers = *p;
                    dcerpc->dcerpchdr.rpc_vers_minor = *(p + 1);
                    if ((dcerpc->dcerpchdr.rpc_vers != 5) ||
                       ((dcerpc->dcerpchdr.rpc_vers_minor != 0) &&
                       (dcerpc->dcerpchdr.rpc_vers_minor != 1))) {
							SCLogDebug("DCERPC Header did not validate");
							SCReturnInt(-1);
                    }
                    dcerpc->dcerpchdr.type = *(p + 2);
                    SCLogDebug("dcerpc->dcerpchdr.type %02x",
                            dcerpc->dcerpchdr.type);
                    dcerpc->dcerpchdr.pfc_flags = *(p + 3);
                    dcerpc->dcerpchdr.packed_drep[0] = *(p + 4);
                    dcerpc->dcerpchdr.packed_drep[1] = *(p + 5);
                    dcerpc->dcerpchdr.packed_drep[2] = *(p + 6);
                    dcerpc->dcerpchdr.packed_drep[3] = *(p + 7);
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpchdr.frag_length = *(p + 8);
                        dcerpc->dcerpchdr.frag_length |= *(p + 9) << 8;
                        dcerpc->dcerpchdr.auth_length = *(p + 10);
                        dcerpc->dcerpchdr.auth_length |= *(p + 11) << 8;
                        dcerpc->dcerpchdr.call_id = (uint32_t) *(p + 12) << 24;
                        dcerpc->dcerpchdr.call_id |= (uint32_t) *(p + 13) << 16;
                        dcerpc->dcerpchdr.call_id |= (uint32_t) *(p + 14) << 8;
                        dcerpc->dcerpchdr.call_id |= (uint32_t) *(p + 15);
                    } else {
                        dcerpc->dcerpchdr.frag_length = *(p + 8) << 8;
                        dcerpc->dcerpchdr.frag_length |= *(p + 9);
                        dcerpc->dcerpchdr.auth_length = *(p + 10) << 8;
                        dcerpc->dcerpchdr.auth_length |= *(p + 11);
                        dcerpc->dcerpchdr.call_id = (uint32_t) *(p + 12);
                        dcerpc->dcerpchdr.call_id |= (uint32_t) *(p + 13) << 8;
                        dcerpc->dcerpchdr.call_id |= (uint32_t) *(p + 14) << 16;
                        dcerpc->dcerpchdr.call_id |= (uint32_t) *(p + 15) << 24;
                    }
                    dcerpc->bytesprocessed = DCERPC_HDR_LEN;
                    SCReturnInt(16);
                    break;
                } else {
                    dcerpc->dcerpchdr.rpc_vers = *(p++);
                    if (!(--input_len))
                        break;
                }
                /* fall through */
            case 1:
                dcerpc->dcerpchdr.rpc_vers_minor = *(p++);
                if ((dcerpc->dcerpchdr.rpc_vers != 5) ||
                    ((dcerpc->dcerpchdr.rpc_vers_minor != 0) &&
                    (dcerpc->dcerpchdr.rpc_vers_minor != 1))) {
						SCLogDebug("DCERPC Header did not validate");
						SCReturnInt(-1);
                }
                if (!(--input_len))
                    break;
                /* fall through */
            case 2:
                dcerpc->dcerpchdr.type = *(p++);
                SCLogDebug("dcerpc->dcerpchdr.type %02x",
                        dcerpc->dcerpchdr.type);
                if (!(--input_len))
                    break;
                /* fall through */
            case 3:
                dcerpc->dcerpchdr.pfc_flags = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 4:
                dcerpc->dcerpchdr.packed_drep[0] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 5:
                dcerpc->dcerpchdr.packed_drep[1] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 6:
                dcerpc->dcerpchdr.packed_drep[2] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 7:
                dcerpc->dcerpchdr.packed_drep[3] = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 8:
                dcerpc->dcerpchdr.frag_length = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 9:
                dcerpc->dcerpchdr.frag_length |= *(p++) << 8;
                if (!(--input_len))
                    break;
                /* fall through */
            case 10:
                dcerpc->dcerpchdr.auth_length = *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 11:
                dcerpc->dcerpchdr.auth_length |= *(p++) << 8;
                if (!(--input_len))
                    break;
                /* fall through */
            case 12:
                dcerpc->dcerpchdr.call_id = (uint32_t) *(p++);
                if (!(--input_len))
                    break;
                /* fall through */
            case 13:
                dcerpc->dcerpchdr.call_id |= (uint32_t) *(p++) << 8;
                if (!(--input_len))
                    break;
                /* fall through */
            case 14:
                dcerpc->dcerpchdr.call_id |= (uint32_t) *(p++) << 16;
                if (!(--input_len))
                    break;
                /* fall through */
            case 15:
                dcerpc->dcerpchdr.call_id |= (uint32_t) *(p++) << 24;
                if (!(dcerpc->dcerpchdr.packed_drep[0] & 0x10)) {
                    dcerpc->dcerpchdr.frag_length = SCByteSwap16(dcerpc->dcerpchdr.frag_length);
                    dcerpc->dcerpchdr.auth_length = SCByteSwap16(dcerpc->dcerpchdr.auth_length);
                    dcerpc->dcerpchdr.call_id = SCByteSwap32(dcerpc->dcerpchdr.call_id);
                }
                --input_len;
                break;
            default:
                dcerpc->bytesprocessed++;
                SCReturnInt(1);
        }
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnInt((p - input));
}

static inline void DCERPCResetParsingState(DCERPC *dcerpc)
{
    dcerpc->bytesprocessed = 0;
    dcerpc->dcerpcbindbindack.ctxbytesprocessed = 0;

    return;
}

static inline void DCERPCResetStub(DCERPC *dcerpc)
{
    if (dcerpc->dcerpchdr.type == REQUEST) {
        SCLogDebug("REQUEST resetting stub");
        dcerpc->dcerpcrequest.stub_data_buffer_len = 0;
        dcerpc->dcerpcrequest.stub_data_buffer_reset = false;
    } else if (dcerpc->dcerpchdr.type == RESPONSE) {
        SCLogDebug("RESPONSE resetting stub");
        dcerpc->dcerpcresponse.stub_data_buffer_len = 0;
        dcerpc->dcerpcresponse.stub_data_buffer_reset = false;
    }

    return;
}

static inline int DCERPCThrowOutExtraData(DCERPC *dcerpc, const uint8_t *input,
                                           uint16_t input_len)
{
    int parsed = 0;
    /* the function always assumes that
     * dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length */
    if (input_len > (dcerpc->dcerpchdr.frag_length - dcerpc->bytesprocessed)) {
        parsed = dcerpc->dcerpchdr.frag_length - dcerpc->bytesprocessed;
    } else {
        parsed = input_len;
    }
    dcerpc->bytesprocessed += parsed;

    return parsed;
}

/**
 * \todo - Currently the parser is very generic.  Enable target based
 *         reassembly.
 *       - Disable reiniting tailq for mid and last bind/alter_context pdus.
 *       - Use a PM to search for subsequent 05 00 when we see an inconsistent
 *         pdu.  This should be done for each platform based on how it handles
 *         a condition where it has receives a segment with 2 pdus, while the
 *         first pdu in the segment is corrupt.
 */
int32_t DCERPCParser(DCERPC *dcerpc, const uint8_t *input, uint32_t input_len)
{
    SCEnter();

    uint32_t retval = 0;
    uint32_t parsed = 0;
    int hdrretval = 0;

    /* temporary use.  we will get rid of this later, once we have ironed out
     * all the endless loops cases */
    int counter = 0;

    while (input_len) {
        /* in case we have any corner cases remainging, we have this */
        if (counter++ == 30) {
            SCLogDebug("Somehow seem to be stuck inside the dce "
                    "parser for quite sometime.  Let's get out of here.");
            DCERPCResetParsingState(dcerpc);
            SCReturnInt(0);
        }

        while (dcerpc->bytesprocessed < DCERPC_HDR_LEN && input_len) {
            hdrretval = DCERPCParseHeader(dcerpc, input + parsed, input_len);
            if (hdrretval == -1 || hdrretval > (int32_t)input_len) {
                SCLogDebug("Error parsing dce header.  Discarding "
                        "PDU and reseting parsing state to parse next PDU");
                /* error parsing pdu header.  Let's clear the dce state */
                DCERPCResetParsingState(dcerpc);
                SCReturnInt(0);
            } else {
                parsed += hdrretval;
                input_len -= hdrretval;
            }
        }
        SCLogDebug("Done with DCERPCParseHeader bytesprocessed %u/%u left %u",
                   dcerpc->bytesprocessed, dcerpc->dcerpchdr.frag_length, input_len);
#if 0
        printf("Done with DCERPCParseHeader bytesprocessed %u/%u input_len left %u\n",
               dcerpc->bytesprocessed, dcerpc->dcerpchdr.frag_length, input_len);
        printf("\nDCERPC Version:\t%u\n", dcerpc->dcerpchdr.rpc_vers);
        printf("DCERPC Version Minor:\t%u\n", dcerpc->dcerpchdr.rpc_vers_minor);
        printf("DCERPC Type:\t%u\n", dcerpc->dcerpchdr.type);
        printf("DCERPC Flags:\t0x%02x\n", dcerpc->dcerpchdr.pfc_flags);
        printf("DCERPC Packed Drep:\t%02x %02x %02x %02x\n",
               dcerpc->dcerpchdr.packed_drep[0], dcerpc->dcerpchdr.packed_drep[1],
               dcerpc->dcerpchdr.packed_drep[2], dcerpc->dcerpchdr.packed_drep[3]);
        printf("DCERPC Frag Length:\t0x%04x %u\n",
               dcerpc->dcerpchdr.frag_length, dcerpc->dcerpchdr.frag_length);
        printf("DCERPC Auth Length:\t0x%04x\n", dcerpc->dcerpchdr.auth_length);
        printf("DCERPC Call Id:\t0x%08x\n", dcerpc->dcerpchdr.call_id);
#endif

        /* check if we have parsed the entire input passed in the header parser.
         * If we have, time to leave */
        if (input_len == 0) {
            if (dcerpc->bytesprocessed < 10) {

            } else {
                if (dcerpc->bytesprocessed >= dcerpc->dcerpchdr.frag_length) {
                    SCLogDebug("Weird DCE PDU");
                    DCERPCResetParsingState(dcerpc);
                }
            }
            SCReturnInt(parsed);
        }
        switch (dcerpc->dcerpchdr.type) {
            case BIND:
            case ALTER_CONTEXT:
                while (dcerpc->bytesprocessed < DCERPC_HDR_LEN + 12
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length
                       && input_len) {
                    retval = DCERPCParseBIND(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                    } else if (input_len) {
                        SCLogDebug("Error Parsing DCERPC %s PDU",
                                   (dcerpc->dcerpchdr.type == BIND) ?
                                   "BIND" : "ALTER_CONTEXT");
                        parsed = 0;
                        (void)parsed; /* for scan-build */
                        input_len = 0;
                        (void)input_len; /* for scan-build */
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }
                SCLogDebug("Done with DCERPCParseBIND bytesprocessed %u/%u numctxitems %u",
                           dcerpc->bytesprocessed, dcerpc->dcerpchdr.frag_length,
                           dcerpc->dcerpcbindbindack.numctxitems);
                while (dcerpc->dcerpcbindbindack.numctxitemsleft && dcerpc->bytesprocessed
                       < dcerpc->dcerpchdr.frag_length && input_len) {
                    retval = DCERPCParseBINDCTXItem(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        if (dcerpc->dcerpcbindbindack.ctxbytesprocessed == 44) {
                            dcerpc->dcerpcbindbindack.ctxbytesprocessed = 0;
                        }
                        parsed += retval;
                        input_len -= retval;
                        SCLogDebug("BIND processed %u/%u ctxitems %u/%u input_len left %u\n",
                                   dcerpc->bytesprocessed,
                                   dcerpc->dcerpchdr.frag_length,
                                   dcerpc->dcerpcbindbindack.numctxitemsleft,
                                   dcerpc->dcerpcbindbindack.numctxitems, input_len);
                    } else if (input_len) {
                        //parsed -= input_len;
                        SCLogDebug("Error Parsing CTX Item %u\n", parsed);
                        parsed = 0;
                        (void)parsed; /* for scan-build */
                        input_len = 0;
                        (void)input_len; /* for scan-build */
                        dcerpc->dcerpcbindbindack.numctxitemsleft = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }
                if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                    DCERPCResetParsingState(dcerpc);
                } else if (dcerpc->bytesprocessed > dcerpc->dcerpchdr.frag_length) {
                    DCERPCResetParsingState(dcerpc);
                    SCReturnInt(0);
                } else {
                    /* temporary fix */
                    if (input_len) {
                        retval = DCERPCThrowOutExtraData(dcerpc, input + parsed,
                                                         input_len);
                        if (retval && retval <= input_len) {
                            input_len -= retval;
                            parsed += retval;
                            if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                                DCERPCResetParsingState(dcerpc);
                            }
                        } else {
                            SCLogDebug("Error Parsing DCERPC");
                            parsed = 0;
                            (void)parsed; /* for scan-build */
                            input_len = 0;
                            (void)input_len; /* for scan-build */
                            DCERPCResetParsingState(dcerpc);
                            SCReturnInt(0);
                        }
                    }
                }
                break;

            case BIND_ACK:
            case ALTER_CONTEXT_RESP:
                while (dcerpc->bytesprocessed < DCERPC_HDR_LEN + 9
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length
                       && input_len) {
                    retval = DCERPCParseBINDACK(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                        SCLogDebug("DCERPCParseBINDACK processed %u/%u input_len left %u",
                                   dcerpc->bytesprocessed,
                                   dcerpc->dcerpchdr.frag_length, input_len);
                    } else if (input_len) {
                        SCLogDebug("Error parsing %s\n",
                                   (dcerpc->dcerpchdr.type == BIND_ACK) ?
                                   "BIND_ACK" : "ALTER_CONTEXT_RESP");
                        parsed = 0;
                        (void)parsed; /* for scan-build */
                        input_len = 0;
                        (void)input_len; /* for scan-build */
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                while (dcerpc->bytesprocessed < DCERPC_HDR_LEN + 10
                       + dcerpc->dcerpcbindbindack.secondaryaddrlen
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length && input_len) {
                    retval = DCERPCParseSecondaryAddr(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                        SCLogDebug("DCERPCParseSecondaryAddr %u/%u left %u secondaryaddr len(%u)",
                                   dcerpc->bytesprocessed, dcerpc->dcerpchdr.frag_length, input_len,
                                   dcerpc->dcerpcbindbindack.secondaryaddrlen);
                    } else if (input_len) {
                        SCLogDebug("Error parsing Secondary Address");
                        parsed = 0;
                        (void)parsed; /* for scan-build */
                        input_len = 0;
                        (void)input_len; /* for scan-build */
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                if (dcerpc->bytesprocessed == DCERPC_HDR_LEN + 10
                    + dcerpc->dcerpcbindbindack.secondaryaddrlen) {
                    if (dcerpc->bytesprocessed % 4) {
                        dcerpc->pad = (4 - dcerpc->bytesprocessed % 4);
                        dcerpc->padleft = dcerpc->pad;
                    }
                }

                while (dcerpc->bytesprocessed < DCERPC_HDR_LEN + 10
                       + dcerpc->dcerpcbindbindack.secondaryaddrlen + dcerpc->pad
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length && input_len) {
                    retval = PaddingParser(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                        SCLogDebug("PaddingParser %u/%u left %u pad(%u)",
                                   dcerpc->bytesprocessed, dcerpc->dcerpchdr.frag_length, input_len,
                                   dcerpc->pad);
                    } else if (input_len) {
                        SCLogDebug("Error parsing DCERPC Padding");
                        parsed = 0;
                        (void)parsed; /* for scan-build */
                        input_len = 0;
                        (void)input_len; /* for scan-build */
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                while (dcerpc->bytesprocessed >= DCERPC_HDR_LEN + 10 + dcerpc->pad
                       + dcerpc->dcerpcbindbindack.secondaryaddrlen && dcerpc->bytesprocessed
                       < DCERPC_HDR_LEN + 14 + dcerpc->pad + dcerpc->dcerpcbindbindack.secondaryaddrlen
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length && input_len) {
                    retval = DCERPCGetCTXItems(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                        SCLogDebug("DCERPCGetCTXItems %u/%u (%u)", dcerpc->bytesprocessed,
                                   dcerpc->dcerpchdr.frag_length, dcerpc->dcerpcbindbindack.numctxitems);
                    } else if (input_len) {
                        SCLogDebug("Error parsing CTX Items");
                        parsed = 0;
                        (void)parsed; /* for scan-build */
                        input_len = 0;
                        (void)input_len; /* for scan-build */
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                if (dcerpc->bytesprocessed == DCERPC_HDR_LEN + 14 + dcerpc->pad
                    + dcerpc->dcerpcbindbindack.secondaryaddrlen) {
                    dcerpc->dcerpcbindbindack.ctxbytesprocessed = 0;
                }

                while (dcerpc->dcerpcbindbindack.numctxitemsleft && dcerpc->bytesprocessed
                       < dcerpc->dcerpchdr.frag_length && input_len) {
                    retval = DCERPCParseBINDACKCTXItem(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        if (dcerpc->dcerpcbindbindack.ctxbytesprocessed == 24) {
                            dcerpc->dcerpcbindbindack.ctxbytesprocessed = 0;
                        }
                        parsed += retval;
                        input_len -= retval;
                    } else if (input_len) {
                        SCLogDebug("Error parsing CTX Items");
                        parsed = 0;
                        (void)parsed; /* for scan-build */
                        input_len = 0;
                        (void)input_len; /* for scan-build */
                        dcerpc->dcerpcbindbindack.numctxitemsleft = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }
                SCLogDebug("BINDACK processed %u/%u input_len left %u",
                           dcerpc->bytesprocessed,
                           dcerpc->dcerpchdr.frag_length, input_len);

                if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                    /* response and request done */
                    if (dcerpc->dcerpchdr.type == BIND_ACK) {
                        /* update transaction id */
                        dcerpc->transaction_id++;
                        SCLogDebug("transaction_id updated to %"PRIu16,
                                   dcerpc->transaction_id);
                    }
                    DCERPCResetParsingState(dcerpc);
                } else if (dcerpc->bytesprocessed > dcerpc->dcerpchdr.frag_length) {
                    DCERPCResetParsingState(dcerpc);
                    SCReturnInt(0);
                } else {
                    /* temporary fix */
                    if (input_len) {
                        retval = DCERPCThrowOutExtraData(dcerpc, input + parsed,
                                                             input_len);
                        if (retval && retval <= input_len) {
                            input_len -= retval;
                            parsed += retval;
                            if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                                DCERPCResetParsingState(dcerpc);
                            }
                        } else {
                            SCLogDebug("Error Parsing DCERPC");
                            parsed = 0;
                            (void)parsed; /* for scan-build */
                            input_len = 0;
                            (void)input_len; /* for scan-build */
                            DCERPCResetParsingState(dcerpc);
                            SCReturnInt(0);
                        }
                    }
                }
                break;

            case REQUEST:
            case RESPONSE:
                SCLogDebug("parsing DCERPC %s",
                        (dcerpc->dcerpchdr.type == REQUEST) ? "REQUEST" : "RESPONSE");
                if ((dcerpc->dcerpchdr.type == REQUEST &&
                            dcerpc->dcerpcrequest.stub_data_buffer_reset) ||
                    (dcerpc->dcerpchdr.type == RESPONSE &&
                            dcerpc->dcerpcresponse.stub_data_buffer_reset))
                {
                    DCERPCResetStub(dcerpc);
                }

                while (dcerpc->bytesprocessed < DCERPC_HDR_LEN + 8
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length
                       && input_len) {
                    retval = DCERPCParseREQUEST(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                        dcerpc->padleft = dcerpc->dcerpchdr.frag_length - dcerpc->bytesprocessed;
                    } else if (input_len) {
                        SCLogDebug("Error parsing DCERPC %s",
                                   (dcerpc->dcerpchdr.type == REQUEST) ? "REQUEST" : "RESPONSE");
                        parsed = 0;
                        (void)parsed; /* for scan-build */
                        dcerpc->padleft = 0;
                        input_len = 0;
                        (void)input_len; /* for scan-build */
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                while (dcerpc->bytesprocessed >= DCERPC_HDR_LEN + 8
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length
                       && dcerpc->padleft && input_len) {
                    retval = StubDataParser(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                    } else if (input_len) {
                        SCLogDebug("Error parsing DCERPC Stub Data");
                        parsed = 0;
                        (void)parsed; /* for scan-build */
                        input_len = 0;
                        (void)input_len; /* for scan-build */
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                if (dcerpc->dcerpchdr.type == REQUEST) {
                    SCLogDebug("REQUEST processed %u frag length %u opnum %u input_len %u", dcerpc->bytesprocessed,
                               dcerpc->dcerpchdr.frag_length, dcerpc->dcerpcrequest.opnum, input_len);
                } else {
                    SCLogDebug("RESPONSE processed %u frag length %u opnum %u input_len %u", dcerpc->bytesprocessed,
                               dcerpc->dcerpchdr.frag_length, dcerpc->dcerpcrequest.opnum, input_len);
                }

                /* don't see how we could break the parser for request pdus, by
                 * pusing bytesprocessed beyond frag_length.  Let's have the
                 * check anyways */
                if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                    DCERPCResetParsingState(dcerpc);
                } else if (dcerpc->bytesprocessed > dcerpc->dcerpchdr.frag_length) {
                    DCERPCResetParsingState(dcerpc);
                    SCReturnInt(0);
                } else {
                    /* temporary fix */
                    if (input_len) {
                        retval = DCERPCThrowOutExtraData(dcerpc, input + parsed,
                                                             input_len);
                        if (retval && retval <= input_len) {
                            input_len -= retval;
                            parsed += retval;
                            if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                                DCERPCResetParsingState(dcerpc);
                            }
                        } else {
                            SCLogDebug("Error Parsing DCERPC");
                            parsed = 0;
                            (void)parsed; /* for scan-build */
                            input_len = 0;
                            (void)input_len; /* for scan-build */
                            DCERPCResetParsingState(dcerpc);
                            SCReturnInt(0);
                        }
                    }
                }

                /* response and request done */
                if (dcerpc->dcerpchdr.type == RESPONSE) {
                    /* update transaction id */
                    dcerpc->transaction_id++;
                    SCLogDebug("transaction_id updated to %"PRIu16,
                               dcerpc->transaction_id);
                }
                break;

            default:
                SCLogDebug("DCERPC Type 0x%02x not implemented yet", dcerpc->dcerpchdr.type);
                retval = DCERPCThrowOutExtraData(dcerpc, input + parsed,
                                                 input_len);
                if (retval && retval <= input_len) {
                    input_len -= retval;
                    parsed += retval;
                    if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                        DCERPCResetParsingState(dcerpc);
                    }
                } else {
                    SCLogDebug("Error Parsing DCERPC");
                    parsed = 0;
                    (void)parsed; /* for scan-build */
                    input_len = 0;
                    (void)input_len; /* for scan-build */
                    DCERPCResetParsingState(dcerpc);
                    SCReturnInt(0);
                }
                break;
        }
    }

    SCReturnInt(parsed);
}

static int DCERPCParse(Flow *f, void *dcerpc_state,
                       AppLayerParserState *pstate,
                       const uint8_t *input, uint32_t input_len,
                       void *local_data, int dir)
{
    SCEnter();

    int32_t retval = 0;
    DCERPCState *sstate = (DCERPCState *) dcerpc_state;

    if (input == NULL && AppLayerParserStateIssetFlag(pstate, APP_LAYER_PARSER_EOF)) {
        SCReturnInt(1);
    } else if (input == NULL || input_len == 0) {
        SCReturnInt(-1);
    }

    if (sstate->dcerpc.bytesprocessed != 0 && sstate->data_needed_for_dir != dir) {
        SCReturnInt(-1);
    }

    retval = DCERPCParser(&sstate->dcerpc, input, input_len);
    if (retval == -1) {
        SCReturnInt(0);
    }

    sstate->data_needed_for_dir = dir;

    if (pstate == NULL)
        SCReturnInt(-1);

    SCReturnInt(1);
}

static int DCERPCParseRequest(Flow *f, void *dcerpc_state,
                              AppLayerParserState *pstate,
                              const uint8_t *input, uint32_t input_len,
                              void *local_data, const uint8_t flags)
{
    return DCERPCParse(f, dcerpc_state, pstate, input, input_len,
                       local_data, 0);
}

static int DCERPCParseResponse(Flow *f, void *dcerpc_state,
                               AppLayerParserState *pstate,
                               const uint8_t *input, uint32_t input_len,
                               void *local_data, const uint8_t flags)
{
    return DCERPCParse(f, dcerpc_state, pstate, input, input_len,
                       local_data, 1);
}

void DCERPCInit(DCERPC *dcerpc)
{
    dcerpc->transaction_id = 1;

    TAILQ_INIT(&dcerpc->dcerpcbindbindack.uuid_list);
    TAILQ_INIT(&dcerpc->dcerpcbindbindack.accepted_uuid_list);
}

static void *DCERPCStateAlloc(void)
{
    SCEnter();

    DCERPCState *s = SCCalloc(1, sizeof(DCERPCState));
    if (unlikely(s == NULL)) {
        SCReturnPtr(NULL, "void");
    }

    DCERPCInit(&s->dcerpc);

    SCReturnPtr((void *)s, "void");
}

void DCERPCUuidListFree(DCERPCUuidEntryList *list)
{
    DCERPCUuidEntry *entry;

    while ((entry = TAILQ_FIRST(list))) {
        TAILQ_REMOVE(list, entry, next);
        SCFree(entry);
    }
}

void DCERPCCleanup(DCERPC *dcerpc)
{
    DCERPCUuidListFree(&dcerpc->dcerpcbindbindack.uuid_list);
    DCERPCUuidListFree(&dcerpc->dcerpcbindbindack.accepted_uuid_list);

    if (dcerpc->dcerpcrequest.stub_data_buffer != NULL) {
        SCFree(dcerpc->dcerpcrequest.stub_data_buffer);
        dcerpc->dcerpcrequest.stub_data_buffer = NULL;
        dcerpc->dcerpcrequest.stub_data_buffer_len = 0;
    }
    if (dcerpc->dcerpcresponse.stub_data_buffer != NULL) {
        SCFree(dcerpc->dcerpcresponse.stub_data_buffer);
        dcerpc->dcerpcresponse.stub_data_buffer = NULL;
        dcerpc->dcerpcresponse.stub_data_buffer_len = 0;
    }
}

static void DCERPCStateFree(void *s)
{
    DCERPCState *sstate = (DCERPCState *) s;

    DCERPCCleanup(&sstate->dcerpc);

    if (sstate->de_state != NULL) {
        DetectEngineStateFree(sstate->de_state);
    }

    SCFree(s);
}

static int DCERPCSetTxDetectState(void *vtx, DetectEngineState *de_state)
{
    DCERPCState *dce_state = (DCERPCState *)vtx;
    dce_state->de_state = de_state;
    return 0;
}

static DetectEngineState *DCERPCGetTxDetectState(void *vtx)
{
    DCERPCState *dce_state = (DCERPCState *)vtx;
    return dce_state->de_state;
}

static void DCERPCStateTransactionFree(void *state, uint64_t tx_id)
{
    /* do nothing */
}

static void *DCERPCGetTx(void *state, uint64_t tx_id)
{
    DCERPCState *dce_state = (DCERPCState *)state;
    return dce_state;
}

static uint64_t DCERPCGetTxCnt(void *state)
{
    /* single tx */
    return 1;
}

static int DCERPCGetAlstateProgressCompletionStatus(uint8_t direction)
{
    return 1;
}

static int DCERPCGetAlstateProgress(void *tx, uint8_t direction)
{
    return 0;
}

static void DCERPCSetTxDetectFlags(void *vtx, uint8_t dir, uint64_t flags)
{
    DCERPCState *dcerpc_state = (DCERPCState *)vtx;
    if (dir & STREAM_TOSERVER) {
        dcerpc_state->detect_flags_ts = flags;
    } else {
        dcerpc_state->detect_flags_tc = flags;
    }
}

static uint64_t DCERPCGetTxDetectFlags(void *vtx, uint8_t dir)
{
    DCERPCState *dcerpc_state = (DCERPCState *)vtx;
    if (dir & STREAM_TOSERVER) {
        return dcerpc_state->detect_flags_ts;
    } else {
        return dcerpc_state->detect_flags_tc;
    }
}

static int DCERPCRegisterPatternsForProtocolDetection(void)
{
    if (AppLayerProtoDetectPMRegisterPatternCS(IPPROTO_TCP, ALPROTO_DCERPC,
                                               "|05 00|", 2, 0, STREAM_TOSERVER) < 0)
    {
        return -1;
    }
    if (AppLayerProtoDetectPMRegisterPatternCS(IPPROTO_TCP, ALPROTO_DCERPC,
                                               "|05 00|", 2, 0, STREAM_TOCLIENT) < 0)
    {
        return -1;
    }

    return 0;
}

void RegisterDCERPCParsers(void)
{
    const char *proto_name = "dcerpc";

    if (AppLayerProtoDetectConfProtoDetectionEnabled("tcp", proto_name)) {
        AppLayerProtoDetectRegisterProtocol(ALPROTO_DCERPC, proto_name);
        if (DCERPCRegisterPatternsForProtocolDetection() < 0)
            return;
    } else {
        SCLogInfo("Protocol detection and parser disabled for %s protocol.",
                  proto_name);
        return;
    }

    if (AppLayerParserConfParserEnabled("tcp", proto_name)) {
        AppLayerParserRegisterParser(IPPROTO_TCP, ALPROTO_DCERPC, STREAM_TOSERVER,
                                     DCERPCParseRequest);
        AppLayerParserRegisterParser(IPPROTO_TCP, ALPROTO_DCERPC, STREAM_TOCLIENT,
                                     DCERPCParseResponse);
        AppLayerParserRegisterStateFuncs(IPPROTO_TCP, ALPROTO_DCERPC, DCERPCStateAlloc,
                                         DCERPCStateFree);
        AppLayerParserRegisterParserAcceptableDataDirection(IPPROTO_TCP, ALPROTO_DCERPC, STREAM_TOSERVER);


        AppLayerParserRegisterTxFreeFunc(IPPROTO_TCP, ALPROTO_DCERPC, DCERPCStateTransactionFree);

        AppLayerParserRegisterDetectStateFuncs(IPPROTO_TCP, ALPROTO_DCERPC,
                                               DCERPCGetTxDetectState, DCERPCSetTxDetectState);

        AppLayerParserRegisterGetTx(IPPROTO_TCP, ALPROTO_DCERPC, DCERPCGetTx);

        AppLayerParserRegisterGetTxCnt(IPPROTO_TCP, ALPROTO_DCERPC, DCERPCGetTxCnt);

        AppLayerParserRegisterGetStateProgressFunc(IPPROTO_TCP, ALPROTO_DCERPC, DCERPCGetAlstateProgress);

        AppLayerParserRegisterGetStateProgressCompletionStatus(ALPROTO_DCERPC,
                                                               DCERPCGetAlstateProgressCompletionStatus);
        AppLayerParserRegisterDetectFlagsFuncs(IPPROTO_TCP, ALPROTO_DCERPC,
                DCERPCGetTxDetectFlags, DCERPCSetTxDetectFlags);
    } else {
        SCLogInfo("Parsed disabled for %s protocol. Protocol detection"
                  "still on.", proto_name);
    }
#ifdef UNITTESTS
    AppLayerParserRegisterProtocolUnittests(IPPROTO_TCP, ALPROTO_DCERPC, DCERPCParserRegisterTests);
#endif

    return;
}

/* UNITTESTS */
#ifdef UNITTESTS
/** \test DCERPC Header Parsing and  BIND / BIND_ACK multiple UUID handling
*/

/* set this to 1 to see problem */

static int DCERPCParserTest01(void)
{
    int result = 1;
    Flow f;
    uint8_t dcerpcbind[] = {
        0x05, 0x00,
        0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x3c, 0x04,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd0, 0x16,
        0xd0, 0x16, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x2c, 0xd0,
        0x28, 0xda, 0x76, 0x91, 0xf6, 0x6e, 0xcb, 0x0f,
        0xbf, 0x85, 0xcd, 0x9b, 0xf6, 0x39, 0x01, 0x00,
        0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x01, 0x00, 0x2c, 0x75, 0xce, 0x7e, 0x82, 0x3b,
        0x06, 0xac, 0x1b, 0xf0, 0xf5, 0xb7, 0xa7, 0xf7,
        0x28, 0xaf, 0x05, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x01, 0x00, 0xe3, 0xb2,
        0x10, 0xd1, 0xd0, 0x0c, 0xcc, 0x3d, 0x2f, 0x80,
        0x20, 0x7c, 0xef, 0xe7, 0x09, 0xe0, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x01, 0x00, 0xde, 0x85, 0x70, 0xc4, 0x02, 0x7c,
        0x60, 0x23, 0x67, 0x0c, 0x22, 0xbf, 0x18, 0x36,
        0x79, 0x17, 0x01, 0x00, 0x02, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x01, 0x00, 0x41, 0x65,
        0x29, 0x51, 0xaa, 0xe7, 0x7b, 0xa8, 0xf2, 0x37,
        0x0b, 0xd0, 0x3f, 0xb3, 0x36, 0xed, 0x05, 0x00,
        0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x05, 0x00,
        0x01, 0x00, 0x14, 0x96, 0x80, 0x01, 0x2e, 0x78,
        0xfb, 0x5d, 0xb4, 0x3c, 0x14, 0xb3, 0x3d, 0xaa,
        0x02, 0xfb, 0x06, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x01, 0x00, 0x3b, 0x04,
        0x68, 0x3e, 0x63, 0xfe, 0x9f, 0xd8, 0x64, 0x55,
        0xcd, 0xe7, 0x39, 0xaf, 0x98, 0x9f, 0x03, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x07, 0x00,
        0x01, 0x00, 0x16, 0x7a, 0x4f, 0x1b, 0xdb, 0x25,
        0x92, 0x55, 0xdd, 0xae, 0x9e, 0x5b, 0x3e, 0x93,
        0x66, 0x93, 0x04, 0x00, 0x01, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x01, 0x00, 0xe8, 0xa4,
        0x8a, 0xcf, 0x95, 0x6c, 0xc7, 0x8f, 0x14, 0xcc,
        0x56, 0xfc, 0x7b, 0x5f, 0x4f, 0xe8, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x01, 0x00, 0xd8, 0xda, 0xfb, 0xbc, 0xa2, 0x55,
        0x6f, 0x5d, 0xc0, 0x2d, 0x88, 0x6f, 0x00, 0x17,
        0x52, 0x8d, 0x06, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x0a, 0x00, 0x01, 0x00, 0x3f, 0x17,
        0x55, 0x0c, 0xf4, 0x23, 0x3c, 0xca, 0xe6, 0xa0,
        0xaa, 0xcc, 0xb5, 0xe3, 0xf9, 0xce, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x0b, 0x00,
        0x01, 0x00, 0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1,
        0xd0, 0x11, 0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9,
        0x2e, 0xf5, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x0c, 0x00, 0x01, 0x00, 0xc9, 0x9f,
        0x3e, 0x6e, 0x82, 0x0a, 0x2b, 0x28, 0x37, 0x78,
        0xe1, 0x13, 0x70, 0x05, 0x38, 0x4d, 0x01, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x0d, 0x00,
        0x01, 0x00, 0x11, 0xaa, 0x4b, 0x15, 0xdf, 0xa6,
        0x86, 0x3f, 0xfb, 0xe0, 0x09, 0xb7, 0xf8, 0x56,
        0xd2, 0x3f, 0x05, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x0e, 0x00, 0x01, 0x00, 0xee, 0x99,
        0xc4, 0x25, 0x11, 0xe4, 0x95, 0x62, 0x29, 0xfa,
        0xfd, 0x26, 0x57, 0x02, 0xf1, 0xce, 0x03, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x0f, 0x00,
        0x01, 0x00, 0xba, 0x81, 0x9e, 0x1a, 0xdf, 0x2b,
        0xba, 0xe4, 0xd3, 0x17, 0x41, 0x60, 0x6d, 0x2d,
        0x9e, 0x28, 0x03, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x10, 0x00, 0x01, 0x00, 0xa0, 0x24,
        0x03, 0x9a, 0xa9, 0x99, 0xfb, 0xbe, 0x49, 0x11,
        0xad, 0x77, 0x30, 0xaa, 0xbc, 0xb6, 0x02, 0x00,
        0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x11, 0x00,
        0x01, 0x00, 0x32, 0x04, 0x7e, 0xae, 0xec, 0x28,
        0xd1, 0x55, 0x83, 0x4e, 0xc3, 0x47, 0x5d, 0x1d,
        0xc6, 0x65, 0x02, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x12, 0x00, 0x01, 0x00, 0xc6, 0xa4,
        0x81, 0x48, 0x66, 0x2a, 0x74, 0x7d, 0x56, 0x6e,
        0xc5, 0x1d, 0x19, 0xf2, 0xb5, 0xb6, 0x03, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00,
        0x01, 0x00, 0xcb, 0xae, 0xb3, 0xc0, 0x0c, 0xf4,
        0xa4, 0x5e, 0x91, 0x72, 0xdd, 0x53, 0x24, 0x70,
        0x89, 0x02, 0x05, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x14, 0x00, 0x01, 0x00, 0xb8, 0xd0,
        0xa0, 0x1a, 0x5e, 0x7a, 0x2d, 0xfe, 0x35, 0xc6,
        0x7d, 0x08, 0x0d, 0x33, 0x73, 0x18, 0x02, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x15, 0x00,
        0x01, 0x00, 0x21, 0xd3, 0xaa, 0x09, 0x03, 0xa7,
        0x0b, 0xc2, 0x06, 0x45, 0xd9, 0x6c, 0x75, 0xc2,
        0x15, 0xa8, 0x01, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x16, 0x00, 0x01, 0x00, 0xe1, 0xbd,
        0x59, 0xfc, 0xbc, 0xa9, 0x95, 0xc2, 0x68, 0x79,
        0xf3, 0x75, 0xe0, 0xae, 0x6c, 0xe5, 0x04, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x17, 0x00,
        0x01, 0x00, 0x06, 0x52, 0xb4, 0x71, 0x70, 0x15,
        0x4e, 0xf5, 0x7f, 0x08, 0x86, 0x14, 0xe6, 0x17,
        0xd5, 0x97, 0x04, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00};

    uint8_t dcerpcbindack[] = {
        0x05, 0x00, 0x0c, 0x03,
        0x10, 0x00, 0x00, 0x00, 0x6c, 0x02, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xb8, 0x10, 0xb8, 0x10,
        0xce, 0x47, 0x00, 0x00, 0x0c, 0x00, 0x5c, 0x50,
        0x49, 0x50, 0x45, 0x5c, 0x6c, 0x73, 0x61, 0x73,
        0x73, 0x00, 0xf6, 0x6e, 0x18, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    uint8_t dcerpcrequest[] = {
        0x05, 0x00, 0x00, 0x00, 0x10,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00, 0x0b,
        0x00, 0x09, 0x00, 0x45, 0x00, 0x2c, 0x00, 0x4d,
        0x00, 0x73, 0x00, 0x53, 0x00, 0x59, 0x00, 0x2a,
        0x00, 0x4a, 0x00, 0x7a, 0x00, 0x3e, 0x00, 0x58,
        0x00, 0x21, 0x00, 0x4a, 0x00, 0x30, 0x00, 0x41,
        0x00, 0x4b, 0x00, 0x4b, 0x00, 0x3c, 0x00, 0x48,
        0x00, 0x24, 0x00, 0x38, 0x00, 0x54, 0x00, 0x60,
        0x00, 0x2d, 0x00, 0x29, 0x00, 0x64, 0x00, 0x5b,
        0x00, 0x77, 0x00, 0x3a, 0x00, 0x4c, 0x00, 0x24,
        0x00, 0x23, 0x00, 0x66, 0x00, 0x43, 0x00, 0x68,
        0x00, 0x22, 0x00, 0x55, 0x00, 0x29, 0x00, 0x2c,
        0x00, 0x4f, 0x00, 0x5a, 0x00, 0x50, 0x00, 0x61,
        0x00, 0x2a, 0x00, 0x6f, 0x00, 0x2f, 0x00, 0x4d,
        0x00, 0x68, 0x00, 0x3a, 0x00, 0x5c, 0x00, 0x67,
        0x00, 0x68, 0x00, 0x68, 0x00, 0x49, 0x00, 0x45,
        0x00, 0x4c, 0x00, 0x72, 0x00, 0x53, 0x00, 0x4c,
        0x00, 0x25, 0x00, 0x4d, 0x00, 0x67, 0x00, 0x2e,
        0x00, 0x4f, 0x00, 0x64, 0x00, 0x61, 0x00, 0x73,
        0x00, 0x24, 0x00, 0x46, 0x00, 0x35, 0x00, 0x2e,
        0x00, 0x45, 0x00, 0x6f, 0x00, 0x40, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x38, 0x00, 0x47, 0x00, 0x71,
        0x00, 0x5a, 0x00, 0x37, 0x00, 0x7a, 0x00, 0x35,
        0x00, 0x6b, 0x00, 0x3c, 0x00, 0x26, 0x00, 0x37,
        0x00, 0x69, 0x00, 0x75, 0x00, 0x36, 0x00, 0x37,
        0x00, 0x47, 0x00, 0x21, 0x00, 0x2d, 0x00, 0x69,
        0x00, 0x37, 0x00, 0x78, 0x00, 0x5f, 0x00, 0x72,
        0x00, 0x4b, 0x00, 0x5c, 0x00, 0x74, 0x00, 0x3e,
        0x00, 0x52, 0x00, 0x7a, 0x00, 0x49, 0x00, 0x31,
        0x00, 0x5a, 0x00, 0x7b, 0x00, 0x29, 0x00, 0x3b,
        0x00, 0x78, 0x00, 0x3b, 0x00, 0x55, 0x00, 0x3e,
        0x00, 0x35, 0x00, 0x2b, 0x00, 0x4e, 0x00, 0x4f,
        0x00, 0x59, 0x00, 0x38, 0x00, 0x2a, 0x00, 0x59,
        0x00, 0x6b, 0x00, 0x42, 0x00, 0x4c, 0x00, 0x3e,
        0x00, 0x6a, 0x00, 0x49, 0x00, 0x2c, 0x00, 0x79,
        0x00, 0x6e, 0x00, 0x35, 0x00, 0x4f, 0x00, 0x49,
        0x00, 0x55, 0x00, 0x35, 0x00, 0x61, 0x00, 0x72,
        0x00, 0x77, 0x00, 0x38, 0x00, 0x32, 0x00, 0x24,
        0x00, 0x46, 0x00, 0x32, 0x00, 0x32, 0x00, 0x27,
        0x00, 0x64, 0x00, 0x5a, 0x00, 0x77, 0x00, 0x2e,
        0x00, 0x37, 0x00, 0x77, 0x00, 0x2e, 0x00, 0x28,
        0x00, 0x63, 0x00, 0x4f, 0x00, 0x67, 0x00, 0x64,
        0x00, 0x39, 0x00, 0x37, 0x00, 0x31, 0x00, 0x30,
        0x00, 0x28, 0x00, 0x2e, 0x00, 0x6f, 0x00, 0x3e,
        0x00, 0x59, 0x00, 0x28, 0x00, 0x67, 0x00, 0x52,
        0x00, 0x35, 0x00, 0x5a, 0x00, 0x7c, 0x00, 0x56,
        0x00, 0x6a, 0x00, 0x5c, 0x00, 0x3c, 0x00, 0x30,
        0x00, 0x59, 0x00, 0x5c, 0x00, 0x5e, 0x00, 0x38,
        0x00, 0x54, 0x00, 0x5c, 0x00, 0x5b, 0x00, 0x42,
        0x00, 0x62, 0x00, 0x70, 0x00, 0x34, 0x00, 0x5c,
        0x00, 0x57, 0x00, 0x7a, 0x00, 0x4b, 0x00, 0x2f,
        0x00, 0x6b, 0x00, 0x6a, 0x00, 0x4f, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x52, 0x00, 0x36, 0x00, 0x27,
        0x00, 0x30, 0x00, 0x6d, 0x00, 0x4a, 0x00, 0x30,
        0x00, 0x78, 0x00, 0x46, 0x00, 0x65, 0x00, 0x4e,
        0x00, 0x29, 0x00, 0x66, 0x00, 0x3f, 0x00, 0x72,
        0x00, 0x71, 0x00, 0x75, 0x00, 0x4c, 0x00, 0x2b,
        0x00, 0x5c, 0x00, 0x46, 0x00, 0x52, 0x00, 0x7b,
        0x00, 0x5c, 0x00, 0x69, 0x00, 0x66, 0x00, 0x56,
        0x00, 0x31, 0x00, 0x2d, 0x00, 0x72, 0x00, 0x61,
        0x00, 0x68, 0x00, 0x28, 0x00, 0x7d, 0x00, 0x58,
        0x00, 0x2a, 0x00, 0x7b, 0x00, 0x28, 0x00, 0x5b,
        0x00, 0x54, 0x00, 0x3a, 0x00, 0x26, 0x00, 0x52,
        0x00, 0x44, 0x00, 0x60, 0x00, 0x50, 0x00, 0x65,
        0x00, 0x48, 0x00, 0x7d, 0x00, 0x2a, 0x00, 0x74,
        0x00, 0x49, 0x00, 0x7b, 0x00, 0x21, 0x00, 0x61,
        0x00, 0x52, 0x00, 0x43, 0x00, 0x5f, 0x00, 0x5a,
        0x00, 0x74, 0x00, 0x5c, 0x00, 0x62, 0x00, 0x68,
        0x00, 0x6c, 0x00, 0x6c, 0x00, 0x2b, 0x00, 0x6f,
        0x00, 0x7c, 0x00, 0x42, 0x00, 0x67, 0x00, 0x32,
        0x00, 0x58, 0x00, 0x35, 0x00, 0x30, 0x00, 0x2f,
        0x00, 0x2d, 0x00, 0x60, 0x00, 0x62, 0x00, 0x51,
        0x00, 0x2a, 0x00, 0x30, 0x00, 0x31, 0x00, 0x48,
        0x00, 0x5b, 0x00, 0x5b, 0x00, 0x5d, 0x00, 0x25,
        0x00, 0x58, 0x00, 0x4a, 0x00, 0x76, 0x00, 0x32,
        0x00, 0x62, 0x00, 0x27, 0x00, 0x42, 0x00, 0x40,
        0x00, 0x53, 0x00, 0x7c, 0x00, 0x7d, 0x00, 0x50,
        0x00, 0x3d, 0x00, 0x40, 0x00, 0x76, 0x00, 0x38,
        0x00, 0x58, 0x00, 0x39, 0x00, 0x63, 0x00, 0x3c,
        0x00, 0x5b, 0x00, 0x23, 0x00, 0x53, 0x00, 0x7a,
        0x00, 0x54, 0x00, 0x74, 0x00, 0x61, 0x00, 0x76,
        0x00, 0x4a, 0x00, 0x3e, 0x00, 0x33, 0x00, 0x75,
        0x00, 0x66, 0x00, 0x2d, 0x00, 0x48, 0x00, 0x33,
        0x00, 0x71, 0x00, 0x76, 0x00, 0x48, 0x00, 0x71,
        0x00, 0x41, 0x00, 0x6f, 0x00, 0x2a, 0x00, 0x67,
        0x00, 0x70, 0x00, 0x21, 0x00, 0x70, 0x00, 0x4b,
        0x00, 0x52, 0x00, 0x58, 0x00, 0x68, 0x00, 0x23,
        0x00, 0x39, 0x00, 0x46, 0x00, 0x4d, 0x00, 0x51,
        0x00, 0x57, 0x00, 0x3a, 0x00, 0x79, 0x00, 0x7b,
        0x00, 0x6c, 0x00, 0x55, 0x00, 0x33, 0x00, 0x65,
        0x00, 0x49, 0x00, 0x72, 0x00, 0x30, 0x00, 0x4f,
        0x00, 0x41, 0x00, 0x6e, 0x00, 0x31, 0x00, 0x4a,
        0x00, 0x60, 0x00, 0x79, 0x00, 0x70, 0x00, 0x4f,
        0x00, 0x58, 0x00, 0x75, 0x00, 0x44, 0x00, 0x59,
        0x00, 0x58, 0x00, 0x46, 0x00, 0x3d, 0x00, 0x46,
        0x00, 0x74, 0x00, 0x51, 0x00, 0x57, 0x00, 0x6e,
        0x00, 0x2d, 0x00, 0x47, 0x00, 0x23, 0x00, 0x45,
        0x00, 0x60, 0x00, 0x4c, 0x00, 0x72, 0x00, 0x4e,
        0x00, 0x74, 0x00, 0x40, 0x00, 0x76, 0x00, 0x75,
        0x00, 0x74, 0x00, 0x56, 0x00, 0x44, 0x00, 0x29,
        0x00, 0x62, 0x00, 0x58, 0x00, 0x31, 0x00, 0x78,
        0x00, 0x32, 0x00, 0x52, 0x00, 0x4a, 0x00, 0x6b,
        0x00, 0x55, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x6f,
        0x00, 0x4a, 0x00, 0x54, 0x00, 0x7d, 0x00, 0x68,
        0x00, 0x3f, 0x00, 0x28, 0x00, 0x21, 0x00, 0x53,
        0x00, 0x48, 0x00, 0x5a, 0x00, 0x34, 0x00, 0x36,
        0x00, 0x35, 0x00, 0x64, 0x00, 0x4e, 0x00, 0x75,
        0x00, 0x69, 0x00, 0x23, 0x00, 0x75, 0x00, 0x55,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x2f, 0x00, 0x73,
        0x00, 0x62, 0x00, 0x6f, 0x00, 0x37, 0x00, 0x4e,
        0x00, 0x25, 0x00, 0x25, 0x00, 0x21, 0x00, 0x3d,
        0x00, 0x3c, 0x00, 0x71, 0x00, 0x3e, 0x00, 0x3f,
        0x00, 0x30, 0x00, 0x36, 0x00, 0x62, 0x00, 0x63,
        0x00, 0x53, 0x00, 0x54, 0x00, 0x5d, 0x00, 0x61,
        0x00, 0x4c, 0x00, 0x28, 0x00, 0x2b, 0x00, 0x4c,
        0x00, 0x4e, 0x00, 0x66, 0x00, 0x5f, 0x00, 0x4b,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x45, 0x00, 0x37,
        0x00, 0x28, 0x00, 0x56, 0x00, 0x36, 0x00, 0x6a,
        0x00, 0x3e, 0x00, 0x64, 0x00, 0x34, 0x00, 0x6a,
        0x00, 0x7d, 0x00, 0x4a, 0x00, 0x66, 0x00, 0x7a,
        0x00, 0x3e, 0x00, 0x75, 0x00, 0x38, 0x00, 0x7b,
        0x00, 0x42, 0x00, 0x76, 0x00, 0x29, 0x00, 0x4c,
        0x00, 0x65, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x4b,
        0x00, 0x2b, 0x00, 0x51, 0x00, 0x47, 0x00, 0x22,
        0x00, 0x48, 0x00, 0x3d, 0x00, 0x49, 0x00, 0x44,
        0x00, 0x5d, 0x00, 0x59, 0x00, 0x63, 0x00, 0x5c,
        0x00, 0x24, 0x00, 0x35, 0x00, 0x34, 0x00, 0x70,
        0x00, 0x69, 0x00};
    uint32_t requestlen = sizeof(dcerpcrequest);

    uint32_t bindlen = sizeof(dcerpcbind);
    uint32_t bindacklen = sizeof(dcerpcbindack);
    TcpSession ssn;
    DCERPCUuidEntry *uuid_entry;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    int r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                                STREAM_TOSERVER | STREAM_START, dcerpcbind,
                                bindlen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.rpc_vers != 5) {
        printf("expected dcerpc version 0x05, got 0x%02x : ",
                dcerpc_state->dcerpc.dcerpchdr.rpc_vers);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.type != BIND) {
        printf("expected dcerpc type 0x%02x , got 0x%02x : ", BIND, dcerpc_state->dcerpc.dcerpchdr.type);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.frag_length != 1084) {
        printf("expected dcerpc frag_length 0x%02x , got 0x%02x : ", 1084, dcerpc_state->dcerpc.dcerpchdr.frag_length);
        result = 0;
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOCLIENT, dcerpcbindack, bindacklen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);
    if (dcerpc_state->dcerpc.dcerpchdr.type != BIND_ACK) {
        printf("expected dcerpc type 0x%02x , got 0x%02x : ", BIND_ACK, dcerpc_state->dcerpc.dcerpchdr.type);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.frag_length != 620) {
        printf("expected dcerpc frag_length 0x%02x , got 0x%02x : ", 620, dcerpc_state->dcerpc.dcerpchdr.frag_length);
        result = 0;
        goto end;
    }
    TAILQ_FOREACH(uuid_entry, &dcerpc_state->dcerpc.dcerpcbindbindack.uuid_list, next) {
        printUUID("BIND_ACK", uuid_entry);
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER | STREAM_EOF, dcerpcrequest,
                            requestlen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);
    if (dcerpc_state->dcerpc.dcerpchdr.type != REQUEST) {
        printf("expected dcerpc type 0x%02x , got 0x%02x : ", REQUEST, dcerpc_state->dcerpc.dcerpchdr.type);
        result = 0;
        goto end;
    }
end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/** \test DCERPC Request decoding and opnum parsing.
*/
static int DCERPCParserTest02(void)
{
    int result = 1;
    Flow f;
    uint8_t dcerpcrequest[] = {
        0x05, 0x00, 0x00, 0x00, 0x10,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00, 0x0b,
        0x00, 0x09, 0x00, 0x45, 0x00, 0x2c, 0x00, 0x4d,
        0x00, 0x73, 0x00, 0x53, 0x00, 0x59, 0x00, 0x2a,
        0x00, 0x4a, 0x00, 0x7a, 0x00, 0x3e, 0x00, 0x58,
        0x00, 0x21, 0x00, 0x4a, 0x00, 0x30, 0x00, 0x41,
        0x00, 0x4b, 0x00, 0x4b, 0x00, 0x3c, 0x00, 0x48,
        0x00, 0x24, 0x00, 0x38, 0x00, 0x54, 0x00, 0x60,
        0x00, 0x2d, 0x00, 0x29, 0x00, 0x64, 0x00, 0x5b,
        0x00, 0x77, 0x00, 0x3a, 0x00, 0x4c, 0x00, 0x24,
        0x00, 0x23, 0x00, 0x66, 0x00, 0x43, 0x00, 0x68,
        0x00, 0x22, 0x00, 0x55, 0x00, 0x29, 0x00, 0x2c,
        0x00, 0x4f, 0x00, 0x5a, 0x00, 0x50, 0x00, 0x61,
        0x00, 0x2a, 0x00, 0x6f, 0x00, 0x2f, 0x00, 0x4d,
        0x00, 0x68, 0x00, 0x3a, 0x00, 0x5c, 0x00, 0x67,
        0x00, 0x68, 0x00, 0x68, 0x00, 0x49, 0x00, 0x45,
        0x00, 0x4c, 0x00, 0x72, 0x00, 0x53, 0x00, 0x4c,
        0x00, 0x25, 0x00, 0x4d, 0x00, 0x67, 0x00, 0x2e,
        0x00, 0x4f, 0x00, 0x64, 0x00, 0x61, 0x00, 0x73,
        0x00, 0x24, 0x00, 0x46, 0x00, 0x35, 0x00, 0x2e,
        0x00, 0x45, 0x00, 0x6f, 0x00, 0x40, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x38, 0x00, 0x47, 0x00, 0x71,
        0x00, 0x5a, 0x00, 0x37, 0x00, 0x7a, 0x00, 0x35,
        0x00, 0x6b, 0x00, 0x3c, 0x00, 0x26, 0x00, 0x37,
        0x00, 0x69, 0x00, 0x75, 0x00, 0x36, 0x00, 0x37,
        0x00, 0x47, 0x00, 0x21, 0x00, 0x2d, 0x00, 0x69,
        0x00, 0x37, 0x00, 0x78, 0x00, 0x5f, 0x00, 0x72,
        0x00, 0x4b, 0x00, 0x5c, 0x00, 0x74, 0x00, 0x3e,
        0x00, 0x52, 0x00, 0x7a, 0x00, 0x49, 0x00, 0x31,
        0x00, 0x5a, 0x00, 0x7b, 0x00, 0x29, 0x00, 0x3b,
        0x00, 0x78, 0x00, 0x3b, 0x00, 0x55, 0x00, 0x3e,
        0x00, 0x35, 0x00, 0x2b, 0x00, 0x4e, 0x00, 0x4f,
        0x00, 0x59, 0x00, 0x38, 0x00, 0x2a, 0x00, 0x59,
        0x00, 0x6b, 0x00, 0x42, 0x00, 0x4c, 0x00, 0x3e,
        0x00, 0x6a, 0x00, 0x49, 0x00, 0x2c, 0x00, 0x79,
        0x00, 0x6e, 0x00, 0x35, 0x00, 0x4f, 0x00, 0x49,
        0x00, 0x55, 0x00, 0x35, 0x00, 0x61, 0x00, 0x72,
        0x00, 0x77, 0x00, 0x38, 0x00, 0x32, 0x00, 0x24,
        0x00, 0x46, 0x00, 0x32, 0x00, 0x32, 0x00, 0x27,
        0x00, 0x64, 0x00, 0x5a, 0x00, 0x77, 0x00, 0x2e,
        0x00, 0x37, 0x00, 0x77, 0x00, 0x2e, 0x00, 0x28,
        0x00, 0x63, 0x00, 0x4f, 0x00, 0x67, 0x00, 0x64,
        0x00, 0x39, 0x00, 0x37, 0x00, 0x31, 0x00, 0x30,
        0x00, 0x28, 0x00, 0x2e, 0x00, 0x6f, 0x00, 0x3e,
        0x00, 0x59, 0x00, 0x28, 0x00, 0x67, 0x00, 0x52,
        0x00, 0x35, 0x00, 0x5a, 0x00, 0x7c, 0x00, 0x56,
        0x00, 0x6a, 0x00, 0x5c, 0x00, 0x3c, 0x00, 0x30,
        0x00, 0x59, 0x00, 0x5c, 0x00, 0x5e, 0x00, 0x38,
        0x00, 0x54, 0x00, 0x5c, 0x00, 0x5b, 0x00, 0x42,
        0x00, 0x62, 0x00, 0x70, 0x00, 0x34, 0x00, 0x5c,
        0x00, 0x57, 0x00, 0x7a, 0x00, 0x4b, 0x00, 0x2f,
        0x00, 0x6b, 0x00, 0x6a, 0x00, 0x4f, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x52, 0x00, 0x36, 0x00, 0x27,
        0x00, 0x30, 0x00, 0x6d, 0x00, 0x4a, 0x00, 0x30,
        0x00, 0x78, 0x00, 0x46, 0x00, 0x65, 0x00, 0x4e,
        0x00, 0x29, 0x00, 0x66, 0x00, 0x3f, 0x00, 0x72,
        0x00, 0x71, 0x00, 0x75, 0x00, 0x4c, 0x00, 0x2b,
        0x00, 0x5c, 0x00, 0x46, 0x00, 0x52, 0x00, 0x7b,
        0x00, 0x5c, 0x00, 0x69, 0x00, 0x66, 0x00, 0x56,
        0x00, 0x31, 0x00, 0x2d, 0x00, 0x72, 0x00, 0x61,
        0x00, 0x68, 0x00, 0x28, 0x00, 0x7d, 0x00, 0x58,
        0x00, 0x2a, 0x00, 0x7b, 0x00, 0x28, 0x00, 0x5b,
        0x00, 0x54, 0x00, 0x3a, 0x00, 0x26, 0x00, 0x52,
        0x00, 0x44, 0x00, 0x60, 0x00, 0x50, 0x00, 0x65,
        0x00, 0x48, 0x00, 0x7d, 0x00, 0x2a, 0x00, 0x74,
        0x00, 0x49, 0x00, 0x7b, 0x00, 0x21, 0x00, 0x61,
        0x00, 0x52, 0x00, 0x43, 0x00, 0x5f, 0x00, 0x5a,
        0x00, 0x74, 0x00, 0x5c, 0x00, 0x62, 0x00, 0x68,
        0x00, 0x6c, 0x00, 0x6c, 0x00, 0x2b, 0x00, 0x6f,
        0x00, 0x7c, 0x00, 0x42, 0x00, 0x67, 0x00, 0x32,
        0x00, 0x58, 0x00, 0x35, 0x00, 0x30, 0x00, 0x2f,
        0x00, 0x2d, 0x00, 0x60, 0x00, 0x62, 0x00, 0x51,
        0x00, 0x2a, 0x00, 0x30, 0x00, 0x31, 0x00, 0x48,
        0x00, 0x5b, 0x00, 0x5b, 0x00, 0x5d, 0x00, 0x25,
        0x00, 0x58, 0x00, 0x4a, 0x00, 0x76, 0x00, 0x32,
        0x00, 0x62, 0x00, 0x27, 0x00, 0x42, 0x00, 0x40,
        0x00, 0x53, 0x00, 0x7c, 0x00, 0x7d, 0x00, 0x50,
        0x00, 0x3d, 0x00, 0x40, 0x00, 0x76, 0x00, 0x38,
        0x00, 0x58, 0x00, 0x39, 0x00, 0x63, 0x00, 0x3c,
        0x00, 0x5b, 0x00, 0x23, 0x00, 0x53, 0x00, 0x7a,
        0x00, 0x54, 0x00, 0x74, 0x00, 0x61, 0x00, 0x76,
        0x00, 0x4a, 0x00, 0x3e, 0x00, 0x33, 0x00, 0x75,
        0x00, 0x66, 0x00, 0x2d, 0x00, 0x48, 0x00, 0x33,
        0x00, 0x71, 0x00, 0x76, 0x00, 0x48, 0x00, 0x71,
        0x00, 0x41, 0x00, 0x6f, 0x00, 0x2a, 0x00, 0x67,
        0x00, 0x70, 0x00, 0x21, 0x00, 0x70, 0x00, 0x4b,
        0x00, 0x52, 0x00, 0x58, 0x00, 0x68, 0x00, 0x23,
        0x00, 0x39, 0x00, 0x46, 0x00, 0x4d, 0x00, 0x51,
        0x00, 0x57, 0x00, 0x3a, 0x00, 0x79, 0x00, 0x7b,
        0x00, 0x6c, 0x00, 0x55, 0x00, 0x33, 0x00, 0x65,
        0x00, 0x49, 0x00, 0x72, 0x00, 0x30, 0x00, 0x4f,
        0x00, 0x41, 0x00, 0x6e, 0x00, 0x31, 0x00, 0x4a,
        0x00, 0x60, 0x00, 0x79, 0x00, 0x70, 0x00, 0x4f,
        0x00, 0x58, 0x00, 0x75, 0x00, 0x44, 0x00, 0x59,
        0x00, 0x58, 0x00, 0x46, 0x00, 0x3d, 0x00, 0x46,
        0x00, 0x74, 0x00, 0x51, 0x00, 0x57, 0x00, 0x6e,
        0x00, 0x2d, 0x00, 0x47, 0x00, 0x23, 0x00, 0x45,
        0x00, 0x60, 0x00, 0x4c, 0x00, 0x72, 0x00, 0x4e,
        0x00, 0x74, 0x00, 0x40, 0x00, 0x76, 0x00, 0x75,
        0x00, 0x74, 0x00, 0x56, 0x00, 0x44, 0x00, 0x29,
        0x00, 0x62, 0x00, 0x58, 0x00, 0x31, 0x00, 0x78,
        0x00, 0x32, 0x00, 0x52, 0x00, 0x4a, 0x00, 0x6b,
        0x00, 0x55, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x6f,
        0x00, 0x4a, 0x00, 0x54, 0x00, 0x7d, 0x00, 0x68,
        0x00, 0x3f, 0x00, 0x28, 0x00, 0x21, 0x00, 0x53,
        0x00, 0x48, 0x00, 0x5a, 0x00, 0x34, 0x00, 0x36,
        0x00, 0x35, 0x00, 0x64, 0x00, 0x4e, 0x00, 0x75,
        0x00, 0x69, 0x00, 0x23, 0x00, 0x75, 0x00, 0x55,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x2f, 0x00, 0x73,
        0x00, 0x62, 0x00, 0x6f, 0x00, 0x37, 0x00, 0x4e,
        0x00, 0x25, 0x00, 0x25, 0x00, 0x21, 0x00, 0x3d,
        0x00, 0x3c, 0x00, 0x71, 0x00, 0x3e, 0x00, 0x3f,
        0x00, 0x30, 0x00, 0x36, 0x00, 0x62, 0x00, 0x63,
        0x00, 0x53, 0x00, 0x54, 0x00, 0x5d, 0x00, 0x61,
        0x00, 0x4c, 0x00, 0x28, 0x00, 0x2b, 0x00, 0x4c,
        0x00, 0x4e, 0x00, 0x66, 0x00, 0x5f, 0x00, 0x4b,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x45, 0x00, 0x37,
        0x00, 0x28, 0x00, 0x56, 0x00, 0x36, 0x00, 0x6a,
        0x00, 0x3e, 0x00, 0x64, 0x00, 0x34, 0x00, 0x6a,
        0x00, 0x7d, 0x00, 0x4a, 0x00, 0x66, 0x00, 0x7a,
        0x00, 0x3e, 0x00, 0x75, 0x00, 0x38, 0x00, 0x7b,
        0x00, 0x42, 0x00, 0x76, 0x00, 0x29, 0x00, 0x4c,
        0x00, 0x65, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x4b,
        0x00, 0x2b, 0x00, 0x51, 0x00, 0x47, 0x00, 0x22,
        0x00, 0x48, 0x00, 0x3d, 0x00, 0x49, 0x00, 0x44,
        0x00, 0x5d, 0x00, 0x59, 0x00, 0x63, 0x00, 0x5c,
        0x00, 0x24, 0x00, 0x35, 0x00, 0x34, 0x00, 0x70,
        0x00, 0x69, 0x00};
    uint32_t requestlen = sizeof(dcerpcrequest);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    int r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                                STREAM_TOSERVER | STREAM_START,
                                dcerpcrequest,
                                requestlen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.rpc_vers != 5) {
        printf("expected dcerpc version 0x05, got 0x%02x : ",
                dcerpc_state->dcerpc.dcerpchdr.rpc_vers);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.type != REQUEST) {
        printf("expected dcerpc type 0x%02x , got 0x%02x : ", REQUEST, dcerpc_state->dcerpc.dcerpchdr.type);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.frag_length != 1024) {
        printf("expected dcerpc frag_length 0x%02x , got 0x%02x : ", 1024, dcerpc_state->dcerpc.dcerpchdr.frag_length);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpcrequest.opnum != 9) {
        printf("expected dcerpc opnum 0x%02x , got 0x%02x : ", 9, dcerpc_state->dcerpc.dcerpcrequest.opnum);
        result = 0;
        goto end;
    }

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/** \test Test endianness handling
*/
static int DCERPCParserTest03(void)
{
    int result = 1;
    Flow f;
    uint8_t dcerpcrequest[] = {
        0x05, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x09, 0x45, 0x00, 0x2c, 0x00, 0x4d,
        0x00, 0x73, 0x00, 0x53, 0x00, 0x59, 0x00, 0x2a,
        0x00, 0x4a, 0x00, 0x7a, 0x00, 0x3e, 0x00, 0x58,
        0x00, 0x21, 0x00, 0x4a, 0x00, 0x30, 0x00, 0x41,
        0x00, 0x4b, 0x00, 0x4b, 0x00, 0x3c, 0x00, 0x48,
        0x00, 0x24, 0x00, 0x38, 0x00, 0x54, 0x00, 0x60,
        0x00, 0x2d, 0x00, 0x29, 0x00, 0x64, 0x00, 0x5b,
        0x00, 0x77, 0x00, 0x3a, 0x00, 0x4c, 0x00, 0x24,
        0x00, 0x23, 0x00, 0x66, 0x00, 0x43, 0x00, 0x68,
        0x00, 0x22, 0x00, 0x55, 0x00, 0x29, 0x00, 0x2c,
        0x00, 0x4f, 0x00, 0x5a, 0x00, 0x50, 0x00, 0x61,
        0x00, 0x2a, 0x00, 0x6f, 0x00, 0x2f, 0x00, 0x4d,
        0x00, 0x68, 0x00, 0x3a, 0x00, 0x5c, 0x00, 0x67,
        0x00, 0x68, 0x00, 0x68, 0x00, 0x49, 0x00, 0x45,
        0x00, 0x4c, 0x00, 0x72, 0x00, 0x53, 0x00, 0x4c,
        0x00, 0x25, 0x00, 0x4d, 0x00, 0x67, 0x00, 0x2e,
        0x00, 0x4f, 0x00, 0x64, 0x00, 0x61, 0x00, 0x73,
        0x00, 0x24, 0x00, 0x46, 0x00, 0x35, 0x00, 0x2e,
        0x00, 0x45, 0x00, 0x6f, 0x00, 0x40, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x38, 0x00, 0x47, 0x00, 0x71,
        0x00, 0x5a, 0x00, 0x37, 0x00, 0x7a, 0x00, 0x35,
        0x00, 0x6b, 0x00, 0x3c, 0x00, 0x26, 0x00, 0x37,
        0x00, 0x69, 0x00, 0x75, 0x00, 0x36, 0x00, 0x37,
        0x00, 0x47, 0x00, 0x21, 0x00, 0x2d, 0x00, 0x69,
        0x00, 0x37, 0x00, 0x78, 0x00, 0x5f, 0x00, 0x72,
        0x00, 0x4b, 0x00, 0x5c, 0x00, 0x74, 0x00, 0x3e,
        0x00, 0x52, 0x00, 0x7a, 0x00, 0x49, 0x00, 0x31,
        0x00, 0x5a, 0x00, 0x7b, 0x00, 0x29, 0x00, 0x3b,
        0x00, 0x78, 0x00, 0x3b, 0x00, 0x55, 0x00, 0x3e,
        0x00, 0x35, 0x00, 0x2b, 0x00, 0x4e, 0x00, 0x4f,
        0x00, 0x59, 0x00, 0x38, 0x00, 0x2a, 0x00, 0x59,
        0x00, 0x6b, 0x00, 0x42, 0x00, 0x4c, 0x00, 0x3e,
        0x00, 0x6a, 0x00, 0x49, 0x00, 0x2c, 0x00, 0x79,
        0x00, 0x6e, 0x00, 0x35, 0x00, 0x4f, 0x00, 0x49,
        0x00, 0x55, 0x00, 0x35, 0x00, 0x61, 0x00, 0x72,
        0x00, 0x77, 0x00, 0x38, 0x00, 0x32, 0x00, 0x24,
        0x00, 0x46, 0x00, 0x32, 0x00, 0x32, 0x00, 0x27,
        0x00, 0x64, 0x00, 0x5a, 0x00, 0x77, 0x00, 0x2e,
        0x00, 0x37, 0x00, 0x77, 0x00, 0x2e, 0x00, 0x28,
        0x00, 0x63, 0x00, 0x4f, 0x00, 0x67, 0x00, 0x64,
        0x00, 0x39, 0x00, 0x37, 0x00, 0x31, 0x00, 0x30,
        0x00, 0x28, 0x00, 0x2e, 0x00, 0x6f, 0x00, 0x3e,
        0x00, 0x59, 0x00, 0x28, 0x00, 0x67, 0x00, 0x52,
        0x00, 0x35, 0x00, 0x5a, 0x00, 0x7c, 0x00, 0x56,
        0x00, 0x6a, 0x00, 0x5c, 0x00, 0x3c, 0x00, 0x30,
        0x00, 0x59, 0x00, 0x5c, 0x00, 0x5e, 0x00, 0x38,
        0x00, 0x54, 0x00, 0x5c, 0x00, 0x5b, 0x00, 0x42,
        0x00, 0x62, 0x00, 0x70, 0x00, 0x34, 0x00, 0x5c,
        0x00, 0x57, 0x00, 0x7a, 0x00, 0x4b, 0x00, 0x2f,
        0x00, 0x6b, 0x00, 0x6a, 0x00, 0x4f, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x52, 0x00, 0x36, 0x00, 0x27,
        0x00, 0x30, 0x00, 0x6d, 0x00, 0x4a, 0x00, 0x30,
        0x00, 0x78, 0x00, 0x46, 0x00, 0x65, 0x00, 0x4e,
        0x00, 0x29, 0x00, 0x66, 0x00, 0x3f, 0x00, 0x72,
        0x00, 0x71, 0x00, 0x75, 0x00, 0x4c, 0x00, 0x2b,
        0x00, 0x5c, 0x00, 0x46, 0x00, 0x52, 0x00, 0x7b,
        0x00, 0x5c, 0x00, 0x69, 0x00, 0x66, 0x00, 0x56,
        0x00, 0x31, 0x00, 0x2d, 0x00, 0x72, 0x00, 0x61,
        0x00, 0x68, 0x00, 0x28, 0x00, 0x7d, 0x00, 0x58,
        0x00, 0x2a, 0x00, 0x7b, 0x00, 0x28, 0x00, 0x5b,
        0x00, 0x54, 0x00, 0x3a, 0x00, 0x26, 0x00, 0x52,
        0x00, 0x44, 0x00, 0x60, 0x00, 0x50, 0x00, 0x65,
        0x00, 0x48, 0x00, 0x7d, 0x00, 0x2a, 0x00, 0x74,
        0x00, 0x49, 0x00, 0x7b, 0x00, 0x21, 0x00, 0x61,
        0x00, 0x52, 0x00, 0x43, 0x00, 0x5f, 0x00, 0x5a,
        0x00, 0x74, 0x00, 0x5c, 0x00, 0x62, 0x00, 0x68,
        0x00, 0x6c, 0x00, 0x6c, 0x00, 0x2b, 0x00, 0x6f,
        0x00, 0x7c, 0x00, 0x42, 0x00, 0x67, 0x00, 0x32,
        0x00, 0x58, 0x00, 0x35, 0x00, 0x30, 0x00, 0x2f,
        0x00, 0x2d, 0x00, 0x60, 0x00, 0x62, 0x00, 0x51,
        0x00, 0x2a, 0x00, 0x30, 0x00, 0x31, 0x00, 0x48,
        0x00, 0x5b, 0x00, 0x5b, 0x00, 0x5d, 0x00, 0x25,
        0x00, 0x58, 0x00, 0x4a, 0x00, 0x76, 0x00, 0x32,
        0x00, 0x62, 0x00, 0x27, 0x00, 0x42, 0x00, 0x40,
        0x00, 0x53, 0x00, 0x7c, 0x00, 0x7d, 0x00, 0x50,
        0x00, 0x3d, 0x00, 0x40, 0x00, 0x76, 0x00, 0x38,
        0x00, 0x58, 0x00, 0x39, 0x00, 0x63, 0x00, 0x3c,
        0x00, 0x5b, 0x00, 0x23, 0x00, 0x53, 0x00, 0x7a,
        0x00, 0x54, 0x00, 0x74, 0x00, 0x61, 0x00, 0x76,
        0x00, 0x4a, 0x00, 0x3e, 0x00, 0x33, 0x00, 0x75,
        0x00, 0x66, 0x00, 0x2d, 0x00, 0x48, 0x00, 0x33,
        0x00, 0x71, 0x00, 0x76, 0x00, 0x48, 0x00, 0x71,
        0x00, 0x41, 0x00, 0x6f, 0x00, 0x2a, 0x00, 0x67,
        0x00, 0x70, 0x00, 0x21, 0x00, 0x70, 0x00, 0x4b,
        0x00, 0x52, 0x00, 0x58, 0x00, 0x68, 0x00, 0x23,
        0x00, 0x39, 0x00, 0x46, 0x00, 0x4d, 0x00, 0x51,
        0x00, 0x57, 0x00, 0x3a, 0x00, 0x79, 0x00, 0x7b,
        0x00, 0x6c, 0x00, 0x55, 0x00, 0x33, 0x00, 0x65,
        0x00, 0x49, 0x00, 0x72, 0x00, 0x30, 0x00, 0x4f,
        0x00, 0x41, 0x00, 0x6e, 0x00, 0x31, 0x00, 0x4a,
        0x00, 0x60, 0x00, 0x79, 0x00, 0x70, 0x00, 0x4f,
        0x00, 0x58, 0x00, 0x75, 0x00, 0x44, 0x00, 0x59,
        0x00, 0x58, 0x00, 0x46, 0x00, 0x3d, 0x00, 0x46,
        0x00, 0x74, 0x00, 0x51, 0x00, 0x57, 0x00, 0x6e,
        0x00, 0x2d, 0x00, 0x47, 0x00, 0x23, 0x00, 0x45,
        0x00, 0x60, 0x00, 0x4c, 0x00, 0x72, 0x00, 0x4e,
        0x00, 0x74, 0x00, 0x40, 0x00, 0x76, 0x00, 0x75,
        0x00, 0x74, 0x00, 0x56, 0x00, 0x44, 0x00, 0x29,
        0x00, 0x62, 0x00, 0x58, 0x00, 0x31, 0x00, 0x78,
        0x00, 0x32, 0x00, 0x52, 0x00, 0x4a, 0x00, 0x6b,
        0x00, 0x55, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x6f,
        0x00, 0x4a, 0x00, 0x54, 0x00, 0x7d, 0x00, 0x68,
        0x00, 0x3f, 0x00, 0x28, 0x00, 0x21, 0x00, 0x53,
        0x00, 0x48, 0x00, 0x5a, 0x00, 0x34, 0x00, 0x36,
        0x00, 0x35, 0x00, 0x64, 0x00, 0x4e, 0x00, 0x75,
        0x00, 0x69, 0x00, 0x23, 0x00, 0x75, 0x00, 0x55,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x2f, 0x00, 0x73,
        0x00, 0x62, 0x00, 0x6f, 0x00, 0x37, 0x00, 0x4e,
        0x00, 0x25, 0x00, 0x25, 0x00, 0x21, 0x00, 0x3d,
        0x00, 0x3c, 0x00, 0x71, 0x00, 0x3e, 0x00, 0x3f,
        0x00, 0x30, 0x00, 0x36, 0x00, 0x62, 0x00, 0x63,
        0x00, 0x53, 0x00, 0x54, 0x00, 0x5d, 0x00, 0x61,
        0x00, 0x4c, 0x00, 0x28, 0x00, 0x2b, 0x00, 0x4c,
        0x00, 0x4e, 0x00, 0x66, 0x00, 0x5f, 0x00, 0x4b,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x45, 0x00, 0x37,
        0x00, 0x28, 0x00, 0x56, 0x00, 0x36, 0x00, 0x6a,
        0x00, 0x3e, 0x00, 0x64, 0x00, 0x34, 0x00, 0x6a,
        0x00, 0x7d, 0x00, 0x4a, 0x00, 0x66, 0x00, 0x7a,
        0x00, 0x3e, 0x00, 0x75, 0x00, 0x38, 0x00, 0x7b,
        0x00, 0x42, 0x00, 0x76, 0x00, 0x29, 0x00, 0x4c,
        0x00, 0x65, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x4b,
        0x00, 0x2b, 0x00, 0x51, 0x00, 0x47, 0x00, 0x22,
        0x00, 0x48, 0x00, 0x3d, 0x00, 0x49, 0x00, 0x44,
        0x00, 0x5d, 0x00, 0x59, 0x00, 0x63, 0x00, 0x5c,
        0x00, 0x24, 0x00, 0x35, 0x00, 0x34, 0x00, 0x70,
        0x00, 0x69, 0x00};
    uint32_t requestlen = sizeof(dcerpcrequest);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    int r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                                STREAM_TOSERVER | STREAM_START,
                                dcerpcrequest,
                                requestlen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.packed_drep[0] != 0x01) {
        printf("expected dcerpc data representation 0x01, got 0x%02x : ",
                dcerpc_state->dcerpc.dcerpchdr.packed_drep[0]);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.frag_length != 1024) {
        printf("expected dcerpc frag_length 0x%02x , got 0x%02x : ", 1024, dcerpc_state->dcerpc.dcerpchdr.frag_length);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpcrequest.opnum != 9) {
        printf("expected dcerpc opnum 0x%02x , got 0x%02x : ", 9, dcerpc_state->dcerpc.dcerpcrequest.opnum);
        result = 0;
        goto end;
    }
end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test General test.
 */
static int DCERPCParserTest05(void)
{
    int result = 1;
    Flow f;
    int r = 0;
    uint8_t bind1[] = {
        0x05, 0x00, 0x0b, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0xb8, 0x4a, 0x9f, 0x4d, 0x1c, 0x7d, 0xcf, 0x11,
        0x86, 0x1e, 0x00, 0x20, 0xaf, 0x6e, 0x7c, 0x57,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind1_len = sizeof(bind1);

    uint8_t bind2[] = {
        0x05, 0x00, 0x0b, 0x02, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
        0xb8, 0x4a, 0x9f, 0x4d, 0x1c, 0x7d, 0xcf, 0x11,
        0x86, 0x1e, 0x00, 0x20, 0xaf, 0x6e, 0x7c, 0x67,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind2_len = sizeof(bind2);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER | STREAM_START, bind1, bind1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    DCERPCUuidEntry *item = NULL;
    int m = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.uuid_list, next) {
        printf("%d ", m);
        printUUID("BIND",item);
        m++;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, bind2, bind2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    item = NULL;
    m = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.uuid_list, next) {
        printf("%d ", m);
        printUUID("BIND",item);
        m++;
    }

    /* we will need this test later for fragged bind pdus.  keep it */
    result = 1;

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented bind PDU(one PDU which is frag'ed)
 */
static int DCERPCParserTest06(void)
{
    int result = 1;
    Flow f;
    int r = 0;
    uint8_t bind1[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0xdc, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0xc7, 0x70, 0x0d, 0x3e, 0x71, 0x37, 0x39, 0x0d,
        0x3a, 0x4f, 0xd3, 0xdc, 0xca, 0x49, 0xe8, 0xa3,
        0x05, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x01, 0x00, 0x84, 0xb6, 0x55, 0x75,
        0xdb, 0x9e, 0xba, 0x54, 0x56, 0xd3, 0x45, 0x10,
        0xb7, 0x7a, 0x2a, 0xe2, 0x04, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x6e, 0x39, 0x21, 0x24, 0x70, 0x6f, 0x41, 0x57,
        0x54, 0x70, 0xb8, 0xc3, 0x5e, 0x89, 0x3b, 0x43,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x01, 0x00, 0x39, 0x6a, 0x86, 0x5d,
        0x24, 0x0f, 0xd2, 0xf7, 0xb6, 0xce, 0x95, 0x9c,
        0x54, 0x1d, 0x3a, 0xdb, 0x02, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0x12, 0xa5, 0xdd, 0xc5, 0x55, 0xce, 0xc3, 0x46,
        0xbd, 0xa0, 0x94, 0x39, 0x3c, 0x0d, 0x9b, 0x5b,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x01, 0x00, 0x87, 0x1c, 0x8b, 0x6e,
        0x11, 0xa8, 0x67, 0x98, 0xd4, 0x5d, 0xf6, 0x8a,
        0x2f, 0x33, 0x24, 0x7b, 0x05, 0x00, 0x03, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x06, 0x00, 0x01, 0x00,
        0x9b, 0x82, 0x13, 0xd1, 0x28, 0xe0, 0x63, 0xf3,
        0x62, 0xee, 0x76, 0x73, 0xf9, 0xac, 0x3d, 0x2e,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x01, 0x00, 0xa9, 0xd4, 0x73, 0xf2,
        0xed, 0xad, 0xe8, 0x82, 0xf8, 0xcf, 0x9d, 0x9f,
        0x66, 0xe6, 0x43, 0x37, 0x02, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x01, 0x00,
        0x06, 0x2b, 0x85, 0x38, 0x4f, 0x73, 0x96, 0xb1,
        0x73, 0xe1, 0x59, 0xbe, 0x9d, 0xe2, 0x6c, 0x07,
        0x05, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60};
    uint32_t bind1_len = sizeof(bind1);

    uint8_t bind2[] = {
        0x02, 0x00, 0x00, 0x00, 0x09, 0x00, 0x01, 0x00,
        0xbf, 0xfa, 0xbb, 0xa4, 0x9e, 0x5c, 0x80, 0x61,
        0xb5, 0x8b, 0x79, 0x69, 0xa6, 0x32, 0x88, 0x77,
        0x01, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0a, 0x00, 0x01, 0x00, 0x39, 0xa8, 0x2c, 0x39,
        0x73, 0x50, 0x06, 0x8d, 0xf2, 0x37, 0x1e, 0x1e,
        0xa8, 0x8f, 0x46, 0x98, 0x02, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x01, 0x00,
        0x91, 0x13, 0xd0, 0xa7, 0xef, 0xc4, 0xa7, 0x96,
        0x0c, 0x4a, 0x0d, 0x29, 0x80, 0xd3, 0xfe, 0xbf,
        0x00, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x01, 0x00, 0xcc, 0x2b, 0x55, 0x1d,
        0xd4, 0xa4, 0x0d, 0xfb, 0xcb, 0x6f, 0x86, 0x36,
        0xa6, 0x57, 0xc3, 0x21, 0x02, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x01, 0x00,
        0x43, 0x7b, 0x07, 0xee, 0x85, 0xa8, 0xb9, 0x3a,
        0x0f, 0xf9, 0x83, 0x70, 0xe6, 0x0b, 0x4f, 0x33,
        0x02, 0x00, 0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0e, 0x00, 0x01, 0x00, 0x9c, 0x6a, 0x15, 0x8c,
        0xd6, 0x9c, 0xa6, 0xc3, 0xb2, 0x9e, 0x62, 0x9f,
        0x3d, 0x8e, 0x47, 0x73, 0x02, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x01, 0x00,
        0xc8, 0x4f, 0x32, 0x4b, 0x70, 0x16, 0xd3, 0x01,
        0x12, 0x78, 0x5a, 0x47, 0xbf, 0x6e, 0xe1, 0x88,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind2_len = sizeof(bind2);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER | STREAM_START, bind1, bind1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 420);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.ctxbytesprocessed == 40);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.numctxitems == 16);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.numctxitemsleft == 8);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, bind2, bind2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.ctxbytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.numctxitems == 16);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.numctxitemsleft == 0);

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented bind PDU(one PDU which is frag'ed).
 */
static int DCERPCParserTest07(void)
{
    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x2C, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x09, 0x0A, 0x0B, 0x0C
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x0D, 0x0E
    };
    uint32_t request2_len = sizeof(request2);

    uint8_t request3[] = {
        0x0F, 0x10, 0x11, 0x12, 0x13, 0x14
    };
    uint32_t request3_len = sizeof(request3);

    Flow f;
    int r = 0;
    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();
    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));
    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;
    StreamTcpInitConfig(TRUE);

    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER | STREAM_START, request1,
                            request1_len);
    FAIL_IF(r != 0);

    DCERPCState *dcerpc_state = f.alstate;
    FAIL_IF_NULL(dcerpc_state);

    FAIL_IF_NOT(dcerpc_state->dcerpc.bytesprocessed == 36);
    FAIL_IF_NOT(dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL);
    FAIL_IF_NOT(dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 12);

    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, request2, request2_len);
    FAIL_IF(r != 0);

    FAIL_IF_NOT(dcerpc_state->dcerpc.bytesprocessed == 38);
    FAIL_IF_NOT(dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL);
    FAIL_IF_NOT(dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 14);

    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, request3, request3_len);
    FAIL_IF(r != 0);

    FAIL_IF_NOT(dcerpc_state->dcerpc.bytesprocessed == 0);
    FAIL_IF_NOT(dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL);
    FAIL_IF_NOT(dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 20);

    AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    PASS;
}

/**
 * \test DCERPC fragmented bind PDU(one PDU which is frag'ed).
 */
static int DCERPCParserTest08(void)
{
    int result = 1;
    Flow f;
    int r = 0;
    uint8_t request[] = {
        0x05, 0x02, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x2C, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x09, 0x0A, 0x0B, 0x0C,
    };
    uint32_t request_len = sizeof(request);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER | STREAM_START, request,
                            request_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer == NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 0);

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented bind PDU(one PDU which is frag'ed).
 */
static int DCERPCParserTest09(void)
{
    int result = 1;
    Flow f;
    int r = 0;
    uint8_t request[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x2C, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x09, 0x0A, 0x0B, 0x0C,
    };
    uint32_t request_len = sizeof(request);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER | STREAM_START, request,
                            request_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 36);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len ==  12);

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented PDU.
 */
static int DCERPCParserTest10(void)
{
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t fault[] = {
        0x05, 0x00, 0x03, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0xf7, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
    uint32_t fault_len = sizeof(fault);

    uint8_t request1[] = {
        0x05, 0x00
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0x24, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x02,
        0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A,
        0x0B, 0x0C
    };
    uint32_t request2_len = sizeof(request2);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER | STREAM_START, fault, fault_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    result &= (dcerpc_state->dcerpc.bytesprocessed == 2);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer == NULL);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 12);

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented PDU.
 */
static int DCERPCParserTest11(void)
{
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x09, 0x0A, 0x0B, 0x0C
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00
    };
    uint32_t request2_len = sizeof(request2);

    uint8_t request3[] = {
        0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0x26, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x02,
        0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A,
        0x0B, 0x0C, 0xFF, 0xFF
    };
    uint32_t request3_len = sizeof(request3);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len ==  12);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    result &= (dcerpc_state->dcerpc.bytesprocessed == 2);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, request3, request3_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len ==  14);

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented PDU.
 */
static int DCERPCParserTest12(void)
{
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bind_ack1[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x48, 0x1a, 0x00, 0x00,
    };
    uint32_t bind_ack1_len = sizeof(bind_ack1);

    uint8_t bind_ack2[] = {
        0x0c, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6c, 0x73, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack2_len = sizeof(bind_ack2);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOCLIENT, bind_ack1, bind_ack1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 24);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOCLIENT, bind_ack2, bind_ack2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test Check if the parser accepts bind pdus that have context ids starting
 *       from a non-zero value.
 */
static int DCERPCParserTest13(void)
{
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bindbuf[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x7f, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
        0xa0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bindbuf_len = sizeof(bindbuf);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, bindbuf, bindbuf_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.numctxitems == 1);
    if (result == 0)
        goto end;

    result = 0;
    uint8_t ctx_uuid_from_pcap[16] = {
        0x00, 0x00, 0x01, 0xa0, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46};
    DCERPCUuidEntry *item = NULL;
    int internal_id = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (ctx_uuid_from_pcap[i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        result = 1;
        result &= (item->internal_id == internal_id++);
    }

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test Check for another endless loop with bind pdus.
 */
static int DCERPCParserTest14(void)
{
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bindbuf[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x4A, 0x00, 0x00, 0x00, 0x7f, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
        0xa0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x01, 0x02, 0x03, 0x04, 0xFF /* ka boom - endless loop */
    };
    uint32_t bindbuf_len = sizeof(bindbuf);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, bindbuf, bindbuf_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test Check for another endless loop for bind_ack pdus.
 */
static int DCERPCParserTest15(void)
{
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x3e, 0x00, 0x00, 0x00, 0x7f, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0xfd, 0x04, 0x01, 0x00,
        0x04, 0x00, 0x31, 0x33, 0x35, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x01, 0x02, 0x03, 0x04, 0xFF
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOCLIENT, bind_ack, bind_ack_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test Check for correct internal ids for bind_acks.
 */
static int DCERPCParserTest16(void)
{
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bind1[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x58, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x50, 0x08, 0x43, 0x95, 0x43, 0x5a, 0x8b, 0xb2,
        0xf4, 0xc5, 0xb9, 0xee, 0x67, 0x55, 0x7c, 0x19,
        0x00, 0x00, 0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x01, 0x00, 0xda, 0xc2, 0xbc, 0x9b,
        0x35, 0x2e, 0xd4, 0xc9, 0x1f, 0x85, 0x01, 0xe6,
        0x4e, 0x5a, 0x5e, 0xd4, 0x04, 0x00, 0x03, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0xb2, 0x97, 0xcc, 0x14, 0x6f, 0x70, 0x0d, 0xa5,
        0x33, 0xd7, 0xf4, 0xe3, 0x8e, 0xb2, 0x2a, 0x1e,
        0x05, 0x00, 0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x01, 0x00, 0x96, 0x4e, 0xa6, 0xf6,
        0xb2, 0x4b, 0xae, 0xb3, 0x21, 0xf4, 0x97, 0x7c,
        0xcd, 0xa7, 0x08, 0xb0, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0xbc, 0xc0, 0xf7, 0x71, 0x3f, 0x71, 0x54, 0x44,
        0x22, 0xa8, 0x55, 0x0f, 0x98, 0x83, 0x1f, 0xfe,
        0x04, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x01, 0x00, 0xbe, 0x52, 0xf2, 0x58,
        0x4a, 0xc3, 0xb5, 0xd0, 0xba, 0xac, 0xda, 0xf0,
        0x12, 0x99, 0x38, 0x6e, 0x04, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x06, 0x00, 0x01, 0x00,
        0xdb, 0xfa, 0x73, 0x01, 0xb3, 0x81, 0x01, 0xd4,
        0x7f, 0xa0, 0x36, 0xb1, 0x97, 0xae, 0x29, 0x7f,
        0x01, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x01, 0x00, 0x89, 0xbe, 0x41, 0x1d,
        0x38, 0x75, 0xf5, 0xb5, 0xad, 0x27, 0x73, 0xf1,
        0xb0, 0x7a, 0x28, 0x82, 0x05, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x01, 0x00,
        0xf6, 0x87, 0x09, 0x93, 0xb8, 0xa8, 0x20, 0xc4,
        0xb8, 0x63, 0xe6, 0x95, 0xed, 0x59, 0xee, 0x3f,
        0x05, 0x00, 0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x09, 0x00, 0x01, 0x00, 0x92, 0x77, 0x92, 0x68,
        0x3e, 0xa4, 0xbc, 0x3f, 0x44, 0x33, 0x0e, 0xb8,
        0x33, 0x0a, 0x2f, 0xdf, 0x01, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x01, 0x00,
        0xa1, 0x03, 0xd2, 0xa9, 0xd2, 0x16, 0xc9, 0x89,
        0x67, 0x18, 0x3e, 0xb1, 0xee, 0x6b, 0xf9, 0x18,
        0x02, 0x00, 0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x01, 0x00, 0x2f, 0x09, 0x5e, 0x74,
        0xec, 0xa0, 0xbb, 0xc1, 0x60, 0x18, 0xf1, 0x93,
        0x04, 0x17, 0x11, 0xf9, 0x01, 0x00, 0x03, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x01, 0x00,
        0xc8, 0x4f, 0x32, 0x4b, 0x70, 0x16, 0xd3, 0x01,
        0x12, 0x78, 0x5a, 0x47, 0xbf, 0x6e, 0xe1, 0x88,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind1_len = sizeof(bind1);

    uint8_t bind_ack1[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x64, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0xc1, 0x2b, 0x00, 0x00,
        0x0e, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x62, 0x72, 0x6f, 0x77, 0x73, 0x65, 0x72, 0x00,
        0x0d, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack1_len = sizeof(bind_ack1);

    uint8_t bind2[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0xdc, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0xc7, 0x70, 0x0d, 0x3e, 0x71, 0x37, 0x39, 0x0d,
        0x3a, 0x4f, 0xd3, 0xdc, 0xca, 0x49, 0xe8, 0xa3,
        0x05, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x01, 0x00, 0x84, 0xb6, 0x55, 0x75,
        0xdb, 0x9e, 0xba, 0x54, 0x56, 0xd3, 0x45, 0x10,
        0xb7, 0x7a, 0x2a, 0xe2, 0x04, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x6e, 0x39, 0x21, 0x24, 0x70, 0x6f, 0x41, 0x57,
        0x54, 0x70, 0xb8, 0xc3, 0x5e, 0x89, 0x3b, 0x43,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x01, 0x00, 0x39, 0x6a, 0x86, 0x5d,
        0x24, 0x0f, 0xd2, 0xf7, 0xb6, 0xce, 0x95, 0x9c,
        0x54, 0x1d, 0x3a, 0xdb, 0x02, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0x12, 0xa5, 0xdd, 0xc5, 0x55, 0xce, 0xc3, 0x46,
        0xbd, 0xa0, 0x94, 0x39, 0x3c, 0x0d, 0x9b, 0x5b,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x01, 0x00, 0x87, 0x1c, 0x8b, 0x6e,
        0x11, 0xa8, 0x67, 0x98, 0xd4, 0x5d, 0xf6, 0x8a,
        0x2f, 0x33, 0x24, 0x7b, 0x05, 0x00, 0x03, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x06, 0x00, 0x01, 0x00,
        0x9b, 0x82, 0x13, 0xd1, 0x28, 0xe0, 0x63, 0xf3,
        0x62, 0xee, 0x76, 0x73, 0xf9, 0xac, 0x3d, 0x2e,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x01, 0x00, 0xa9, 0xd4, 0x73, 0xf2,
        0xed, 0xad, 0xe8, 0x82, 0xf8, 0xcf, 0x9d, 0x9f,
        0x66, 0xe6, 0x43, 0x37, 0x02, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x01, 0x00,
        0x06, 0x2b, 0x85, 0x38, 0x4f, 0x73, 0x96, 0xb1,
        0x73, 0xe1, 0x59, 0xbe, 0x9d, 0xe2, 0x6c, 0x07,
        0x05, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x09, 0x00, 0x01, 0x00, 0xbf, 0xfa, 0xbb, 0xa4,
        0x9e, 0x5c, 0x80, 0x61, 0xb5, 0x8b, 0x79, 0x69,
        0xa6, 0x32, 0x88, 0x77, 0x01, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x01, 0x00,
        0x39, 0xa8, 0x2c, 0x39, 0x73, 0x50, 0x06, 0x8d,
        0xf2, 0x37, 0x1e, 0x1e, 0xa8, 0x8f, 0x46, 0x98,
        0x02, 0x00, 0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x01, 0x00, 0x91, 0x13, 0xd0, 0xa7,
        0xef, 0xc4, 0xa7, 0x96, 0x0c, 0x4a, 0x0d, 0x29,
        0x80, 0xd3, 0xfe, 0xbf, 0x00, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x01, 0x00,
        0xcc, 0x2b, 0x55, 0x1d, 0xd4, 0xa4, 0x0d, 0xfb,
        0xcb, 0x6f, 0x86, 0x36, 0xa6, 0x57, 0xc3, 0x21,
        0x02, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0d, 0x00, 0x01, 0x00, 0x43, 0x7b, 0x07, 0xee,
        0x85, 0xa8, 0xb9, 0x3a, 0x0f, 0xf9, 0x83, 0x70,
        0xe6, 0x0b, 0x4f, 0x33, 0x02, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x01, 0x00,
        0x9c, 0x6a, 0x15, 0x8c, 0xd6, 0x9c, 0xa6, 0xc3,
        0xb2, 0x9e, 0x62, 0x9f, 0x3d, 0x8e, 0x47, 0x73,
        0x02, 0x00, 0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0f, 0x00, 0x01, 0x00, 0xc8, 0x4f, 0x32, 0x4b,
        0x70, 0x16, 0xd3, 0x01, 0x12, 0x78, 0x5a, 0x47,
        0xbf, 0x6e, 0xe1, 0x88, 0x03, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind2_len = sizeof(bind2);

    uint8_t bind_ack2[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0xac, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0xc2, 0x2b, 0x00, 0x00,
        0x0e, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x62, 0x72, 0x6f, 0x77, 0x73, 0x65, 0x72, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack2_len = sizeof(bind_ack2);

    uint8_t bind3[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x2c, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0xa4, 0x7f, 0x8e, 0xc6, 0xef, 0x56, 0x9b, 0x63,
        0x92, 0xfa, 0x08, 0xb3, 0x35, 0xe2, 0xa5, 0x81,
        0x00, 0x00, 0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x01, 0x00, 0x9f, 0xfc, 0x78, 0xd2,
        0x5f, 0x16, 0x0b, 0xbc, 0xc6, 0xdb, 0x5d, 0xef,
        0xde, 0x54, 0xa2, 0x6f, 0x04, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x78, 0xb8, 0x96, 0xc7, 0x2f, 0xda, 0x11, 0x6b,
        0xd1, 0x28, 0x68, 0xe1, 0xd6, 0x71, 0xac, 0x9d,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x01, 0x00, 0xcf, 0xf4, 0xd7, 0x37,
        0x03, 0xda, 0xcc, 0xe3, 0x3e, 0x34, 0x7f, 0x67,
        0x99, 0x91, 0x41, 0x3d, 0x01, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0x48, 0xeb, 0x32, 0xf0, 0x27, 0xd5, 0x9d, 0xd0,
        0x1e, 0xc6, 0x48, 0x46, 0x97, 0xe9, 0xdb, 0x09,
        0x05, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x01, 0x00, 0x82, 0xec, 0x0d, 0x08,
        0xf2, 0x8f, 0x22, 0x57, 0x42, 0x9b, 0xce, 0xa8,
        0x74, 0x16, 0xc6, 0xec, 0x00, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x06, 0x00, 0x01, 0x00,
        0x2e, 0x00, 0x70, 0x44, 0xee, 0xc9, 0x30, 0x6b,
        0xf4, 0x34, 0x1e, 0x3d, 0x35, 0x0f, 0xf7, 0xf7,
        0x00, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x01, 0x00, 0x59, 0x04, 0x39, 0x3f,
        0x59, 0x87, 0x14, 0x0e, 0x76, 0x8d, 0x17, 0xc2,
        0x47, 0xfa, 0x67, 0x7f, 0x04, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x01, 0x00,
        0x30, 0xd6, 0xed, 0x2e, 0x57, 0xfa, 0xf4, 0x72,
        0x6c, 0x10, 0x0d, 0xe5, 0x51, 0x7f, 0xd0, 0x39,
        0x02, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x09, 0x00, 0x01, 0x00, 0xea, 0x8b, 0x84, 0x4d,
        0x44, 0x43, 0xc1, 0x94, 0x75, 0xe2, 0x81, 0x48,
        0xd8, 0x77, 0xd9, 0xce, 0x05, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x01, 0x00,
        0x89, 0x4f, 0xe7, 0x95, 0xa3, 0xc1, 0x62, 0x36,
        0x26, 0x9e, 0x67, 0xdb, 0x2c, 0x52, 0x89, 0xd3,
        0x01, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x01, 0x00, 0x78, 0x56, 0x34, 0x12,
        0x34, 0x12, 0xcd, 0xab, 0xef, 0x00, 0x01, 0x23,
        0x45, 0x67, 0x89, 0xab, 0x01, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind3_len = sizeof(bind3);

    uint8_t bind_ack3[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x4c, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x1a, 0x33, 0x00, 0x00,
        0x0e, 0x00, 0x5c, 0x70, 0x69, 0x70, 0x65, 0x5c,
        0x73, 0x70, 0x6f, 0x6f, 0x6c, 0x73, 0x73, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack3_len = sizeof(bind_ack3);

    TcpSession ssn;
    DCERPCUuidEntry *item = NULL;
    int count = 0;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    uint8_t accepted_uuids[3][16] = {
        {0x4b, 0x32, 0x4f, 0xc8, 0x16, 0x70, 0x01, 0xd3,
         0x12, 0x78, 0x5a, 0x47, 0xbf, 0x6e, 0xe1, 0x88},
        {0x4b, 0x32, 0x4f, 0xc8, 0x16, 0x70, 0x01, 0xd3,
         0x12, 0x78, 0x5a, 0x47, 0xbf, 0x6e, 0xe1, 0x88},
        {0x12, 0x34, 0x56, 0x78, 0x12, 0x34, 0xab, 0xcd,
         0xef, 0x00, 0x01, 0x23, 0x45, 0x67, 0x89, 0xab}
    };

    uint16_t accepted_ctxids[3] = {12, 15, 11};

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, bind1, bind1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOCLIENT, bind_ack1, bind_ack1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (accepted_uuids[0][i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        if (accepted_ctxids[0] != item->ctxid) {
            result = 0;
            goto end;
        }
        count++;
    }
    if (count != 1) {
        result = 0;
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, bind2, bind2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        count++;
    }
    if (count != 0) {
        result = 0;
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOCLIENT, bind_ack2, bind_ack2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (accepted_uuids[1][i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        if (accepted_ctxids[1] != item->ctxid) {
            result = 0;
            goto end;
        }
        count++;
    }
    if (count != 1) {
        result = 0;
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, bind3, bind3_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        count++;
    }
    if (count != 0) {
        result = 0;
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOCLIENT, bind_ack3, bind_ack3_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (accepted_uuids[2][i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        if (accepted_ctxids[2] != item->ctxid) {
            result = 0;
            goto end;
        }
        count++;
    }
    if (count != 1) {
        result = 0;
        goto end;
    }

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}


/**
 * \test Check for correct internal ids for bind_acks + alter_contexts
 */
static int DCERPCParserTest17(void)
{
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bindbuf[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x40, 0xfd, 0x2c, 0x34, 0x6c, 0x3c, 0xce, 0x11,
        0xa8, 0x93, 0x08, 0x00, 0x2b, 0x2e, 0x9c, 0x6d,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bindbuf_len = sizeof(bindbuf);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x7d, 0xd8, 0x00, 0x00,
        0x0d, 0x00, 0x5c, 0x70, 0x69, 0x70, 0x65, 0x5c,
        0x6c, 0x6c, 0x73, 0x72, 0x70, 0x63, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t alter_context[] = {
        0x05, 0x00, 0x0e, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
        0xd0, 0x4c, 0x67, 0x57, 0x00, 0x52, 0xce, 0x11,
        0xa8, 0x97, 0x08, 0x00, 0x2b, 0x2e, 0x9c, 0x6d,
        0x01, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t alter_context_len = sizeof(alter_context);

    uint8_t alter_context_resp[] = {
        0x05, 0x00, 0x0f, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x7d, 0xd8, 0x00, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t alter_context_resp_len = sizeof(alter_context_resp);


    TcpSession ssn;
    DCERPCUuidEntry *item = NULL;
    int count = 0;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    uint8_t accepted_uuids[2][16] = {
        {0x57, 0x67, 0x4c, 0xd0, 0x52, 0x00, 0x11, 0xce,
         0xa8, 0x97, 0x08, 0x00, 0x2b, 0x2e, 0x9c, 0x6d},
        {0x34, 0x2c, 0xfd, 0x40, 0x3c, 0x6c, 0x11, 0xce,
         0xa8, 0x93, 0x08, 0x00, 0x2b, 0x2e, 0x9c, 0x6d},
    };

    uint16_t accepted_ctxids[2] = {1, 0};

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, bindbuf, bindbuf_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOCLIENT, bind_ack, bind_ack_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (accepted_uuids[1][i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        if (accepted_ctxids[1] != item->ctxid) {
            result = 0;
            goto end;
        }
        count++;
    }
    if (count != 1) {
        result = 0;
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, alter_context, alter_context_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        count++;
    }
    if (count != 1) {
        result = 0;
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOCLIENT, alter_context_resp,
                            alter_context_resp_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (accepted_uuids[count][i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        if (accepted_ctxids[count] != item->ctxid) {
            result = 0;
            goto end;
        }
        count++;
    }
    if (count != 2) {
        result = 0;
        goto end;
    }

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented PDU.
 */
static int DCERPCParserTest18(void)
{
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x26, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00,
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x02,
        0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A,
        0x0B, 0x0C, 0xFF, 0xFF
    };
    uint32_t request2_len = sizeof(request2);

    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 18);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer == NULL);

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len ==  14);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.opnum == 2);

end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

static int DCERPCParserTest19(void)
{
    int result = 0;
    Flow f;
    uint8_t dcerpcbind[] = {
        0x05, 0x00,
        0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x3c, 0x04,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd0, 0x16,
        0xd0, 0x16, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x2c, 0xd0,
        0x28, 0xda, 0x76, 0x91, 0xf6, 0x6e, 0xcb, 0x0f,
        0xbf, 0x85, 0xcd, 0x9b, 0xf6, 0x39, 0x01, 0x00,
        0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x01, 0x00, 0x2c, 0x75, 0xce, 0x7e, 0x82, 0x3b,
        0x06, 0xac, 0x1b, 0xf0, 0xf5, 0xb7, 0xa7, 0xf7,
        0x28, 0xaf, 0x05, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x01, 0x00, 0xe3, 0xb2,
        0x10, 0xd1, 0xd0, 0x0c, 0xcc, 0x3d, 0x2f, 0x80,
        0x20, 0x7c, 0xef, 0xe7, 0x09, 0xe0, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x01, 0x00, 0xde, 0x85, 0x70, 0xc4, 0x02, 0x7c,
        0x60, 0x23, 0x67, 0x0c, 0x22, 0xbf, 0x18, 0x36,
        0x79, 0x17, 0x01, 0x00, 0x02, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x01, 0x00, 0x41, 0x65,
        0x29, 0x51, 0xaa, 0xe7, 0x7b, 0xa8, 0xf2, 0x37,
        0x0b, 0xd0, 0x3f, 0xb3, 0x36, 0xed, 0x05, 0x00,
        0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x05, 0x00,
        0x01, 0x00, 0x14, 0x96, 0x80, 0x01, 0x2e, 0x78,
        0xfb, 0x5d, 0xb4, 0x3c, 0x14, 0xb3, 0x3d, 0xaa,
        0x02, 0xfb, 0x06, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x01, 0x00, 0x3b, 0x04,
        0x68, 0x3e, 0x63, 0xfe, 0x9f, 0xd8, 0x64, 0x55,
        0xcd, 0xe7, 0x39, 0xaf, 0x98, 0x9f, 0x03, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x07, 0x00,
        0x01, 0x00, 0x16, 0x7a, 0x4f, 0x1b, 0xdb, 0x25,
        0x92, 0x55, 0xdd, 0xae, 0x9e, 0x5b, 0x3e, 0x93,
        0x66, 0x93, 0x04, 0x00, 0x01, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x01, 0x00, 0xe8, 0xa4,
        0x8a, 0xcf, 0x95, 0x6c, 0xc7, 0x8f, 0x14, 0xcc,
        0x56, 0xfc, 0x7b, 0x5f, 0x4f, 0xe8, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x01, 0x00, 0xd8, 0xda, 0xfb, 0xbc, 0xa2, 0x55,
        0x6f, 0x5d, 0xc0, 0x2d, 0x88, 0x6f, 0x00, 0x17,
        0x52, 0x8d, 0x06, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x0a, 0x00, 0x01, 0x00, 0x3f, 0x17,
        0x55, 0x0c, 0xf4, 0x23, 0x3c, 0xca, 0xe6, 0xa0,
        0xaa, 0xcc, 0xb5, 0xe3, 0xf9, 0xce, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x0b, 0x00,
        0x01, 0x00, 0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1,
        0xd0, 0x11, 0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9,
        0x2e, 0xf5, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x0c, 0x00, 0x01, 0x00, 0xc9, 0x9f,
        0x3e, 0x6e, 0x82, 0x0a, 0x2b, 0x28, 0x37, 0x78,
        0xe1, 0x13, 0x70, 0x05, 0x38, 0x4d, 0x01, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x0d, 0x00,
        0x01, 0x00, 0x11, 0xaa, 0x4b, 0x15, 0xdf, 0xa6,
        0x86, 0x3f, 0xfb, 0xe0, 0x09, 0xb7, 0xf8, 0x56,
        0xd2, 0x3f, 0x05, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x0e, 0x00, 0x01, 0x00, 0xee, 0x99,
        0xc4, 0x25, 0x11, 0xe4, 0x95, 0x62, 0x29, 0xfa,
        0xfd, 0x26, 0x57, 0x02, 0xf1, 0xce, 0x03, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x0f, 0x00,
        0x01, 0x00, 0xba, 0x81, 0x9e, 0x1a, 0xdf, 0x2b,
        0xba, 0xe4, 0xd3, 0x17, 0x41, 0x60, 0x6d, 0x2d,
        0x9e, 0x28, 0x03, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x10, 0x00, 0x01, 0x00, 0xa0, 0x24,
        0x03, 0x9a, 0xa9, 0x99, 0xfb, 0xbe, 0x49, 0x11,
        0xad, 0x77, 0x30, 0xaa, 0xbc, 0xb6, 0x02, 0x00,
        0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x11, 0x00,
        0x01, 0x00, 0x32, 0x04, 0x7e, 0xae, 0xec, 0x28,
        0xd1, 0x55, 0x83, 0x4e, 0xc3, 0x47, 0x5d, 0x1d,
        0xc6, 0x65, 0x02, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x12, 0x00, 0x01, 0x00, 0xc6, 0xa4,
        0x81, 0x48, 0x66, 0x2a, 0x74, 0x7d, 0x56, 0x6e,
        0xc5, 0x1d, 0x19, 0xf2, 0xb5, 0xb6, 0x03, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00,
        0x01, 0x00, 0xcb, 0xae, 0xb3, 0xc0, 0x0c, 0xf4,
        0xa4, 0x5e, 0x91, 0x72, 0xdd, 0x53, 0x24, 0x70,
        0x89, 0x02, 0x05, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x14, 0x00, 0x01, 0x00, 0xb8, 0xd0,
        0xa0, 0x1a, 0x5e, 0x7a, 0x2d, 0xfe, 0x35, 0xc6,
        0x7d, 0x08, 0x0d, 0x33, 0x73, 0x18, 0x02, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
    };

    uint8_t dcerpcbindack[] = {
        0x05, 0x00, 0x0c, 0x03,
        0x10, 0x00, 0x00, 0x00, 0x6c, 0x02, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xb8, 0x10, 0xb8, 0x10,
        0xce, 0x47, 0x00, 0x00, 0x0c, 0x00, 0x5c, 0x50,
        0x49, 0x50, 0x45, 0x5c, 0x6c, 0x73, 0x61, 0x73,
        0x73, 0x00, 0xf6, 0x6e, 0x18, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    uint32_t bindlen = sizeof(dcerpcbind);
    uint32_t bindacklen = sizeof(dcerpcbindack);
    TcpSession ssn;
    AppLayerParserThreadCtx *alp_tctx = AppLayerParserThreadCtxAlloc();

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.proto = IPPROTO_TCP;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);

    FLOWLOCK_WRLOCK(&f);
    int r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                                STREAM_TOSERVER | STREAM_START, dcerpcbind,
                                bindlen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        goto end;
    }

    if (dcerpc_state->dcerpc.bytesprocessed == 0) {
        printf("request - dce parser bytesprocessed should not be 0.\n");
        goto end;
    }

    FLOWLOCK_WRLOCK(&f);
    r = AppLayerParserParse(NULL, alp_tctx, &f, ALPROTO_DCERPC,
                            STREAM_TOCLIENT, dcerpcbindack, bindacklen);
    if (r == 0) {
        printf("dce parser didn't return fail\n");
        FLOWLOCK_UNLOCK(&f);
        goto end;
    }
    FLOWLOCK_UNLOCK(&f);

    result = 1;
end:
    if (alp_tctx != NULL)
        AppLayerParserThreadCtxFree(alp_tctx);
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

#endif /* UNITTESTS */

void DCERPCParserRegisterTests(void)
{
#ifdef UNITTESTS
    UtRegisterTest("DCERPCParserTest01", DCERPCParserTest01);
    UtRegisterTest("DCERPCParserTest02", DCERPCParserTest02);
    UtRegisterTest("DCERPCParserTest03", DCERPCParserTest03);
    UtRegisterTest("DCERPCParserTest05", DCERPCParserTest05);
    UtRegisterTest("DCERPCParserTest06", DCERPCParserTest06);
    UtRegisterTest("DCERPCParserTest07", DCERPCParserTest07);
    UtRegisterTest("DCERPCParserTest08", DCERPCParserTest08);
    UtRegisterTest("DCERPCParserTest09", DCERPCParserTest09);
    UtRegisterTest("DCERPCParserTest10", DCERPCParserTest10);
    UtRegisterTest("DCERPCParserTest11", DCERPCParserTest11);
    UtRegisterTest("DCERPCParserTest12", DCERPCParserTest12);
    UtRegisterTest("DCERPCParserTest13", DCERPCParserTest13);
    UtRegisterTest("DCERPCParserTest14", DCERPCParserTest14);
    UtRegisterTest("DCERPCParserTest15", DCERPCParserTest15);
    UtRegisterTest("DCERPCParserTest16", DCERPCParserTest16);
    UtRegisterTest("DCERPCParserTest17", DCERPCParserTest17);
    UtRegisterTest("DCERPCParserTest18", DCERPCParserTest18);
    UtRegisterTest("DCERPCParserTest19", DCERPCParserTest19);
#endif /* UNITTESTS */

    return;
}
