/* Copyright (C) 2007-2011 Open Information Security Foundation
 *
 * You can copy, redistribute or modify this Program under the terms of
 * the GNU General Public License version 2 as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * version 2 along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

/**
 * \file
 *
 * \author Victor Julien <victor@inliniac.net>
 *
 * Wrappers and tests for libmagic usage.
 *
 * Libmagic's API is not thread safe. The data the pointer returned by
 * magic_buffer is overwritten by the next magic_buffer call. This is
 * why we need to lock calls and copy the returned string.
 */

#include "suricata-common.h"
#include "conf.h"

#include "util-unittest.h"

#include <magic.h>

static magic_t g_magic_ctx = NULL;
static SCMutex g_magic_lock;

/**
 *  \brief Initialize the "magic" context.
 */
int MagicInit(void) {
    BUG_ON(g_magic_ctx != NULL);

    SCEnter();

    char *filename = NULL;
    FILE *fd = NULL;

    SCMutexInit(&g_magic_lock, NULL);
    SCMutexLock(&g_magic_lock);

    g_magic_ctx = magic_open(0);
    if (g_magic_ctx == NULL) {
        SCLogError(SC_ERR_MAGIC_OPEN, "magic_open failed: %s", magic_error(g_magic_ctx));
        goto error;
    }

    (void)ConfGet("magic-file", &filename);
    if (filename != NULL) {
        SCLogInfo("using magic-file %s", filename);

        if ( (fd = fopen(filename, "r")) == NULL) {
            SCLogWarning(SC_ERR_FOPEN, "Error opening file: \"%s\": %s", filename, strerror(errno));
            goto error;
        }
        fclose(fd);
    }

    if (magic_load(g_magic_ctx, filename) != 0) {
        SCLogError(SC_ERR_MAGIC_LOAD, "magic_load failed: %s", magic_error(g_magic_ctx));
        goto error;
    }

    SCMutexUnlock(&g_magic_lock);
    SCReturnInt(0);

error:
    if (g_magic_ctx != NULL) {
        magic_close(g_magic_ctx);
        g_magic_ctx = NULL;
    }

    SCMutexUnlock(&g_magic_lock);
    SCReturnInt(-1);
}

/**
 *  \brief Find the magic value for a buffer.
 *
 *  \param buf the buffer
 *  \param buflen length of the buffer
 *
 *  \retval result pointer to null terminated string
 */
char *MagicLookup(uint8_t *buf, uint32_t buflen) {
    const char *result = NULL;
    char *magic = NULL;

    SCMutexLock(&g_magic_lock);

    if (buf != NULL && buflen > 0) {
        result = magic_buffer(g_magic_ctx, (void *)buf, (size_t)buflen);
        if (result != NULL) {
            magic = SCStrdup(result);
        }
    }

    SCMutexUnlock(&g_magic_lock);
    SCReturnPtr(magic, "const char");
}

void MagicDeinit(void) {
    SCMutexLock(&g_magic_lock);
    if (g_magic_ctx != NULL) {
        magic_close(g_magic_ctx);
        g_magic_ctx = NULL;
    }
    SCMutexUnlock(&g_magic_lock);
    SCMutexDestroy(&g_magic_lock);
}

#ifdef UNITTESTS

/** \test magic lib calls -- init */
int MagicInitTest01(void) {
    magic_t magic_ctx;

    magic_ctx = magic_open(0);

    if (magic_load(magic_ctx, NULL) == -1)
        return 0;

    magic_close(magic_ctx);
    return 1;
}

/** \test magic init through api */
int MagicInitTest02(void) {
    if (g_magic_ctx != NULL) {
        printf("g_magic_ctx != NULL at start of the test: ");
        return 0;
    }

    MagicInit();

    if (g_magic_ctx == NULL) {
        printf("g_magic_ctx == NULL: ");
        return 0;
    }

    MagicDeinit();

    if (g_magic_ctx != NULL) {
        printf("g_magic_ctx != NULL at end of the test: ");
        return 0;
    }

    return 1;
}

/** \test magic lib calls -- lookup */
int MagicDetectTest01(void) {
    magic_t magic_ctx;
    char *result = NULL;
    char buffer[] = { 0x25, 'P', 'D', 'F', '-', '1', '.', '3', 0x0d, 0x0a};
    size_t buffer_len = sizeof(buffer);
    int retval = 0;

    magic_ctx = magic_open(0);

    if (magic_load(magic_ctx, NULL) == -1)
        return 0;

    result = (char *)magic_buffer(magic_ctx, (void *)buffer, buffer_len);
    if (result == NULL || strncmp(result, "PDF document", 12) != 0) {
        printf("result %p:%s, not \"PDF document\": ", result,result?result:"(null)");
        goto end;
    }

    retval = 1;
end:
    magic_close(magic_ctx);
    return retval;
}

/** \test magic lib calls -- lookup */
int MagicDetectTest02(void) {
    magic_t magic_ctx;
    char *result = NULL;

    char buffer[] = {
        0xd0, 0xcf, 0x11, 0xe0, 0xa1, 0xb1, 0x1a, 0xe1,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x3e, 0x00, 0x03, 0x00, 0xfe, 0xff, 0x09, 0x00,

        0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x96, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x10, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00,

        0x01, 0x00, 0x00, 0x00, 0xfe, 0xff, 0xff, 0xff,
        0x00, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00, 0x00,
        0x97, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    };
    size_t buffer_len = sizeof(buffer);
    int retval = 0;

    magic_ctx = magic_open(0);

    if (magic_load(magic_ctx, NULL) == -1)
        return 0;

    result = (char *)magic_buffer(magic_ctx, (void *)buffer, buffer_len);
    if (result == NULL || strcmp(result, "Microsoft Office Document") != 0) {
        printf("result %p:%s, not \"Microsoft Office Document\": ", result,result?result:"(null)");
        goto end;
    }

    retval = 1;
end:
    magic_close(magic_ctx);
    return retval;
}

/** \test magic lib calls -- lookup */
int MagicDetectTest03(void) {
    magic_t magic_ctx;
    char *result = NULL;

    char buffer[] = {
        0x50, 0x4b, 0x03, 0x04, 0x14, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x0b, 0x55, 0x2a, 0x36, 0x5e, 0xc6,
        0x32, 0x0c, 0x27, 0x00, 0x00, 0x00, 0x27, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x6d, 0x69,

        0x6d, 0x65, 0x74, 0x79, 0x70, 0x65, 0x61, 0x70,
        0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
        0x6e, 0x2f, 0x76, 0x6e, 0x64, 0x2e, 0x6f, 0x61,
        0x73, 0x69, 0x73, 0x2e, 0x6f, 0x70, 0x65, 0x6e,

        0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74,
        0x2e, 0x74, 0x65, 0x78, 0x74, 0x50, 0x4b, 0x03,
        0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b,
        0x55, 0x2a, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00,

        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a,
        0x00, 0x00, 0x00, 0x43, 0x6f, 0x6e, 0x66, 0x69,
        0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e,
        0x73, 0x32, 0x2f, 0x73, 0x74, 0x61, 0x74, 0x75,

        0x73, 0x62, 0x61, 0x72, 0x2f, 0x50, 0x4b, 0x03,
        0x04, 0x14, 0x00, 0x08, 0x00, 0x08, 0x00, 0x0b,
    };
    size_t buffer_len = sizeof(buffer);
    int retval = 0;

    magic_ctx = magic_open(0);

    if (magic_load(magic_ctx, NULL) == -1)
        return 0;

    result = (char *)magic_buffer(magic_ctx, (void *)buffer, buffer_len);
    if (result == NULL || strcmp(result, "OpenDocument Text") != 0) {
        printf("result %p:%s, not \"OpenDocument Text\": ", result,result?result:"(null)");
        goto end;
    }

    retval = 1;
end:
    magic_close(magic_ctx);
    return retval;
}

/** \test magic lib calls -- lookup */
int MagicDetectTest04(void) {
    magic_t magic_ctx;
    char *result = NULL;

    char buffer[] = {
        0x50, 0x4b, 0x03, 0x04, 0x14, 0x00, 0x00, 0x08,
        0x00, 0x00, 0x52, 0x7b, 0x86, 0x3c, 0x8b, 0x70,
        0x96, 0x08, 0x1c, 0x00, 0x00, 0x00, 0x1c, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x6d, 0x69,

        0x6d, 0x65, 0x74, 0x79, 0x70, 0x65, 0x61, 0x70,
        0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
        0x6e, 0x2f, 0x76, 0x6e, 0x64, 0x2e, 0x73, 0x75,
        0x6e, 0x2e, 0x78, 0x6d, 0x6c, 0x2e, 0x62, 0x61,

        0x73, 0x65, 0x50, 0x4b, 0x03, 0x04, 0x14, 0x00,
        0x00, 0x08, 0x00, 0x00, 0x52, 0x7b, 0x86, 0x3c,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,

        0x4d, 0x45, 0x54, 0x41, 0x2d, 0x49, 0x4e, 0x46,
        0x2f, 0x50, 0x4b, 0x03, 0x04, 0x14, 0x00, 0x00,
        0x08, 0x08, 0x00, 0xa8, 0x42, 0x1d, 0x37, 0x5d,
        0xa7, 0xb2, 0xc1, 0xde, 0x01, 0x00, 0x00, 0x7e,

        0x04, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x63,
        0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2e, 0x78,
        0x6d, 0x6c, 0x95, 0x54, 0x4d, 0x6f, 0xdb, 0x30,
        0x0c, 0xbd, 0xe7, 0x57, 0x18, 0x02, 0x06, 0x6c,

        0x07, 0xc5, 0xe9, 0xb6, 0xc3, 0x22, 0xc4, 0x29,
        0x86, 0x7d, 0x00, 0x05, 0x8a, 0x9d, 0xb2, 0x43,
        0x8f, 0xb2, 0x24, 0xa7, 0xc2, 0x64, 0xc9, 0x15,
    };
    size_t buffer_len = sizeof(buffer);
    int retval = 0;

    magic_ctx = magic_open(0);

    if (magic_load(magic_ctx, NULL) == -1)
        return 0;

    result = (char *)magic_buffer(magic_ctx, (void *)buffer, buffer_len);
    if (result == NULL || strcmp(result, "OpenOffice.org 1.x Database file") != 0) {
        printf("result %p:%s, not \"OpenOffice.org 1.x Database file\": ", result,result?result:"(null)");
        goto end;
    }

    retval = 1;
end:
    magic_close(magic_ctx);
    return retval;
}

/** \test magic api calls -- lookup */
int MagicDetectTest05(void) {
    const char *result = NULL;
    uint8_t buffer[] = { 0x25, 'P', 'D', 'F', '-', '1', '.', '3', 0x0d, 0x0a};
    size_t buffer_len = sizeof(buffer);
    int retval = 0;

    MagicInit();

    result = MagicLookup(buffer, buffer_len);
    if (result == NULL || strncmp(result, "PDF document", 12) != 0) {
        printf("result %p:%s, not \"PDF document\": ", result,result?result:"(null)");
        goto end;
    }

    retval = 1;
end:
    MagicDeinit();
    return retval;
}

/** \test magic api calls -- lookup */
int MagicDetectTest06(void) {
    const char *result = NULL;
    uint8_t buffer[] = {
        0xd0, 0xcf, 0x11, 0xe0, 0xa1, 0xb1, 0x1a, 0xe1,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x3e, 0x00, 0x03, 0x00, 0xfe, 0xff, 0x09, 0x00,

        0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x96, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x10, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00,

        0x01, 0x00, 0x00, 0x00, 0xfe, 0xff, 0xff, 0xff,
        0x00, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00, 0x00,
        0x97, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    };
    size_t buffer_len = sizeof(buffer);
    int retval = 0;

    MagicInit();

    result = MagicLookup(buffer, buffer_len);
    if (result == NULL || strcmp(result, "Microsoft Office Document") != 0) {
        printf("result %p:%s, not \"Microsoft Office Document\": ", result,result?result:"(null)");
        goto end;
    }

    retval = 1;

end:
    MagicDeinit();
    return retval;
}

/** \test magic api calls -- lookup */
int MagicDetectTest07(void) {
    const char *result = NULL;
    uint8_t buffer[] = {
        0x50, 0x4b, 0x03, 0x04, 0x14, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x0b, 0x55, 0x2a, 0x36, 0x5e, 0xc6,
        0x32, 0x0c, 0x27, 0x00, 0x00, 0x00, 0x27, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x6d, 0x69,

        0x6d, 0x65, 0x74, 0x79, 0x70, 0x65, 0x61, 0x70,
        0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
        0x6e, 0x2f, 0x76, 0x6e, 0x64, 0x2e, 0x6f, 0x61,
        0x73, 0x69, 0x73, 0x2e, 0x6f, 0x70, 0x65, 0x6e,

        0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74,
        0x2e, 0x74, 0x65, 0x78, 0x74, 0x50, 0x4b, 0x03,
        0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b,
        0x55, 0x2a, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00,

        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a,
        0x00, 0x00, 0x00, 0x43, 0x6f, 0x6e, 0x66, 0x69,
        0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e,
        0x73, 0x32, 0x2f, 0x73, 0x74, 0x61, 0x74, 0x75,

        0x73, 0x62, 0x61, 0x72, 0x2f, 0x50, 0x4b, 0x03,
        0x04, 0x14, 0x00, 0x08, 0x00, 0x08, 0x00, 0x0b,
    };
    size_t buffer_len = sizeof(buffer);
    int retval = 0;

    MagicInit();

    result = MagicLookup(buffer, buffer_len);
    if (result == NULL || strcmp(result, "OpenDocument Text") != 0) {
        printf("result %p:%s, not \"OpenDocument Text\": ", result,result?result:"(null)");
        goto end;
    }

    retval = 1;
end:
    MagicDeinit();
    return retval;
}

/** \test magic api calls -- lookup */
int MagicDetectTest08(void) {
    const char *result = NULL;
    uint8_t buffer[] = {
        0x50, 0x4b, 0x03, 0x04, 0x14, 0x00, 0x00, 0x08,
        0x00, 0x00, 0x52, 0x7b, 0x86, 0x3c, 0x8b, 0x70,
        0x96, 0x08, 0x1c, 0x00, 0x00, 0x00, 0x1c, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x6d, 0x69,

        0x6d, 0x65, 0x74, 0x79, 0x70, 0x65, 0x61, 0x70,
        0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f,
        0x6e, 0x2f, 0x76, 0x6e, 0x64, 0x2e, 0x73, 0x75,
        0x6e, 0x2e, 0x78, 0x6d, 0x6c, 0x2e, 0x62, 0x61,

        0x73, 0x65, 0x50, 0x4b, 0x03, 0x04, 0x14, 0x00,
        0x00, 0x08, 0x00, 0x00, 0x52, 0x7b, 0x86, 0x3c,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,

        0x4d, 0x45, 0x54, 0x41, 0x2d, 0x49, 0x4e, 0x46,
        0x2f, 0x50, 0x4b, 0x03, 0x04, 0x14, 0x00, 0x00,
        0x08, 0x08, 0x00, 0xa8, 0x42, 0x1d, 0x37, 0x5d,
        0xa7, 0xb2, 0xc1, 0xde, 0x01, 0x00, 0x00, 0x7e,

        0x04, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x63,
        0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2e, 0x78,
        0x6d, 0x6c, 0x95, 0x54, 0x4d, 0x6f, 0xdb, 0x30,

        0x0c, 0xbd, 0xe7, 0x57, 0x18, 0x02, 0x06, 0x6c,
        0x07, 0xc5, 0xe9, 0xb6, 0xc3, 0x22, 0xc4, 0x29,
        0x86, 0x7d, 0x00, 0x05, 0x8a, 0x9d, 0xb2, 0x43,
        0x8f, 0xb2, 0x24, 0xa7, 0xc2, 0x64, 0xc9, 0x15,
    };
    size_t buffer_len = sizeof(buffer);
    int retval = 0;

    MagicInit();

    result = MagicLookup(buffer, buffer_len);
    if (result == NULL || strcmp(result, "OpenOffice.org 1.x Database file") != 0) {
        printf("result %p:%s, not \"OpenOffice.org 1.x Database file\": ", result,result?result:"(null)");
        goto end;
    }

    retval = 1;
end:
    MagicDeinit();
    return retval;
}

/** \test magic api calls -- make sure memory is shared */
int MagicDetectTest09(void) {
    const char *result1 = NULL;
    const char *result2 = NULL;
    uint8_t buffer[] = { 0x25, 'P', 'D', 'F', '-', '1', '.', '3', 0x0d, 0x0a};
    size_t buffer_len = sizeof(buffer);
    int retval = 0;

    MagicInit();

    result1 = MagicLookup(buffer, buffer_len);
    if (result1 == NULL || strncmp(result1, "PDF document", 12) != 0) {
        printf("result %p:%s, not \"PDF document\": ", result1,result1?result1:"(null)");
        goto end;
    }

    result2 = MagicLookup(buffer, buffer_len);
    if (result2 == NULL || strncmp(result2, "PDF document", 12) != 0) {
        printf("result %p:%s, not \"PDF document\": ", result2,result2?result2:"(null)");
        goto end;
    }

    if (result1 != result2) {
        printf("pointers not equal, weird... %p != %p: ", result1, result2);
        goto end;
    }

    retval = 1;
end:
    MagicDeinit();
    return retval;
}

#endif /* UNITTESTS */


void MagicRegisterTests(void) {
#ifdef UNITTESTS
    UtRegisterTest("MagicInitTest01", MagicInitTest01, 1);
    UtRegisterTest("MagicInitTest02", MagicInitTest02, 1);
    UtRegisterTest("MagicDetectTest01", MagicDetectTest01, 1);
    UtRegisterTest("MagicDetectTest02", MagicDetectTest02, 1);
    UtRegisterTest("MagicDetectTest03", MagicDetectTest03, 1);
    UtRegisterTest("MagicDetectTest04", MagicDetectTest04, 1);
    UtRegisterTest("MagicDetectTest05", MagicDetectTest05, 1);
    UtRegisterTest("MagicDetectTest06", MagicDetectTest06, 1);
    UtRegisterTest("MagicDetectTest07", MagicDetectTest07, 1);
    UtRegisterTest("MagicDetectTest08", MagicDetectTest08, 1);
    /* fails in valgrind, somehow it returns different pointers then.
    UtRegisterTest("MagicDetectTest09", MagicDetectTest09, 1); */
#endif /* UNITTESTS */
}
