//  SuperTux
//  Copyright (C) 2008 Matthias Braun <matze@braunis.de>
//                     Ingo Ruhnke <grumbel@gmail.com>
//
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "supertux/tile_set_parser.hpp"

#include <stdexcept>
#include <sstream>

#include "lisp/list_iterator.hpp"
#include "lisp/parser.hpp"
#include "supertux/tile_set.hpp"
#include "util/file_system.hpp"

TileSetParser::TileSetParser(TileSet& tileset, const std::string& filename) :
  m_tileset(tileset),
  m_filename(filename),
  m_tiles_path()
{
}

void
TileSetParser::parse()
{
  m_tiles_path = FileSystem::dirname(m_filename);

  m_tileset.tiles.resize(1, 0);
  m_tileset.tiles[0] = new Tile();

  lisp::Parser parser;
  const lisp::Lisp* root = parser.parse(m_filename);

  const lisp::Lisp* tiles_lisp = root->get_lisp("supertux-tiles");
  if(!tiles_lisp)
    throw std::runtime_error("file is not a supertux tiles file.");

  lisp::ListIterator iter(tiles_lisp);
  while(iter.next())
  {
    if (iter.item() == "tile")
    {
      parse_tile(*iter.lisp());
    }
    else if (iter.item() == "tilegroup")
    {
      /* tilegroups are only interesting for the editor */
    }
    else if (iter.item() == "tiles")
    {
      parse_tiles(*iter.lisp());
    }
    else
    {
      log_warning << "Unknown symbol '" << iter.item() << "' in tileset file" << std::endl;
    }
  }
}

void
TileSetParser::parse_tile(const Reader& reader)
{
  uint32_t id;
  if (!reader.get("id", id))
  {
    throw std::runtime_error("Missing tile-id.");
  }

  uint32_t attributes = 0;

  bool value = false;
  if(reader.get("solid", value) && value)
    attributes |= Tile::SOLID;
  if(reader.get("unisolid", value) && value)
    attributes |= Tile::UNISOLID | Tile::SOLID;
  if(reader.get("brick", value) && value)
    attributes |= Tile::BRICK;
  if(reader.get("ice", value) && value)
    attributes |= Tile::ICE;
  if(reader.get("water", value) && value)
    attributes |= Tile::WATER;
  if(reader.get("hurts", value) && value)
    attributes |= Tile::HURTS;
  if(reader.get("fire", value) && value)
    attributes |= Tile::FIRE;
  if(reader.get("fullbox", value) && value)
    attributes |= Tile::FULLBOX;
  if(reader.get("coin", value) && value)
    attributes |= Tile::COIN;
  if(reader.get("goal", value) && value)
    attributes |= Tile::GOAL;

  uint32_t data = 0;

  if(reader.get("north", value) && value)
    data |= Tile::WORLDMAP_NORTH;
  if(reader.get("south", value) && value)
    data |= Tile::WORLDMAP_SOUTH;
  if(reader.get("west", value) && value)
    data |= Tile::WORLDMAP_WEST;
  if(reader.get("east", value) && value)
    data |= Tile::WORLDMAP_EAST;
  if(reader.get("stop", value) && value)
    data |= Tile::WORLDMAP_STOP;

  reader.get("data", data);

  float fps = 10;
  reader.get("fps", fps);

  if(reader.get("slope-type", data))
  {
    attributes |= Tile::SOLID | Tile::SLOPE;
  }

  std::vector<Tile::ImageSpec> editor_imagespecs;
  const lisp::Lisp* editor_images;
  editor_images = reader.get_lisp("editor-images");
  if(editor_images)
    editor_imagespecs = parse_tile_images(*editor_images);

  std::vector<Tile::ImageSpec> imagespecs;
  const lisp::Lisp* images;
  images = reader.get_lisp("images");
  if(images)
      imagespecs = parse_tile_images(*images);

  std::unique_ptr<Tile> tile(new Tile(imagespecs, editor_imagespecs, attributes, data, fps));

  if (id >= m_tileset.tiles.size())
    m_tileset.tiles.resize(id+1, 0);

  if (m_tileset.tiles[id] != 0)
  {
    log_warning << "Tile with ID " << id << " redefined" << std::endl;
  }
  else
  {
    m_tileset.tiles[id] = tile.release();
  }
}

std::vector<Tile::ImageSpec>
TileSetParser::parse_tile_images(const Reader& images_lisp)
{
  std::vector<Tile::ImageSpec> imagespecs;

  const lisp::Lisp* list = &images_lisp;
  while(list)
  {
    const lisp::Lisp* cur = list->get_car();

    if(cur->get_type() == lisp::Lisp::TYPE_STRING)
    {
      std::string file;
      cur->get(file);
      imagespecs.push_back(Tile::ImageSpec(m_tiles_path + file, Rectf(0, 0, 0, 0)));
    }
    else if(cur->get_type() == lisp::Lisp::TYPE_CONS &&
            cur->get_car()->get_type() == lisp::Lisp::TYPE_SYMBOL &&
            cur->get_car()->get_symbol() == "region")
    {
      const lisp::Lisp* ptr = cur->get_cdr();

      std::string file;
      float x = 0;
      float y = 0;
      float w = 0;
      float h = 0;
      ptr->get_car()->get(file); ptr = ptr->get_cdr();
      ptr->get_car()->get(x); ptr = ptr->get_cdr();
      ptr->get_car()->get(y); ptr = ptr->get_cdr();
      ptr->get_car()->get(w); ptr = ptr->get_cdr();
      ptr->get_car()->get(h);
      imagespecs.push_back(Tile::ImageSpec(m_tiles_path + file, Rectf(x, y, x+w, y+h)));
    }
    else
    {
      log_warning << "Expected string or list in images tag" << std::endl;
    }

    list = list->get_cdr();
  }

  return imagespecs;
}

void
TileSetParser::parse_tiles(const Reader& reader)
{
  // List of ids (use 0 if the tile should be ignored)
  std::vector<uint32_t> ids;
  // List of attributes of the tile
  std::vector<uint32_t> attributes;
  // List of data for the tiles
  std::vector<uint32_t> datas;
  //List of frames that the tiles come in
  std::vector<std::string> images;
  //List of frames that the editor tiles come in
  std::vector<std::string> editor_images;
  // Name used to report errors.
  std::string image_name;

  // width and height of the image in tile units, this is used for two
  // purposes:
  //  a) so we don't have to load the image here to know its dimensions
  //  b) so that the resulting 'tiles' entry is more robust,
  //  ie. enlarging the image won't break the tile id mapping
  // FIXME: height is actually not used, since width might be enough for
  // all purposes, still feels somewhat more natural this way
  unsigned int width  = 0;
  unsigned int height = 0;

  reader.get("ids",        ids);
  bool has_attributes = reader.get("attributes", attributes);
  bool has_datas = reader.get("datas", datas);

  reader.get("image", images) || reader.get("images", images);
  reader.get("editor-images", editor_images);

  if (images.size() > 0)
    image_name = images[0];
  else
    image_name = "(no image)";

  reader.get("width",      width);
  reader.get("height",     height);

  float fps = 10;
  reader.get("fps",     fps);

  if (width == 0)
  {
    throw std::runtime_error("Width is zero.");
  }
  else if (height == 0)
  {
    throw std::runtime_error("Height is zero.");
  }
  else if (fps < 0)
  {
    throw std::runtime_error("Negative fps.");
  }
  else if (ids.size() != width*height)
  {
    std::ostringstream err;
    err << "Number of ids (" << ids.size() <<  ") and "
      "dimensions of image (" << width << "x" << height << " = " << width*height << ") "
      "differ for image " << image_name;
    throw std::runtime_error(err.str());
  }
  else if (has_attributes && (ids.size() != attributes.size()))
  {
    std::ostringstream err;
    err << "Number of ids (" << ids.size() <<  ") and attributes (" << attributes.size()
        << ") mismatch for image '" << image_name << "', but must be equal";
    throw std::runtime_error(err.str());
  }
  else if (has_datas && ids.size() != datas.size())
  {
    std::ostringstream err;
    err << "Number of ids (" << ids.size() <<  ") and datas (" << datas.size()
        << ") mismatch for image '" << image_name << "', but must be equal";
    throw std::runtime_error(err.str());
  }
  else
  {
    for(std::vector<uint32_t>::size_type i = 0; i < ids.size() && i < width*height; ++i)
    {
      if (ids[i] != 0)
      {
        if (ids[i] >= m_tileset.tiles.size())
          m_tileset.tiles.resize(ids[i]+1, 0);

        int x = 32*(i % width);
        int y = 32*(i / width);

        std::vector<Tile::ImageSpec> imagespecs;
        for(std::vector<std::string>::const_iterator j = images.begin(); j != images.end(); ++j)
        {
          imagespecs.push_back(Tile::ImageSpec(m_tiles_path + *j, Rectf(x, y, x + 32, y + 32)));
        }

        std::vector<Tile::ImageSpec> editor_imagespecs;
        for(std::vector<std::string>::const_iterator j = editor_images.begin(); j != editor_images.end(); ++j)
        {
          editor_imagespecs.push_back(Tile::ImageSpec(m_tiles_path + *j, Rectf(x, y, x + 32, y + 32)));
        }

        std::unique_ptr<Tile> tile(new Tile(imagespecs, editor_imagespecs,
                                          (has_attributes ? attributes[i] : 0), (has_datas ? datas[i] : 0), fps));
        if (m_tileset.tiles[ids[i]] == 0) {
          m_tileset.tiles[ids[i]] = tile.release();
        } else {
          log_warning << "Tile with ID " << ids[i] << " redefined" << std::endl;
        }
      }
    }
  }
}

/* EOF */
