#!/usr/bin/env python3
# Copyright 2015 Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import argparse
import sys
import logging
import signal
from subiquitycore.log import setup_logger
from subiquitycore import __version__ as VERSION
from console_conf.core import ConsoleConf
from subiquitycore.core import ApplicationError
from subiquitycore.ui.frame import SubiquityUI
from subiquitycore.utils import environment_check


# Does console-conf actually need any of this?
ENVIRONMENT = '''
checks:
    write:
        directory:
            - /tmp
    mount:
        directory:
            - /proc
            - /sys
'''


def parse_options(argv):
    parser = argparse.ArgumentParser(
        description=(
            'console-conf - Pre-Ownership Configuration for Ubuntu Core'),
        prog='console-conf')
    parser.add_argument('--dry-run', action='store_true',
                        dest='dry_run',
                        help='menu-only, do not call installer function')
    parser.add_argument('--serial', action='store_true',
                        dest='run_on_serial',
                        help='Run the installer over serial console.')
    parser.add_argument('--machine-config', metavar='CONFIG',
                        dest='machine_config',
                        help="Don't Probe. Use probe data file")
    parser.add_argument('--screens', action='append', dest='screens',
                        default=[])
    parser.add_argument('--answers')
    return parser.parse_args(argv)


LOGDIR = "/var/log/console-conf/"


def main():
    opts = parse_options(sys.argv[1:])
    global LOGDIR
    if opts.dry_run:
        LOGDIR = ".subiquity"
    LOGFILE = setup_logger(dir=LOGDIR)
    logger = logging.getLogger('console_conf')
    logger.info("Starting console-conf v{}".format(VERSION))
    logger.info("Arguments passed: {}".format(sys.argv))

    signal.signal(signal.SIGINT, signal.SIG_IGN)
    signal.signal(signal.SIGQUIT, signal.SIG_IGN)

    env_ok = environment_check(ENVIRONMENT)
    if env_ok is False and not opts.dry_run:
        print('Failed environment check.  '
              'Check {} for errors.'.format(LOGFILE))
        return 1

    ui = SubiquityUI()

    try:
        interface = ConsoleConf(ui, opts)
    except ApplicationError as e:
        logger.exception('Failed to load ConsoleConf interface')
        print(e)
        return 1

    interface.run()


if __name__ == '__main__':
    sys.exit(main())
