/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

#include "TextModelTestRemove.h"

// cppunit
#include <cppunit/TestSuite.h>
#include <cppunit/TestCaller.h>

// sc
#include "sublib/TextModelImpl.h"
#include "sublib/Cursor.h"
#include "sublib/Line.h"
#include "util/String.h"


TextModelTestRemove::TextModelTestRemove() : CppUnit::TestCase("TextModelTestRemove")
{
}

TextModelTestRemove::~TextModelTestRemove()
{
}

void TextModelTestRemove::setUp()
{
  sc::String text( "test" );
  TextModelImpl* impl1 = new TextModelImpl( text );
  TextModelImpl* impl2 = new TextModelImpl( text );

  impl1->addLine( Line(sc::String("line1\x0a"),0,ctCommon) );
  impl1->addLine( Line(sc::String("line2\x0a"),0,ctCommon) );

  impl2->addLine( Line(sc::String("line1\x0d\x0a"),0,ctCommon) );
  impl2->addLine( Line(sc::String("line2\x0d\x0a"),0,ctCommon) );

  _text1 = impl1;
  _text2 = impl2;
}

void TextModelTestRemove::tearDown()
{
  delete _text1;
  delete _text2;
}


CppUnit::Test *TextModelTestRemove::suite()
{
  CppUnit::TestSuite *newSuite = new CppUnit::TestSuite("TextModelTestRemove");

  newSuite->addTest(new CppUnit::TestCaller<TextModelTestRemove>("testRemoveLeft",&TextModelTestRemove::testRemoveLeft) );
  newSuite->addTest(new CppUnit::TestCaller<TextModelTestRemove>("testRemoveRight",&TextModelTestRemove::testRemoveRight) );

  newSuite->addTest(new CppUnit::TestCaller<TextModelTestRemove>("testRemoveLeftLineEnd",&TextModelTestRemove::testRemoveLeftLineEnd) );

  return newSuite;
}

void TextModelTestRemove::testRemoveLeft()
{
  Cursor posBefore(0,3);
  Cursor posAfter(0,2);
  _text1->setCursor( posBefore );
  _text1->setCursor2( posBefore );

  _text1->removeTextLeft();

  {
    Cursor c1 = _text1->getCursor();
    Cursor c2 = _text1->getCursor2();
    CPPUNIT_ASSERT( c1.equalPos(posAfter) );
    CPPUNIT_ASSERT( c2.equalPos(posAfter) );
  }
  {
    const Line& res = _text1->getLine(0);
    CPPUNIT_ASSERT_EQUAL( sc::String("lie1\x0a"), res.getLine() );
  }

  _text1->undo();

  {
    Cursor c1 = _text1->getCursor();
    Cursor c2 = _text1->getCursor2();
    CPPUNIT_ASSERT( c1.equalPos(posBefore) );
    CPPUNIT_ASSERT( c2.equalPos(posBefore) );
  }
  {
    const Line& res = _text1->getLine(0);
    CPPUNIT_ASSERT_EQUAL( sc::String("line1\x0a"), res.getLine() );
  }
}


void TextModelTestRemove::testRemoveRight()
{
  Cursor posBefore(0,3);
  Cursor posAfter(0,3);
  _text1->setCursor( posBefore );
  _text1->setCursor2( posBefore );

  _text1->removeTextRight();

  {
    Cursor c1 = _text1->getCursor();
    Cursor c2 = _text1->getCursor2();
    CPPUNIT_ASSERT( c1.equalPos(posAfter) );
    CPPUNIT_ASSERT( c2.equalPos(posAfter) );
  }
  {
    const Line& res = _text1->getLine(0);
    CPPUNIT_ASSERT_EQUAL( sc::String("lin1\x0a"), res.getLine() );
  }

  _text1->undo();

  {
    Cursor c1 = _text1->getCursor();
    Cursor c2 = _text1->getCursor2();
    CPPUNIT_ASSERT( c1.equalPos(posBefore) );
    CPPUNIT_ASSERT( c2.equalPos(posBefore) );
  }
  {
    const Line& res = _text1->getLine(0);
    CPPUNIT_ASSERT_EQUAL( sc::String("line1\x0a"), res.getLine() );
  }
}


void TextModelTestRemove::testRemoveLeftLineEnd()
{
  Cursor posBefore(1,0);
  Cursor posAfter(0,5);
  _text1->setCursor( posBefore );
  _text1->setCursor2( posBefore );

  _text1->removeTextLeft();

  {
    Cursor c1 = _text1->getCursor();
    Cursor c2 = _text1->getCursor2();
    CPPUNIT_ASSERT( c1.equalPos(posAfter) );
    CPPUNIT_ASSERT( c2.equalPos(posAfter) );
  }
  {
    CPPUNIT_ASSERT( 1 == _text1->getLineCnt() );

    const Line& res = _text1->getLine(0);
    CPPUNIT_ASSERT_EQUAL( sc::String("line1line2\x0a"), res.getLine() );

    const Line& res2 = _text1->getLine(1);
    CPPUNIT_ASSERT_EQUAL( Line::getEmpty().getLine(), res2.getLine() );
  }

  _text1->undo();

  {
    Cursor c1 = _text1->getCursor();
    Cursor c2 = _text1->getCursor2();
    CPPUNIT_ASSERT( c1.equalPos(posBefore) );
    CPPUNIT_ASSERT( c2.equalPos(posBefore) );
  }
  {
    CPPUNIT_ASSERT( 2 == _text1->getLineCnt() );

    const Line& res = _text1->getLine(0);
    CPPUNIT_ASSERT_EQUAL( sc::String("line1\x0a"), res.getLine() );

    const Line& res2 = _text1->getLine(1);
    CPPUNIT_ASSERT_EQUAL( sc::String("line2\x0a"), res2.getLine() );
  }
}

