/*
 * Copyright (C) 2012 Tobias Brunner
 * Hochschule fuer Technik Rapperswil
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

/**
 * @defgroup cmac cmac
 * @{ @ingroup cmac_p
 */

#ifndef CMAC_H_
#define CMAC_H_

#include <crypto/crypters/crypter.h>

typedef struct cmac_t cmac_t;

/**
 * Cipher-based Message Authentication Code (CMAC).
 *
 * This class implements the message authentication algorithm
 * described in RFC 4493.
 */
struct cmac_t {

	/**
	 * Generate message authentication code.
	 *
	 * If buffer is NULL, no result is given back. A next call will
	 * append the data to already supplied data. If buffer is not NULL,
	 * the mac of all apended data is calculated, returned and the internal
	 * state is reset.
	 *
	 * @param data		chunk of data to authenticate
	 * @param buffer	pointer where the generated bytes will be written
	 */
	void (*get_mac) (cmac_t *this, chunk_t data, u_int8_t *buffer);

	/**
	 * Get the block size of this cmac_t object.
	 *
	 * @return			block size in bytes
	 */
	size_t (*get_block_size) (cmac_t *this);

	/**
	 * Set the key for this cmac_t object.
	 *
	 * @param key		key to set
	 */
	void (*set_key) (cmac_t *this, chunk_t key);

	/**
	 * Destroys a cmac_t object.
	 */
	void (*destroy) (cmac_t *this);
};

/**
 * Creates a new cmac_t object.
 *
 * @param algo			underlying crypto algorithm
 * @param key_size		key size to use, if required for algorithm
 * @return				cmac_t object, NULL if not supported
 */
cmac_t *cmac_create(encryption_algorithm_t algo, size_t key_size);

#endif /** CMAC_H_ @}*/
