use core:lang;
use core:asm;
use lang:bs:macro;

/**
 * Class representing a statement in C++.
 */
class Stmt on Compiler {
	// Position of this statement.
	SrcPos pos;

	// Create.
	init(SrcPos pos) {
		init { pos = pos; }
	}

	// Generate code. Returns 'false' if control will never continue from this statement to the next
	// one.
	Bool code(CodeGen gen) : abstract;
}

/**
 * Null-statement. Does nothing.
 */
class NullStmt extends Stmt {
	init(SrcPos pos) {
		init(pos) {}
	}

	Bool code(CodeGen gen) { true; }
}

/**
 * A block that may define new variables.
 *
 * Used as a foundation for other constructs that may create a new scope, but contains custom
 * codegen logic.
 */
class Block extends Stmt {
	// Create from a scope.
	init(SrcPos pos, Scope parent) {
		init(pos) { step = true; }
		scope = blockScope(this, parent);
	}

	// Create from another block.
	init(SrcPos pos, Block parent) {
		init(pos) { step = parent.step; }
		scope = blockScope(this, parent.scope);
	}

	// Scope for this block.
	Scope scope;

	// Emit source references for this block?
	Bool step;

	// Generate code.
	Bool code(CodeGen gen) : override {
		CodeGen child = gen.child();
		gen.l << begin(child.block);

		initVariables(child);

		Bool returns = blockCode(child);

		gen.l << end(child.block);
		returns;
	}

	Bool blockCode(CodeGen gen) : abstract;

	// Variables declared in this block.
	protected Str->LocalVar vars;

	// Order the variables were created, so that we can preserve that for the visualization.
	protected Str[] varOrder;

	void initVariables(CodeGen gen) {
		for (k in varOrder) {
			vars.get(k).create(gen);
		}
	}

	// Add a variable.
	void add(LocalVar var) {
		if (vars.has(var.name))
			throw SyntaxError(var.pos, "There is already a variable named ${var.name} in the current block!");
		vars.put(var.name, var);
		varOrder << var.name;
	}

	// Get a variable.
	Named? variable(Str name) {
		if (vars.has(name))
			return vars.get(name);
		else
			return null;
	}
}

/**
 * Block that is only used to get a scope for instantiating expressions. For example, in global initializers.
 */
class DummyBlock extends Block {
	init(SrcPos pos, Scope parent) {
		init(pos, parent) {}
	}

	Bool blockCode(CodeGen gen) : override {
		throw InternalError("Should not generate code from a DummyBlock!");
	}
}


// Create BlockLookup objects.
private Scope blockScope(Block block, Scope parent) {
	if (t = parent.top) {
		parent.child(BlockLookup(block, t));
	} else {
		// This should generally not happen...
		parent.child(BlockLookup(block));
	}
}

/**
 * Lookup logic for blocks.
 */
class BlockLookup extends NameLookup {
	init(Block block) {
		init() { block = block; }
	}

	init(Block block, NameLookup prev) {
		init(prev) {
			block = block;
		}
	}

	Block block;

	Named? find(SimplePart part, Scope source) {
		if (part.params.empty)
			return block.variable(part.name);

		null;
	}
}

// Execute a sub-block in its own codegen block if required.
package Bool executeStmt(Block inside, CodeGen gen, Stmt s) on Compiler {
	if (s as Block) {
		s.code(gen);
	} else {
		if (inside.step & s.pos.any)
			gen.l << location(s.pos);

		CodeGen child = gen.child();
		gen.l << begin(child.block);

		Bool returns = s.code(child);

		gen.l << end(child.block);
		returns;
	}
}

/**
 * A block that contains expressions.
 */
class StmtBlock extends Block {
	// Create from a scope.
	init(SrcPos pos, Scope parent) {
		init(pos, parent) {}
	}

	// Create from another block.
	init(SrcPos pos, Block parent) {
		init(pos, parent) {}
	}

	// Contents.
	private Stmt[] content;

	// Add statements.
	void add(Stmt stmt) {
		content << stmt;
	}

	// Add statements to the beginning.
	void prepend(Stmt stmt) {
		content.insert(0, stmt);
	}

	// Generate code for statements inside the block.
	Bool blockCode(CodeGen gen) {
		for (s in content) {
			if (!executeStmt(gen, s))
				return false;
		}

		if (step & pos.any) {
			gen.l << location(pos.lastCh);
		}

		true;
	}
}

/**
 * Return statement.
 */
class Return extends Stmt {
	init(SrcPos pos, Block inside) {
		FnRoot root = expectFnRoot(inside, pos);
		init(pos) {
			root = root;
		}

		checkType(Value());
	}

	init(SrcPos pos, Block inside, Expr value) {
		FnRoot root = expectFnRoot(inside, pos);
		Value expect = root.result;
		if (t = expect.type)
			value = expectCast(value, t);

		init(pos) {
			value = value;
			root = root;
		}

		var r = value.result();
		if (!r.nothing())
			checkType(r.type());
	}

	Expr? value;
	FnRoot root;

	Bool code(CodeGen gen) : override {
		if (value) {
			if (root.resultPtr != Var()) {
				codePtr(value, gen);
			} else {
				codePlain(value, gen);
			}
		} else {
			gen.l << fnRet();
		}

		// We never return.
		false;
	}

	// Return in the pointer-parameter.
	private void codePtr(Expr value, CodeGen gen) {
		unless (type = root.result.type) {
			gen.l << fnRet();
			return;
		}

		// Force name resolution now, so that we can see what is happening.
		if (l = value as NameExpr) {
			value = l.resolve();
		}

		// Allow move-ctors for local variables.
		if (l = value as LocalVarAccess) {
			value = LocalVarTmpAccess(l);
		}

		Var ptr = gen.l.createVar(gen.block, sPtr);
		Var offset = gen.l.createVar(gen.block, sInt);

		gen.l << mov(ptrA, root.resultPtr);
		gen.l << mov(ptr, ptrRel(ptrA, Offset(sPtr * 2)));
		gen.l << mov(offset, intRel(ptrA, Offset(sPtr * 3)));
		gen.l << ucast(ptrA, offset);
		gen.l << add(ptr, ptrA);

		if (ctor = cppCtor(type, [value], root.scope)) {
			value = expectCast(value, safeType(ctor.params[2], value.pos));
			CodeResult r(ctor.params[2], gen.block);
			value.code(gen, r);

			gen.l << fnParam(ptrDesc, ptr);
			gen.l << fnParam(intDesc, offset);
			gen.l << fnParam(ptrDesc, r.location(gen));
			gen.l << fnCall(ctor.ref, true);

			gen.l << fnRet();
			return;
		}

		if (ctor = stormCtor(type, [value], root.scope)) {
			value = expectCast(value, safeType(ctor.params[1], value.pos));
			CodeResult r(ctor.params[1], gen.block);
			value.code(gen, r);

			gen.l << fnParam(ptrDesc, ptr);
			gen.l << fnParam(ptrDesc, r.location(gen));
			gen.l << fnCall(ctor.ref, true);

			gen.l << fnRet();
			return;
		}

		throw SyntaxError(pos, "Unable to find a copy- or move constructor for the type ${type}.");
	}

	private void codePlain(Expr value, CodeGen gen) {
		var res = value.result.type;
		CodeResult r(res, gen.block);
		value.code(gen, r);

		if (root.result == Value()) {
			gen.l << fnRet();
		} else if (res.ref) {
			gen.l << fnRetRef(r.location(gen));
		} else {
			gen.l << fnRet(r.location(gen));
		}
	}

	private void checkType(Value type) {
		if (!root.result.mayReferTo(type))
			throw SyntaxError(pos, "Incompatible return type. Got ${type}, expected ${root.result}.");
	}
}

FnRoot expectFnRoot(Block inside, SrcPos pos) on Compiler {
	expectFnRoot(inside.scope, pos);
}

FnRoot expectFnRoot(Scope inside, SrcPos pos) on Compiler {
	unless (x = findFnRoot(inside))
		throw SyntaxError(pos, "This statement must be used inside a function.");
	x;
}

FnRoot? findFnRoot(Block inside) on Compiler {
	findFnRoot(inside.scope);
}

FnRoot? findFnRoot(Scope scope) on Compiler {
	NameLookup? at = scope.top;
	while (curr = at as BlockLookup) {
		if (fn = curr.block as FnRoot) {
			return fn;
		}

		at = curr.parent;
	}

	return null;
}

/**
 * Delete statement. Actually an expression to accommodate "free".
 */
class DeleteStmt extends Expr {
	// If 'array' is true, we expect to free memory previously allocated using 'new[]'.
	init(SrcPos pos, Expr ptr, Bool array) {
		init(pos) {
			ptrExpr = ptr;
			array = array;
		}

		if (!isCppPtr(ptr.result.type))
			throw SyntaxError(pos, "The expression passed to delete must be a pointer.");
	}

	Expr ptrExpr;
	Bool array;

	ExprResult result() : override {
		Value();
	}

	void code(CodeGen gen, CodeResult r) : override {
		// We don't return anything, so we don't touch 'r'.
		genCode(gen);
	}

	void codePtr(CodeGen gen, Type type) : override {
		genCode(gen);
	}

	private void genCode(CodeGen gen) {
		CodeResult result(ptrExpr.result.type.asRef(), gen.block);
		ptrExpr.code(gen, result);

		Var ptr = result.location(gen);

		// Check if this is the start of an allocation.
		gen.l << mov(ptrA, ptr);
		gen.l << fnParam(ptrDesc, ptrRel(ptrA));
		gen.l << fnParam(intDesc, intRel(ptrA, Offset(sPtr)));
		if (array) {
			gen.l << fnCall(named{checkDeleteArray<unsafe:RawPtr, Nat>}.ref, false, byteDesc(), al);
		} else {
			gen.l << fnCall(named{checkDelete<unsafe:RawPtr, Nat>}.ref, false, byteDesc(), al);
		}

		// Check if the check function returned 'true', then we can call the dtor.
		Label end = gen.l.label;
		gen.l << cmp(al, byteConst(0));
		gen.l << jmp(end, CondFlag:ifEqual);

		Type type = safeType(unwrapPtr(ptrExpr.result.type), pos);
		Function dtor = type.cppDtor();
		gen.l << mov(ptrA, ptr);
		gen.l << fnParam(ptrDesc, ptrRel(ptrA));
		gen.l << fnCall(dtor.ref, false);

		gen.l << end;
	}

}

/**
 * Block that does not emit location instructions to disable stepping.
 */
class NoStep extends StmtBlock {
	init(SrcPos pos, Block block) {
		init(pos, block) {}
		step = false;
	}
}
