use core:lang;
use lang:bs;
use lang:bs:macro;

// Logic to generate unique query ID:s that we can later use to cache prepared statements for
// databases. That way, we can instruct the database about our intentions and don't have to rely as
// much on caches in the database.
private Nat nextQueryID on Compiler;

// Generate a connection ID for a particular
private Nat newQueryID() on Compiler {
	return nextQueryID++;
}

private Expr bindParam(SrcPos pos, Block block, Expr statement, Nat id, Expr param) on Compiler {
	Actuals params;
	params.add(NumLiteral(pos, id));
	params.add(param);
	namedExpr(block, pos, "bind", statement, params);
}

// Create a query expression for a typed query. Helper.
private Expr typedQueryExpr(SrcPos pos, Block block, Expr connection, Database contents, Query query) on Compiler {
	query.resolve(ResolveContext(block, contents));
	var q = query.build();

	ExprBlock r(pos, block);

	// Query ID.
	Nat id = newQueryID();

	// Create a prepared statement.
	var prepared = {
		// Call "connection.prepare(id, query)".
		Actuals params;
		params.add(NumLiteral(pos, id));
		params.add(StrLiteral(pos, q.query.toS));
		var stmt = namedExpr(r, pos, "prepare", connection, params);
		var varDecl = Var(r, SStr("statement"), stmt);
		r.add(varDecl);
		LocalVarAccess(pos, varDecl.var);
	};

	// Bind parameters.
	for (i, x in q.bind) {
		r.add(bindParam(pos, r, prepared, i, x));
	}

	// Execute it, and save the result.
	var resultDecl = Var(r, SStr("result"), namedExpr(r, pos, "execute", prepared));
	r.add(resultDecl);

	// Get the result, if desired. Otherwise, finalize the query.
	LocalVarAccess resultAccess(pos, resultDecl.var);
	if (e = query.result(r, resultAccess))
		r.add(e);
	else
		r.add(namedExpr(r, pos, "finalize", resultAccess));

	r;
}

// Create a query expression for an untyped query. Helper.
private Expr untypedQueryExpr(SrcPos pos, Block block, Expr database, Query query) on Compiler {
	query.resolve(ResolveContext(block));
	var q = query.build();

	ExprBlock r(pos, block);

	var prepared = {
		var stmt = namedExpr(r, pos, "prepare", database, Actuals(StrLiteral(pos, q.query.toS)));
		var varDecl = Var(r, SStr("statement"), stmt);
		r.add(varDecl);
		LocalVarAccess(pos, varDecl.var);
	};

	// Bind parameters.
	for (i, x in q.bind) {
		Actuals params;
		params.add(NumLiteral(pos, i));
		params.add(x);
		r.add(namedExpr(r, pos, "bind", prepared, params));
	}

	// Execute it, and save the result.
	var resultDecl = Var(r, SStr("result"), namedExpr(r, pos, "execute", prepared));
	r.add(resultDecl);

	// Get the result, if desired. Otherwise, finalize the query.
	LocalVarAccess resultAccess(pos, resultDecl.var);
	if (e = query.result(r, resultAccess))
		r.add(e);
	else
		r.add(namedExpr(r, pos, "finalize", resultAccess));

	r;
}


// Create a suitable query expression depending on what database connection was used.
Expr queryExpr(SrcPos pos, Block block, Expr expr, Query query) on Compiler {
	var result = expr.result().type();
	if (x = result.type as DatabaseType) {
		// Typed version.
		return typedQueryExpr(pos, block, expr, x.contents, query);
	} else if (Value(named{DBConnection}).mayReferTo(result)) {
		// Untyped version.
		return untypedQueryExpr(pos, block, expr, query);
	} else {
		throw SyntaxError(expr.pos, "Expected a database connection (typed or untyped).");
	}
}

// A Query block.
class QueryBlock extends ExprBlock {
	init(SrcPos pos, Block parent, Expr db) {
		var result = db.result.type;
		if (Value(named{DBConnection}).mayReferTo(result)) {
		} else if (result.type as DatabaseType) {
		} else {
			throw SyntaxError(db.pos, "Expected a database connection (typed or untyped).");
		}

		super(pos, parent);

		Var var(this, SStr(" db", db.pos), db);
		add(var);

		init {
			connection(db.pos, var.var);
		}
	}

	// Variable containing our database connection.
	LocalVarAccess connection;

	// Helper to find a Query block.
	QueryBlock find(Block in) : static {
		NameLookup? at = in.lookup;
		while (l = at as BlockLookup) {
			if (q = l.block as QueryBlock)
				return q;

			at = l.parent;
		}

		throw InternalError("Could not find a parent QueryBlock!");
	}
}

// Create a suitable query expression, assuming we're inside a WITH block.
Expr queryBlockExpr(SrcPos pos, Block block, Query query) on Compiler {
	QueryBlock q = QueryBlock:find(block);
	queryExpr(pos, block, q.connection, query);
}

/**
 * Table name. Optionally creates an alias for the query.
 */
class TableName on Compiler {
	SrcPos pos;
	Str name;
	Str alias;

	// No alias.
	init(SStr name) {
		init { pos = name.pos; name = name.v; alias = name.v; }
	}

	// Create an alias.
	init(SStr name, SStr alias) {
		init { pos = name.pos; name = name.v; alias = alias.v; }
	}

	// Build.
	void build(QueryBuilder b) {
		b.name(name);
		if (name != alias) {
			b.query << " AS ";
			b.name(alias);
		}
	}

	// To string.
	void toS(StrBuf to) {
		to << name;
		if (name != alias)
			to << " AS " << alias;
	}
}

/**
 * Context passed around while resolving a SQL query.
 */
class ResolveContext on Compiler {
	// Current BS block.
	Block block;

	// Contents of the database.
	Database? db;

	// Which tables are visible in the current context? Keys are aliases and not necessarily table names.
	Str->Table visible;

	// Create typed version.
	init(Block block, Database db) {
		init { block = block; db = db; }
	}

	// Create untyped version.
	init(Block block) {
		init { block = block; }
	}

	// Is this a typed query?
	Bool typed() {
		db.any;
	}

	// Add a table to 'current'. Throws if the table is known not to exists.
	Table? addTable(SrcPos pos, Str to) {
		// Untyped version?
		unless (db)
			return null;

		unless (found = db.find(to))
			throw NoSuchTable(pos, to);
		if (visible.has(to))
			throw DuplicateAlias(pos, to);
		visible.put(to, found);
		found;
	}
	Table? addTable(TableName table) {
		unless (db)
			return null;

		unless (found = db.find(table.name))
			throw NoSuchTable(table.pos, table.name);
		if (visible.has(table.alias))
			throw DuplicateAlias(table.pos, table.alias);
		visible.put(table.alias, found);
		found;
	}

	// Result from "resolve"
	value ResolveResult {
		Str table;
		Column column;

		init(Str table, Column column) {
			init { table = table; column = column; }
		}
	}

	// Resolve a column.
	ResolveResult? resolve(SrcPos pos, Str? table, Str column) {
		if (table) {
			if (!visible.has(table))
				return null;
			Table t = visible.get(table);
			if (col = t.find(column))
				return ResolveResult(table, col);
			return null;
		} else {
			ResolveResult? result;
			for (alias, t in visible) {
				if (col = t.find(column)) {
					if (result) {
						throw SyntaxError(pos, "The column name ${column} is ambigous. It exists in tables ${result.table} and ${t.name} at least.");
					} else {
						result = ResolveResult(alias, col);
					}
				}
			}
			return result;
		}
	}
}


/**
 * Query builder. Collects a SQL query string in a StrBuf, as well as BS expressions for any
 * parameters that need to be bound.
 */
class QueryBuilder on Compiler {
	// String builder that contains the final SQL query.
	StrBuf query;

	// Expressions that are used to bind parameters.
	Expr[] bind;

	// Append a properly escaped SQL name to the query.
	void name(Str name) {
		// It seems in standard SQL, double quotes are used for identifiers.
		// I am unable to find anything that describes how to include double quotes in an identifier.
		// Perhaps we should just disallow that (currently it is implicitly disallowed due to the grammar).
		query << "\"" << name << "\"";
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "Query: " << query.toS;
		to << "\nData: " << bind;
	}
}


/**
 * Base class for an SQL query.
 *
 * Note: .toS() looks like SQL, but does not take proper care of edge-cases, so should not be
 * treated as a proper SQL statement.
 */
class Query on Compiler {
	// Position in source.
	SrcPos pos;

	// Create.
	init(SrcPos pos) {
		init() {
			pos = pos;
		}
	}

	// Resolve any unknowns in this query, given a suitable context.
	// Also typechecks as applicable.
	void resolve(ResolveContext context) : abstract;

	// Build a query.
	QueryBuilder build() : abstract;

	// Compute the result of this query, if it should be available.
	Expr? result(Block context, Expr result) {
		null;
	}
}

/**
 * INSERT query.
 */
class InsertQuery extends Query {
	// Table name.
	SStr table;

	// Columns to insert into. If empty, we insert into all columns.
	SStr[] columns;

	// Values to insert.
	SQLExpr[] values;

	// Create, insert all columns.
	init(SrcPos pos, SStr table, SQLExpr[] values) {
		init(pos) {
			table = table;
			values = values;
		}
	}

	// Create, insert into only a subset of columns.
	init(SrcPos pos, SStr table, SStr[] columns, SQLExpr[] values) {
		init(pos) {
			table = table;
			columns = columns;
			values = values;
		}
	}

	// Resolve.
	void resolve(ResolveContext context) : override {
		var table = context.addTable(table.pos, table.v);

		// If we found a table, explicitly specify the column names for easier typechecking later.
		Column[] cols;
		if (table) {
			if (columns.empty) {
				for (c in table.columns)
					columns << SStr(c.name, pos);
			} else {
				// Find all of the columns.
				Set<Str> used;
				for (c in columns) {
					used.put(c.v);
					unless (found = table.find(c.v))
						throw NoSuchColumn(c.pos, c.v, table.name);
					cols << found;
				}

				Bool multiPK = table.multiplePK();

				// Check the ones that are not inserted.
				for (c in table.columns) {
					if (used.has(c.name))
						continue;

					if (!c.hasDefault(multiPK))
						throw SyntaxError(pos, "The column ${c.name} has no default value, and needs to be given a value.");
				}
			}
		}

		if (cols.any & (values.count != cols.count))
			throw SyntaxError(pos, "The number of values does not match the number of columns inserted into.");

		// It does not really make sense to refer to column names in an insert statement...
		context.visible.clear();

		for (Nat i = 0; i < values.count; i++)
			values[i] = values[i].resolve(context);

		for (i, col in cols) {
			Value result(values[i].result);
			if (col.allowNull)
				result = unwrapMaybe(result);

			if (!Value(col.datatype.storm).mayReferTo(result))
				throw SyntaxError(values[i].pos, "Can not store a value of type ${result} in the column \"${col.name}\".");
		}
	}

	// Build the query.
	QueryBuilder build() : override {
		QueryBuilder r;

		r.query << "INSERT INTO ";
		r.name(table.v);
		if (columns.any) {
			r.query << " (";
			for (i, col in columns) {
				if (i > 0)
					r.query << ", ";
				r.name(col.v);
			}
			r.query << ")";
		}
		r.query << " VALUES (";
		for (i, v in values) {
			if (i > 0)
				r.query << ", ";
			v.build(r);
		}
		r.query << ");";

		r;
	}

	// Return the last created row ID.
	Expr? result(Block context, Expr result) {
		namedExpr(context, pos, "lastRowId", result);
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "INSERT INTO " << table.v;
		if (columns.any)
			to << " (" << join(columns, ", ", (x) => x.v) << ")";

		to << " VALUES (" << join(values, ", ") << ");";
	}
}

// Helper to check that a condition returns a boolean. Assumes that the condition was resolved beforehand.
private void checkCondition(SQLExpr condition) on Compiler {
	var result = condition.result;
	if (!Value(named{Bool}).mayReferTo(unwrapMaybe(result)))
		throw SyntaxError(condition.pos, "Expressions in WHERE and ON clauses are expected to return a Bool, not ${result}.");
}

/**
 * UPDATE query.
 */
class UpdateQuery extends Query {
	// Table to update.
	SStr table;

	// Columns to update.
	AssignExpr[] update;

	// Condition, if any.
	SQLExpr? condition;

	// Create.
	init(SrcPos pos, SStr table, AssignExpr[] update, SQLExpr? condition) {
		init(pos) {
			table = table;
			update = update;
			condition = condition;
		}
	}

	// Resolve.
	void resolve(ResolveContext context) : override {
		var table = context.addTable(table.pos, table.v);

		// Resolve all assignments.
		for (x in update) {
			x.value = x.value.resolve(context);
		}

		// Update the expression if suitable.
		if (x = condition) {
			var resolved = x.resolve(context);
			if (context.typed)
				checkCondition(resolved);
			condition = resolved;
		}

		if (table) {
			// Check if all columns exist, and type-check the assignments.
			for (x in update) {
				unless (column = table.find(x.column.v))
					throw NoSuchColumn(x.column.pos, x.column.v, table.name);

				Value result(x.value.result);
				if (!Value(column.datatype.storm).mayReferTo(result))
					throw SyntaxError(x.value.pos, "Cannot assign a ${result} to column ${column.name}");
			}
		}
	}

	// Build the query.
	QueryBuilder build() : override {
		QueryBuilder r;

		r.query << "UPDATE ";
		r.name(table.v);
		r.query << " SET ";
		for (i, x in update) {
			if (i > 0)
				r.query << ", ";
			r.query << x.column.v << " = ";
			x.value.build(r);
		}

		if (condition) {
			r.query << " WHERE ";
			condition.build(r);
		}

		r.query << ";";

		r;
	}

	// Return the number of modified rows.
	Expr? result(Block context, Expr result) {
		namedExpr(context, pos, "changes", result);
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "UPDATE " << table.v << " SET " << join(update, ", ");
		if (condition) {
			to << " WHERE " << condition;
		}
	}
}

/**
 * Value to update in an UPDATE query.
 */
class AssignExpr on Compiler {
	// Column to update.
	SStr column;

	// Value to assign.
	SQLExpr value;

	// Create.
	init(SStr column, SQLExpr value) {
		init { column = column; value = value; }
	}

	// To string.
	void toS(StrBuf to) : override {
		to << column.v << " = " << value;
	}
}


/**
 * DELETE query.
 */
class DeleteQuery extends Query {
	// Table to update.
	SStr table;

	// Condition, if any.
	SQLExpr? condition;

	// Create.
	init(SrcPos pos, SStr table, SQLExpr? condition) {
		init(pos) {
			table = table;
			condition = condition;
		}
	}

	// Resolve.
	void resolve(ResolveContext context) : override {
		context.addTable(table.pos, table.v);

		if (x = condition) {
			var resolved = x.resolve(context);
			if (context.typed)
				checkCondition(resolved);
			condition = resolved;
		}
	}

	// Build the query.
	QueryBuilder build() : override {
		QueryBuilder r;

		r.query << "DELETE FROM ";
		r.name(table.v);
		if (condition) {
			r.query << " WHERE ";
			condition.build(r);
		}
		r.query << ";";

		r;
	}

	// Return the number of modified rows.
	Expr? result(Block context, Expr result) {
		namedExpr(context, pos, "changes", result);
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "DELETE FROM " << table.v;
		if (condition) {
			to << " WHERE " << condition;
		}
	}
}


/**
 * Common parts for SELECT-like queries (SELECT, SELECT ONE and COUNT currently).
 *
 * Handles the logic of JOIN and WHERE clauses.
 */
class SelectBase extends Query {
	// Table.
	TableName table;

	/**
	 * Join statement.
	 */
	class Join on Compiler {
		// Type of join.
		Str type;

		// Table.
		TableName table;

		// Condition for join.
		SQLExpr expr;

		// Create.
		init(Str type, TableName table, SQLExpr expr) {
			init {
				type = type;
				table = table;
				expr = expr;
			}
		}

		// Build.
		void build(QueryBuilder to) {
			to.query << type << " ";
			table.build(to);
			to.query << " ON ";
			expr.build(to);
		}

		// ToS.
		void toS(StrBuf to) : override {
			to << type << " " << table << " ON " << expr;
		}
	}

	// Join clauses.
	Join[] joins;

	// Where clause, if present.
	SQLExpr? condition;

	init(SrcPos pos, TableName table, Join[] joins, SQLExpr? condition) {
		init(pos) {
			table = table;
			joins = joins;
			condition = condition;
		}
	}

	// Resolve.
	void resolve(ResolveContext context) : override {
		context.addTable(table);

		for (j in joins) {
			context.addTable(j.table);
			// TODO: Might be too early, but I think this is how SQL works.
			j.expr = j.expr.resolve(context);
			if (context.typed)
				checkCondition(j.expr);
		}

		if (x = condition) {
			var resolved = x.resolve(context);
			if (context.typed)
				checkCondition(resolved);
			condition = resolved;
		}
	}

	// Build the tail of the query (i.e. JOIN and WHERE clauses).
	protected void buildTail(QueryBuilder to) {
		for (j in joins) {
			to.query << " ";
			j.build(to);
		}

		if (condition) {
			to.query << " WHERE ";
			condition.build(to);
		}
	}

	// To string. Once again, only the tail parts.
	protected void tailToS(StrBuf to) {
		for (j in joins)
			to << j;
		if (condition)
			to << " WHERE " << condition;
	}
}


/**
 * SELECT query.
 */
class SelectQuery extends SelectBase {
	/**
	 * Column to extract.
	 */
	class Column on Compiler {
		// Position
		SrcPos pos;

		// Table (optional).
		Str? table;

		// Column.
		Str column;

		// Optional alternate name.
		Str as;

		// Type of this column, if available.
		Type? type;

		init(SrcPos pos, SStr column) {
			init {
				pos = pos;
				column = column.v;
				as = column.v;
			}
		}

		init(SrcPos pos, SStr table, SStr column) {
			init {
				pos = pos;
				table = table.v;
				column = column.v;
				as = table.v # "_" # column.v;
			}
		}

		init(SrcPos pos, Str? table, Str column) {
			init {
				pos = pos;
				table = table;
				column = column;
				as = column;
			}

			if (table)
				as = table # "_" # column;
		}

		void setAs(SStr to) {
			as = to.v;
		}

		// Build.
		void build(QueryBuilder to) {
			if (table) {
				to.name(table);
				to.query << ".";
			}
			to.name(column);
		}

		// ToS.
		void toS(StrBuf to) : override {
			if (table)
				to << table << ".";
			to << column << " AS " << as;
		}
	}


	/**
	 * Single order by column.
	 */
	class OrderBy on Compiler {
		// Position
		SrcPos pos;

		// Table (optional).
		Str? table;

		// Column.
		Str column;

		// Ascending/descending.
		Bool ascending;

		init(SrcPos pos, SStr? table, SStr column, Bool ascending) {
			Str? t;
			if (table)
				t = table.v;

			init {
				pos = pos;
				table = t;
				column = column.v;
				ascending = ascending;
			}
		}

		// Build.
		void build(QueryBuilder to) {
			if (table) {
				to.name(table);
				to.query << ".";
			}
			to.name(column);

			if (ascending)
				to.query << " ASC";
			else
				to.query << " DESC";
		}

		// To string.
		void toS(StrBuf to) {
			if (table)
				to << table << ".";
			to << column;
			if (ascending)
				to << " ASC";
			else
				to << " DESC";
		}
	}

	// Columns to select. If empty, we assume all columns.
	Column[] cols;

	// Order by.
	OrderBy[] orderBy;

	init(SrcPos pos, TableName table, Column[] cols, Join[] joins, SQLExpr? condition, OrderBy[] orderBy) {
		init(pos, table, joins, condition) {
			cols = cols;
			orderBy = orderBy;
		}
	}

	// Resolve.
	void resolve(ResolveContext context) : override {
		super:resolve(context);

		if (context.visible.any)
			resolveOutput(context);
	}

	// Resolve the output data.
	private void resolveOutput(ResolveContext context) {
		if (cols.empty) {
			// No columns -> we need to add all visible columns.
			for (table in context.visible) {
				for (col in table.columns) {
					Column c = if (joins.empty) {
						Column(pos, null, col.name);
					} else {
						Column(pos, table.name, col.name);
					};
					c.type = resultType(col);
					cols << c;
				}
			}
		} else {
			// We have some columns. Check them.
			for (col in cols) {
				unless (result = context.resolve(col.pos, col.table, col.column)) {
					if (table = col.table)
						throw NoSuchColumn(col.pos, col.column, table);
					else
						throw NoSuchColumn(col.pos, col.column, table.name);
				}

				col.type = resultType(result.column);
			}
		}

		for (col in orderBy) {
			if (context.resolve(col.pos, col.table, col.column).empty) {
				if (table = col.table)
					throw NoSuchColumn(col.pos, col.column, table);
				else
					throw NoSuchColumn(col.pos, col.column, table.name);
			}
		}
	}

	// Figure out the type for a column.
	private Type resultType(sql:Column col) {
		Type t = col.datatype.storm;
		if (col.allowNull) {
			if (maybe = wrapMaybe(t).type)
				return maybe;
		}
		t;
	}

	// Build the query.
	QueryBuilder build() : override {
		QueryBuilder r;

		r.query << "SELECT ";
		if (cols.empty) {
			r.query << "*";
		} else {
			for (i, name in cols) {
				if (i > 0)
					r.query << ", ";
				name.build(r);
			}
		}
		r.query << " FROM ";
		table.build(r);

		buildTail(r);

		if (orderBy.any) {
			r.query << " ORDER BY ";
			for (i, name in orderBy) {
				if (i > 0)
					r.query << ", ";
				name.build(r);
			}
		}

		r.query << ";";
		r;
	}

	// Return a proper iterator.
	Expr? result(Block context, Expr result) {
		Bool noTypeInfo = cols.empty;

		ValParam[] types;
		if (!noTypeInfo) {
			for (c in cols) {
				if (t = c.type)
					types << ValParam(t, c.as);
				else
					noTypeInfo = true;
			}
		}

		// If we don't have type info, just return a plain iterator.
		if (noTypeInfo)
			return namedExpr(context, pos, "iter", result);

		var iterType = getTypedIter(types);
		unless (ctor = iterType.find("__init", [Value(iterType), result.result.type.asRef(false)], Scope()) as Function)
			throw InternalError("Could not find a suitable constructor in the generated type.");

		CtorCall(pos, context.scope, ctor, Actuals(result));
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "SELECT " << join(cols, ", ") << " FROM ";
		table.toS(to);
		tailToS(to);
		if (orderBy.any) {
			to << " ORDER BY ";
			for (i, name in orderBy) {
				if (i > 0)
					to << ", ";
				name.toS(to);
			}
		}
	}
}

/**
 * SELECT ONE query.
 *
 * Works just like SELECT, but returns Maybe<Row> rather than an iterator.
 */
class SelectOneQuery extends SelectQuery {
	init(SrcPos pos, TableName table, Column[] cols, Join[] joins, SQLExpr? condition, OrderBy[] orderBy) {
		init(pos, table, cols, joins, condition, orderBy) {}
	}

	// Modified result. Returns a single row (wrapped in Maybe) rather than an iterator.
	Expr? result(Block context, Expr result) {
		Bool noTypeInfo = cols.empty;

		ValParam[] types;
		if (!noTypeInfo) {
			for (c in cols) {
				if (t = c.type)
					types << ValParam(t, c.as);
				else
					noTypeInfo = true;
			}
		}

		// No type info - just return a Maybe<Row> object.
		if (noTypeInfo)
			return namedExpr(context, pos, "next", result);

		// Otherwise, we want to return either <null> or an instance of the row-type.
		TypedRow t = getTypedRow(types);
		unless (ctor = t.find("__init", [Value(t), Value(named{Row})], Scope()) as Function)
			throw InternalError("Could not find a suitable constructor in the generated type.");

		unless (maybe = wrapMaybe(Value(t)).type)
			throw InternalError("Could not find the maybe type for the generated row type.");

		// Create an if-statement.
		WeakMaybeCast cast(namedExpr(context, pos, "next", result));
		cast.name(SStr("x"));
		If check(context, cast);

		CondSuccess success(pos, context, cast);
		check.trueCode = success;
		if (created = cast.result) {
			success.set(CtorCall(pos, Scope(), ctor, Actuals(LocalVarAccess(pos, created))));
		}

		if (maybeCtor = maybe.find("__init", [Value(maybe, true)], Scope()) as Function) {
			check.falseCode = CtorCall(pos, Scope(), maybeCtor, Actuals());
		}

		check;
	}
}

/**
 * COUNT FROM query.
 *
 * Works like SELECT COUNT(*) FROM, but clearer. Returns a Nat.
 */
class CountQuery extends SelectBase {
	init(SrcPos pos, TableName table, Join[] joins, SQLExpr? condition) {
		init(pos, table, joins, condition) {}
	}

	// Build the query.
	QueryBuilder build() : override {
		QueryBuilder r;

		r.query << "SELECT COUNT(*) FROM ";
		table.build(r);

		buildTail(r);

		r.query << ";";
		r;
	}

	// Return a Nat representing the number of rows.
	Expr? result(Block context, Expr result) {
		pattern(context) {
			if (row = ${result}.next()) {
				row.getLong(0).nat;
			} else {
				// This should not happen...
				0;
			}
		};
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "COUNT FROM ";
		table.toS(to);
		tailToS(to);
	}
}

/**
 * CREATE TABLE query.
 *
 * Only for untyped connections.
 */
class CreateQuery extends Query {
	// Table declaration.
	Table table;

	// Is this a "CREATE IF NOT EXISTS"?
	Bool ifNotExists;

	// Create.
	init(SrcPos pos, Bool ifNotExists, Table table) {
		init(pos) {
			table = table;
			ifNotExists = ifNotExists;
		}
	}

	// Resolve.
	void resolve(ResolveContext context) : override {
		if (context.typed)
			throw SyntaxError(pos, "Can not use CREATE TABLE queries with typed connections.");
	}

	// Build the query.
	QueryBuilder build() : override {
		QueryBuilder r;
		r.query << table.toSQL(ifNotExists, []);
		r;
	}

}

/**
 * CREATE INDEX query.
 *
 * Only for untyped connections.
 */
class IndexQuery extends Query {
	// Index name.
	Str name;

	// Table name.
	SStr table;

	// Columns.
	SStr[] columns;

	// Create.
	init(SrcPos pos, SStr name, SStr table, SStr[] columns) {
		init(pos) {
			name = name.v;
			table = table;
			columns = columns;
		}
	}

	// Create.
	init(SrcPos pos, SStr table, SStr[] columns) {
		StrBuf name;
		name << table.v << "_" << join(columns, "_", (x) => x.v);

		init(pos) {
			name = name.toS;
			table = table;
			columns = columns;
		}
	}

	// Resolve.
	void resolve(ResolveContext context) : override {
		if (context.typed)
			throw SyntaxError(pos, "Can not use CREATE TABLE queries with typed connections.");
	}

	// Build the query.
	QueryBuilder build() : override {
		QueryBuilder r;

		r.query << "CREATE INDEX ";
		r.name(name);
		r.query << " ON ";
		r.name(table.v);
		r.query << "(";
		for (i, c in columns) {
			if (i > 0)
				r.query << ", ";
			r.name(c.v);
		}
		r.query << ");";

		r;
	}

}

/**
 * DROP TABLE query.
 *
 * Only for untyped connections.
 */
class DropQuery extends Query {
	// Table name.
	SStr table;

	// Create.
	init(SrcPos pos, SStr table) {
		init(pos) {
			table = table;
		}
	}

	// Resolve.
	void resolve(ResolveContext context) : override {
		if (context.typed)
			throw SyntaxError(pos, "Can not use CREATE TABLE queries with typed connections.");
	}

	// Build the query.
	QueryBuilder build() : override {
		QueryBuilder r;

		r.query << "DROP TABLE ";
		r.name(table.v);
		r.query << ";";

		r;
	}

}
