/***************************************************************************
 *   Copyright (C) 2005 by Bjoern Erik Nilsen & Fredrik Berg Kjoelstad     *
 *   bjoern.nilsen@bjoernen.com     & fredrikbk@hotmail.com                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "exporttab.h"

#include "graphics/icons/close.xpm"
#include "flexiblelineedit.h"
#include "src/foundation/preferencestool.h"

#include <qlayout.h>
#include <qfiledialog.h>


ExportTab::ExportTab(QWidget *parent) : QFrame(parent)
{
	addButton = NULL;
	removeButton = NULL;
	editButton = NULL;
	browseButton = NULL;
	yesButton = NULL;
	noButton = NULL;
	closeButton = NULL;
	encoderPrefs = NULL;
	encoderTable = NULL;
	startEncoder = NULL;
	stopEncoder = NULL;
	startEncoderLabel = NULL;
	stopEncoderLabel = NULL;
	defaultOutput = NULL;
	defaultOutputLabel = NULL;
	mainGrid = NULL;
	settingsGrid = NULL;
	space3 = NULL;
	rightSpace = NULL;
	leftSpace = NULL;
	askForOutput = NULL;
	infoText = NULL;
	
	selectedEncoder = -1;
	numEncoders = 0;
	
	makeGUI();
}


void ExportTab::makeGUI()
{
	mainGrid = new QGridLayout(this, 1, 1, 3);
	
	infoText = new QLabel(
			tr(
				"<p>Below you can set which program/process stopmotion should use "
				"for encoding the currently active project to a video file.</p>"
				"<p>You can use $IMAGEPATH to represent the image "
				"path (absolute path to the directory where the images can be found). "
				"You should always use <b>$VIDEOFILE</b> to represent the "
				"output file (the videofile generated by the encoder) in the <b>command "
				"line</b>. This is independent of how you decides to "
				"select it (select it for each export or have a default file)."
				"<br></p>"
				"<p> Example with mencoder (converting jpg to mpeg1): <br>"
				"mencoder mf://$IMAGEPATH/*.jpg -mf w=640:h=480:fps=12:type=jpg "
				"-ovc lavc -lavcopts vcodec=mpeg1video -oac copy -o $VIDEOFILE <br>"
			), this);
	infoText->setMinimumWidth(440);
	infoText->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
	
	encoderTable = new QTable(this);
	encoderTable->setMinimumHeight(120);
	encoderTable->setNumCols(3);
	encoderTable->setSelectionMode(QTable::SingleRow);
	encoderTable->horizontalHeader()->setLabel( 0, tr( "Active" ) );
	encoderTable->horizontalHeader()->setLabel( 1, tr( "Name" ) );
	encoderTable->horizontalHeader()->setLabel( 2, tr( "Description" ) );
	encoderTable->adjustColumn(0);
	encoderTable->adjustColumn(1);
	encoderTable->adjustColumn(2);
	connect( encoderTable, SIGNAL(currentChanged(int, int)),
			this, SLOT(activeRowChanged(int)) );
	connect( encoderTable, SIGNAL(valueChanged(int, int)),
			this, SLOT(valueChanged(int, int)) );
	
	addButton = new QPushButton(tr("&Add") ,this);
	addButton->setFocusPolicy( QWidget::NoFocus );
	connect(addButton, SIGNAL(clicked()), this, SLOT(addEncoder()) );
	
	removeButton = new QPushButton(tr("&Remove"), this);
	connect( removeButton, SIGNAL(clicked()), this, SLOT(removeEncoder()) );
	
	editButton = new QPushButton(tr("&Edit"), this);
	QObject::connect( editButton, SIGNAL(clicked()), this, SLOT(editSettings()) );
	
	space3 = new QSpacerItem(0, 10, QSizePolicy::Fixed, QSizePolicy::Fixed);
	rightSpace = new QSpacerItem(5, 0, QSizePolicy::Fixed, QSizePolicy::Fixed);
	leftSpace = new QSpacerItem(5, 0, QSizePolicy::Fixed, QSizePolicy::Fixed);
	
	encoderPrefs = new QGroupBox(this);
	encoderPrefs->setTitle( tr("Encoder settings") );
	encoderPrefs->hide();
	settingsGrid = new QGridLayout(encoderPrefs, 1, 1, 3);
	
	closeButton = new QPushButton(encoderPrefs);
	closeButton->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
	closeButton->setPixmap(closeicon);
	closeButton->setFlat(true);
	connect( closeButton, SIGNAL(clicked()),this, SLOT(closeSettings()) );
	
	askForOutput = new QLabel( 
			tr("Do you want to be asked for an output file everytime "
			"you choose to export?"), encoderPrefs);
			   
	yesButton = new QRadioButton(tr("Yes"), encoderPrefs);
	yesButton->setChecked(true);
	yesButton->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
	connect( yesButton, SIGNAL(clicked()), this, SLOT(setYesButtonOn()) );
	
	noButton = new QRadioButton(tr("No"), encoderPrefs);
	noButton->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
	noButton->setChecked(false);
	connect( noButton, SIGNAL(clicked()), this, SLOT(setNoButtonOn()) );
	
	defaultOutputLabel = new QLabel( tr("Set default output file:"), encoderPrefs );
	defaultOutput = new FlexibleLineEdit(encoderPrefs);
	defaultOutput->setEnabled(false);
	connect( defaultOutput, SIGNAL(textChanged(const QString &)), 
			this, SLOT(setDefaultOutput(const QString &)) );
			
	browseButton = new QPushButton(tr("Browse"), encoderPrefs);
	browseButton->setEnabled(false);
	browseButton->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
	connect( browseButton, SIGNAL(clicked()), this, SLOT(browseFiles()) );
	
	startEncoderLabel = new QLabel( tr("Start encoder:"), encoderPrefs );
	startEncoder = new FlexibleLineEdit(encoderPrefs);
	connect( startEncoder, SIGNAL(textChanged(const QString &)),
			this, SLOT(updateStartString(const QString &)) );
	
	stopEncoderLabel = new QLabel( tr("Stop encoder:"), encoderPrefs );
	stopEncoder = new FlexibleLineEdit(encoderPrefs);
	connect( stopEncoder, SIGNAL(textChanged(const QString &)),
			this, SLOT(updateStopString(const QString &)) );
	
	mainGrid->addWidget(infoText, 0, 1);
	mainGrid->addMultiCellWidget(encoderTable, 1, 4, 1, 1);
	mainGrid->addWidget(addButton, 2, 2);
	mainGrid->addWidget(removeButton, 3, 2);
	mainGrid->addWidget(editButton, 4, 2 );
	mainGrid->addMultiCellWidget(encoderPrefs, 5, 5, 1, 2);
	mainGrid->addItem(space3, 6, 1);
	mainGrid->addItem(rightSpace, 3, 3);
	mainGrid->addItem(leftSpace, 3, 0);

	settingsGrid->addWidget(closeButton, 0, 4);
	settingsGrid->addMultiCellWidget(askForOutput, 1, 1, 0, 3);
	settingsGrid->addWidget(yesButton, 2, 0);
	settingsGrid->addWidget(noButton, 2, 1);
	settingsGrid->addMultiCellWidget(defaultOutputLabel, 3, 3, 0, 3);
	settingsGrid->addMultiCellWidget(defaultOutput, 4, 4, 0, 2);
	settingsGrid->addWidget(browseButton, 4, 3);
	settingsGrid->addMultiCellWidget(startEncoderLabel, 5, 5, 0, 3);
	settingsGrid->addMultiCellWidget(startEncoder, 6, 6, 0, 3);
	settingsGrid->addMultiCellWidget(stopEncoderLabel, 7, 7, 0, 3);
	settingsGrid->addMultiCellWidget(stopEncoder, 8, 8, 0, 3);
}


void ExportTab::initialize()
{
	PreferencesTool *pref = PreferencesTool::get();
	selectedEncoder = pref->getPreference("activeEncoder", 0);

	int numEncoders = pref->getPreference("numEncoders", 0);
	int newRow = 0;
	for(int i = 0; i < numEncoders; i++) {
		newRow = encoderTable->numRows();
		encoderTable->insertRows(newRow);
		encoderTable->setItem( newRow, 0, new QCheckTableItem(encoderTable, "") );
		encoderTable->setText( newRow, 1, QString(pref->getPreference(
				QString("encoderName%1").arg(i).ascii(), "")) );
		encoderTable->setText( newRow, 2, QString( pref->getPreference(
				QString("encoderDescription%1").arg(i).ascii(), "")) );
		startEncoderStrings.push_back(QString( pref->getPreference(
				QString("startEncoder%1").arg(i).ascii(), "")));
		stopEncoderStrings.push_back(QString( pref->getPreference(
				QString("stopEncoder%1").arg(i).ascii(), "")));
		outputFiles.push_back( QString( pref->getPreference( 
				QString("outputFile%1").arg(i).ascii(), "" )));
		
	}
	encoderTable->adjustColumn(1);
	encoderTable->adjustColumn(2);
	
	int width = encoderTable->columnWidth(2);
	int prefWidth = encoderTable->width() - 
			(encoderTable->columnWidth(0) + encoderTable->columnWidth(1));
	if (width < prefWidth) {
		encoderTable->setColumnWidth(2, prefWidth + 4);
	}
	
	int active = pref->getPreference("activeEncoder", -1);
	if(active > -1) {
		((QCheckTableItem*)encoderTable->item(active, 0))->setChecked(true);
	}
}


void ExportTab::apply()
{
	PreferencesTool *prefs = PreferencesTool::get();
	int size = encoderTable->numRows();
	int activeEncoder = -1;
	
	// Deletes removed encoders from the preferencestool.
	int numEncoders = prefs->getPreference("numEncoders", -1);
 	if(numEncoders > size) {
		for(int i = size; i < numEncoders; ++i) {
			prefs->removePreference(QString("encoderName%1").arg(i).ascii());
			prefs->removePreference(QString("encoderDescription%1").arg(i).ascii());
			prefs->removePreference(QString("startEncoder%1").arg(i).ascii());
			prefs->removePreference(QString("stopEncoder%1").arg(i).ascii());
			prefs->removePreference(QString("outputFile%1").arg(i).ascii());
		}
	}
	prefs->setPreference("numEncoders", size, true);
	
	for(int i = 0; i < size; i++) {
		prefs->setPreference(QString("encoderName%1").arg(i), 
				encoderTable->text(i, 1).ascii(), true);
		prefs->setPreference(QString("encoderDescription%1").arg(i),
				encoderTable->text(i, 2).ascii(), true);
		prefs->setPreference(QString("startEncoder%1").arg(i),
				startEncoderStrings[i].ascii(), true);
		prefs->setPreference(QString("stopEncoder%1").arg(i),
				stopEncoderStrings[i].ascii(), true);
		prefs->setPreference(QString("outputFile%1").arg(i),
				outputFiles[i].ascii(), true);
		
		if( ((QCheckTableItem*)encoderTable->item(i, 0))->isChecked() ) {
			activeEncoder = i;
		}
	}
	
	prefs->setPreference("activeEncoder", activeEncoder, true);
}


void ExportTab::addEncoder()
{
	int newRow = encoderTable->numRows();
	encoderTable->insertRows(newRow);
	encoderTable->setItem( newRow, 0, new QCheckTableItem(encoderTable, "") );
	startEncoderStrings.push_back("");
	stopEncoderStrings.push_back("");
	outputFiles.push_back("");
	++numEncoders;
}


void ExportTab::removeEncoder()
{
	if (numEncoders > 0 && selectedEncoder >= 0) {	
		startEncoderStrings.erase(startEncoderStrings.begin() + selectedEncoder);
		stopEncoderStrings.erase(stopEncoderStrings.begin() + selectedEncoder);
		outputFiles.erase(outputFiles.begin() + selectedEncoder);
		encoderTable->removeRow(selectedEncoder);
		--numEncoders;
	}
}


void ExportTab::valueChanged(int row, int column)
{
	switch (column) {
		case 0:
		{
			int size = encoderTable->numRows();
			for(int i = 0; i < size; i++) {
				((QCheckTableItem*)encoderTable->item(i, 0))->setChecked(i == row);
			}
			break;
		}
		case 1: case 2:
		{
			encoderTable->adjustColumn(1);
			encoderTable->adjustColumn(2);
			
			int width = encoderTable->columnWidth(2);
			int prefWidth = encoderTable->width() - 
					(encoderTable->columnWidth(0) + encoderTable->columnWidth(1));
			if (width < prefWidth) {
				encoderTable->setColumnWidth(2, prefWidth - 34);
			}
			
			activeRowChanged(row);
			break;
		}
	}
}


void ExportTab::activeRowChanged(int row)
{
	selectedEncoder = row;
	if(encoderPrefs->isShown()) {
		editSettings();
	}
}


void ExportTab::editSettings()
{
	if(selectedEncoder >= 0) {
		startEncoder->setText(startEncoderStrings[selectedEncoder]);
		stopEncoder->setText(stopEncoderStrings[selectedEncoder]);
		if ( outputFiles[selectedEncoder] == "" ) {
			setYesButtonOn();
		}
		else {
			setNoButtonOn();
		}
		encoderPrefs->show();
	}
}


void ExportTab::closeSettings( )
{
	encoderPrefs->hide();
}


void ExportTab::updateStartString(const QString &txt)
{
	startEncoderStrings[selectedEncoder] = txt;
}


void ExportTab::updateStopString(const QString &txt)
{
	stopEncoderStrings[selectedEncoder] = txt;
}


void ExportTab::setDefaultOutput(const QString &txt)
{
	outputFiles[selectedEncoder] = txt;
}


void ExportTab::setYesButtonOn()
{
	yesButton->setChecked(true);
	noButton->setChecked(false);
	defaultOutput->setText("");
	defaultOutput->setEnabled(false);
	browseButton->setEnabled(false);
	outputFiles[selectedEncoder] = "";
}


void ExportTab::setNoButtonOn()
{
	noButton->setChecked(true);
	yesButton->setChecked(false);
	defaultOutput->setEnabled(true);
	browseButton->setEnabled(true);
	defaultOutput->setText(outputFiles[selectedEncoder]);
}


void ExportTab::browseFiles()
{
	QString file = QFileDialog::getSaveFileName(getenv("PWD"), NULL,
			this, NULL, tr("Choose output file"));
	if ( !file.isEmpty() ) {
		outputFiles[selectedEncoder] = file;
		defaultOutput->setText(file);
	}
}
