package org.stegosuite.ui.gui;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.ResourceBundle;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.stegosuite.image.embedding.EmbeddingMethod;
import org.stegosuite.image.embedding.EmbeddingProgress;
import org.stegosuite.image.format.GIFImage;
import org.stegosuite.image.format.ImageFormat;
import org.stegosuite.image.util.FileUtils;
import org.stegosuite.model.exception.SteganoEmbedException;
import org.stegosuite.model.exception.SteganoExtractException;
import org.stegosuite.model.exception.SteganoImageException;
import org.stegosuite.model.payload.Payload;
import org.stegosuite.model.payload.block.Block;
import org.stegosuite.model.payload.block.FileBlock;
import org.stegosuite.model.payload.block.MessageBlock;
import org.stegosuite.ui.gui.ImageContainer.ImageState;
import org.stegosuite.ui.gui.embedding.Embedding;
import org.stegosuite.ui.gui.embedding.EmbeddingFactory;
import org.stegosuite.ui.gui.embedding.MyGIFShuffle;

/**
 * Contains the GUI for embedding/extracting data.
 */
public class EmbedUi {

	private Composite compositeImage;
	private Text passwordField;
	private Button checkBoxVisualize;
	private ImageContainer imageContainer;
	private Label imageLabel, payloadFileCounter, payloadFileSize;
	private int fileSizeSum = 0;
	private Payload payload;
	private Button embedButton, extractButton;
	private ImageFormat image;
	private Cursor cursor;
	private Embedding embedding;
	private ProgressBar progressBar;
	// private Text pathText;
	private Table fileTable;
	private StyledText messageField;
	private static final Logger LOG = LoggerFactory.getLogger(EmbedUi.class);
	private final ResourceBundle L = ResourceBundle.getBundle("Messages");

	public EmbedUi(Composite c1, GuiComponents components) {

		GuiComponents myComponents = components;
		c1.setLayout(new FillLayout());

		Composite compositeControls = myComponents.createControlsComposite(c1);
		Composite fileEmbedding = myComponents.createFileEmbedding(compositeControls);

		messageField = (StyledText) compositeControls.getChildren()[0];
		fileTable = (Table) fileEmbedding.getChildren()[2];
		payloadFileCounter = (Label) fileEmbedding.getChildren()[0];
		payloadFileSize = (Label) fileEmbedding.getChildren()[1];
		passwordField = myComponents.createPasswordField(compositeControls);

		embedButton = myComponents.createMainButton(compositeControls, L.getString("embed_button"));
		extractButton = myComponents.createMainButton(compositeControls, L.getString("extract_button"));
		((GridData) extractButton.getLayoutData()).horizontalAlignment = SWT.END;

		compositeImage = myComponents.createImageComposite(c1);
		imageLabel = new Label(compositeImage, SWT.NONE);
		imageLabel.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false));

		payload = new Payload();

		compositeImage.addListener(SWT.Resize, event -> {
			if (imageLabel.getImage() != null) {
				imageLabel.setImage(imageContainer.scaleImage());
				compositeImage.layout(true, true);
			}
		});

		final DropTarget dropTarget = new DropTarget(fileTable, DND.DROP_MOVE);
		dropTarget.setTransfer(new Transfer[] { FileTransfer.getInstance() });
		dropTarget.addDropListener(new DropTargetAdapter() {

			@Override
			public void drop(final DropTargetEvent event) {
				final String[] filenames = (String[]) event.data;
				if (filenames[0] != null) {
					payload.getBlocks().add(new FileBlock(filenames[0]));
					addPayloadFile(filenames[0]);
				}
			}
		});

		Menu menu = new Menu(c1.getShell(), SWT.POP_UP);
		fileTable.setMenu(menu);

		MenuItem item1 = new MenuItem(menu, SWT.PUSH);
		item1.setText(L.getString("files_menu_delete"));
		item1.addListener(SWT.Selection, event -> {
			removeSelectedPayloadFile();
		});

		MenuItem item2 = new MenuItem(menu, SWT.PUSH);
		item2.setText(L.getString("files_menu_add"));
		item2.addListener(SWT.Selection, event -> {
			final FileDialog dlg = new FileDialog(c1.getShell(), SWT.OPEN);
			final String filePath = dlg.open();
			if (filePath != null) {
				payload.getBlocks().add(new FileBlock(filePath));
				addPayloadFile(filePath);
			}
		});

		fileTable.addKeyListener(new KeyAdapter() {

			@Override
			public void keyReleased(final KeyEvent e) {
				if (e.keyCode == SWT.DEL) {
					removeSelectedPayloadFile();
				}
			}
		});

		embedButton.addListener(SWT.Selection, event -> {
			payload.getBlocks().add(new MessageBlock(messageField.getText()));

			if (!passwordField.getText().isEmpty()) {
				payload.setSteganoPassword(passwordField.getText());
				payload.setEncryptionPassword(passwordField.getText());
			} else {
				payload.setSteganoPassword(null);
				payload.setEncryptionPassword(null);
			}

			progressBar = new ProgressBar(compositeControls, SWT.SMOOTH);
			progressBar.setSelection(0);

			GridData data = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
			data.horizontalSpan = 2;
			progressBar.setLayoutData(data);
			compositeControls.layout(true, true);

			adjustWindowSize();

			EmbeddingProgress progress = new EmbeddingProgress();
			new EmbeddingProgressObserver(progressBar, progress);

			Gui.setStatusBarMsg("Embedding data...");
			embedButton.setEnabled(false);

			cursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
			c1.getShell().setCursor(cursor);
			if (image.getClass().equals(GIFImage.class)) {
				LOG.debug("Trying to embed with GIFShuffle.");
			}
			new Thread(() -> {
				embed(progress, c1);
			}).start();
		});

		extractButton.addListener(SWT.Selection, event -> {

			if (!passwordField.getText().isEmpty()) {
				payload.setSteganoPassword(passwordField.getText());
				payload.setEncryptionPassword(passwordField.getText());
			} else {
				payload.setSteganoPassword(null);
				payload.setEncryptionPassword(null);
			}

			progressBar = new ProgressBar(compositeControls, SWT.SMOOTH);
			progressBar.setSelection(0);

			GridData data = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
			data.horizontalSpan = 2;
			progressBar.setLayoutData(data);
			compositeControls.layout(true, true);

			adjustWindowSize();

			EmbeddingProgress progress = new EmbeddingProgress();
			new EmbeddingProgressObserver(progressBar, progress);

			Gui.setStatusBarMsg("Extracting data...");
			extractButton.setEnabled(false);

			cursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
			c1.getShell().setCursor(cursor);
			new Thread(() -> {
				extract(progress, c1);
			}).start();
		});
	}

	private void extract(EmbeddingProgress progress, Composite c1) {
		try {
			embedding.extract(payload, progress, (embeddingMethod) -> {
				Display.getDefault().asyncExec(() -> extractingComplete(embeddingMethod));
			});
		} catch (SteganoExtractException e) {

			if (embedding.getEmbeddingMethod() instanceof MyGIFShuffle) {
				LOG.debug("Extracting using GIFShuffle failed. Switching to GIFSortedColorTable.");
				embedding = EmbeddingFactory.GIF_SORTED.newEmbedding(image);
				extract(progress, c1);

			} else {
				Display.getDefault().syncExec(() -> {
					MessageBox dialog = new MessageBox(c1.getShell(), SWT.ICON_ERROR | SWT.OK);
					dialog.setText("Error");
					dialog.setMessage(e.getMessage());
					dialog.open();
					extractButton.setEnabled(true);
					cursor = new Cursor(Display.getDefault(), SWT.CURSOR_ARROW);
					compositeImage.getShell().setCursor(cursor);
					progressBar.dispose();
					Gui.setStatusBarMsg("Extracting aborted.");

				});
			}
		}
	}

	private void embed(EmbeddingProgress progress, Composite c1) {
		try {
			embedding.embed(payload, progress, (embeddingMethod, embeddedImage) -> {
				Display.getDefault().asyncExec(() -> embeddingComplete(embeddingMethod, embeddedImage));
			});
		} catch (SteganoEmbedException e) {

			if (embedding.getEmbeddingMethod() instanceof MyGIFShuffle) {
				LOG.debug("Embedding with GIFShuffle failed. Switching to GIFSortedColorTable.");
				embedding = EmbeddingFactory.GIF_SORTED.newEmbedding(image);
				embed(progress, c1);

			} else {
				Display.getDefault().syncExec(() -> {
					MessageBox dialog = new MessageBox(c1.getShell(), SWT.ICON_ERROR | SWT.OK);
					dialog.setText("Error");
					dialog.setMessage(e.getMessage());
					dialog.open();
					embedButton.setEnabled(true);
					cursor = new Cursor(Display.getDefault(), SWT.CURSOR_ARROW);
					compositeImage.getShell().setCursor(cursor);
					progressBar.dispose();
					Gui.setStatusBarMsg("Embedding aborted.");

				});
			}
		}
	}

	/**
	 * Gets called after the embedding-process is finished. Displays the image, disposes the
	 * progress bar etc.
	 *
	 * @param embeddingMethod
	 * @param embeddedImage
	 */
	public void embeddingComplete(EmbeddingMethod<?> embeddingMethod, ImageFormat embeddedImage) {
		String outputPath = FileUtils.addFileNameSuffix(embeddedImage.getFile().getAbsolutePath(), "_embed");
		try {
			embeddedImage.save(new File(outputPath));
		} catch (SteganoImageException e) {
			e.printStackTrace();
		}
		imageContainer.setImageData(ImageState.STEG, embeddedImage.getImageData());
		if (embeddingMethod.getVisualizer() != null) {
			imageContainer.setImageData(ImageState.STEG_VISUALIZED, embeddingMethod.getVisualizer().getImageData());
		}
		Gui.setStatusBarMsg("Embedding completed. File saved to " + outputPath);
		embedButton.setEnabled(true);
		imageLabel.setImage(imageContainer.scaleImage(ImageState.STEG));
		visualizationCheckbox();
		compositeImage.layout(true, true);
		payload = new Payload();
		fileTable.clearAll();
		fileTable.setItemCount(0);
		fileSizeSum = 0;
		payloadFileSize.setText("");
		payloadFileCounter.setText(fileTable.getItemCount() + " " + L.getString("files_text"));
		cursor = new Cursor(Display.getDefault(), SWT.CURSOR_ARROW);
		compositeImage.getShell().setCursor(cursor);
		progressBar.dispose();
	}

	/**
	 * Gets called after the extracting-process is finished. Displays the message, saves the
	 * extracted files, disposes the progress bar etc.
	 *
	 * @param embeddingMethod
	 */
	public void extractingComplete(EmbeddingMethod<?> embeddingMethod) {
		String outPath = null;
		ListIterator<Block> iterator = payload.getBlocks().listIterator();
		while (iterator.hasNext()) {
			Block block = iterator.next();
			if (block.getIdentifier() == FileBlock.IDENTIFIER) {
				outPath = FileUtils.changeFileName(image.getFile().getAbsolutePath(),
						((FileBlock) block).getFileName());
				try {
					Files.write(Paths.get(outPath), ((FileBlock) block).getFileContent(), StandardOpenOption.CREATE);
				} catch (IOException e) {
					e.printStackTrace();
				}
				iterator.remove();
				iterator.add(new FileBlock(outPath));
				addPayloadFile(outPath);
			} else if (block.getIdentifier() == MessageBlock.IDENTIFIER) {
				String message = ((MessageBlock) block).getMessage();
				messageField.setText(message);
			}
		}

		// for (Block block : payload.getBlocks()) {
		// if (block.getIdentifier() == FileBlock.IDENTIFIER) {
		//
		// outPath = FileUtils.changeFileName(image.getFile().getAbsolutePath(),
		// ((FileBlock) block).getFileName());
		// try {
		// Files.write(Paths.get(outPath), ((FileBlock) block).getFileContent(),
		// StandardOpenOption.CREATE);
		// } catch (IOException e) {
		// e.printStackTrace();
		// }
		// } else if (block.getIdentifier() == MessageBlock.IDENTIFIER) {
		// String message = ((MessageBlock) block).getMessage();
		// messageField.setText(message);
		// }
		// }

		String statusMessage = "Extracting completed.";
		if (outPath != null) {
			statusMessage += " Extracted file saved to " + outPath;
		}
		Gui.setStatusBarMsg(statusMessage);
		imageContainer.setImageData(ImageState.STEG, image.getImageData());
		if (embeddingMethod.getVisualizer() != null) {
			imageContainer.setImageData(ImageState.STEG_VISUALIZED, embeddingMethod.getVisualizer().getImageData());
		}
		extractButton.setEnabled(true);
		visualizationCheckbox();
		// payload = new Payload();
		cursor = new Cursor(Display.getDefault(), SWT.CURSOR_ARROW);
		compositeImage.getShell().setCursor(cursor);
		progressBar.dispose();
	}

	/**
	 * Loads an image from the given path, displays it and start capacity- and noise-calculations.
	 *
	 * @param image The image to load
	 */
	void loadImage(ImageFormat image)
			throws SteganoImageException {
		this.image = image;
		embedButton.setEnabled(false);
		extractButton.setEnabled(false);
		imageContainer = new ImageContainer(compositeImage);

		Image img = imageContainer.loadImage(image.getImageData());
		imageLabel.setImage(img);
		imageLabel.setToolTipText(image.getFile().getAbsolutePath());

		if (image.getClass().equals(GIFImage.class)) {
			embedding = EmbeddingFactory.GIFSHUFFLE.newEmbedding(image);
		}

		messageField.setEnabled(true);

		fileTable.clearAll();
		fileTable.setItemCount(0);
		payloadFileSize.setText("");
		fileSizeSum = 0;
		payloadFileCounter.setText(fileTable.getItemCount() + " " + L.getString("files_text"));
		payload = new Payload();
		visualizationCheckbox();
		compositeImage.layout(true, true);

		Gui.setStatusBarMsg("Searching for homogeneous areas in the image...");
		cursor = new Cursor(Display.getDefault(), SWT.CURSOR_WAIT);
		passwordField.getParent().setCursor(cursor);
		passwordField.getParent().layout(true, true);
		embedding = EmbeddingFactory.getEmbedding(image);

		new Thread(() -> {
			embedding.setPointFilter(1);
			int capacity = embedding.getCapacity();

			Display.getDefault().asyncExec(() -> {
				Gui.setStatusBarMsg(L.getString("statusbar_capacity") + ": " + ImageUtils.formatSize(capacity));
				embedButton.setEnabled(true);
				extractButton.setEnabled(true);
				cursor = new Cursor(Display.getDefault(), SWT.CURSOR_ARROW);
				passwordField.getParent().setCursor(cursor);

			});
		}).start();
	}

	private void visualizationCheckbox() {
		if (imageContainer.getImageData(ImageState.STEG_VISUALIZED) != null) {
			if (checkBoxVisualize == null || checkBoxVisualize.isDisposed()) {
				checkBoxVisualize = new Button(compositeImage, SWT.CHECK);
				checkBoxVisualize.setText(L.getString("visualize_checkbox"));
				checkBoxVisualize.addListener(SWT.Selection, event2 -> {
					ImageState state = checkBoxVisualize.getSelection() ? ImageState.STEG_VISUALIZED : ImageState.STEG;
					imageLabel.setImage(imageContainer.scaleImage(state));
				});
			} else {
				checkBoxVisualize.setSelection(false);
			}
		} else {
			if (checkBoxVisualize != null) {
				checkBoxVisualize.dispose();
			}
		}
	}

	private void adjustWindowSize() {
		final Point newSize = passwordField.getShell().computeSize(passwordField.getShell().getSize().x, SWT.DEFAULT,
				true);
		if (newSize.y > passwordField.getShell().getSize().y) {
			passwordField.getShell().setSize(newSize);
		}
	}

	private void addPayloadFile(String filePath) {
		String filename = Paths.get(filePath).getFileName().toString();
		TableItem item = new TableItem(fileTable, SWT.NONE);
		item.setText(filename);

		String extension = "";
		int i = filename.lastIndexOf('.');
		if (i > 0) {
			extension = filename.substring(i);
		}

		Program p = Program.findProgram(extension);
		if (p != null) {
			ImageData data = p.getImageData();
			if (data != null) {
				Image image = new Image(Display.getDefault(), data);
				item.setImage(image);
			}
		}

		payloadFileCounter.setText(fileTable.getItemCount() + " " + L.getString("files_text"));

		File f = new File(filePath);
		fileSizeSum += f.length();
		payloadFileSize.setText(ImageUtils.formatSize(fileSizeSum));
		item.setText(1, ImageUtils.formatSize(f.length()));
		item.setForeground(1, Display.getDefault().getSystemColor(SWT.COLOR_GRAY));
		final TableColumn[] columns = fileTable.getColumns();
		for (TableColumn column : columns) {
			column.pack();
		}
		payloadFileSize.getParent().layout(true, true);
	}

	private void removeSelectedPayloadFile() {
		if (fileTable.getItemCount() > 0) {
			if (fileTable.getSelectionIndex() >= 0) {
				final TableItem it = fileTable.getItem(fileTable.getSelectionIndex());
				String filename = it.getText();

				fileTable.remove(fileTable.getSelectionIndex());
				Iterator<Block> iterator = payload.getBlocks().iterator();
				while (iterator.hasNext()) {
					Block block = iterator.next();
					if (block.getIdentifier() == FileBlock.IDENTIFIER) {
						FileBlock fBlock = (FileBlock) block;
						String filePath = fBlock.getFileName();
						String filename2 = Paths.get(filePath).getFileName().toString();
						if (filename.equals(filename2)) {
							File f = new File(filePath);
							fileSizeSum -= f.length();
							payloadFileSize.setText(ImageUtils.formatSize(fileSizeSum));
							iterator.remove();
							LOG.debug("Fileblock removed.");
						}
					}
				}
				payloadFileCounter.setText(fileTable.getItemCount() + " " + L.getString("files_text"));
				payloadFileSize.getParent().layout(true, true);
			}
		}
	}
}
