/*
    FileListController.vala
    Copyright (C) 2010 Maia Kozheva <sikon@ubuntu.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

using Gtk;
using Steadyflow.Core;
using Steadyflow.UI;

namespace Steadyflow {

public class FileListController : GLib.Object {
	private static const double REDRAW_SEC = 0.2;

	private TreeView tree;
	private ListStore model;
	private string filter = "";
	private Timer redraw_timer;

	public FileListController (TreeView tree) {
		this.tree = tree;
		redraw_timer = new Timer ();
		redraw_timer.start ();
		
		TreeViewColumn column = new TreeViewColumn ();
		column.expand = true;
		column.clickable = false;
		
		CellRenderer renderer = new DownloadCellRenderer();
		column.pack_start (renderer, true);
		column.set_cell_data_func (renderer, set_cell_data);
		
		model = new ListStore.newv ({ typeof (IDownloadFile) });
		tree.set_model (model);
		tree.append_column (column);
		
		foreach (IDownloadFile file in Services.download.files) {
			connect_file_signals (file);
		}
		
		update_model ();
		Services.download.file_added.connect ((file) => {
			connect_file_signals (file);
			update_model ();
		});
		Services.download.file_removed.connect ((file) => { update_model (); });
		tree.cursor_changed.connect (on_selection_changed);
	}
	
	private IDownloadFile file_from_iter (TreeIter iter) {
		Value value;
		model.get_value (iter, 0, out value);
		return (IDownloadFile) value.get_object();
	}
	
	private void set_cell_data (CellLayout cell_layout, CellRenderer renderer, TreeModel model, TreeIter iter) {
		(renderer as DownloadCellRenderer).file = file_from_iter (iter);
	}
	
	private void connect_file_signals (IDownloadFile file) {
		file.status_changed.connect ((status) => {
			selection_changed (file);
			filelist_changed ();
			tree.queue_draw ();
		});
		file.download_progressed.connect ((size) => { on_progress_notify (); });
	}
	
	private void update_model () {
		model.clear ();
		
		foreach (IDownloadFile file in Services.download.files) {
			if (!file.local_basename.casefold ().contains (filter))
				continue;
		
			TreeIter iter;
			model.prepend (out iter);
			model.set (iter, 0, file, -1);
		}
		
		filelist_changed ();
	}
	
	private void on_progress_notify () {
		if (!tree.get_toplevel ().visible)
			return;
		
		if (redraw_timer.elapsed () >= REDRAW_SEC) {
			redraw_timer.start ();
			tree.queue_draw ();
		}
	}
	
	private void on_selection_changed () {
		selection_changed (get_selected_file ());
	}

	public IDownloadFile? get_selected_file () {
		TreePath path;
		TreeIter iter;
		tree.get_cursor (out path, null);
		
		if (path == null)
			return null;
		
		if (model.get_iter (out iter, path))
			return file_from_iter (iter);
		else
			return null;
	}
	
	public void set_filter (string filter) {
		this.filter = filter.casefold ();
		update_model ();
	}
	
	public signal void filelist_changed ();
	public signal void selection_changed (IDownloadFile? file);
}

}
